/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file transition_layer.cpp
 * \brief Implementation of the bear::engine::transition_layer class.
 * \author Julien Jorge
 */
#include "engine/layer/transition_layer.hpp"

#include "engine/level.hpp"
#include "engine/transition_effect/transition_effect.hpp"

/*----------------------------------------------------------------------------*/
const std::size_t bear::engine::transition_layer::not_an_id(0);
std::size_t bear::engine::transition_layer::s_next_id(1);

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param name The name of the of the layer in the post office.
 */
bear::engine::transition_layer::transition_layer( const std::string& name )
  : communication::messageable(name), m_effect(NULL), m_id(not_an_id)
{

} // transition_layer::transition_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::engine::transition_layer::~transition_layer()
{
  delete m_effect;
} // transition_layer::~transition_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialise the layer.
 */
void bear::engine::transition_layer::build()
{
  get_level_globals().register_item(*this);
} // transition_layer::build()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the components of the layer.
 * \param elapsed_time Elapsed time since the last call.
 */
void bear::engine::transition_layer::progress
( universe::time_type elapsed_time )
{
  if ( m_effect != NULL )
    {
      if ( m_effect->is_finished() )
        {
          delete m_effect;
          m_effect = NULL;
        }
      else
        m_effect->progress( elapsed_time );
    }
} // transition_layer::progress()

/*----------------------------------------------------------------------------*/
/**
 * \brief Render the components of the layer.
 * \param e (out) The scene elements.
 */
void bear::engine::transition_layer::render( scene_element_list& e ) const
{
  if ( m_effect != NULL )
    m_effect->render( e );
} // transition_layer::render()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a key had been pressed.
 * \param key The value of the pressed key.
 */
bool bear::engine::transition_layer::key_pressed( const input::key_info& key )
{
  if ( m_effect != NULL )
    return m_effect->key_pressed( key );
  else
    return false;
} // transition_layer::key_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a key is maintained.
 * \param key The value of the pressed key.
 */
bool bear::engine::transition_layer::key_maintained
( const input::key_info& key )
{
  if ( m_effect != NULL )
    return m_effect->key_maintained( key );
  else
    return false;
} // transition_layer::key_maintained()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a key had been released.
 * \param key The value of the pressed key.
 */
bool bear::engine::transition_layer::key_released( const input::key_info& key )
{
  if ( m_effect != NULL )
    return m_effect->key_released( key );
  else
    return false;
} // transition_layer::key_released()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a character has been entered.
 * \param key The value of the pressed key.
 */
bool bear::engine::transition_layer::char_pressed( const input::key_info& key )
{
  if ( m_effect != NULL )
    return m_effect->char_pressed(key);
  else
    return false;
} // transition_layer::char_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a joystick button had been pressed.
 * \param button The value of the pressed button.
 * \param joy_index The index of the joystick.
 */
bool bear::engine::transition_layer::button_pressed
( input::joystick::joy_code button, unsigned int joy_index )
{
  if ( m_effect != NULL )
    return m_effect->button_pressed( button, joy_index );
  else
    return false;
} // transition_layer::button_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a joystick button is maintained.
 * \param button The value of the pressed button.
 * \param joy_index The index of the joystick.
 */
bool bear::engine::transition_layer::button_maintained
( input::joystick::joy_code button, unsigned int joy_index )
{
  if ( m_effect != NULL )
    return m_effect->button_maintained( button, joy_index );
  else
    return false;
} // transition_layer::button_maintained()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a joystick button had been released.
 * \param button The value of the pressed button.
 * \param joy_index The index of the joystick.
 */
bool bear::engine::transition_layer::button_released
( input::joystick::joy_code button, unsigned int joy_index )
{
  if ( m_effect != NULL )
    return m_effect->button_released( button, joy_index );
  else
    return false;
} // transition_layer::button_released()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a mouse button has been pressed.
 * \param pos The current position of the cursor.
 * \param key The value of the pressed button.
 */
bool bear::engine::transition_layer::mouse_pressed
( input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  if ( m_effect != NULL )
    return m_effect->mouse_pressed( key, pos );
  else
    return false;
} // transition_layer::mouse_pressed()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a mouse button is maintained.
 * \param pos The current position of the cursor.
 * \param key The value of the maintained button.
 */
bool bear::engine::transition_layer::mouse_maintained
( input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  if ( m_effect != NULL )
    return m_effect->mouse_maintained( key, pos );
  else
    return false;
} // transition_layer::mouse_maintained()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that a mouse button has been released.
 * \param pos The current position of the cursor.
 * \param key The value of the released button.
 */
bool bear::engine::transition_layer::mouse_released
( input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  if ( m_effect != NULL )
    return m_effect->mouse_released( key, pos );
  else
    return false;
} // transition_layer::mouse_released()

/*----------------------------------------------------------------------------*/
/**
 * \brief Inform the layer that the mouse has been moved.
 * \param pos The new position of the mouse.
 */
bool bear::engine::transition_layer::mouse_move
( const claw::math::coordinate_2d<unsigned int>& pos )
{
  if ( m_effect != NULL )
    return m_effect->mouse_move( pos );
  else
    return false;
} // transition_layer::mouse_move()

/*----------------------------------------------------------------------------*/
/**
 * \brief Erase an effect.
 * \param id The identifier of the effect to erase.
 */
void bear::engine::transition_layer::erase_effect( std::size_t id )
{
  if ( id == m_id )
    {
      delete m_effect;
      m_effect = NULL;
      m_id = not_an_id;
    }
} // transition_layer::erase_effect()

/*----------------------------------------------------------------------------*/
/**
 * \brief Push an effect in the layer.
 * \param e The effect. It will be deleted by the layer.
 * \remark This method call transition_layer::build().
 * \remark The new effect replaces the current one.
 */
std::size_t bear::engine::transition_layer::push_effect( transition_effect* e )
{
  m_id = not_an_id;

  if ( m_effect != NULL )
    {
      delete m_effect;
      m_effect = NULL;
    }

  if ( e != NULL )
    {
      m_id = s_next_id;
      ++s_next_id;
      e->set_layer(*this);
      e->build();
    }

  m_effect = e;
  return m_id;
} // transition_layer::push_effect()
