/*
	 pingpong - Really free server for Amateur Radio convers
	 Copyright (C) 2005 Joop Stakenborg <pg4i@amsat.org>

	 This program is free software; you can redistribute it and/or modify
	 it under the terms of the GNU General Public License as published by
	 the Free Software Foundation; either version 2, or (at your option)
	 any later version.

	 This program is distributed in the hope that it will be useful,
	 but WITHOUT ANY WARRANTY; without even the implied warranty of
	 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the
	 GNU General Public License for more details.

	 You should have received a copy of the GNU General Public License
	 along with this program; if not, write to the Free Software Foundation,
	 Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.	
*/

#include <ctype.h>
#include <syslog.h>
#include <glib.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "pingpong.h"
#include "types.h"
#include "messages.h"
#include "init.h"
#include "client.h"
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

GList *client_list = NULL, *channel_list = NULL;
extern gchar *confdir;
extern gchar *buffer;

gboolean create_client(gint fd)
{
	Client *pos;
	FILE *fp;
	gint c;
	gboolean issue_exist =	TRUE;
	GString *sendmsg = g_string_new("* ");
	gchar *issue;

	if ((g_list_length(client_list) + 1) == MAX_CLIENTS) return FALSE;

	pos = g_new0(Client, 1);
	pos->fd = fd;
	client_list = g_list_append(client_list, pos);

	/* send pingpong.issue */
	if (confdir) issue = g_strdup_printf("%s/pingpong.issue", confdir);
		else issue = g_strdup_printf("%s/pingpong.issue", PINGPONG_SYSCONFDIR);
	if ((fp = fopen(issue, "r")) == NULL) issue_exist = FALSE;
	if (issue_exist)
	{
		while((c = fgetc(fp)) != EOF)
		{
			sendmsg = g_string_append_c(sendmsg, c);
			if (c == '\n')
			{
				send_msg(pos->fd, PINGPONG_MSG, sendmsg->str);
				sendmsg = g_string_new("* ");
			}
		}
		fclose(fp);
	}
	g_string_free(sendmsg, TRUE);

	syslog (LOG_INFO, "client %p created\n", pos);
	return TRUE;
}

void remove_client(Client *client, gchar *msg)
{
	Client *pos;
	guint i, j, k;

	syslog (LOG_INFO, "removing client %p ('%s')\n",
		client, client->name ? client->name : "noname");

	close(client->fd);

	/* send a message to all users that use the same channel that the user is leaving */
	if (client->name)
	{
		for (i = 0; i < g_list_length(client_list); i++)
		{
			pos = g_list_nth_data(client_list, i);
			for (j = 0; j < g_list_length(pos->channel_list); j++)
				for (k = 0; k < g_list_length(client->channel_list); k++)
				{
				if (GPOINTER_TO_INT(g_list_nth_data(client->channel_list, k)) == 
						GPOINTER_TO_INT(g_list_nth_data(pos->channel_list, j)))
					send_left_msg(pos->fd, client->name, pos->channel, msg);
				}
		}
	}

	/* remove client from list */
	client_list = g_list_remove(client_list, client);
}

gboolean set_name(Client *client)
{
	Client *pos;
	guint i, j, listlen;
	gint c, channel = -1, channelusers = 1;
	GString *sendmsg = g_string_new("");
	gchar *motd, **namesplit = NULL, nowtime[16], *name = NULL;
	gboolean motd_exist = TRUE;
	FILE *fp;
	size_t timeres;
	time_t now;

	if (!buffer || client->name) return FALSE;

	/* split up name to see if a channel was entered */
	namesplit = g_strsplit(buffer, " ", 0);
	if (namesplit[0])
	{ /* "/n <name>" */
		name = g_strdup(namesplit[0]);
		if (namesplit[1]) { /* "/n <name> <channel>" */
			if (g_strcasecmp(namesplit[1], "0") == 0)
				channel = 0;
			else
			{
				if ((channel = atoi(namesplit[1])) == 0)
					channel = getdefaultchannel();
			}
		}
	}
	g_strfreev(namesplit);
	if (strlen(name) > 10) return FALSE;
	if (channel == -1) channel = getdefaultchannel();
	g_strdown(name);

	syslog (LOG_INFO, "client %p is '%s' on channel %d\n", client, name, channel);

	/* send joined announcement to all users on this channel and count users */
	listlen = g_list_length(client_list);
	for (i = 0; i < listlen; i++)
	{
		pos = g_list_nth_data(client_list, i);
		if (pos->name)
		{
			for (j = 0; j < g_list_length(pos->channel_list); j++)
			{
				if (channel == GPOINTER_TO_INT(g_list_nth_data(pos->channel_list, j)))
				{
					send_joined_msg(pos->fd, name, channel);
					channelusers++;
				}
			}
		}
	}

	/* store the new name and channel */
	client->name = g_strdup(name);
	client->channel = channel;
	client->channel_list = g_list_append(client->channel_list,	GINT_TO_POINTER(channel));

	
	/* greeting */
	g_string_sprintf(sendmsg, "%s @ %s version %s\n", 
		PACKAGE, getservername(), VERSION);
	send_msg(client->fd, PINGPONG_MSG, sendmsg->str);

	/* send pingpong.motd */
	if (confdir) motd = g_strdup_printf("%s/pingpong.motd", confdir);
		else motd = g_strdup_printf("%s/pingpong.motd", PINGPONG_SYSCONFDIR);
	if ((fp = fopen(motd, "r")) == NULL) motd_exist = FALSE;
	sendmsg = g_string_new("* ");
	if (motd_exist)
	{
		while((c = fgetc(fp)) != EOF)
		{
			sendmsg = g_string_append_c(sendmsg, c);
			if (c == '\n')
			{
				send_msg(client->fd, PINGPONG_MSG, sendmsg->str);
				sendmsg = g_string_new("* ");
			}
		}
		fclose(fp);
	}

	/* send number of online users */
	if (listlen == 1) 
		g_string_sprintf(sendmsg, "*** There is 1 user online\n");
	else
	g_string_sprintf(sendmsg, "*** There are %d users online\n", listlen);
	send_msg(client->fd, PINGPONG_MSG, sendmsg->str);

	/* send channel and number of users on channel */
	now = time(NULL);
	timeres = strftime(nowtime, 16, "%H:%M", localtime(&now));
	if (channelusers == 1)
		g_string_sprintf(sendmsg, "*** You created a new channel %d\n", channel);
	else
		g_string_sprintf(sendmsg, "*** (%s) You are now talking to channel %d. There are %d users.\n", nowtime, channel, channelusers);
	send_msg(client->fd, PINGPONG_MSG, sendmsg->str);

	g_free(motd);
	g_string_free(sendmsg, TRUE);
	return TRUE;
}

void goto_channel(Client *client)
{
	gchar nowtime[16], *sendmsg;
	size_t timeres;
	time_t now;
	gint gotochannel = -1, bufferchannel, channelusers = 1;
	guint i, listlen;
	Client *pos;

	now = time(NULL);
	timeres = strftime(nowtime, 16, "%H:%M", localtime(&now));

	listlen = g_list_length(client_list);

	/* empty "/c" was typed, send information about this channel */
	if (!buffer)
	{
		if (listlen == 1)
			sendmsg = g_strdup_printf("*** (%s) You are talking to channel %d. There is 1 user.\n",
				nowtime, client->channel);
		else
			sendmsg = g_strdup_printf("*** (%s) You are talking to channel %d. There are %d user.\n",
				nowtime, client->channel, listlen);
		send_msg(client->fd, PINGPONG_MSG, sendmsg);
	}
	else
	{ /* buffer holds the channel number */
		if (g_strcasecmp(buffer, "0") == 0) gotochannel = 0;
		bufferchannel = atoi(buffer);
		if (bufferchannel == 0 || bufferchannel > 32767)
		{
			sendmsg = g_strdup_printf("*** (%s) Channel numbers must be in the range 0..32767.\n", nowtime);
			send_msg(client->fd, PINGPONG_MSG, sendmsg);
			return;
		}
		if (gotochannel == -1) gotochannel = bufferchannel;

		/* in case we try to join the channel we are already on */
		if (gotochannel == client->channel)
		{
			sendmsg = g_strdup_printf("*** (%s) Channel %d is already default.\n", nowtime, gotochannel);
			send_msg(client->fd, PINGPONG_MSG, sendmsg);
			return;
		}

		/* send joined announcement to all users and count users on this channel */
		for (i = 0; i < listlen; i++)
		{
			pos = g_list_nth_data(client_list, i);
			if (pos->name && pos->channel == gotochannel)
			{
				send_joined_msg(pos->fd, client->name, gotochannel);
				channelusers++;
			}
		}

		client->channel = gotochannel;
		client->channel_list = g_list_append(client->channel_list,	GINT_TO_POINTER(gotochannel));

		if (channelusers == 1)
			sendmsg = g_strdup_printf("*** (%s) You are now talking to channel %d. You're alone.\n", 
				nowtime, gotochannel);
		else
			sendmsg = g_strdup_printf("*** (%s) You are now talking to channel %d. There are %d users.\n", 
				nowtime, gotochannel, channelusers);
		send_msg(client->fd, PINGPONG_MSG, sendmsg);
	}
}

void leave_channel(Client *client)
{
	gchar nowtime[16], *sendmsg;
	size_t timeres;
	time_t now;
	guint i, j, listlen;
	gpointer channeltoleave;
	gint channelentered = -1, bufferchannel = -1;
	Client *pos;

	now = time(NULL);
	timeres = strftime(nowtime, 16, "%H:%M", localtime(&now));
	listlen = g_list_length(client->channel_list);

	/* if we were only active on one channel we quit */
	if (listlen == 1)
	{
		remove_client(client, PINGPONG_CMD_LEAVE);
		return;
	}

	/* just "/l" will leave the last channel */
	if (!buffer)
	{
		channeltoleave = g_list_nth_data(client->channel_list, listlen - 1);
		client->channel_list = g_list_remove(client->channel_list, channeltoleave);
		client->channel = GPOINTER_TO_INT(g_list_nth_data(client->channel_list, listlen - 2));
		sendmsg = g_strdup_printf("*** (%s) Default channel is now %d.\n", nowtime, client->channel);
		send_msg(client->fd, PINGPONG_MSG, sendmsg);
	}
	else
	{ /* you can leave the channel you are on, or leave a channel in the channel list*/
		if (g_strcasecmp(buffer, "0") == 0) channelentered = 0;
		else bufferchannel = atoi(buffer);
		if (bufferchannel == 0 || bufferchannel > 32767)
		{
			sendmsg = g_strdup_printf("*** (%s) Channel numbers must be in the range 0..32767.\n", nowtime);
			send_msg(client->fd, PINGPONG_MSG, sendmsg);
			return;
		}
		if (channelentered == -1) channelentered = bufferchannel;

		/* find the pointer in the channel list associated with the channel */
		for (i = 0; i < listlen; i++)
			if (channelentered == GPOINTER_TO_INT(g_list_nth_data(client->channel_list, i)))
				break;
		channeltoleave = g_list_nth_data(client->channel_list, i);

		if (client->channel == channelentered)
		{ /*the same as just typing "/l" */
			client->channel_list = g_list_remove(client->channel_list, channeltoleave);
			client->channel = GPOINTER_TO_INT(g_list_nth_data(client->channel_list, listlen - 2));
			sendmsg = g_strdup_printf("*** (%s) Default channel is now %d.\n", nowtime, client->channel);
			send_msg(client->fd, PINGPONG_MSG, sendmsg);
		} 
		else
		{
			client->channel_list = g_list_remove(client->channel_list, channeltoleave);
			sendmsg = g_strdup_printf("*** (%s) Left channel %d.\n", nowtime, channelentered);
			send_msg(client->fd, PINGPONG_MSG, sendmsg);

			/* send a message to all users that use the same channel that the user is leaving */
			for (i = 0; i < g_list_length(client_list); i++)
			{
				pos = g_list_nth_data(client_list, i);
				if (pos == client) continue;
				for (j = 0; j < g_list_length(pos->channel_list); j++)
				{
					if (channelentered == GPOINTER_TO_INT(g_list_nth_data(pos->channel_list, j)))
						send_left_msg(pos->fd, client->name, channelentered, PINGPONG_CMD_LEAVE);
				}
			}
		}
	}
}
