/*
 * Permafrost - Physical modelling framework
 *
 * Copyright (C) 2009, 2010 Stefano D'Angelo <zanga.mail@gmail.com>
 *
 * See the COPYING file for license conditions.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "src/types.h"
#include "src/util.h"
#include "src/list.h"
#include "src/macro.h"

void
port_dump(void *data, void *context)
{
	struct port *p;

	p = (struct port *)data;

	printf("    %s %s `%s'\n",
	       (p->sync == port_sync_sync) ? "sync"
	       : ((p->sync == port_sync_async) ? "async" : "unknown"),
	       (p->type == port_type_input) ? "input"
	        : ((p->type == port_type_output) ? "output"
	           : ((p->type == port_type_w) ? "w_port" : "k_port")), p->id);
}

void
component_dump(void *data, void *context)
{
	struct component *c;

	c = (struct component *)data;

	printf("    %s `%s' `%s'\n",
	       c->type.t.is_macro ? "macro" : "block",
	       c->type.t.is_macro ? ((struct system *)c->type.t.p)->id
	       : ((struct block *)c->type.t.p)->id, c->id);
}

void
connection_dump(void *data, void *context)
{
	struct connection *c;

	c = (struct connection *)data;

	printf("    %s%s%s%s = ",
	       (c->output.c.p.component != NULL) ? c->output.c.p.component->id
	       : "", (c->output.c.p.component != NULL) ? "." : "",
	       c->output.c.p.port->id,
	       (c->output.type == conn_elem_type_port) ? ""
	       : ((c->output.type == conn_elem_type_in) ? "in" : "out"));

	if (c->input.type == conn_elem_type_value)
	  {
		printf("%f\n", c->input.c.value);
		return;
	  }

	printf("%s%s%s%s\n",
	       (c->input.c.p.component != NULL)
	       ? c->input.c.p.component->id : "",
	       (c->input.c.p.component != NULL) ? "." : "",
	       c->input.c.p.port->id,
	       (c->input.type == conn_elem_type_port) ? ""
	       : ((c->input.type == conn_elem_type_in) ? "in" : "out"));
}

void
system_dump(struct system *system)
{
	printf("system `%s':\n", system->id);

	printf("  ports:\n");
	list_for_each(system->ports, port_dump, NULL);

	printf("  components:\n");
	list_for_each(system->components, component_dump, NULL);

	printf("  connections:\n");
	list_for_each(system->connections, connection_dump, NULL);
}

static int
port_cmp_id(void *p, void *id)
{
	return strcmp(((struct port *)p)->id, (char *)id);
}

static int
component_cmp_id(void *c, void *id)
{
	return strcmp(((struct component *)c)->id, (char *)id);
}

static void
port_copy(void *data, void *context)
{
	struct port *port, *p;
	list_t list;

	port = (struct port *)data;
	list = (list_t)context;

	p = xmalloc(sizeof(struct port));

	p->sync = port->sync;
	p->type = port->type;
	p->id = xmalloc(strlen(port->id) + 1);
	strcpy(p->id, port->id);

	list_append(list, p);
}

struct component_add_context
  {
	list_t	 list;
	char	*prefix;
  };

static void
component_add(void *data, void *context)
{
	struct component *comp, *c;
	struct component_add_context *ctx;
	struct component_add_context c_ctx;

	comp = (struct component *)data;
	ctx = (struct component_add_context *)context;

	if (!comp->type.t.is_macro)
	  {
		c = xmalloc(sizeof(struct component));

		c->id = xmalloc(
			((ctx->prefix != NULL) ? strlen(ctx->prefix) : 0)
			+ strlen(comp->id) + 1);
		if (ctx->prefix != NULL)
			strcpy(c->id, ctx->prefix);
		else
			*c->id = '\0';
		strcat(c->id, comp->id);

		c->type.t.is_macro = 0;
		c->type.t.p = comp->type.t.p;

		list_append(ctx->list, c);

		return;
	  }

	c_ctx.prefix = xmalloc(((ctx->prefix != NULL) ? strlen(ctx->prefix) : 0)
			       + strlen(comp->id) + 2);
	if (ctx->prefix != NULL)
		strcpy(c_ctx.prefix, ctx->prefix);
	else
		*c_ctx.prefix = '\0';
	strcat(c_ctx.prefix, comp->id);
	strcat(c_ctx.prefix, ".");
	c_ctx.list = ctx->list;

	list_for_each(((struct system *)comp->type.t.p)->components,
		      component_add, &c_ctx);

	free(c_ctx.prefix);
}

struct conn_add_to_analogous_list_context
  {
	struct conn_elem	*elem;
	list_t			 list;
	list_t			 ports;
	list_t			 components;
	struct component	*base;
	char			*prefix;
  };

static list_t
conn_elem_analogous(struct conn_elem *elem, list_t ports, list_t components,
		    struct component *base, char *prefix);

static void
conn_add_to_analogous_list(void *data, void *context)
{
	struct connection *conn;
	struct conn_add_to_analogous_list_context *ctx;
	list_t l;

	conn = (struct connection *)data;
	ctx = (struct conn_add_to_analogous_list_context *)context;

	if (conn->output.c.p.port == ctx->elem->c.p.port)
	  {
		l = conn_elem_analogous(&conn->input, ctx->ports,
					ctx->components, ctx->base,
					ctx->prefix);
		list_merge(ctx->list, l);
	  }

	if (conn->input.type == conn_elem_type_value)
		return;

	if (conn->input.c.p.port == ctx->elem->c.p.port)
	  {
		l = conn_elem_analogous(&conn->output, ctx->ports,
					ctx->components, ctx->base,
					ctx->prefix);
		list_merge(ctx->list, l);
	  }
}

static list_t
conn_elem_analogous(struct conn_elem *elem, list_t ports, list_t components,
		    struct component *base, char *prefix)
{
	list_t ret;
	struct conn_elem *e;
	struct conn_add_to_analogous_list_context ctx;
	char *id;

	ret = list_new();
	if (elem->type == conn_elem_type_value)
	  {
		e = xmalloc(sizeof(struct conn_elem));

		e->type = conn_elem_type_value;
		e->c.value = elem->c.value;

		list_append(ret, e);
	  }
	else if (elem->c.p.component == NULL)
	  {
		if (base != NULL)
			return ret;

		e = xmalloc(sizeof(struct conn_elem));

		e->type = elem->type;
		e->c.p.component = NULL;
		e->c.p.port = list_find(ports, port_cmp_id, elem->c.p.port->id);

		list_append(ret, e);
	  }
	else if (!elem->c.p.component->type.t.is_macro)
	  {
		e = xmalloc(sizeof(struct conn_elem));

		e->type = elem->type;

		id = xmalloc(strlen(prefix) + strlen(elem->c.p.component->id)
			     + 1);
		strcpy(id, prefix);
		strcat(id, elem->c.p.component->id);
		e->c.p.component = list_find(components, component_cmp_id, id);
		free(id);

		e->c.p.port = list_find(
			((struct block *)elem->c.p.component->type.t.p)->ports,
			port_cmp_id, elem->c.p.port->id);

		list_append(ret, e);
	  }
	else
	  {
		ctx.elem = elem;
		ctx.list = ret;
		ctx.ports = ports;
		ctx.components = components;
		ctx.base = base;
		ctx.prefix = xmalloc(strlen(prefix)
				     + strlen(elem->c.p.component->id) + 2);
		sprintf(ctx.prefix, "%s%s.", prefix, elem->c.p.component->id);
		list_for_each(
			((struct system *)elem->c.p.component->type.t.p)->connections,
			conn_add_to_analogous_list, &ctx);
		free(ctx.prefix);
	  }

	return ret;
}

struct conn_add_context
  {
	list_t			 list;
	struct conn_elem	*output;
  };

static void
conn_add(void *data, void *context)
{
	struct conn_elem *input;
	struct conn_add_context *ctx;
	struct connection *conn;

	input = (struct conn_elem *)data;
	ctx = (struct conn_add_context *)context;

	conn = xmalloc(sizeof(struct connection));

	if (ctx->output->type == conn_elem_type_value)
	  {
		conn->input = *ctx->output;
		conn->output = *input;
	  }
	else if (input->type == conn_elem_type_value)
	  {
		conn->input = *input;
		conn->output = *ctx->output;
	  }
	else if (((ctx->output->c.p.component == NULL)
	          && (ctx->output->c.p.port->type == port_type_input))
		 || ((ctx->output->c.p.component != NULL)
		     && (ctx->output->c.p.port->type == port_type_output))
		 || ((input->c.p.component == NULL)
		     && (input->c.p.port->type == port_type_output))
		 || ((input->c.p.component != NULL)
		     && (input->c.p.port->type == port_type_input)))
	  {
		conn->input = *ctx->output;
		conn->output = *input;
	  }
	else
	  {
		conn->input = *input;
		conn->output = *ctx->output;
	  }
	free(ctx->output);

	list_append(ctx->list, conn);
}

struct conns_add_context
  {
	list_t	list;
	list_t	inputs;
  };

static void
conns_add(void *data, void *context)
{
	struct conn_elem *output;
	struct conns_add_context *ctx;
	struct conn_add_context c_ctx;

	output = (struct conn_elem *)data;
	ctx = (struct conns_add_context *)context;

	c_ctx.list = ctx->list;
	c_ctx.output = output;
	list_for_each(ctx->inputs, conn_add, &c_ctx);
}

static void
free_data(void *data, void *context)
{
	free(data);
}

struct
connection_add_context
  {
	list_t			 list;
	list_t			 ports;
	list_t			 components;
	struct component	*base;
	char			*prefix;
  };

static void
connection_add(void *data, void *context)
{
	struct connection *conn;
	struct connection_add_context *ctx;
	struct conns_add_context c_ctx;
	list_t outputs, inputs;

	conn = (struct connection *)data;
	ctx = (struct connection_add_context *)context;

	outputs = conn_elem_analogous(&conn->output, ctx->ports,
				      ctx->components, ctx->base, ctx->prefix);
	inputs = conn_elem_analogous(&conn->input, ctx->ports, ctx->components,
				     ctx->base, ctx->prefix);

	c_ctx.list = ctx->list;
	c_ctx.inputs = inputs;
	list_for_each(outputs, conns_add, &c_ctx);
	list_for_each(inputs, free_data, NULL);
}

static void
connection_add_macro(void *data, void *context)
{
	struct component *comp;
	struct connection_add_context *ctx;
	struct connection_add_context m_ctx;

	comp = (struct component *)data;
	ctx = (struct connection_add_context *)context;

	if (!comp->type.t.is_macro)
		return;

	m_ctx.list = ctx->list;
	m_ctx.ports = ((struct system *)comp->type.t.p)->ports;
	m_ctx.components = ctx->components;
	m_ctx.base = comp;
	m_ctx.prefix = xmalloc(strlen(ctx->prefix) + strlen(comp->id) + 2);
	sprintf(m_ctx.prefix, "%s%s.", ctx->prefix, comp->id);
	list_for_each(((struct system *)comp->type.t.p)->connections,
		      connection_add, &m_ctx);
	list_for_each(((struct system *)comp->type.t.p)->components,
		      connection_add_macro, &m_ctx);
	free(m_ctx.prefix);
}

struct system *
macro_to_blocks(struct system *system)
{
	struct system *s;
	struct component_add_context ctx;
	struct connection_add_context c_ctx;

	s = xmalloc(sizeof(struct system));

	s->id = xmalloc(strlen(system->id) + 1);
	strcpy(s->id, system->id);

	s->ports = list_new();
	list_for_each(system->ports, port_copy, s->ports);

	s->components = list_new();
	ctx.prefix = NULL;
	ctx.list = s->components;
	list_for_each(system->components, component_add, &ctx);

	s->connections = list_new();
	c_ctx.list = s->connections;
	c_ctx.ports = s->ports;
	c_ctx.components = s->components;
	c_ctx.base = NULL;
	c_ctx.prefix = "";
	list_for_each(system->connections, connection_add, &c_ctx);

	list_for_each(system->components, connection_add_macro, &c_ctx);

	return s;
}
