/*
 * Permafrost - Physical modelling framework
 *
 * Copyright (C) 2009, 2010 Stefano D'Angelo <zanga.mail@gmail.com>
 *
 * See the COPYING file for license conditions.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <float.h>

#include "src/types.h"
#include "src/list.h"
#include "src/util.h"
#include "src/expr.h"

enum elem_type
  {
	elem_type_value,
	elem_type_sample_rate,
	elem_type_signal_id,
	elem_type_signal_port,
	elem_type_signal_ref,
	elem_type_add,
	elem_type_sub,
	elem_type_mul,
	elem_type_div,
	elem_type_plus,
	elem_type_minus,
	elem_type_call_id,
	elem_type_call_ext_func
  };

struct elem
  {
	enum elem_type				 type;
	union
	  {
		double				 value;
		struct
		  {
			struct expr_signal_id	 id;
			struct port		*port;
			struct scheduled_expr	*ref;
			expr_t			 delay;
			expr_t			 delay_max;
		  } signal;
		struct
		  {
			char			*id;
			struct ext_func		*ext_func;
			list_t			 args;
		  } func;
	  } v;
  };

struct _expr
  {
	list_t	stack;
  };

expr_t
expr_new_value(double value)
{
	struct _expr *ret;
	struct elem *elem;

	ret = xmalloc(sizeof(struct _expr));
	elem = xmalloc(sizeof(struct elem));
	ret->stack = list_new();

	elem->type = elem_type_value;
	elem->v.value = value;

	list_push(ret->stack, elem);

	return ret;
}

expr_t
expr_new_sample_rate()
{
	struct _expr *ret;
	struct elem *elem;

	ret = xmalloc(sizeof(struct _expr));
	elem = xmalloc(sizeof(struct elem));
	ret->stack = list_new();

	elem->type = elem_type_sample_rate;

	list_push(ret->stack, elem);

	return ret;
}

expr_t
expr_new_signal(char *id, char in, expr_t delay, expr_t delay_max)
{
	struct _expr *ret;
	struct elem *elem;

	ret = xmalloc(sizeof(struct _expr));
	elem = xmalloc(sizeof(struct elem));
	ret->stack = list_new();

	elem->type = elem_type_signal_id;
	elem->v.signal.id.id = id;
	elem->v.signal.id.in = in;
	elem->v.signal.delay = delay;
	elem->v.signal.delay_max = delay_max;

	list_push(ret->stack, elem);

	return ret;
}

expr_t
expr_new_signal_ref(struct scheduled_expr *ref, expr_t delay, expr_t delay_max)
{
	struct _expr *ret;
	struct elem *elem;

	ret = xmalloc(sizeof(struct _expr));
	elem = xmalloc(sizeof(struct elem));
	ret->stack = list_new();

	elem->type = elem_type_signal_ref;
	elem->v.signal.ref = ref;
	elem->v.signal.delay = delay;
	elem->v.signal.delay_max = delay_max;

	list_push(ret->stack, elem);

	return ret;
}

expr_t
expr_new_call(char *id, list_t args)
{
	struct _expr *ret;
	struct elem *elem;

	ret = xmalloc(sizeof(struct _expr));
	elem = xmalloc(sizeof(struct elem));
	ret->stack = list_new();

	elem->type = elem_type_call_id;
	elem->v.func.id = id;
	elem->v.func.args = args;

	list_push(ret->stack, elem);

	return ret;
}

void
expr_push_add(expr_t expr, expr_t expr2)
{
	struct elem *elem;

	elem = xmalloc(sizeof(struct elem));

	elem->type = elem_type_add;

	list_merge(expr2->stack, expr->stack);
	expr->stack = expr2->stack;
	list_push(expr->stack, elem);

	free(expr2);
}

void
expr_push_sub(expr_t expr, expr_t expr2)
{
	struct elem *elem;

	elem = xmalloc(sizeof(struct elem));

	elem->type = elem_type_sub;

	list_merge(expr2->stack, expr->stack);
	expr->stack = expr2->stack;
	list_push(expr->stack, elem);

	free(expr2);
}

void
expr_push_mul(expr_t expr, expr_t expr2)
{
	struct elem *elem;

	elem = xmalloc(sizeof(struct elem));

	elem->type = elem_type_mul;

	list_merge(expr2->stack, expr->stack);
	expr->stack = expr2->stack;
	list_push(expr->stack, elem);

	free(expr2);
}

void
expr_push_div(expr_t expr, expr_t expr2)
{
	struct elem *elem;

	elem = xmalloc(sizeof(struct elem));

	elem->type = elem_type_div;

	list_merge(expr2->stack, expr->stack);
	expr->stack = expr2->stack;
	list_push(expr->stack, elem);

	free(expr2);
}

void
expr_push_plus(expr_t expr)
{
	struct elem *elem;

	elem = xmalloc(sizeof(struct elem));

	elem->type = elem_type_plus;

	list_push(expr->stack, elem);
}

void
expr_push_minus(expr_t expr)
{
	struct elem *elem;

	elem = xmalloc(sizeof(struct elem));

	elem->type = elem_type_minus;

	list_push(expr->stack, elem);
}

char
expr_top_is_sign(expr_t expr)
{
	struct elem *elem;

	elem = list_get_last_data(expr->stack);

	return (elem->type == elem_type_plus)
	       || (elem->type == elem_type_minus);
}

static int
port_cmp_id(void *p, void *id)
{
	return strcmp(((struct port *)p)->id, (char *)id);
}

struct bind_to_ports_args_context
  {
	list_t			 ports;
	struct expr_signal_id	*id;
  };

static void
bind_to_ports_args(void *data, void *context)
{
	expr_t expr;
	struct bind_to_ports_args_context *ctx;

	expr = (expr_t)data;
	ctx = (struct bind_to_ports_args_context *)context;

	ctx->id = expr_bind_to_ports(expr, ctx->ports);
}

struct bind_to_ports_context
  {
	struct expr_signal_id	*id;
	list_t			 ports;
  };

static void
bind_to_ports(void *data, void *context)
{
	struct elem *elem;
	struct bind_to_ports_context *ctx;
	struct port *port;
	struct bind_to_ports_args_context a_ctx;

	elem = (struct elem *)data;
	ctx = (struct bind_to_ports_context *)context;

	if (ctx->id != NULL)
		return;

	if ((elem->type == elem_type_signal_id)
	    || (elem->type == elem_type_signal_port)
	    || (elem->type == elem_type_signal_ref))
	  {
		if (elem->v.signal.delay != NULL)
		  {
			ctx->id = expr_bind_to_ports(elem->v.signal.delay,
						     ctx->ports);
			if (ctx->id != NULL)
				return;
		  }
		if (elem->v.signal.delay_max != NULL)
		  {
			ctx->id = expr_bind_to_ports(elem->v.signal.delay_max,
						     ctx->ports);
			if (ctx->id != NULL)
				return;
		  }
	  }
	else if ((elem->type == elem_type_call_id)
		 || (elem->type == elem_type_call_ext_func))
	  {
		a_ctx.ports = ctx->ports;
		a_ctx.id = NULL;
		list_for_each(elem->v.func.args, bind_to_ports_args, &a_ctx);

		if (a_ctx.id != NULL)
		  {
			ctx->id = a_ctx.id;
			return;
		  }
	  }

	if (elem->type != elem_type_signal_id)
		return;

	port = list_find(ctx->ports, port_cmp_id, elem->v.signal.id.id);
	if (port == NULL)
		return;

	if ((elem->v.signal.id.in && (port->type != port_type_w)
	     && (port->type != port_type_k))
	    || (!elem->v.signal.id.in && (port->type != port_type_input)
		&& (port->type != port_type_output)))
	  {
		ctx->id = &elem->v.signal.id;
		return;
	  }

	free(elem->v.signal.id.id);
	elem->type = elem_type_signal_port;
	elem->v.signal.port = port;
}

struct expr_signal_id *
expr_bind_to_ports(expr_t expr, list_t ports)
{
	struct bind_to_ports_context ctx;

	ctx.id = NULL;
	ctx.ports = ports;
	list_for_each(expr->stack, bind_to_ports, &ctx);

	return ctx.id;
}

static int
const_v_cmp_id(void *v, void *id)
{
	return strcmp(((struct const_v *)v)->id, (char *)id);
}

static void
bind_consts_args(void *data, void *context)
{
	expr_t expr;
	list_t consts;

	expr = (expr_t)data;
	consts = (list_t)context;

	expr_bind_consts(expr, consts);
}

static void
bind_consts(void *data, void *context)
{
	struct elem *elem;
	list_t consts;
	struct const_v *v;

	elem = (struct elem *)data;
	consts = (list_t)context;

	if ((elem->type == elem_type_signal_id)
	    || (elem->type == elem_type_signal_port)
	    || (elem->type == elem_type_signal_ref))
	  {
		if (elem->v.signal.delay != NULL)
		  {
			expr_bind_consts(elem->v.signal.delay, consts);
			if (elem->v.signal.delay_max != NULL)
				expr_bind_consts(elem->v.signal.delay_max,
						 consts);
			return;
		  }
	  }
	else if ((elem->type == elem_type_call_id)
		 || (elem->type == elem_type_call_ext_func))
	  {
		list_for_each(elem->v.func.args, bind_consts_args, consts);
		return;
	  }

	if (elem->type != elem_type_signal_id)
		return;
	if (elem->v.signal.id.in)
		return;

	v = list_find(consts, const_v_cmp_id, elem->v.signal.id.id);
	if (v == NULL)
		return;

	free(elem->v.signal.id.id);
	elem->type = elem_type_value;
	elem->v.value = v->value;
}

void
expr_bind_consts(expr_t expr, list_t consts)
{
	list_for_each(expr->stack, bind_consts, consts);
}

static int
ext_func_cmp_id(void *f, void *id)
{
	return strcmp(((struct ext_func *)f)->id, (char *)id);
}

struct bind_ext_funcs_context
  {
	char	*ret;
	list_t	 funcs;
  };

static void
bind_ext_funcs_args(void *data, void *context)
{
	expr_t expr;
	struct bind_ext_funcs_context *ctx;

	expr = (expr_t)data;
	ctx = (struct bind_ext_funcs_context *)context;

	ctx->ret = expr_bind_ext_funcs(expr, ctx->funcs);
}

static void
bind_ext_funcs(void *data, void *context)
{
	struct elem *elem;
	struct bind_ext_funcs_context *ctx;
	struct ext_func *func;

	elem = (struct elem *)data;
	ctx = (struct bind_ext_funcs_context *)context;

	if (ctx->ret != NULL)
		return;

	if ((elem->type == elem_type_signal_id)
	    || (elem->type == elem_type_signal_port)
	    || (elem->type == elem_type_signal_ref))
	  {
		if (elem->v.signal.delay != NULL)
		  {
			ctx->ret = expr_bind_ext_funcs(elem->v.signal.delay,
						       ctx->funcs);
			if (ctx->ret != NULL)
				return;

			if (elem->v.signal.delay_max != NULL)
				ctx->ret = expr_bind_ext_funcs(
					elem->v.signal.delay_max, ctx->funcs);
			return;
		  }
	  }
	else if ((elem->type == elem_type_call_id)
		 || (elem->type == elem_type_call_ext_func))
		list_for_each(elem->v.func.args, bind_ext_funcs_args, ctx);

	if (elem->type != elem_type_call_id)
		return;

	func = list_find(ctx->funcs, ext_func_cmp_id, elem->v.func.id);
	if (func == NULL)
	  {
		ctx->ret = elem->v.func.id;
		return;
	  }

	free(elem->v.func.id);
	elem->type = elem_type_call_ext_func;
	elem->v.func.ext_func = func;
}

char *
expr_bind_ext_funcs(expr_t expr, list_t ext_funcs)
{
	struct bind_ext_funcs_context ctx;

	ctx.ret = NULL;
	ctx.funcs = ext_funcs;
	list_for_each(expr->stack, bind_ext_funcs, &ctx);

	return ctx.ret;
}

static void
expr_find_unbinded_args(void *data, void *context)
{
	expr_t expr;
	struct expr_signal_id **unbinded;

	expr = (expr_t)data;
	unbinded = (struct expr_signal_id **)context;

	if (*unbinded != NULL)
		return;

	*unbinded = expr_find_unbinded(expr);
}

static void
elem_is_unbinded(void *data, void *context)
{
	struct elem *elem;
	struct expr_signal_id **unbinded;

	elem = (struct elem *)data;
	unbinded = (struct expr_signal_id **)context;

	if (*unbinded != NULL)
		return;

	if (elem->type == elem_type_signal_id)
		*unbinded = &elem->v.signal.id;
	else if ((elem->type == elem_type_signal_port)
		 || (elem->type == elem_type_signal_ref))
	  {
		if (elem->v.signal.delay != NULL)
			*unbinded = expr_find_unbinded(elem->v.signal.delay);
		if (*unbinded != NULL)
			return;
		if (elem->v.signal.delay_max != NULL)
			*unbinded = expr_find_unbinded(
				elem->v.signal.delay_max);
	  }
	else if ((elem->type == elem_type_call_id)
		 || (elem->type == elem_type_call_ext_func))
		  list_for_each(elem->v.func.args, expr_find_unbinded_args,
				unbinded);
}

struct expr_signal_id *
expr_find_unbinded(expr_t expr)
{
	struct expr_signal_id *ret;

	ret = NULL;
	list_for_each(expr->stack, elem_is_unbinded, &ret);

	return ret;
}

static void
check_ext_func_args(void *data, void *context)
{
	struct elem *elem;
	struct ext_func **ret;

	elem = (struct elem *)data;
	ret = (struct ext_func **)context;

	if (*ret != NULL)
		return;

	if ((elem->type == elem_type_signal_id)
	    || (elem->type == elem_type_signal_port)
	    || (elem->type == elem_type_signal_ref))
	  {
		if (elem->v.signal.delay != NULL)
		  {
			*ret = expr_check_ext_func_args(elem->v.signal.delay);
			if (*ret != NULL)
				return;
			if (elem->v.signal.delay_max != NULL)
				*ret = expr_check_ext_func_args(
					elem->v.signal.delay_max);
		  }
	  }
	else if (elem->type == elem_type_call_ext_func)
	  {
		if (list_get_n_elems(elem->v.func.args)
		    != elem->v.func.ext_func->n_args)
			*ret = elem->v.func.ext_func;
	  }
}

struct ext_func *
expr_check_ext_func_args(expr_t expr)
{
	struct ext_func *ret;

	ret = NULL;
	list_for_each(expr->stack, check_ext_func_args, &ret);

	return ret;
}

static int
ptr_cmp(void *d1, void *d2)
{
	return (char *)d1 - (char *)d2;
}

static void
merge_inputs(void *data, void *context)
{
	struct port *port;
	list_t list;

	port = (struct port *)data;
	list = (list_t)context;

	if (list_find(list, ptr_cmp, port) != NULL)
		return;

	list_append(list, port);
}

static void
merge_inputs_args(void *data, void *context)
{
	expr_t expr;
	list_t list;
	list_t l;

	expr = (expr_t)data;
	list = (list_t)context;

	l = expr_get_inputs(expr);
	list_for_each(l, merge_inputs, list);
}

static void
add_input(void *data, void *context)
{
	struct elem *elem;
	list_t list;
	list_t l;

	elem = (struct elem *)data;
	list = (list_t)context;

	if (elem->type == elem_type_signal_port)
	  {
		if (list_find(list, ptr_cmp, elem->v.signal.port) != NULL)
			return;

		list_append(list, elem->v.signal.port);

		if (elem->v.signal.delay != NULL)
		  {
			l = expr_get_inputs(elem->v.signal.delay);
			list_for_each(l, merge_inputs, list);
		  }
		if (elem->v.signal.delay_max != NULL)
		  {
			l = expr_get_inputs(elem->v.signal.delay_max);
			list_for_each(l, merge_inputs, list);
		  }
	  }
	else if (elem->type == elem_type_call_ext_func)
		list_for_each(elem->v.func.args, merge_inputs_args, list);
}

list_t
expr_get_inputs(expr_t expr)
{
	list_t ret;

	ret = list_new();

	list_for_each(expr->stack, add_input, ret);

	return ret;
}

static void
merge_refs(void *data, void *context)
{
	struct scheduled_expr *ref;
	list_t list;

	ref = (struct scheduled_expr *)data;
	list = (list_t)context;

	if (list_find(list, ptr_cmp, ref) != NULL)
		return;

	list_append(list, ref);
}

static void
merge_refs_args(void *data, void *context)
{
	expr_t expr;
	list_t list;
	list_t l;

	expr = (expr_t)data;
	list = (list_t)context;

	l = expr_get_refs(expr);
	list_for_each(l, merge_refs, list);
}

static void
add_ref(void *data, void *context)
{
	struct elem *elem;
	list_t list;
	list_t l;

	elem = (struct elem *)data;
	list = (list_t)context;

	if (elem->type == elem_type_signal_ref)
	  {
		list_append(list, elem->v.signal.ref);

		if (elem->v.signal.delay != NULL)
		  {
			l = expr_get_refs(elem->v.signal.delay);
			list_for_each(l, merge_refs, list);
		  }
		/*if (elem->v.signal.delay_max != NULL)
		  {
			l = expr_get_refs(elem->v.signal.delay_max);
			list_for_each(l, merge_refs, list);
		  }*/
	  }
	else if (elem->type == elem_type_call_ext_func)
		list_for_each(elem->v.func.args, merge_refs_args, list);
}

list_t
expr_get_refs(expr_t expr)
{
	list_t ret;

	ret = list_new();

	list_for_each(expr->stack, add_ref, ret);

	return ret;
}

static void
merge_df_refs_args(void *data, void *context)
{
	expr_t expr;
	list_t list;
	list_t l;

	expr = (expr_t)data;
	list = (list_t)context;

	l = expr_get_df_refs(expr);
	list_for_each(l, merge_refs, list);
}

static void
add_df_ref(void *data, void *context)
{
	struct elem *elem;
	list_t list;
	list_t l;

	elem = (struct elem *)data;
	list = (list_t)context;

	if (elem->type == elem_type_signal_ref)
	  {
		if (elem->v.signal.delay != NULL)
		  {
			l = expr_get_df_refs(elem->v.signal.delay);
			list_for_each(l, merge_refs, list);
			/*if (elem->v.signal.delay_max != NULL)
			  {
				l = expr_get_df_refs(elem->v.signal.delay_max);
				list_for_each(l, merge_refs, list);
			  }*/
		  }
		else
			list_append(list, elem->v.signal.ref);
	  }
	else if (elem->type == elem_type_call_ext_func)
		list_for_each(elem->v.func.args, merge_df_refs_args, list);
}

list_t
expr_get_df_refs(expr_t expr)
{
	list_t ret;

	ret = list_new();

	list_for_each(expr->stack, add_df_ref, ret);

	return ret;
}

struct add_delay_context
  {
	list_t			 list;
	struct scheduled_expr	*ref;
  };

static void
add_delay_args(void *data, void *context)
{
	expr_t expr;
	struct add_delay_context *ctx;

	expr = (expr_t)data;
	ctx = (struct add_delay_context *)context;

	list_merge(ctx->list, expr_get_delays(expr, ctx->ref));
}

static void
add_delay(void *data, void *context)
{
	struct elem *elem;
	struct add_delay_context *ctx;
	struct delay *d;

	elem = (struct elem *)data;
	ctx = (struct add_delay_context *)context;

	if (elem->type == elem_type_signal_ref)
	  {
		if (elem->v.signal.ref == ctx->ref)
		  {
			if (elem->v.signal.delay != NULL)
			  {
				d = xmalloc(sizeof(struct delay));
				d->delay = elem->v.signal.delay;
				d->delay_max = elem->v.signal.delay_max;
				list_append(ctx->list, d);
			  }
		  }
		else if (elem->v.signal.delay != NULL)
			list_merge(ctx->list,
				   expr_get_delays(elem->v.signal.delay,
						   ctx->ref));
	  }
	else if (elem->type == elem_type_call_ext_func)
		list_for_each(elem->v.func.args, add_delay_args, ctx);
}

list_t
expr_get_delays(expr_t expr, struct scheduled_expr *ref)
{
	struct add_delay_context ctx;

	ctx.list = list_new();
	ctx.ref = ref;
	list_for_each(expr->stack, add_delay, &ctx);

	return ctx.list;
}

static void
merge_ext_funcs(void *data, void *context)
{
	struct ext_func *func;
	list_t list;

	func = (struct ext_func *)data;
	list = (list_t)context;

	if (list_find(list, ptr_cmp, func) == NULL)
		list_append(list, func);
}

static void
merge_ext_funcs_args(void *data, void *context)
{
	expr_t expr;
	list_t list;
	list_t l;

	expr = (expr_t)data;
	list = (list_t)context;

	l = expr_get_ext_funcs(expr);
	list_for_each(l, merge_ext_funcs, list);
}

static void
add_ext_func(void *data, void *context)
{
	struct elem *elem;
	list_t list;
	list_t l;

	elem = (struct elem *)data;
	list = (list_t)context;

	if (elem->type == elem_type_signal_ref)
	  {
		if (elem->v.signal.delay != NULL)
		  {
			l = expr_get_ext_funcs(elem->v.signal.delay);
			list_for_each(l, merge_ext_funcs, list);
		  }
		if (elem->v.signal.delay_max != NULL)
		  {
			l = expr_get_ext_funcs(elem->v.signal.delay_max);
			list_for_each(l, merge_ext_funcs, list);
		  }
	  }
	else if (elem->type == elem_type_call_ext_func)
	  {
		if (list_find(list, ptr_cmp, elem->v.func.ext_func) == NULL)
			list_append(list, elem->v.func.ext_func);
		list_for_each(elem->v.func.args, merge_ext_funcs_args, list);
	  }
}

list_t
expr_get_ext_funcs(expr_t expr)
{
	list_t ret;

	ret = list_new();
	list_for_each(expr->stack, add_ext_func, ret);

	return ret;
}

static void
arg_copy(void *data, void *context)
{
	expr_t expr;
	list_t list;

	expr = (expr_t)data;
	list = (list_t)context;

	list_append(list, expr_copy(expr));
}

static void
elem_copy(void *data, void *context)
{
	struct elem *elem, *e;
	list_t stack;

	elem = (struct elem *)data;
	stack = (list_t)context;

	e = xmalloc(sizeof(struct elem));
	*e = *elem;

	if (elem->type == elem_type_signal_id)
		e->v.signal.id.id = copy_str(elem->v.signal.id.id);
	else if (elem->type == elem_type_call_id)
		e->v.func.id = copy_str(elem->v.func.id);

	if ((elem->type == elem_type_signal_id)
	    || (elem->type == elem_type_signal_port)
	    || (elem->type == elem_type_signal_ref))
	  {
		if (elem->v.signal.delay != NULL)
			e->v.signal.delay = expr_copy(elem->v.signal.delay);
		else
			e->v.signal.delay = NULL;

		if (elem->v.signal.delay_max != NULL)
			e->v.signal.delay_max =
				expr_copy(elem->v.signal.delay_max);
		else
			e->v.signal.delay_max = NULL;
	  }
	else if ((elem->type == elem_type_call_id)
		 || (elem->type == elem_type_call_ext_func))
	  {
		e->v.func.args = list_new();
		list_for_each(elem->v.func.args, arg_copy, e->v.func.args);
	  }

	list_append(stack, e);
}

expr_t
expr_copy(expr_t expr)
{
	struct _expr *ret;

	ret = xmalloc(sizeof(struct _expr));
	ret->stack = list_new();

	list_for_each(expr->stack, elem_copy, ret->stack);

	return ret;
}

struct port_to_value_context
  {
	struct port	*port;
	double		 value;
  };

static void
port_to_value_args(void *data, void *context)
{
	expr_t expr;
	struct port_to_value_context *ctx;

	expr = (expr_t)data;
	ctx = (struct port_to_value_context *)context;

	expr_port_to_value(expr, ctx->port, ctx->value);
}

static void
port_to_value(void *data, void *context)
{
	struct elem *elem;
	struct port_to_value_context *ctx;

	elem = (struct elem *)data;
	ctx = (struct port_to_value_context *)context;

	if ((elem->type == elem_type_signal_id)
	    || (elem->type == elem_type_signal_port)
	    || (elem->type == elem_type_signal_ref))
	  {
		if (elem->v.signal.delay != NULL)
			expr_port_to_value(elem->v.signal.delay, ctx->port,
					   ctx->value);
		if (elem->v.signal.delay_max != NULL)
			expr_port_to_value(elem->v.signal.delay_max, ctx->port,
					   ctx->value);
	  }
	else if ((elem->type == elem_type_call_id)
		 || (elem->type == elem_type_call_ext_func))
		list_for_each(elem->v.func.args, port_to_value_args, ctx);

	if (elem->type == elem_type_signal_port)
	  {
		if (elem->v.signal.port != ctx->port)
			return;

		if (elem->v.signal.delay != NULL)
			expr_free(elem->v.signal.delay);
		if (elem->v.signal.delay_max != NULL)
			expr_free(elem->v.signal.delay_max);

		elem->type = elem_type_value;
		elem->v.value = ctx->value;
	  }
}

void
expr_port_to_value(expr_t expr, struct port *port, double value)
{
	struct port_to_value_context ctx;

	ctx.port = port;
	ctx.value = value;
	list_for_each(expr->stack, port_to_value, &ctx);
}

struct port_to_ref_context
  {
	struct port		*port;
	struct scheduled_expr	*ref;
  };

static void
port_to_ref_args(void *data, void *context)
{
	expr_t expr;
	struct port_to_ref_context *ctx;

	expr = (expr_t)data;
	ctx = (struct port_to_ref_context *)context;

	expr_port_to_ref(expr, ctx->port, ctx->ref);
}

static void
port_to_ref(void *data, void *context)
{
	struct elem *elem;
	struct port_to_ref_context *ctx;

	elem = (struct elem *)data;
	ctx = (struct port_to_ref_context *)context;

	if ((elem->type == elem_type_signal_id)
	    || (elem->type == elem_type_signal_port)
	    || (elem->type == elem_type_signal_ref))
	  {
		if (elem->v.signal.delay != NULL)
			expr_port_to_ref(elem->v.signal.delay, ctx->port,
					 ctx->ref);
		if (elem->v.signal.delay_max != NULL)
			expr_port_to_ref(elem->v.signal.delay_max, ctx->port,
					 ctx->ref);
	  }
	else if ((elem->type == elem_type_call_id)
		 || (elem->type == elem_type_call_ext_func))
		list_for_each(elem->v.func.args, port_to_ref_args, ctx);

	if (elem->type == elem_type_signal_port)
	  {
		if (elem->v.signal.port != ctx->port)
			return;

		elem->type = elem_type_signal_ref;
		elem->v.signal.ref = ctx->ref;
	  }	
}

void
expr_port_to_ref(expr_t expr, struct port *port, struct scheduled_expr *ref)
{
	struct port_to_ref_context ctx;

	ctx.port = port;
	ctx.ref = ref;
	list_for_each(expr->stack, port_to_ref, &ctx);
}

static void
arg_contains_ref(void *data, void *context)
{
	expr_t expr;
	char *found;

	expr = (expr_t)data;
	found = (char *)context;

	if (*found)
		return;

	*found = expr_contains_ref(expr);
}

static void
contains_ref(void *data, void *context)
{
	struct elem *elem;
	char *found;

	elem = (struct elem *)data;
	found = (char *)context;

	if (*found)
		return;

	if (elem->type == elem_type_signal_ref)
		*found = 1;
	else if ((elem->type == elem_type_call_id)
		 || (elem->type == elem_type_call_ext_func))
		list_for_each(elem->v.func.args, arg_contains_ref, found);
}

char
expr_contains_ref(expr_t expr)
{
	char ref_found;

	ref_found = 0;
	list_for_each(expr->stack, contains_ref, &ref_found);

	return ref_found;
}

static void
arg_contains_sample_rate(void *data, void *context)
{
	expr_t expr;
	char *found;

	expr = (expr_t)data;
	found = (char *)context;

	if (*found)
		return;

	*found = expr_contains_sample_rate(expr);
}

static void
contains_sample_rate(void *data, void *context)
{
	struct elem *elem;
	char *found;

	elem = (struct elem *)data;
	found = (char *)context;

	if (*found)
		return;

	if (elem->type == elem_type_signal_ref)
	  {
		if (elem->v.signal.delay != NULL)
			*found = expr_contains_sample_rate(
				elem->v.signal.delay);
		if (*found)
			return;

		if (elem->v.signal.delay_max != NULL)
			*found = expr_contains_sample_rate(
				elem->v.signal.delay_max);
	  }
	else if (elem->type == elem_type_sample_rate)
		*found = 1;
	else if ((elem->type == elem_type_call_id)
		 || (elem->type == elem_type_call_ext_func))
		list_for_each(elem->v.func.args, arg_contains_sample_rate,
			      found);
}

char
expr_contains_sample_rate(expr_t expr)
{
	char sr_found;

	sr_found = 0;
	list_for_each(expr->stack, contains_sample_rate, &sr_found);

	return sr_found;
}

struct is_const_context
  {
	list_t	stack;
	char	ret;
  };

static void
is_const(void *data, void *context)
{
	struct elem *elem;
	struct is_const_context *ctx;

	elem = (struct elem *)data;
	ctx = (struct is_const_context *)context;

	if (!ctx->ret)
		return;

	if ((elem->type == elem_type_sample_rate)
	    || (elem->type == elem_type_call_ext_func))
	  {
		ctx->ret = 0;
		return;
	  }

	if (elem->type == elem_type_signal_ref)
	  {
		if (elem->v.signal.ref->component == NULL)
		  {
			ctx->ret = 0;
			return;
		  }

		if (list_find(ctx->stack, ptr_cmp, elem->v.signal.ref->expr)
		    != NULL)
		  {
			ctx->ret = 0;
			return;
		  }

		list_push(ctx->stack, elem->v.signal.ref->expr);
		list_for_each(elem->v.signal.ref->expr->stack, is_const, ctx);
		list_pop(ctx->stack);
	  }
}

char
expr_is_const(expr_t expr)
{
	struct is_const_context ctx;

	ctx.ret = 1;
	ctx.stack = list_new();
	list_push(ctx.stack, expr);
	list_for_each(expr->stack, is_const, &ctx);
	list_free(ctx.stack);

	return ctx.ret;
}

static void is_rt_const(void *data, void *context);

static void
arg_is_rt_const(void *data, void *context)
{
	expr_t expr;
	struct is_const_context *ctx;

	expr = (expr_t)data;
	ctx = (struct is_const_context *)context;

	if (!ctx->ret)
		return;

	list_push(ctx->stack, expr);
	list_for_each(expr->stack, is_rt_const, ctx);
	list_pop(ctx->stack);
}

static void
is_rt_const(void *data, void *context)
{
	struct elem *elem;
	struct is_const_context *ctx;

	elem = (struct elem *)data;
	ctx = (struct is_const_context *)context;

	if (!ctx->ret)
		return;

	if (elem->type == elem_type_signal_ref)
	  {
		if (elem->v.signal.ref->component == NULL)
		  {
			ctx->ret = 0;
			return;
		  }

		if (list_find(ctx->stack, ptr_cmp, elem->v.signal.ref->expr)
		    != NULL)
		  {
			ctx->ret = 0;
			return;
		  }

		list_push(ctx->stack, elem->v.signal.ref->expr);
		list_for_each(elem->v.signal.ref->expr->stack, is_rt_const,
			      ctx);
		list_pop(ctx->stack);
	  }
	else if (elem->type == elem_type_call_ext_func)
		list_for_each(elem->v.func.args, arg_is_rt_const, ctx);
}

char
expr_is_rt_const(expr_t expr)
{
	struct is_const_context ctx;

	ctx.ret = 1;
	ctx.stack = list_new();
	list_push(ctx.stack, expr);
	list_for_each(expr->stack, is_rt_const, &ctx);
	list_free(ctx.stack);

	return ctx.ret;
}

static double
eval(expr_t expr)
{
	struct elem *elem;
	double val;

	elem = list_pop(expr->stack);

	switch(elem->type)
	  {
		case elem_type_value:
			val = elem->v.value;
			break;
		case elem_type_add:
			val = eval(expr);
			val += eval(expr);
			break;
		case elem_type_sub:
			val = eval(expr);
			val -= eval(expr);
			break;
		case elem_type_mul:
			val = eval(expr);
			val *= eval(expr);
			break;
		case elem_type_div:
			val = eval(expr);
			val /= eval(expr);
			break;
		case elem_type_plus:
			val = eval(expr);
			break;
		case elem_type_minus:
			val = -eval(expr);
			break;
		case elem_type_signal_ref:
			val = expr_eval(elem->v.signal.ref->expr);
			break;
		default:
			/* this should never happen */
			fprintf(stderr, "error: oops in eval()\n");
			list_push(expr->stack, elem);
			expr_dump(expr);
			exit(EXIT_FAILURE);
			break;
	  }

	return val;
}

double
expr_eval(expr_t expr)
{
	expr_t eval_expr;
	double val;

	eval_expr = expr_copy(expr);

	val = eval(eval_expr);

	expr_free(eval_expr);

	return val;
}

static void
expr_free_args(void *data, void *context)
{
	expr_free((expr_t)data);
}

static void
free_elem(void *data, void *context)
{
	struct elem *elem;

	elem = (struct elem *)data;

	if (elem->type == elem_type_signal_id)
		free(elem->v.signal.id.id);
	else if (elem->type == elem_type_call_id)
		free(elem->v.func.id);

	if ((elem->type == elem_type_signal_id)
	    || (elem->type == elem_type_signal_port)
	    || (elem->type == elem_type_signal_ref))
	  {
		if (elem->v.signal.delay != NULL)
			expr_free(elem->v.signal.delay);
		if (elem->v.signal.delay_max != NULL)
			expr_free(elem->v.signal.delay_max);
	  }
	else if ((elem->type == elem_type_call_id)
		 || (elem->type == elem_type_call_ext_func))
	  {
		list_for_each(elem->v.func.args, expr_free_args, NULL);
		list_free(elem->v.func.args);
	  }

	free(elem);
}

void
expr_free(expr_t expr)
{
	list_for_each(expr->stack, free_elem, NULL);
	list_free(expr->stack);
	free(expr);
}

static void
arg_dump(void *data, void *context)
{
	expr_t expr;
	char *first;

	expr = (expr_t)data;
	first = (char *)context;

	if (*first)
		*first = 0;
	else
		printf(", ");

	expr_dump(expr);
}

static void
elem_dump(list_t stack)
{
	struct elem *e;
	char first;

	e = list_pop(stack);

	switch (e->type)
	  {
		case elem_type_value:
			printf("%.*f", DBL_DIG, e->v.value);
			break;
		case elem_type_sample_rate:
			printf("sample_rate");
			break;
		case elem_type_signal_ref:
			printf("%s%s%s%s",
			       (e->v.signal.ref->component != NULL)
			       ? e->v.signal.ref->component->id : "",
			       (e->v.signal.ref->component != NULL) ? "." : "",
			       e->v.signal.ref->port->id,
			       ((e->v.signal.ref->port->type == port_type_w)
			        || (e->v.signal.ref->port->type == port_type_k))
			       ? ".in" : "");
			if (e->v.signal.delay != NULL)
			  {
				printf("[");
				expr_dump(e->v.signal.delay);
				if (e->v.signal.delay_max != NULL)
				  {
					printf(", ");
					expr_dump(e->v.signal.delay_max);
				  }
				printf("]");
			  }
			break;
		case elem_type_call_ext_func:
			printf("%s(", e->v.func.ext_func->id);
			first = 1;
			list_for_each(e->v.func.args, arg_dump, &first);
			printf(")");
			break;
		case elem_type_add:
			printf("(");
			elem_dump(stack);
			printf(") + (");
			elem_dump(stack);
			printf(")");
			break;
		case elem_type_sub:
			printf("(");
			elem_dump(stack);
			printf(") - (");
			elem_dump(stack);
			printf(")");
			break;
		case elem_type_mul:
			printf("(");
			elem_dump(stack);
			printf(") * (");
			elem_dump(stack);
			printf(")");
			break;
		case elem_type_div:
			printf("(");
			elem_dump(stack);
			printf(") / (");
			elem_dump(stack);
			printf(")");
			break;
		case elem_type_plus:
			printf("+(");
			elem_dump(stack);
			printf(")");
			break;
		case elem_type_minus:
			printf("-(");
			elem_dump(stack);
			printf(")");
			break;
		default:
			printf("NOOOOOO %d (%d)!", e->type, elem_type_value);
			break;
	  }
}

void
expr_dump(expr_t expr)
{
	expr_t expr2;

	expr2 = expr_copy(expr);

	elem_dump(expr2->stack);

	expr_free(expr2);
}

struct arg_c_compile_context
  {
	FILE	*fp;
	char	 first;
	char	 inside_run;
  };

static void
arg_c_compile(void *data, void *context)
{
	expr_t expr;
	struct arg_c_compile_context *ctx;

	expr = (expr_t)data;
	ctx = (struct arg_c_compile_context *)context;

	if (ctx->first)
		ctx->first = 0;
	else
		fprintf(ctx->fp, ", ");

	expr_c_compile(expr, ctx->fp, ctx->inside_run);
	expr_free(expr);
}

static void
elem_c_compile(list_t stack, FILE *fp, char inside_run)
{
	struct elem *e;
	struct arg_c_compile_context ctx;

	e = list_pop(stack);

	switch (e->type)
	  {
		case elem_type_value:
			fprintf(fp, "%.*f", DBL_DIG, e->v.value);
			break;
		case elem_type_sample_rate:
			fprintf(fp, "%ssample_rate",
				inside_run ? "plugin->" : "");
			break;
		case elem_type_signal_ref:
			if (!inside_run)
			  {
				expr_c_compile(e->v.signal.ref->expr, fp, 0);
				if (e->v.signal.delay != NULL)
					expr_free(e->v.signal.delay);
				if (e->v.signal.delay_max != NULL)
					expr_free(e->v.signal.delay_max);
				break;
			  }

			if (e->v.signal.delay == NULL)
			  {
				if (e->v.signal.ref->component == NULL)
					fprintf(fp, "plugin->port_%s%s",
						e->v.signal.ref->port->id,
						(e->v.signal.ref->port->sync
						 == port_sync_sync)
						? "[i]" : "[0]");
				else
					fprintf(fp, "plugin->val_%lu",
						e->v.signal.ref->index);
			  }
			else if (e->v.signal.ref->db_type
				 == delay_buf_type_single)
			  {
				fprintf(fp, "plugin->buf_%lu",
					e->v.signal.ref->index);
				expr_free(e->v.signal.delay);
				if (e->v.signal.delay_max != NULL)
					expr_free(e->v.signal.delay_max);
			  }
			else if (e->v.signal.ref->db_type
				 == delay_buf_type_fixed)
			  {
				fprintf(fp, "plugin->buf_%lu[pmf_delay(",
					e->v.signal.ref->index);
				expr_c_compile(e->v.signal.delay, fp,
					       inside_run);
				fprintf(fp, ", plugin->buf_%lu_cur, %lu)]",
					e->v.signal.ref->index,
					(size_t)e->v.signal.ref->delay_min);
				expr_free(e->v.signal.delay);
				if (e->v.signal.delay_max != NULL)
					expr_free(e->v.signal.delay_max);
			  }
			else
			  {
				fprintf(fp, "plugin->buf_%lu[pmf_delay(",
					e->v.signal.ref->index);
				expr_c_compile(e->v.signal.delay, fp,
					       inside_run);
				fprintf(fp, ", plugin->buf_%lu_cur, "
					"plugin->buf_%lu_len - 1)]",
					e->v.signal.ref->index,
					e->v.signal.ref->index);
				expr_free(e->v.signal.delay);
				if (e->v.signal.delay_max != NULL)
					expr_free(e->v.signal.delay_max);
			  }
			break;
		case elem_type_call_ext_func:
			fprintf(fp, "%s(", e->v.func.ext_func->f_id);
			ctx.fp = fp;
			ctx.first = 1;
			ctx.inside_run = inside_run;
			list_for_each(e->v.func.args, arg_c_compile, &ctx);
			list_free(e->v.func.args);
			fprintf(fp, ")");
			break;
		case elem_type_add:
			fprintf(fp, "(");
			elem_c_compile(stack, fp, inside_run);
			fprintf(fp, ") + (");
			elem_c_compile(stack, fp, inside_run);
			fprintf(fp, ")");
			break;
		case elem_type_sub:
			fprintf(fp, "(");
			elem_c_compile(stack, fp, inside_run);
			fprintf(fp, ") - (");
			elem_c_compile(stack, fp, inside_run);
			fprintf(fp, ")");
			break;
		case elem_type_mul:
			fprintf(fp, "(");
			elem_c_compile(stack, fp, inside_run);
			fprintf(fp, ") * (");
			elem_c_compile(stack, fp, inside_run);
			fprintf(fp, ")");
			break;
		case elem_type_div:
			fprintf(fp, "(");
			elem_c_compile(stack, fp, inside_run);
			fprintf(fp, ") / (");
			elem_c_compile(stack, fp, inside_run);
			fprintf(fp, ")");
			break;
		case elem_type_plus:
			fprintf(fp, "+(");
			elem_c_compile(stack, fp, inside_run);
			fprintf(fp, ")");
			break;
		case elem_type_minus:
			fprintf(fp, "-(");
			elem_c_compile(stack, fp, inside_run);
			fprintf(fp, ")");
			break;
		default:
			break;
	  }

	free(e);
}

void
expr_c_compile(expr_t expr, FILE *fp, char inside_run)
{
	expr_t expr2;

	expr2 = expr_copy(expr);

	elem_c_compile(expr2->stack, fp, inside_run);

	expr_free(expr2);
}
