/*=========================================================================

  Program:   ParaView
  Module:    vtkSMAnimationSceneProxy.h

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See Copyright.txt or http://www.paraview.org/HTML/Copyright.html for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
// .NAME vtkSMAnimationSceneProxy - proxy for vtkAnimationScene
// .SECTION Description
// Proxy for animation scene. A scene is an animation setup that can be played.
// Also supports writing out animation images (movie) and animation geometry.
// Like all animation proxies, this is a client side proxy with not server 
// side VTK objects created.
// .SECTION See Also
// vtkAnimationScene vtkSMAnimationCueProxy

#ifndef __vtkSMAnimationSceneProxy_h
#define __vtkSMAnimationSceneProxy_h

#include "vtkSMAnimationCueProxy.h"

class vtkAnimationScene;
class vtkSMViewProxy;
class vtkSMTimeKeeperProxy;
class vtkCommand;

class VTK_EXPORT vtkSMAnimationSceneProxy : public vtkSMAnimationCueProxy
{
public:
  static vtkSMAnimationSceneProxy* New();
  vtkTypeMacro(vtkSMAnimationSceneProxy, vtkSMAnimationCueProxy);
  void PrintSelf(ostream& os, vtkIndent indent);

  // Description:
  // Add view module that is involved in the animation generated by this scene.
  // When playing animation, the scene proxy will call Render()
  // and CacheUpdate() on view modules that it is aware of. Also, while saving,
  // geometry or images, the scene considers only the view modules it is aware of.
  void AddViewModule(vtkSMViewProxy*);
  void RemoveViewModule(vtkSMViewProxy*);
  void RemoveAllViewModules();

  // Description:
  // API to get the view modules.
  unsigned int GetNumberOfViewModules();
  vtkSMViewProxy* GetViewModule(unsigned int cc);
 
  // Description:
  // Method to set the current time. This updates the proxies to reflect the state
  // at the indicated time.
  void SetAnimationTime(double time);

  // Description:
  // Returns the current animation time.
  double GetAnimationTime();

  // Description:
  // Get/Set the cache limit (in kilobytes) for each process. If cache size
  // grows beyond the limit, no caching is done on any of the processes.
  vtkGetMacro(CacheLimit, int);
  vtkSetMacro(CacheLimit, int);

  // Description:
  // Set if caching is enabled.
  // This method synchronizes the cahcing flag on every cue.
  vtkSetMacro(Caching, int);
  vtkGetMacro(Caching, int);

  // Description:
  // Add/Remove animation cue proxy.
  void AddCueProxy(vtkSMAnimationCueProxy*);
  void RemoveCueProxy(vtkSMAnimationCueProxy*);

  // Description:
  // Set the time keeper. Time keeper is used to obtain the information about
  // timesteps. This is required to play animation in "Snap To Timesteps" mode.
  void SetTimeKeeper(vtkSMTimeKeeperProxy*);
  vtkGetObjectMacro(TimeKeeper, vtkSMTimeKeeperProxy);

  // Description:
  // Lock the start time. When locked, the StartTime won't be automatically
  // updated when data time changes.
  vtkSetMacro(LockStartTime, bool);
  vtkGetMacro(LockStartTime, bool);
  vtkBooleanMacro(LockStartTime, bool);

  // Description:
  // Lock the end time. When locked, the EndTime won't be automatically updated
  // when the data time changes.
  vtkSetMacro(LockEndTime, bool);
  vtkGetMacro(LockEndTime, bool);
  vtkBooleanMacro(LockEndTime, bool);

  // Description:
  // Overridden to ensure that correct end-times are set on the scene. 
  virtual void UpdateVTKObjects();
//BTX
protected:
  vtkSMAnimationSceneProxy();
  ~vtkSMAnimationSceneProxy();

  // Description:
  // Create VTK Objects.
  virtual void CreateVTKObjects();

  virtual void UpdateVTKObjects(vtkClientServerStream& stream)
    { this->Superclass::UpdateVTKObjects(stream); }

  // Description:
  // Callbacks for corresponding Cue events. The argument must be 
  // casted to vtkAnimationCue::AnimationCueInfo.
  virtual void StartCueInternal(void* info);
  virtual void TickInternal(void* info);
  virtual void EndCueInternal(void* info);
  void CacheUpdate(void* info);

  // Description:
  // Called when the timekeeper's time range changes.
  void TimeKeeperTimeRangeChanged();
  void TimeKeeperTimestepsChanged();

  // Description:
  // Check if the current cache size on all processes is within limits.
  bool CheckCacheSizeWithinLimit();

  int Caching;

  friend class vtkSMAnimationSceneImageWriter;
  int OverrideStillRender;
  vtkSetMacro(OverrideStillRender, int);

  int CacheLimit; // in KiloBytes.

  vtkSMProxy* AnimationPlayer;
  vtkSMTimeKeeperProxy* TimeKeeper;

  bool LockEndTime;
  bool LockStartTime;

private:
  vtkSMAnimationSceneProxy(const vtkSMAnimationSceneProxy&); // Not implemented.
  void operator=(const vtkSMAnimationSceneProxy&); // Not implemented.

  class vtkInternals;
  vtkInternals* Internals;

  vtkCommand* TimeRangeObserver;
  vtkCommand* TimestepValuesObserver;

  class vtkPlayerObserver;
  friend class vtkPlayerObserver;
  vtkPlayerObserver* PlayerObserver;

  // Called when player begins playing animation.
  void OnStartPlay();

  // Called when player is done with playing animation.
  void OnEndPlay();

  // Used to prevent calls SetAnimationTime() during an animation tick.
  bool InTick;
//ETX
};


#endif

