/*
 * Copyright (c) 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
 
using Nemerle.Collections;
using Nemerle.Utility;

using Nemerle.Compiler.Parsetree;

using TT = Nemerle.Compiler.Typedtree;

namespace Nemerle.Compiler 
{
  public module ConstantFolder
  {
    internal is_known_operator (name : string) : bool
    {
      | "+"
      | "-"
      | "*"
      | "/"
      | "%"
      | "%^"
      | "%|"
      | "&"
      | "^"
      | "|"
      | "%&"
      | "&&"
      | "||"
      | "!"
      | ">>"
      | "<<"
      | "~" => true
      | _ => false
    }

    UnsignedValue (lit : Literal.Integer) : ulong
    {
      if (lit.is_negative)
        unchecked ( (- (lit.val :> long)) :> ulong & MaxValueMask (lit.treat_as))
      else lit.val
    }

    CheckInteger (l : Literal.Integer) : Literal.Integer
    {
      when (l.is_negative && l.val > 0x8000000000000000UL)
        Message.Error ("the operation overflows at compile-time during"
                       " constants folding in checked mode");
      l
    }

    internal FoldLiterals (is_checked : bool, op : string,
                           lit1 : Literal, lit2 : Literal) : Literal
    {
      match ((lit1, lit2)) {
        | (Literal.Integer as l1, Literal.Integer as l2) =>
          FoldLiterals (is_checked, op, l1, l2)
          
        | (Literal.String (s1), Literal.String (s2)) when op == "+" =>
          Literal.String (s1 + s2)
          
        | _ => 
          null
          // Message.Warning ($ "evil literals $l1 $l2");
          // assert (false)
      }
    }

    internal FoldLiterals (is_checked : bool, op : string,
                           lit1 : Literal.Integer, lit2 : Literal.Integer) : Literal.Integer
    {
      //Message.Debug ($ "fold $lit1 $op $lit2");
      def subtr (v1, v2, is_negative) {
        perform_check (is_checked, fun () {
          if (v1 >= v2)
            Literal.Integer (v1 - v2, is_negative, null)
          else
            Literal.Integer (v2 - v1, !is_negative, null)
        })
      }

      def positive (v) {
        Literal.Integer (v, false, null).WithProperType () :> Literal.Integer
      }
      
      def val1 = lit1.val;
      def val2 = lit2.val;

      def val1u = UnsignedValue (lit1);
      def val2u = UnsignedValue (lit2);

      def result_sign = (lit1.is_negative || lit2.is_negative) &&
                        !(lit1.is_negative && lit2.is_negative);
     
      def res =
        match (op) {
          | "+"  => 
            if (lit2.is_negative)
              FoldLiterals (is_checked, "-", lit1, positive (val2))
            else if (lit1.is_negative)
              FoldLiterals (is_checked, "-", lit2, positive (val1))
            else
              positive (val1 + val2)
                      
          | "-"  =>
            if (lit2.is_negative)
              FoldLiterals (is_checked, "+", lit1, positive (val2))
            else if (lit1.is_negative)
              Literal.Integer (val1 + val2, true, null)
            else
              subtr (val1, val2, lit1.is_negative)
                      
          | "*"  => 
            def tmp = val1 * val2; // workaround mono bug #74726
            Literal.Integer (tmp, result_sign, null)

          | "/"  => Literal.Integer (val1 / val2, result_sign, null)
          | "%"  => Literal.Integer (val1 % val2, result_sign, null)

          // FIXME
          | "%^"  
          | "^"  => ConvertSignTo (val1u ^ val2u, lit1.treat_as)
            
          | "|"            
          | "%|" => ConvertSignTo (val1u | val2u, lit1.treat_as)

          | "&"  
          | "%&" => ConvertSignTo (val1u & val2u, lit1.treat_as)
            
          | ">>" => ConvertSignTo (val1u >> (val2u & 0x3F) :> int, lit1.treat_as)
          | "<<" => ConvertSignTo (val1u << (val2u & 0x3F) :> int, lit1.treat_as)
            
          | _ => null
          // Util.ice ("wrong operator " + op);
        }

      if (res != null) {
        // Message.Debug ($ "fold $lit1 [$val1] $op $lit2 [$val2] = $res $result_sign $((res:>Literal.Integer).is_negative)");
        CheckInteger (res).WithProperType () :> Literal.Integer
      }
      else null
    }

    MaxValueMask (t : MType) : ulong
    {
      if (t.Equals (InternalType.SByte))
        0xff : ulong
      else if (t.Equals (InternalType.Byte))
        0xff : ulong
      else if (t.Equals (InternalType.Int16))
        0xffff : ulong
      else if (t.Equals (InternalType.UInt16))
        0xffff : ulong
      else if (t.Equals (InternalType.Int32))
        0xffffffffUL : ulong
      else if (t.Equals (InternalType.UInt32))
        0xffffffffUL : ulong
      else if (t.Equals (InternalType.Int64))
        0xffffffffffffffffUL : ulong
      else if (t.Equals (InternalType.UInt64))
        0xffffffffffffffffUL : ulong
      else assert (false)
    }

    MaxSignedValueMask (t : MType) : ulong
    {
      if (t.Equals (InternalType.SByte))
        0x7f : ulong
      else if (t.Equals (InternalType.Byte))
        0xff : ulong
      else if (t.Equals (InternalType.Int16))
        0x7fff : ulong
      else if (t.Equals (InternalType.UInt16))
        0xffff : ulong
      else if (t.Equals (InternalType.Int32))
        0x7fffffffUL : ulong
      else if (t.Equals (InternalType.UInt32))
        0xffffffffUL : ulong
      else if (t.Equals (InternalType.Int64))
        0x7fffffffffffffffUL : ulong
      else if (t.Equals (InternalType.UInt64))
        0xffffffffffffffffUL : ulong
      else assert (false)
    }

    ConvertSignTo (val : ulong, t : MType.Class) : Literal.Integer
    {
      unchecked {
        mutable newval = 0 : long;
        newval =
          if (t.Equals (InternalType.SByte))
            (val :> sbyte) :> long
          else if (t.Equals (InternalType.Int16))
            (val :> short) :> long
          else if (t.Equals (InternalType.Int32))
            (val :> int) :> long
          else if (t.Equals (InternalType.Int64))
            (val :> long)
          else
            assert (false);

        if (newval == long.MinValue)
          Literal.Integer (0x8000000000000000UL, true, t)
        else
          Literal.Integer (System.Math.Abs (newval) :> ulong, newval < 0, t)
      }
    }

    
    internal FoldLiteral (_is_checked : bool, op : string, lit : Literal.Integer) : Literal.Integer
    {
      // Message.Debug ($ "fold: $op $lit");
      match (op) {
        | "-" =>
          CheckInteger (Literal.Integer (lit.val, ! lit.is_negative, null).WithProperType () 
                        :> Literal.Integer)

        | "~" =>
          def t = lit.treat_as;
          def v = lit.val;
          if (MaxSignedValueMask (t) == MaxValueMask (t)) {
              assert (! lit.is_negative);
              CheckInteger (Literal.Integer (~v & MaxValueMask (t), false, t))
          } else {
            def v' =
              if (lit.is_negative) unchecked (v - 1)
              else unchecked (v + 1);
            CheckInteger (Literal.Integer (v' & MaxSignedValueMask (t), ! lit.is_negative, t))
          }

        | "+" => lit
        | _ => null
      }
    }
    
    emit_flag_warning (t : TypeInfo) : void
    {
      when ((t.GetTydecl () is TT.TypeDeclaration.Enum) && 
            ! t.HasAttribute (InternalType.FlagsAttribute_tc))
        Message.Warning (10004, $ "using bitwise operator on enum type `$(t)' "
                           "that hasn't got [Flags] attribute");
    }


    internal FieldValueAsLiteral (field : IField) : option [Literal]
    {
      def decl_type = field.DeclaringType;
      decl_type.HasBeenUsed = true;
      def enum_ty =
        match (decl_type.GetTydecl ()) {
          | TT.TypeDeclaration.Enum => Some (decl_type)
          | _ => None ()
        };
      def lit = field.GetValue ();
      field.HasBeenUsed = true;
      match (enum_ty) {
        | Some (tc) when ! (lit is Literal.Enum) => 
          Some (Literal.Enum (lit :> Literal.Integer, tc))
        | _ => Some (lit)
      }
    }

    internal FieldValueAsPureLiteral (field : IField) : Literal
    {
      def decl_type = field.DeclaringType;
      decl_type.HasBeenUsed = true;
      field.HasBeenUsed = true;
      field.GetValue ()
    }
    
    literal_field_value (env : GlobalEnv, expr : PExpr) : PExpr
    {
      match (Util.QidOfExpr (expr)) {
        | Some ((id, name)) =>
          match (name.GetEnv (env).LookupSymbol (id, null)) {
            | [fld is IField] when fld.IsLiteral =>
              match (FieldValueAsLiteral (fld)) {
                | None => expr
                | Some (lit) => PExpr.Literal (expr.loc, lit)
              }
            | _ => expr
          }
        | None => expr
      }
    }

    get_literal (expr : TT.TExpr) : Literal
    {
      | TT.TExpr.Literal (lit) =>
        // we are not interested in other literals
        match (lit) {
          | Literal.Integer
          | Literal.String => lit
          | _ => null
        }
      | _ => null
    }

    
    literal_of_expr (e : PExpr) : option [Literal]
    {
       | PExpr.Literal (l) => Some (l)
       | PExpr.Typed (TT.TExpr.Literal (l)) => Some (l)
       | _ => None ()
    }
    
    fold_unary_operator (is_checked : bool, name : string, 
                         e1 : PExpr, expr : PExpr) : PExpr
    {
      match (literal_of_expr (e1)) {
        | Some (Literal.Enum (lit, enum_ty)) when name == "~" =>
          emit_flag_warning (enum_ty);
          def res = FoldLiteral (is_checked, name, lit);
          PExpr.Literal (expr.loc, Literal.Enum (res, enum_ty))

        | Some (Literal.Integer as lit) when is_known_operator (name) =>
          def res = FoldLiteral (is_checked, name, lit);
          PExpr.Literal (expr.loc, res)

        | _ => expr
      }
    }

    fold_binary_operator (is_checked : bool, name : string, e1 : PExpr, e2 : PExpr, 
                          expr : PExpr) : PExpr
    {
      // Message.Debug ($"$e1   $name   $e2    ($expr)");
      match ((literal_of_expr (e1), literal_of_expr (e2))) {
        | (Some (l1), Some (l2)) =>
          def (l1, l2, enum_ty) =
            match ((l1, l2)) {
              | (Literal.Enum (l1, t1), Literal.Enum (l2, t2)) 
                when t1.Equals (t2) =>
                (l1, l2, Some (t1))
              | _ => (l1, l2, None ())
            };
          def lit = FoldLiterals (is_checked, name, l1, l2);
          if (lit == null)
            expr
          else
            match ((enum_ty, name)) {
              | (None, _) => PExpr.Literal (expr.loc, lit)
                
              | (Some (t), "|")
              | (Some (t), "&")
              | (Some (t), "^")
              | (Some (t), "%|")
              | (Some (t), "%&")
              | (Some (t), "%^") =>
                emit_flag_warning (t);
                def lit = if (lit is Literal.Enum) lit else Literal.Enum (lit :> Literal.Integer, t);
                PExpr.Literal (expr.loc, lit)
              | (Some, _) =>
                // other operators not allowed on enums
                expr
            }
        | _ => expr
      }
    }
    
    fold_constants (is_checked : bool, env : GlobalEnv, expr : PExpr, recurse : bool) : PExpr
    {
      try {
        match (expr) {
          | <[ $(name : dyn) ($e1, $e2) ]> when is_known_operator (name) =>
            if (recurse)
              fold_binary_operator (is_checked,
                                    name, 
                                    fold_constants (is_checked, env, e1, true), 
                                    fold_constants (is_checked, env, e2, true), 
                                    expr)
            else
              fold_binary_operator (is_checked, name, e1, e2, expr)

          | <[ $(name : dyn) ($e1) ]> when is_known_operator (name) =>
            if (recurse)
              fold_unary_operator (is_checked, name, fold_constants (is_checked, env, e1, true), expr)
            else
              fold_unary_operator (is_checked, name, e1, expr)

          | <[ $_o . $_f ]>
          | <[ $(_ : name) ]> => 
            literal_field_value (env, expr)

          | _ => expr
        }
      }
      catch {
        | _ is System.DivideByZeroException =>
          Message.Error ("division by zero during constants folding");
          null
        
        | _ is System.OverflowException => // FIXME: use guards in catching
          if (is_checked) {
            Message.Error ("the operation overflows at compile-time during"
                           " constants folding in checked mode");
            null
          }
          else
            Util.ice ("overflow exception")
      }
    }

    perform_check ['a] (is_checked : bool, f : void -> 'a) : 'a where 'a : class
    {
      try {
        f ()
      }
      catch {
        | _ is System.DivideByZeroException =>
          Message.Error ("division by zero during constants folding");
          null
        
        | _ is System.OverflowException => // FIXME: use guards in catching
          if (is_checked) {
            Message.Error ("the operation overflows at compile-time during"
                           " constants folding in checked mode");
            null
          }
          else
            Util.ice ("overflow exception")
      }
    }
    
    
    public FoldTyped (name : string, p : TT.TExpr, is_checked : bool) : TT.TExpr
    {
      //Message.Debug ($"ft: $name $p");
      match (get_literal (p)) {
        | Literal.Integer as lit =>
          def res = perform_check (is_checked, fun () {
            FoldLiteral (is_checked, name, lit)
          });
          if (res == null)
            null
          else
            //Message.Debug ($"ft: re=$res");
            TT.TExpr.Literal (res.treat_as, res)

        | _ => 
          //Message.Debug ($"ft: null");
          null
      }
    } 

    public FoldTyped (name : string, p1 : TT.TExpr, p2 : TT.TExpr,
                      is_checked : bool) : TT.TExpr
    {
      match ((get_literal (p1), get_literal (p2))) {
        | (null, _) | (_, null) => null
        | (l1, l2) =>
          def res = perform_check (is_checked, fun () {
            FoldLiterals (is_checked, name, l1, l2)
          });
          if (res == null)
            null
          else {
            def ty =
              match (res) {
                | Literal.Integer (_, _, t) => t
                | _ => p1.Type
              }
            TT.TExpr.Literal (ty, res)
          }
      }
    } 
    
    public FoldConstants (env : GlobalEnv, expr : PExpr) : PExpr
    {
      fold_constants (false, env, expr, true)
    }
  }
}

