unit ApplicationDataModule;

interface

uses
  gnugettext, SysUtils, Classes, MyxError, Options, Forms, AuxFuncs, Dialogs,
  myx_public_interface, myx_util_public_interface, StrUtils, StdActns, TntStdActns, ActnList,
  TntActnList;

type
  TApplicationDM = class(TDataModule)
    procedure DataModuleCreate(Sender: TObject);
    procedure DataModuleDestroy(Sender: TObject);

    procedure CheckCommandlineParameter;

    procedure OnApplicationException(Sender: TObject; E: Exception);

    function GetLastFileDialogPaths(DlgName: WideString): WideString;
    procedure SetLastFileDialogPaths(DlgName: WideString; Path: WideString);
  private
    FOptions: IOptionProvider;
    FLastFileDialogPaths: TStringList;
  public
    property Options: IOptionProvider read FOptions;
  end;

  EMyxQueryBrowserLibError = class(EMyxLibraryError);

var
  ApplicationDM: TApplicationDM;

//----------------------------------------------------------------------------------------------------------------------

implementation

{$R *.dfm}

type
  TWorkbenchOptions = class(TMyxOptions)
  protected
    procedure SetFixedOptions; override;
    procedure SetDefaultOptions; override;
  public
    constructor Create; override;
    destructor Destroy; override;

    procedure LoadOptions; override;
    procedure StoreOptions; override;
  end;

//----------------------------------------------------------------------------------------------------------------------

procedure TApplicationDM.DataModuleCreate(Sender: TObject);

var
  s: WideString;
  
begin
  FLastFileDialogPaths := TStringList.Create;

  // Exception handling.
  Application.OnException := OnApplicationException;

  // DLL Version check.
  if (libmysqlx_PUBLIC_INTERFACE_VERSION <> myx_get_public_interface_version) then
    s := Format(_('There is a incompatible version of the ' +
      'library %s installed (Version %s). Please update the library to version %s.'),
      ['libmysqlx.dll', FormatLibraryVersion(myx_get_public_interface_version),
      FormatLibraryVersion(libmysqlx_PUBLIC_INTERFACE_VERSION)]) + #13#10#13#10;

  FOptions := TWorkbenchOptions.Create;

  if (S <> '') and (ShowModalDialog('Library version mismatch', Trim(s), myx_mtError, 'Quit' + #13#10 + 'Ignore') = 1) then
    Application.Terminate;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TApplicationDM.DataModuleDestroy(Sender: TObject);

begin
  // Release reference so the options get stored.
  FOptions := nil;
  FLastFileDialogPaths.Free;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TApplicationDM.CheckCommandlineParameter;

var
  I: integer;
  
begin
  I := 1;
  while I <= ParamCount do
  begin
    if WideSameText(ParamStr(I), '-verbose') then
      FOptions.OptionAsBoolean['Verbose'] := True
    else
      if WideSameText(ParamStr(I), '-debug') then
        FOptions.OptionAsBoolean['RemoteDebug'] := True
      else
        if (WideSameText(ParamStr(I), '-jvm')) and (I < ParamCount) then
        begin
          FOptions.OptionAsString['JvmLibrary'] := ParamStr(I + 1);
          Inc(I);
        end
        else
          if (WideSameText(ParamStr(I), '-script')) and (I < ParamCount) then
          begin
            FOptions.OptionAsString['StartScriptFile'] := ParamStr(I + 1);
            inc(i);
          end
          else
            if (WideSameText(ParamStr(I), '-initscript')) and (I < ParamCount) then
            begin
              FOptions.OptionAsString['InitScriptFile'] := ParamStr(I + 1);
              Inc(I);
            end
            else
              FOptions.OptionAsString['StartModelFile'] := ParamStr(I);

    inc(i);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TApplicationDM.OnApplicationException(Sender: TObject; E: Exception);

begin
  if not (Application.Terminated) then
    ShowError(E);
end;

//----------------------------------------------------------------------------------------------------------------------

function TApplicationDM.GetLastFileDialogPaths(DlgName: WideString): WideString;

begin
  Result := FLastFileDialogPaths.Values['DlgName'];
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TApplicationDM.SetLastFileDialogPaths(DlgName: WideString; Path: WideString);

begin
  FLastFileDialogPaths.Values['DlgName'] := Path;
end;

//----------------- TWorkbenchOptions ----------------------------------------------------------------------------------

constructor TWorkbenchOptions.Create;

begin
  inherited Create;
end;

//----------------------------------------------------------------------------------------------------------------------

destructor TWorkbenchOptions.Destroy;

begin
  inherited Destroy;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchOptions.LoadOptions;

var
  POptions: PMYX_APPLICATION_OPTIONS;
  Options: TMYX_APPLICATION_OPTIONS;
  i, j: integer;
  error: MYX_LIB_ERROR;
  OptionGroupName, OptionName, OptionValue: WideString;
  ExePath: WideString;
  Provider: IOptionProvider;
  
begin
  // Access this class via the interface to have access to the getter and setter methods for options.
  Provider := Self;
  
  ExePath := ExtractFilePath(Application.ExeName);

  // Initialize Values
  ApplicationDM.FLastFileDialogPaths.Text := '';

  Provider.OptionAsInteger['SideBarWidth'] := 230;
  Provider.OptionAsBoolean['CreateWindowsStyleLineBreaks'] := True;
  Provider.OptionAsBoolean['UseToolbarGradient'] := True;

  // Read options file.
  if FileExists(MYXCommonOptions.UserDataDir + 'mysqlwb_options.xml') then
  begin
    POptions := myx_get_application_options(MYXCommonOptions.UserDataDir + 'mysqlwb_options.xml', @error);
    try
      if error <> MYX_NO_ERROR then
      begin
        ShowModalDialog('XML Error', 'Error while loading Options file ''' +
          MYXCommonOptions.UserDataDir + 'mysqlwb_options.xml' + ''''#13#10 +
          'Error Number ' + IntToStr(Ord(error)),
          myx_mtError);
      end
      else
      begin
        Options := TMYX_APPLICATION_OPTIONS.Create(POptions);
        try
          for i := 0 to Options.option_groups.Count - 1 do
            for j := 0 to Options.option_groups[i].name_value_pairs.Count - 1 do
            begin
              OptionGroupName := Options.option_groups[i].name;
              OptionName := Options.option_groups[i].name_value_pairs[j].name;
              OptionValue := Options.option_groups[i].name_value_pairs[j].value;

              Provider.OptionAsString[OptionName] := OptionValue;
              if SameText(OptionGroupName, 'General') then
              begin
                if SameText(OptionName, 'LastFileDialogPaths') then
                  ApplicationDM.FLastFileDialogPaths.Text := AnsiReplaceText(OptionValue, '', #13#10);
              end
              else
                if SameText(OptionGroupName, 'WindowPos') then
                  WindowPosList.AddObject(OptionName, TMyxWindowPos.Create(OptionValue));
            end;
        finally
          Options.Free;
        end;
      end;
    finally
      //Free Application Options
      myx_free_application_options(POptions);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchOptions.SetDefaultOptions;

begin
  inherited;

end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchOptions.SetFixedOptions;

begin
  inherited;

end;

//----------------------------------------------------------------------------------------------------------------------

procedure TWorkbenchOptions.StoreOptions;

var
  Options: TMYX_APPLICATION_OPTIONS;
  OptionGroup: TMYX_OPTION_GROUP;
  ExePath: WideString;
  I: integer;

begin
  if MYXCommonOptions.UserDataDir <> '' then
  begin
    ExePath := ExtractFilePath(Application.ExeName);

    //Create Application Options
    Options := TMYX_APPLICATION_OPTIONS.create;
    try
      //-----------------------------------------------
      //Create Option Group
      OptionGroup := TMYX_OPTION_GROUP.create('General');
      Options.option_groups.Add(OptionGroup);

      AddParam(OptionGroup, 'LastFileDialogPaths', AnsiReplaceText(ApplicationDM.FLastFileDialogPaths.Text, #13#10, ''));

      // Window positions option group.
      OptionGroup := TMYX_OPTION_GROUP.create('WindowPos');
      Options.option_groups.Add(OptionGroup);

      //Store all window positions
      for I := 0 to WindowPosList.Count - 1 do
        AddParam(OptionGroup, WindowPosList[i],
          TMyxWindowPos(WindowPosList.Objects[i]).AsWideString);

      // Convert all options (stored only in the list) to the needed name-value-pairs.
      StoreListOptions(Options.option_groups);

      //-----------------------------------------------
      //Store Options to file
      myx_store_application_options(Options.get_record_pointer,
        MYXCommonOptions.UserDataDir + 'mysqlwb_options.xml');
    finally
      Options.Free;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

end.

