unit ModelPrintSetup;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ComCtrls, TntComCtrls, StdCtrls, TntStdCtrls, Buttons,
  TntButtons, ExtCtrls, TntExtCtrls, TntForms, MyxBaseForm, Contnrs;

const
  MaxPreviewSize = 2000;
  
type
  // A set of values that comprise one preset.
  TPrintPreset = class
  private
    FName: WideString;            // The name of the preset.
    FDeviceName: string;          // The name of the printer device.
    FHPagesCount: Integer;        // Number of horizontal pages for printing.
    FVPagesCount: Integer;        // Number of vertical pages for printing.
    FPrinterHandle: THandle;
    FDeviceMode: PDeviceMode;
    FDeviceModeSize: Integer;
    function GetPrinterProperties: WideString;
    procedure SetPrinterProperties(const Value: WideString);
  public
    constructor Create(Name: WideString; Device: string; HCount, VCount: Integer); virtual;
    destructor Destroy; override;

    procedure Activate;
    property PrinterProperties: WideString read GetPrinterProperties write SetPrinterProperties;
  end;

  // A user defined list of printer presets.
  TPresetList = class(TObjectList)
  private
    FStoredSettings: array of PDeviceMode;
    function GetPreset(Index: Integer): TPrintPreset;
    procedure SetPreset(Index: Integer; const Value: TPrintPreset);
  public
    constructor Create; virtual;
    destructor Destroy; override;

    procedure BuildPresetList(const Source: WideString);
    procedure PresetAdd(Name: WideString; Device: string; HCount, VCount: Integer; Properties: WideString);
    procedure PresetRemove(Index: Integer);
    function StorePresetList: WideString;

    property Items[Index: Integer]: TPrintPreset read GetPreset write SetPreset; default;
  end;
  
  TPageSelection = array of Boolean;

  TModelPrintSetupForm = class(TMyxBaseForm)
    TntGroupBox1: TTntGroupBox;
    TntLabel1: TTntLabel;
    PrintPresetComboBox: TTntComboBox;
    TntLabel2: TTntLabel;
    PrinterComboBox: TTntComboBox;
    CancelButton: TTntButton;
    TntButton1: TTntButton;
    SettingsPageControl: TTntPageControl;
    PreviewTabSheet: TTntTabSheet;
    ModelPaintBox: TTntPaintBox;
    HPagesTrackBar: TTntTrackBar;
    TntTabSheet2: TTntTabSheet;
    TntCheckBox1: TTntCheckBox;
    PrinterSetupButton: TTntButton;
    AddPresetButton: TTntSpeedButton;
    RemovePresetButton: TTntSpeedButton;
    TntGroupBox2: TTntGroupBox;
    HPagesEdit: TTntEdit;
    HPagesCountUpDown: TTntUpDown;
    TntLabel4: TTntLabel;
    TntLabel5: TTntLabel;
    VPagesEdit: TTntEdit;
    VPagesCountUpDown: TTntUpDown;
    VPagesTrackBar: TTntTrackBar;
    SetupDialog: TPrinterSetupDialog;
    procedure PrinterSetupButtonClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure PrintPresetComboBoxChange(Sender: TObject);
    procedure PagesEditChange(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure AddPresetButtonClick(Sender: TObject);
    procedure RemovePresetButtonClick(Sender: TObject);
    procedure ModelPaintBoxPaint(Sender: TObject);
    procedure HPagesTrackBarChange(Sender: TObject);
    procedure VPagesTrackBarChange(Sender: TObject);
    procedure PreviewTabSheetResize(Sender: TObject);
    procedure PrinterComboBoxChange(Sender: TObject);
    procedure ModelPaintBoxMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure HPagesCountUpDownChanging(Sender: TObject;
      var AllowChange: Boolean);
    procedure VPagesCountUpDownChanging(Sender: TObject;
      var AllowChange: Boolean);
  private
    FPresets: TPresetList;
    FModelImage: TBitmap;
    FBackImage: TBitmap;
    FSelection: TPageSelection;
  public
    procedure PresetsLoad(const Source: WideString);
    function PresetsSerialize: WideString;
    procedure PreviewUse(Preview: TBitmap);
    procedure PrintParametersGet(var HCount, VCount: Integer; var Selection: TPageSelection);
  end;

var
  ModelPrintSetupForm: TModelPrintSetupForm;

//----------------------------------------------------------------------------------------------------------------------

implementation

{$R *.dfm}

uses
  Printers, WinSpool, Math, LexicalTools, GraphicTools;

const
  // These separator values should never appear in a preset name.
  PresetSeparator: WideChar = #$273A;
  PresetValueSeparator: WideChar = #$273D;
  
//----------------- TPrintPreset ---------------------------------------------------------------------------------------

constructor TPrintPreset.Create(Name: WideString; Device: string;  HCount, VCount: Integer);

var
  Dummy: TDeviceMode;
  
begin
  inherited Create;

  FName := Name;
  FDeviceName := Device;
  FHPagesCount := HCount;
  FVPagesCount := VCount;

  // Get all available info for the printer. No error checking takes place here because
  // we only use printers that were enumerated by the Printer function. These
  // printer are already checked, so we can safely assume they are ok.
  OpenPrinter(PChar(FDeviceName), FPrinterHandle, nil);
  FDeviceModeSize := DocumentProperties(0, FPrinterHandle, PChar(FDeviceName), Dummy, Dummy, 0);
  FDeviceMode := AllocMem(FDeviceModeSize);
  if DocumentProperties(0, FPrinterHandle, PChar(FDeviceName), FDeviceMode^, FDeviceMode^, DM_OUT_BUFFER) < 0 then
  begin
    FreeMem(FDeviceMode);
    FDeviceMode := nil;
    FDeviceModeSize := 0;
  end
end;

//----------------------------------------------------------------------------------------------------------------------

destructor TPrintPreset.Destroy;

begin
  if Assigned(FDeviceMode) then
    FreeMem(FDeviceMode);

  ClosePrinter(FPrinterHandle);
  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

function TPrintPreset.GetPrinterProperties: WideString;

// Converts the device mode structure into a string of hex numbers.

begin
  if Assigned(FDeviceMode) then
  begin
    SetLength(Result, 2 * FDeviceModeSize);
    BinToHex(FDeviceMode, PWideChar(Result), FDeviceModeSize);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TPrintPreset.SetPrinterProperties(const Value: WideString);

// Converts the given string, which must consist of hex numbers into a device mode structure and applies this to
// the printer.

begin
  if Assigned(FDeviceMode) then
    HexToBin(PWideChar(Value), FDeviceMode, Min(FDeviceModeSize, Length(Value) div 2));
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TPrintPreset.Activate;

// Activates the given preset by setting the printer properties as given in the device mode structure.

begin
  if Assigned(FDeviceMode) then
    DocumentProperties(0, FPrinterHandle, PChar(FDeviceName), FDeviceMode^, FDeviceMode^, DM_OUT_BUFFER or DM_OUT_BUFFER);
end;

//----------------- TPresetList ----------------------------------------------------------------------------------------

constructor TPresetList.Create;

var
  I: Integer;
  Handle: THandle;
  DeviceMode: PDeviceMode;
  Dummy: TDeviceMode;
  Size: Integer;
  Device: PChar;
  PrinterDefaults: TPrinterDefaults;

begin
  inherited Create(True);                                      

  // Capture current printer settings.
  SetLength(FStoredSettings, Printer.Printers.Count);
  for I := 0 to Length(FStoredSettings) - 1 do
  begin
    Device := PChar(Printer.Printers[I]);
    PrinterDefaults.pDataType := nil;
    PrinterDefaults.pDevMode := nil;
    PrinterDefaults.DesiredAccess := PRINTER_ALL_ACCESS;

    if OpenPrinter(Device, Handle, @PrinterDefaults) then
    begin
      Size := DocumentProperties(0, Handle, Device, Dummy, Dummy, 0);
      if Size > 0 then
      begin
        DeviceMode := AllocMem(Size);
        if DocumentProperties(0, Handle, PChar(Device), DeviceMode^, DeviceMode^, DM_OUT_BUFFER) >= 0 then
          FStoredSettings[I] := DeviceMode;
      end;
      ClosePrinter(Handle);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

destructor TPresetList.Destroy;

var
  I: Integer;
  Handle: THandle;
  Device: PChar;
  PrinterDefaults: TPrinterDefaults;

begin
  // Restore previous printer settings.
  for I := 0 to Length(FStoredSettings) - 1 do
    if Assigned(FStoredSettings[I]) then
    begin
      Device := PChar(Printer.Printers[I]);
      PrinterDefaults.pDataType := nil;
      PrinterDefaults.pDevMode := nil;
      PrinterDefaults.DesiredAccess := PRINTER_ALL_ACCESS;
      if OpenPrinter(Device, Handle, @PrinterDefaults) then
      begin
        DocumentProperties(0, Handle, Device, FStoredSettings[I]^, FStoredSettings[I]^,
          DM_OUT_BUFFER or DM_IN_BUFFER);
        FreeMem(FStoredSettings[I]);
        ClosePrinter(Handle);
      end;
    end;

  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

function TPresetList.GetPreset(Index: Integer): TPrintPreset;

begin
  Result := inherited Items[Index] as TPrintPreset;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TPresetList.BuildPresetList(const Source: WideString);

// Fills the list from a previously saved list.

var
  EntryHead: PWideChar;
  EntryTail: PWideChar;
  ValueHead: PWideChar;
  ValueTail: PWideChar;
  Name: WideString;
  PrinterName: WideString;
  S: WideString;
  HCount: Integer;
  VCount: Integer;
  Properties: WideString;
  
begin
  // Clear current list and add the default entry.
  Clear;
  PresetAdd('<default>', Printer.Printers[Printer.PrinterIndex], 2, 2, '');

  EntryHead := PWideChar(Source);
  while EntryHead^ <> #0 do
  begin
    EntryTail := EntryHead;
    while (EntryTail^ <> #0) and (EntryTail^ <> PresetSeparator) do
      Inc(EntryTail);

    ValueHead := EntryHead;
    ValueTail := ValueHead;
    while (ValueTail^ <> PresetValueSeparator) and (ValueTail <> EntryTail) do
      Inc(ValueTail);
    SetString(Name, ValueHead, ValueTail - ValueHead);

    if Name <> '' then
    begin
      // Ok, we have a valid preset name here. So fill in some default values and then look what the source gives us.
      // Finally add a new preset entry.
      HCount := 2;
      VCount := 2;

      if ValueTail <> EntryTail then
      begin
        ValueHead := ValueTail + 1;
        ValueTail := ValueHead;
        while (ValueTail^ <> PresetValueSeparator) and (ValueTail <> EntryTail) do
          Inc(ValueTail);
        SetString(PrinterName, ValueHead, ValueTail - ValueHead);

        if ValueTail <> EntryTail then
        begin
          ValueHead := ValueTail + 1;
          ValueTail := ValueHead;
          while (ValueTail^ <> PresetValueSeparator) and (ValueTail <> EntryTail) do
            Inc(ValueTail);
          SetString(S, ValueHead, ValueTail - ValueHead);
          HCount := StrToIntDef(S, 2); // String will be converted to ANSI here, but that's ok. We only have a number.

          if ValueTail <> EntryTail then
          begin
            ValueHead := ValueTail + 1;
            ValueTail := ValueHead;
            while (ValueTail^ <> PresetValueSeparator) and (ValueTail <> EntryTail) do
              Inc(ValueTail);
            SetString(S, ValueHead, ValueTail - ValueHead);
            VCount := StrToIntDef(S, 2); // Conversion as for HCount.

            if ValueTail <> EntryTail then
            begin
              ValueHead := ValueTail + 1;
              ValueTail := ValueHead;
              while (ValueTail^ <> PresetValueSeparator) and (ValueTail <> EntryTail) do
                Inc(ValueTail);
              SetString(Properties, ValueHead, ValueTail - ValueHead);
            end;
          end;
        end;
      end;
      PresetAdd(Name, PrinterName, HCount, VCount, Properties);
    end;

    EntryHead := EntryTail;
    if EntryHead^ <> #0 then
      Inc(EntryHead);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TPresetList.PresetAdd(Name: WideString; Device: string; HCount, VCount: Integer; Properties: WideString);

// Add a new preset to the list.

var
  Preset: TPrintPreset;

begin
  Preset := TPrintPreset.Create(Name, Device, HCount, VCount);
  if Properties <> '' then
    Preset.PrinterProperties := Properties;
  Add(Preset);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TPresetList.PresetRemove(Index: Integer);

// Remove the entry with the given index from list.

begin
  Delete(Index);
end;

//----------------------------------------------------------------------------------------------------------------------

function TPresetList.StorePresetList: WideString;

// Streams the current preset list into a string.
// Note: the default entry is not streamed.

var
  I: Integer;
  S: WideString;
  Preset: TPrintPreset;
  
begin
  Result := '';
  for I := 1 to Count - 1 do
  begin
    Preset := Self[I];
    S := WideFormat('%1:s%0:s%2:s%0:s%3:d%0:s%4:d%0:s', [PresetValueSeparator, Preset.FName, Preset.FDeviceName,
      Preset.FHPagesCount, Preset.FVPagesCount]);
    Result := Result + S + Preset.PrinterProperties + PresetSeparator;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TPresetList.SetPreset(Index: Integer; const Value: TPrintPreset);

begin
  inherited Items[Index] := Value;
end;

//----------------- TModelPrintForm ------------------------------------------------------------------------------------

procedure TModelPrintSetupForm.PrinterSetupButtonClick(Sender: TObject);

begin
  Printer.PrinterIndex := PrinterComboBox.ItemIndex;
  if SetupDialog.Execute then
    ModelPaintBox.Refresh;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TModelPrintSetupForm.FormCreate(Sender: TObject);

begin
  FPresets := TPresetList.Create;

  // Initialize list of printers.
  PrinterComboBox.Items.Assign(Printer.Printers);
  PrinterComboBox.ItemIndex := Printer.PrinterIndex;

  ModelPaintBox.ControlStyle := ModelPaintBox.ControlStyle + [csOpaque];
  SetLength(FSelection, VPagesCountUpDown.Position * HPagesCountUpDown.Position);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TModelPrintSetupForm.FormDestroy(Sender: TObject);

begin
  FPresets.Free;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TModelPrintSetupForm.PrintPresetComboBoxChange(Sender: TObject);

var
  Index: Integer;
  Preset: TPrintPreset;
  
begin
  // Try locating the given preset name in the list. Further actions depend on whether we find it or not.
  Index := PrintPresetComboBox.Items.IndexOf(PrintPresetComboBox.Text);

  if Index < 0 then
  begin
    // An unknown name. Allow adding a new entry.
    AddPresetButton.Enabled := Trim(PrintPresetComboBox.Text) <> '';
    RemovePresetButton.Enabled := False;
  end
  else
  begin
    AddPresetButton.Enabled := False;

    // Don't allow deleting the default entry.
    RemovePresetButton.Enabled := Index > 0;

    Preset := FPresets[Index];
    Printer.PrinterIndex := Printer.Printers.IndexOf(Preset.FDeviceName);
    Preset.Activate;

    // Must be retrieved from Printer, as -1 as default printer is translated to the real printer index by this class.
    PrinterComboBox.ItemIndex := Printer.PrinterIndex;
    HPagesCountUpDown.Position := Preset.FHPagesCount;
    VPagesCountUpDown.Position := Preset.FVPagesCount;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TModelPrintSetupForm.PagesEditChange(Sender: TObject);

// Validate the page specifiers.

var
  Value: Integer;

begin
  with Sender as TTntEdit do
    if Text <> '' then
    begin
      Value := StrToIntDef(Text, 1);
      if Sender = HPagesEdit then
        HPagesTrackBar.Position := Value
      else
        VPagesTrackBar.Position := Value;
    end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TModelPrintSetupForm.PresetsLoad(const Source: WideString);

// Loads a list of presets from the given source string.

var
  I: Integer;
  
begin
  FPresets.BuildPresetList(Source);

  PrintPresetComboBox.Clear;
  for I := 0 to FPresets.Count - 1 do
    PrintPresetComboBox.AddItem(FPresets[I].FName, nil);

  // There is always one entry (the default entry).
  PrintPresetComboBox.ItemIndex := 0;
  FPresets[0].Activate;
end;

//----------------------------------------------------------------------------------------------------------------------

function TModelPrintSetupForm.PresetsSerialize: WideString;

begin
  Result := FPresets.StorePresetList;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TModelPrintSetupForm.AddPresetButtonClick(Sender: TObject);

begin
  FPresets.PresetAdd(PrintPresetComboBox.Text, Printer.Printers[PrinterComboBox.ItemIndex], HPagesCountUpDown.Position,
    VPagesCountUpDown.Position, '');
  PrintPresetComboBox.Items.Add(PrintPresetComboBox.Text);
  PrintPresetComboBox.ItemIndex := PrintPresetComboBox.Items.Count - 1;
  AddPresetButton.Enabled := False;
  RemovePresetButton.Enabled := True;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TModelPrintSetupForm.RemovePresetButtonClick(Sender: TObject);

var
  LastIndex: Integer;

begin
  FPresets.PresetRemove(PrintPresetComboBox.ItemIndex);
  LastIndex := PrintPresetComboBox.ItemIndex;
  PrintPresetComboBox.Items.Delete(LastIndex);
  PrintPresetComboBox.ItemIndex := LastIndex - 1;
  PrintPresetComboBoxChange(nil);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TModelPrintSetupForm.ModelPaintBoxPaint(Sender: TObject);

var
  ScaleX: Single;
  ScaleY: Single;
  HOffset: Single;                // The horizontal distance of the vertical lines.
  VOffset: Single;                // The vertical distance of the horizontal lines.
  GridWidth: Integer;             // The width of the canvas covered by printer pages (in paint box coordinates).
  GridHeight: Integer;            // Dito for the height.
  PrintWidth: Integer;            // The entire print width in printer coordinates.
  PrintHeight: Integer;           // Dito for the height.
  X: Single;                      // Full precision coordinate.
  Y: Single;
  XInt: Integer;                  // Rounded coordinates
  YInt: Integer;
  Page: Integer;

begin
  if FBackImage = nil then
  begin
    FBackImage := TBitmap.Create;
    FBackImage.PixelFormat := pf32Bit;
  end;

  if (FBackImage.Width <> ModelPaintBox.Width) or (FBackImage.Height <> ModelPaintBox.Height) then
  begin
    // Resize back image to the same size as the paint box.
    FBackImage.Height := 0;
    FBackImage.Width := ModelPaintBox.Width;
    FBackImage.Height := ModelPaintBox.Height;
  end;

  if Assigned(FModelImage) then
    FBackImage.Canvas.StretchDraw(ModelPaintBox.ClientRect, FModelImage);

  PrintWidth := HPagesCountUpDown.Position * Printer.PageWidth;
  PrintHeight := VPagesCountUpDown.Position * Printer.PageHeight;
  ScaleX := ModelPaintBox.Width / PrintWidth;
  ScaleY := ModelPaintBox.Height / PrintHeight;
  if ScaleX < ScaleY then
  begin
    GridWidth := ModelPaintBox.Width - 1;
    GridHeight := Round(ScaleX * PrintHeight) - 1;
  end
  else
  begin
    GridWidth := Round(ScaleY * PrintWidth) - 1;
    GridHeight := ModelPaintBox.Height - 1;
  end;

  HOffset := GridWidth / HPagesCountUpDown.Position;
  VOffset := GridHeight / VPagesCountUpDown.Position;

  with FBackImage.Canvas do
  begin
    Brush.Color := $808080;
    FrameRect(Rect(0, 0, FBackImage.Width, FBackImage.Height));

    // For the text background.
    Brush.Color := $E0E0E0;

    Pen.Color := $A0A0A0;
    Pen.Style := psDot;
    if HOffset > 0 then
    begin
      X := HOffset;
      XInt := Round(HOffset);
      while XInt < GridWidth do
      begin
        MoveTo(XInt, 0);
        LineTo(XInt, GridHeight);
        X := X + HOffset;
        XInt := Round(X);
      end;
    end;

    if VOffset > 0 then
    begin
      Font.Name := 'Small';
      Font.Size := 7;
      Page := 1;
      Y := 0;
      YInt := 0;
      while YInt < GridHeight do
      begin
        if Y > 0 then
        begin
          MoveTo(0, YInt);
          LineTo(GridWidth, YInt);
        end;
        
        // Draw page numbers here.
        X := 3;
        XInt := 3;
        while XInt < GridWidth do
        begin
          TextOut(XInt, YInt + 3, IntToStr(Page));
          if FSelection[Page - 1] then
            AlphaBlendPixel(0, Handle, Rect(XInt - 3, YInt, Round(X + HOffset - 3), Round(Y + VOffset)), Point(0, 0),
              bmConstantAlphaAndColor, 64, $A09080);

          Inc(Page);
          X := X + HOffset;
          XInt := Round(X);
        end;

        Y := Y + VOffset;
        YInt := Round(Y);
      end;
    end;
    Brush.Color := $FFA000;
    FrameRect(Rect(0, 0, GridWidth + 1, GridHeight + 1));
  end;

  ModelPaintBox.Canvas.Draw(0, 0, FBackImage);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TModelPrintSetupForm.PreviewUse(Preview: TBitmap);

begin
  FModelImage := Preview;
  Invalidate;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TModelPrintSetupForm.HPagesTrackBarChange(Sender: TObject);

var
  NewSize: Integer;

begin
  HPagesCountUpDown.Position := HPagesTrackBar.Position;
  NewSize := VPagesCountUpDown.Position * HPagesTrackBar.Position;
  if NewSize <> Length(FSelection) then
    SetLength(FSelection, NewSize);
  ModelPaintBox.Invalidate;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TModelPrintSetupForm.VPagesTrackBarChange(Sender: TObject);

var
  NewSize: Integer;

begin
  VPagesCountUpDown.Position := VPagesTrackBar.Position;
  NewSize := VPagesCountUpDown.Position * HPagesTrackBar.Position;
  if NewSize <> Length(FSelection) then
    SetLength(FSelection, NewSize);
  ModelPaintBox.Invalidate;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TModelPrintSetupForm.VPagesCountUpDownChanging(Sender: TObject; var AllowChange: Boolean);

begin
  VPagesTrackBar.Position := VPagesCountUpDown.Position;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TModelPrintSetupForm.HPagesCountUpDownChanging(Sender: TObject; var AllowChange: Boolean);

begin
  HPagesTrackBar.Position := HPagesCountUpDown.Position;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TModelPrintSetupForm.PreviewTabSheetResize(Sender: TObject);

// Computes maximum area for the model paint box. Ensures the aspect ratio is kept constant.

const
  Border = 5;
  
var
  SheetRect: TRect;
  ScaleX: Single;
  ScaleY: Single;

begin
  if Assigned(FModelImage) and (PreviewTabSheet.Width > 2 * Border) and (PreviewTabSheet.Height > 2 * Border) then
  begin
    SheetRect := PreviewTabSheet.ClientRect;
    InflateRect(SheetRect, -Border, -Border);
    Dec(SheetRect.Right, VPagesTrackBar.Width);
    Dec(SheetRect.Bottom, HPagesTrackBar.Height);

    ScaleX := (SheetRect.Right - SheetRect.Left) / FModelImage.Width;
    ScaleY := (SheetRect.Bottom - SheetRect.Top) / FModelImage.Height;

    // The lesser scale factor determines the overall scale factor to make the whole image visible.
    if ScaleX < ScaleY then
    begin
      ModelPaintBox.Width := Round(ScaleX * FModelImage.Width);
      ModelPaintBox.Height := Round(ScaleX * FModelImage.Height);

      ModelPaintBox.Top := Border + (SheetRect.Bottom - SheetRect.Top - ModelPaintBox.Height) div 2;
      ModelPaintBox.Left := Border;
    end
    else
    begin
      ModelPaintBox.Width := Round(ScaleY * FModelImage.Width);
      ModelPaintBox.Height := Round(ScaleY * FModelImage.Height);

      ModelPaintBox.Top := Border;
      ModelPaintBox.Left := Border + (SheetRect.Right - SheetRect.Left - ModelPaintBox.Width) div 2;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TModelPrintSetupForm.PrinterComboBoxChange(Sender: TObject);

begin
  ModelPaintBox.Refresh;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TModelPrintSetupForm.PrintParametersGet(var HCount, VCount: Integer; var Selection: TPageSelection);

begin
  HCount := HPagesCountUpDown.Position;
  VCount := VPagesCountUpDown.Position;
  Selection := FSelection;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TModelPrintSetupForm.ModelPaintBoxMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X,
  Y: Integer);

var
  Page: Integer;
  ScaleX: Single;
  ScaleY: Single;
  HOffset: Single;                // The horizontal distance of the vertical lines.
  VOffset: Single;                // The vertical distance of the horizontal lines.
  GridWidth: Integer;             // The width of the canvas covered by printer pages (in paint box coordinates).
  GridHeight: Integer;            // Dito for the height.
  PrintWidth: Integer;            // The entire print width in printer coordinates.
  PrintHeight: Integer;           // Dito for the height.
  
begin
  PrintWidth := HPagesCountUpDown.Position * Printer.PageWidth;
  PrintHeight := VPagesCountUpDown.Position * Printer.PageHeight;
  ScaleX := ModelPaintBox.Width / PrintWidth;
  ScaleY := ModelPaintBox.Height / PrintHeight;
  if ScaleX < ScaleY then
  begin
    GridWidth := ModelPaintBox.Width - 1;
    GridHeight := Round(ScaleX * PrintHeight) - 1;
  end
  else
  begin
    GridWidth := Round(ScaleY * PrintWidth) - 1;
    GridHeight := ModelPaintBox.Height - 1;
  end;

  HOffset := GridWidth / HPagesCountUpDown.Position;
  VOffset := GridHeight / VPagesCountUpDown.Position;

  if (X <= GridWidth) and (Y <= GridHeight) then
  begin
    Page := X div Round(HOffset) + (Y div Round(VOffset)) * HPagesCountUpDown.Position;
    FSelection[Page] := not FSelection[Page];
    ModelPaintBox.Invalidate;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

end.
