# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2007-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>

"""
The implementation of a merging search engine for elisa.
"""

from elisa.core import common
from elisa.core.components.resource_provider import ResourceProvider
from elisa.core.plugin_registry import PluginStatusMessage

from elisa.core.utils import defer
from elisa.core.utils.cancellable_defer import cancellable_deferred_iterator

from elisa.plugins.search.models import BaseSearchResultModel

from twisted.internet import task


class SearchMetaresourceProvider(ResourceProvider):

    # the entry point to find the 'Searchers'
    entry_point = 'elisa.plugins.search.searchers'

    supported_uri = '^elisa://search/.*'
    
    default_config = {'default_searcher' : 'DBSearcher'}
    config_doc = {'default_searcher':
                  'The default searcher is the searcher that is always ' \
                  'asked first for the search result and fills the reference' \
                  ' model with data' }

    def __init__(self):
        super(SearchMetaresourceProvider, self).__init__()
        self._searchers_by_path = {}
        self._default_searcher = None

    def initialize(self):
        bus = common.application.bus
        bus.register(self._plugin_status_changed_cb, PluginStatusMessage)
        return super(SearchMetaresourceProvider, self).initialize()

    def clean(self):
        bus = common.application.bus
        bus.unregister(self._plugin_status_changed_cb)
        return super(SearchMetaresourceProvider, self).clean()

    def _plugin_status_changed_cb(self, message, sender):
        # Notification that the status of a plugin has changed.
        plugin_registry = common.application.plugin_registry
        plugin = plugin_registry.get_plugin_by_name(message.plugin_name)
        searchers = plugin.get_entry_map(self.entry_point)
        if searchers:
            # It may mean that a new plugin has been installed, in
            # which case we want to add it(s) searcher(s) in our
            # cache.
            if message.action == message.ActionType.ENABLED:
                return self._load_searchers(searchers)
            else:
                # It may also mean that a plugin has been deactivated,
                # in which case want to remove it(s) searcher(s) from
                # our cache.
                return self._unload_searchers(searchers)

    def _add_searcher(self, searcher, name):
        searcher.name = name.lower()
        if name == self.config['default_searcher']:
            self._default_searcher = searcher
            return

        searchers = self._searchers_by_path
        for path in searcher.paths:
            # this is fairly quick
            searchers.setdefault(path, []).append(searcher)

    def _remove_searcher(self, name):
        for key in self._searchers_by_path.keys():
            searchers = self._searchers_by_path[key]
            searchers = [searcher for searcher in searchers
                         if searcher.name != name.lower()]
            if searchers:
                self._searchers_by_path[key] = searchers
            else:
                del self._searchers_by_path[key]

    def _load_searchers(self, searchers):
        """
        Load a set of searchers. This private method is called when a
        new plugin containing searchers has been enabled by the plugin
        registry.

        @param searchers: searchers to register
        @type searchers: C{dict} of C{str} (searcher name),
                         C{pkg_resources.EntryPoint}
        @rtype: C{elisa.core.utils.cancellable_defer.CancellableDeferred}
        """
        plugin_registry = common.application.plugin_registry

        def failed(failure, name):
            self.warning("Creating %s failed: %s" % (name, failure))
            return None

        def iterate():
            for name, entry in searchers.iteritems():
                self.debug("creating searcher %s" % name)
                path = "%s:%s" % (entry.module_name, ".".join(entry.attrs))
                dfr = plugin_registry.create_component(path)
                dfr.addCallback(self._add_searcher, name)
                dfr.addErrback(failed, name)
                yield dfr

        return task.coiterate(iterate())

    def _unload_searchers(self, searchers):
        """
        Unload a set of searchers. This private method is called when a
        plugin containing searchers has been disabled by the plugin
        registry.

        @param searchers: searchers to unregister
        @type searchers: C{dict} of C{str} (searcher name),
                         C{pkg_resources.EntryPoint}
        @rtype: C{elisa.core.utils.cancellable_defer.CancellableDeferred}
        """

        def iterate():
            for name in searchers.keys():
                self.debug("Unloading searcher %s" % name)
                yield self._remove_searcher(name)

        return task.coiterate(iterate())

    def get(self, uri, context_model=None):
        """
        Make a search request. The context model is ingored.
        """

        # uri.path always starts with a slash
        search_path = uri.path[1:].split('/')[0]

        # Figure out which searchers to use, depending on what was limitations
        # were put in place by the paramenters in the URI.

        only_searchers = uri.get_param('only_searchers', None)
        if only_searchers is not None:
            selected_searchers_names = only_searchers.split(',')
        else:
            selected_searchers_names = []              

        only_default_searcher = uri.get_param('only_default', 'false').lower()
        only_default_searcher = (only_default_searcher == 'true')

        if only_default_searcher and self._default_searcher is None:
            msg = "You can only use the only_default option when there's a" \
                  " default searcher installed."
            return None, defer.fail(TypeError(msg))  

        if only_default_searcher and len(selected_searchers_names) > 0 \
           and self._default_searcher and \
           not self._default_searcher.name in selected_searchers_names:
            msg = "You can only pass the option only_default or only_searchers" \
                  " but not both of them."
            return None, defer.fail(TypeError(msg))

        if not only_default_searcher:
              selected_searchers = self._get_searchers_list(search_path,
                                                            selected_searchers_names)
        else:
              selected_searchers = []

        if not self._default_searcher is None and \
           (len(selected_searchers) == 0 or \
           self._default_searcher.name in selected_searchers_names):
              selected_searchers.append(self._default_searcher)
              
        # Retrieve the model to be used for storing results for a search of this
        # media type.
        #
        # FIXME: TODO: The way we do it now is by assuming the model for a
        # certain media type ('audio', 'video', etc or custom ones) will be
        # always the same for all searchers that implement it. Therefore we can
        # just ask for it to the first Searcher and then use it for all of them
        # and finally return it.
        # However there's no guarantee for this, so weird things may happen.
        # Since we don't really use aggregation right now, this doesn't really
        # terribly matter, so it will stay this way for now.

        model = None
        for searcher in selected_searchers:
              model_kl = searcher.get_model_for_path(search_path)
              if not model_kl is None:
                  model = model_kl()
                  break

        if model is None:
              msg = "None of the Searchers available for this search provided" \
                    " a model suitable for returning the results."
              return None, defer.fail(NotImplementedError(msg))

        # Finally call the searchers and return the results
        dfr = self._call_searchers(selected_searchers, uri, model)

        # the API of get says we have to return the model in the last callback
        dfr.addCallback(lambda r: model)
        return model, dfr

    def _get_searchers_list(self, search_type, selected_searchers):
        try:
            searchers = self._searchers_by_path[search_type]
        except KeyError:
            # no searchers found
            return []

        if selected_searchers:
            # there are only certain searchers allowed
            searchers = filter(lambda s: s.name in selected_searchers,
                               searchers)

        return searchers

    def _call_searchers(self, searchers, uri, model):

        def failed(failure, uri):
            if failure.type is defer.CancelledError:
                return
            msg = "A part of the search for %s failed: %s. Skipping"
            self.warning(msg % (uri, failure))
            return None

        def cancel_list(deferred):
            # cancel all the deferreds that we got from the searchers
            for search_dfr in searchers_dfrs:
                try:
                    search_dfr.cancel()
                except (defer.AlreadyCalledError, AttributeError):
                    # might already called or is an ordinary deferred so it has
                    # no cancel method
                    pass
            
            # remove the reference we have to the searchers dfrs
            searchers_dfrs[:] = []

        # the place to store the dfrs that we receive from the searchers
        searchers_dfrs = []

        # Finally run the search on all the searches.
        for searcher in searchers:
            cur_dfr = searcher.search(uri, model)
            cur_dfr.addErrback(failed, uri)

            # we just append it to the list of searcher_dfrs as we don't
            # want to wait for any of them before requesting the next one
            searchers_dfrs.append(cur_dfr)

            # after the iteration over the searchers we want to wait for the
            # searchers deferreds before callbacking. So we put them into a
            # DeferredList
            dfr_list = defer.DeferredList(searchers_dfrs)

        # and make it still cancellable by wrapping that it in a
        # CancellableDeferred
        dfr_lst_cancellable = defer.Deferred(cancel_list)
        dfr_list.chainDeferred(dfr_lst_cancellable)

        # Please notice that we don't really care what the deferred list actually
        # returns, because we're going to return (from the caller) the entire
        # model anyway.
        return dfr_lst_cancellable

    def _clean_searchers(self, res):
        all_searchers = []
        for searchers in self._searchers_by_path.values():
            all_searchers.extend(searchers)

        dfrs = []
        for searcher in set(all_searchers):
            dfrs.append(searcher.clean())

        return defer.DeferredList(dfrs)

    def clean(self):
        dfr = super(SearchMetaresourceProvider, self).clean()
        dfr.addCallback(self._clean_searchers)
        return dfr
