# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Author: Florian Boucault <florian@fluendo.com>

import math
from elisa.plugins.pigment.animation.implicit import DECELERATE

class Transition(object):
    """
    Define a visual, animated transition between two pictures. One picture
    should disappear, one should appear.
    """

    @staticmethod
    def apply(start_picture, end_picture, interval):
        """
        Apply the visual transition between L{start_picture} and L{end_picture}

        @param start_picture: picture that should disappear
        @type start_picture:  L{elisa.plugins.pigment.animation.implicit.AnimatedObject}
        @param end_picture:   picture that should appear
        @type end_picture:    L{elisa.plugins.pigment.animation.implicit.AnimatedObject}
        @param interval:      time in seconds during which the transition should
                              happen
        @type interval:       C{float}
        """
        raise NotImplementedError()


class CrossfadeTransition(Transition):

    @staticmethod
    def apply(start_picture, end_picture, interval):
        if end_picture != None:
            end_picture.opacity = 255

        if start_picture != None:
            start_picture.opacity = 0


class FlipTransition(Transition):

    @staticmethod
    def apply(start_picture, end_picture, interval):
        if end_picture != None:
            end_picture.object.ry = math.pi
            end_picture.z = [-end_picture.absolute_width, 0.0]
            end_picture.ry = 0.0
            end_picture.opacity = 255

        if start_picture != None:
            start_picture.z = [-start_picture.absolute_width, 0.0]
            start_picture.ry = -math.pi
            start_picture.opacity = 0


class FadeToRedTransition(Transition):

    @staticmethod
    def apply(start_picture, end_picture, interval):
        if end_picture != None:
            end_picture.object.fg_color = (255, 0, 0, 255)
            end_picture.fg_color = (255, 255, 255, 255)
            end_picture.opacity = 255

        if start_picture != None:
            start_picture.fg_color = (255, 0, 0, 255)
            start_picture.opacity = 0



class SlideTransition(Transition):

    @staticmethod
    def apply(start_picture, end_picture, interval):
        if end_picture != None:
            end_picture.update_animation_settings(transformation=DECELERATE)
            end_picture.object.x = 1.0
            end_picture.x = 0.0
            end_picture.opacity = 255

        if start_picture != None:
            start_picture.update_animation_settings(transformation=DECELERATE)
            start_picture.x = -1.2
            start_picture.opacity = 0



class CubeTransition(Transition):

    @staticmethod
    def apply(start_picture, end_picture, interval):
        if end_picture != None:
            end_picture.object.ry = math.pi/2.0
            end_picture.object.position = (0.5, 0.0, 0.0)
            end_picture.ry = 0.0
            end_picture.position = (0.0, 0.0, 0.0)
            end_picture.opacity = 255

        if start_picture != None:
            start_picture.ry = -math.pi/2.0
            start_picture.position = (-0.5, 0.0, 0.0)
            start_picture.opacity = 0


import random

class PostcardTransition(Transition):

    @staticmethod
    def apply(start_picture, end_picture, interval):
        if end_picture != None:
#            end_picture.object.border_width = 2.0
#            grey = 220
#            end_picture.object.border_outer_color = (grey, grey, grey, 255)
            end_picture.object.size = (0.8, 0.8)
            end_picture.object.position = (0.1, 0.1, 0.0)
            end_picture.object.ry = 0.01
            end_picture.opacity = 255
            end_picture.setup_next_animations(duration=2.0*interval*1000)
            end_picture.ry = random.randrange(30, 51)/200.0
            end_picture.rx = random.randrange(30, 51)/200.0
            end_picture.setup_next_animations(duration=1000)

        if start_picture != None:
            start_picture.opacity = 0
