(* Copyright (C) 2004-2005, HELM Team.
 * 
 * This file is part of HELM, an Hypertextual, Electronic
 * Library of Mathematics, developed at the Computer Science
 * Department, University of Bologna, Italy.
 * 
 * HELM is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * HELM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with HELM; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA  02111-1307, USA.
 * 
 * For details, see the HELM World-Wide-Web page,
 * http://helm.cs.unibo.it/
 *)

(* $Id: lexiconEngine.ml 8728 2008-06-17 14:18:12Z tassi $ *)

let out = ref ignore

let set_callback f = out := f

(* lexicon file name * ma file name *)
exception IncludedFileNotCompiled of string * string 
exception MetadataNotFound of string        (* file name *)

type status = {
  aliases: DisambiguateTypes.environment;         (** disambiguation aliases *)
  multi_aliases: DisambiguateTypes.multiple_environment;
  lexicon_content_rev: LexiconMarshal.lexicon;
  notation_ids: CicNotation.notation_id list;      (** in-scope notation ids *)
}

let initial_status = {
  aliases = DisambiguateTypes.Environment.empty;
  multi_aliases = DisambiguateTypes.Environment.empty;
  lexicon_content_rev = [];
  notation_ids = [];
}

let add_lexicon_content cmds status =
  let content = status.lexicon_content_rev in
  let content' =
    List.fold_right
     (fun cmd acc -> 
        match cmd with
        | LexiconAst.Alias _ 
        | LexiconAst.Include _ 
        | LexiconAst.Notation _ -> cmd :: (List.filter ((<>) cmd) acc)
        | LexiconAst.Interpretation _ -> if List.exists ((=) cmd) acc then acc else cmd::acc)
     cmds content
  in
(*   prerr_endline ("new lexicon content: " ^ String.concat " " (List.map
    LexiconAstPp.pp_command content')); *)
  { status with lexicon_content_rev = content' }

let set_proof_aliases mode status new_aliases =
 if mode = LexiconAst.WithoutPreferences then
   status 
 else
   let commands_of_aliases =
     List.map
      (fun alias -> LexiconAst.Alias (HExtlib.dummy_floc, alias))
   in
   let aliases =
    List.fold_left (fun acc (d,c) -> DisambiguateTypes.Environment.add d c acc)
     status.aliases new_aliases in
   let multi_aliases =
    List.fold_left (fun acc (d,c) -> DisambiguateTypes.Environment.cons d c acc)
     status.multi_aliases new_aliases in
   let new_status =
     { status with multi_aliases = multi_aliases ; aliases = aliases}
   in
   if new_aliases = [] then
     new_status
   else
     let aliases = 
       DisambiguatePp.aliases_of_domain_and_codomain_items_list new_aliases
     in
     let status = 
       add_lexicon_content (commands_of_aliases aliases) new_status 
     in
     status


let rec eval_command ?(mode=LexiconAst.WithPreferences) status cmd =
 let cmd =
  match cmd with
  | LexiconAst.Interpretation (loc, dsc, (symbol, args), cic_appl_pattern) ->
     let rec disambiguate =
      function
         CicNotationPt.ApplPattern l ->
          CicNotationPt.ApplPattern (List.map disambiguate l)
       | CicNotationPt.VarPattern id
          when not
           (List.exists
            (function (CicNotationPt.IdentArg (_,id')) -> id'=id) args)
          ->
           let item = DisambiguateTypes.Id id in
            (try
              let t =
               snd (DisambiguateTypes.Environment.find item status.aliases)
                status.aliases "" [] in
              let uri = CicUtil.uri_of_term t in
               CicNotationPt.UriPattern uri
             with Not_found -> 
              prerr_endline ("Domain item not found: " ^ 
               (DisambiguateTypes.string_of_domain_item item));
              assert false)
       | p -> p
     in
      LexiconAst.Interpretation
       (loc, dsc, (symbol, args), disambiguate cic_appl_pattern)
  | _-> cmd
 in
 !out cmd;
 let notation_ids' = CicNotation.process_notation cmd in
 let status =
   { status with notation_ids = notation_ids' @ status.notation_ids } in
  match cmd with
  | LexiconAst.Include (loc, baseuri, mode, fullpath) ->
     let lexiconpath_rw, lexiconpath_r = 
       LibraryMisc.lexicon_file_of_baseuri 
         ~must_exist:false ~writable:true ~baseuri,
       LibraryMisc.lexicon_file_of_baseuri 
         ~must_exist:false ~writable:false ~baseuri
     in
     let lexiconpath = 
       if Sys.file_exists lexiconpath_rw then lexiconpath_rw else
         if Sys.file_exists lexiconpath_r then lexiconpath_r else
          raise (IncludedFileNotCompiled (lexiconpath_rw,fullpath))
     in
     let lexicon = LexiconMarshal.load_lexicon lexiconpath in
     let status = List.fold_left (eval_command ~mode) status lexicon in
     status
  | LexiconAst.Alias (loc, spec) -> 
     let diff =
      (*CSC: Warning: this code should be factorized with the corresponding
             code in DisambiguatePp *)
      match spec with
      | LexiconAst.Ident_alias (id,uri) -> 
         [DisambiguateTypes.Id id,
          (uri,(fun _ _ _-> CicUtil.term_of_uri(UriManager.uri_of_string uri)))]
      | LexiconAst.Symbol_alias (symb, instance, desc) ->
         [DisambiguateTypes.Symbol (symb,instance),
          DisambiguateChoices.lookup_symbol_by_dsc symb desc]
      | LexiconAst.Number_alias (instance,desc) ->
         [DisambiguateTypes.Num instance,
          DisambiguateChoices.lookup_num_by_dsc desc]
     in
      set_proof_aliases mode status diff
  | LexiconAst.Interpretation (_, dsc, (symbol, _), _) as stm ->
      let status = add_lexicon_content [stm] status in
      let diff =
       try
        [DisambiguateTypes.Symbol (symbol, 0),
          DisambiguateChoices.lookup_symbol_by_dsc symbol dsc]
       with
        DisambiguateChoices.Choice_not_found msg ->
          prerr_endline (Lazy.force msg);
          assert false
      in
      let status = set_proof_aliases mode status diff in
      status
  | LexiconAst.Notation _ as stm ->
      add_lexicon_content [stm] status

let eval_command = eval_command ?mode:None

let set_proof_aliases  = set_proof_aliases LexiconAst.WithPreferences
  
