(* Copyright (C) 2003-2005, HELM Team.
 * 
 * This file is part of HELM, an Hypertextual, Electronic
 * Library of Mathematics, developed at the Computer Science
 * Department, University of Bologna, Italy.
 * 
 * HELM is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * HELM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with HELM; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA  02111-1307, USA.
 * 
 * For details, see the HELM World-Wide-Web page,
 * http://cs.unibo.it/helm/.
 *)

module C    = Cic
module I    = CicInspect
module S    = CicSubstitution
module TC   = CicTypeChecker 
module Un   = CicUniv
module UM   = UriManager
module Obj  = LibraryObjects
module HObj = HelmLibraryObjects
module A    = Cic2acic
module Ut   = CicUtil
module E    = CicEnvironment
module Pp   = CicPp
module PEH  = ProofEngineHelpers
module HEL  = HExtlib
module DTI  = DoubleTypeInference

module Cl   = ProceduralClassify
module T    = ProceduralTypes
module Cn   = ProceduralConversion
module H    = ProceduralHelpers

type status = {
   sorts : (C.id, A.sort_kind) Hashtbl.t;
   types : (C.id, A.anntypes) Hashtbl.t;
   max_depth: int option;
   depth: int;
   context: C.context;
   case: int list
}

let debug = false

(* helpers ******************************************************************)

let split2_last l1 l2 =
try
   let n = pred (List.length l1) in
   let before1, after1 = HEL.split_nth n l1 in
   let before2, after2 = HEL.split_nth n l2 in
   before1, before2, List.hd after1, List.hd after2
with Invalid_argument _ -> failwith "A2P.split2_last"
   
let string_of_head = function
   | C.ASort _         -> "sort"
   | C.AConst _        -> "const"
   | C.AMutInd _       -> "mutind"
   | C.AMutConstruct _ -> "mutconstruct"
   | C.AVar _          -> "var"
   | C.ARel _          -> "rel"
   | C.AProd _         -> "prod"
   | C.ALambda _       -> "lambda"
   | C.ALetIn _        -> "letin"
   | C.AFix _          -> "fix"
   | C.ACoFix _        -> "cofix"
   | C.AAppl _         -> "appl"
   | C.ACast _         -> "cast"
   | C.AMutCase _      -> "mutcase"
   | C.AMeta _         -> "meta"
   | C.AImplicit _     -> "implict"

let next st = {st with depth = succ st.depth}

let add st entry = {st with context = entry :: st.context}

let push st = {st with case = 1 :: st.case}

let inc st =
   {st with case = match st.case with 
      | []       -> assert false
      | hd :: tl -> succ hd :: tl
   }

let case st str =
   let case = String.concat "." (List.rev_map string_of_int st.case) in
   Printf.sprintf "case %s: %s" case str

let test_depth st =
try   
   let msg = Printf.sprintf "Depth %u: " st.depth in
   match st.max_depth with
      | None   -> true, "" 
      | Some d -> if st.depth < d then true, msg else false, "DEPTH EXCEDED: "
with Invalid_argument _ -> failwith "A2P.test_depth"

let is_rewrite_right = function
   | C.AConst (_, uri, []) ->
      UM.eq uri HObj.Logic.eq_ind_r_URI || Obj.is_eq_ind_r_URI uri
   | _                     -> false

let is_rewrite_left = function
   | C.AConst (_, uri, []) ->
      UM.eq uri HObj.Logic.eq_ind_URI || Obj.is_eq_ind_URI uri
   | _                     -> false

let is_fwd_rewrite_right hd tl =
   if is_rewrite_right hd then match List.nth tl 3 with
      | C.ARel _ -> true
      | _        -> false
   else false

let is_fwd_rewrite_left hd tl =
   if is_rewrite_left hd then match List.nth tl 3 with
      | C.ARel _ -> true
      | _        -> false
   else false

let get_inner_types st v =
try
   let id = Ut.id_of_annterm v in
   try match Hashtbl.find st.types id with
      | {A.annsynthesized = st; A.annexpected = Some et} -> Some (st, et)
      | {A.annsynthesized = st; A.annexpected = None}    -> Some (st, st)
   with Not_found -> None
with Invalid_argument _ -> failwith "A2P.get_inner_types"
(*
let get_inner_sort st v =
try
   let id = Ut.id_of_annterm v in
   try Hashtbl.find st.sorts id
   with Not_found -> `Type (CicUniv.fresh())
with Invalid_argument _ -> failwith "A2P.get_sort"
*)
let get_type msg st bo =
try   
   let ty, _ = TC.type_of_aux' [] st.context (H.cic bo) Un.oblivion_ugraph in
   ty
with e -> failwith (msg ^ ": " ^ Printexc.to_string e)

let get_entry st id =
   let rec aux = function
      | []                                        -> assert false
      | Some (C.Name name, e) :: _ when name = id -> e
      | _ :: tl                                   -> aux tl
   in
   aux st.context

let get_ind_names uri tno =
try   
   let ts = match E.get_obj Un.oblivion_ugraph uri with
      | C.InductiveDefinition (ts, _, _, _), _ -> ts 
      | _                                      -> assert false
   in
   match List.nth ts tno with
      | (_, _, _, cs) -> List.map fst cs  
with Invalid_argument _ -> failwith "A2P.get_ind_names"

let string_of_atomic = function
   | C.ARel (_, _, _, s)               -> s
   | C.AVar (_, uri, _)                -> H.name_of_uri uri None None
   | C.AConst (_, uri, _)              -> H.name_of_uri uri None None
   | C.AMutInd (_, uri, i, _)          -> H.name_of_uri uri (Some i) None
   | C.AMutConstruct (_, uri, i, j, _) -> H.name_of_uri uri (Some i) (Some j)
   | _                                 -> ""

let get_sub_names head l =
   let s = string_of_atomic head in
   if s = "" then [] else
   let map (names, i) _ = 
      let name = Printf.sprintf "%s_%u" s i in name :: names, succ i
   in
   let names, _ = List.fold_left map ([], 1) l in 
   List.rev names

(* proof construction *******************************************************)

let anonymous_premise = C.Name "PREMISE"

let mk_exp_args hd tl classes synth =
   let meta id = C.AImplicit (id, None) in
   let map v (cl, b) =
      if I.overlaps synth cl && b then v else meta ""
   in
   let rec aux = function
      | [] -> []
      | hd :: tl -> if hd = meta "" then aux tl else List.rev (hd :: tl)
   in
   let args = T.list_rev_map2 map tl classes in
   let args = aux args in
   if args = [] then hd else C.AAppl ("", hd :: args)

let mk_convert st ?name sty ety note =
   let e = Cn.hole "" in
   let csty, cety = H.cic sty, H.cic ety in
   let script = 
      if debug then
         let sname = match name with None -> "" | Some (id, _) -> id in
         let note = Printf.sprintf "%s: %s\nSINTH: %s\nEXP: %s"
            note sname (Pp.ppterm csty) (Pp.ppterm cety)
	 in 
	 [T.Note note]
      else []
   in
   assert (Ut.is_sober st.context csty); 
   assert (Ut.is_sober st.context cety);
   if Ut.alpha_equivalence csty cety then script else 
   let sty, ety = H.acic_bc st.context sty, H.acic_bc st.context ety in
   match name with
      | None         -> T.Change (sty, ety, None, e, "") :: script
      | Some (id, i) -> 
         begin match get_entry st id with
	    | C.Def _  -> assert false (* T.ClearBody (id, "") :: script *)
	    | C.Decl _ -> 
	       T.Change (ety, sty, Some (id, Some id), e, "") :: script 
         end

let convert st ?name v = 
   match get_inner_types st v with
      | None            -> 
         if debug then [T.Note "NORMAL: NO INNER TYPES"] else []
      | Some (sty, ety) -> mk_convert st ?name sty ety "NORMAL"

let convert_elim st ?name t v pattern =
   match t, get_inner_types st t, get_inner_types st v with
      | _, None, _
      | _, _, None                                            -> [(* T.Note "ELIM: NO INNER TYPES"*)]
      | C.AAppl (_, hd :: tl), Some (tsty, _), Some (vsty, _) ->
         let where = List.hd (List.rev tl) in
         let cty = Cn.elim_inferred_type 
	     st.context (H.cic vsty) (H.cic where) (H.cic hd) (H.cic pattern)
	 in
         mk_convert st ?name (Cn.fake_annotate "" st.context cty) tsty "ELIM"
      | _, Some _, Some _                                     -> assert false
	  
let get_intro = function 
   | C.Anonymous -> None
   | C.Name s    -> Some s

let mk_preamble st what script =
   convert st what @ script   

let mk_arg st = function
   | C.ARel (_, _, i, name) as what -> convert st ~name:(name, i) what
   | _                              -> []

let mk_fwd_rewrite st dtext name tl direction v t ity =
   let compare premise = function
      | None   -> true
      | Some s -> s = premise
   in
   assert (List.length tl = 6);
   let what, where, predicate = List.nth tl 5, List.nth tl 3, List.nth tl 2 in
   let e = Cn.mk_pattern 1 predicate in
   if (Cn.does_not_occur e) then st, [] else 
   match where with
      | C.ARel (_, _, i, premise) as w ->
(*         let _script = convert_elim st ~name:(premise, i) v w e in *) 
         let script name =
            let where = Some (premise, name) in
	    let script = mk_arg st what @ mk_arg st w (* @ script *) in
	    T.Rewrite (direction, what, where, e, dtext) :: script
	 in
	 if DTI.does_not_occur (succ i) (H.cic t) || compare premise name then
	    {st with context = Cn.clear st.context premise}, script name
	 else begin
	    assert (Ut.is_sober st.context (H.cic ity));
	    let ity = H.acic_bc st.context ity in
	    let br1 = [T.Id ""] in
	    let br2 = List.rev (T.Apply (w, "assumption") :: script None) in
	    let text = "non linear rewrite" in
	    st, [T.Branch ([br2; br1], ""); T.Cut (name, ity, text)]
	 end
      | _                         -> assert false

let mk_rewrite st dtext where qs tl direction t = 
   assert (List.length tl = 5);
   let predicate = List.nth tl 2 in
   let e = Cn.mk_pattern 1 predicate in
   let script = [T.Branch (qs, "")] in
   if (Cn.does_not_occur e) then script else 
(*   let script = convert_elim st t t e in *)
   T.Rewrite (direction, where, None, e, dtext) :: script

let rec proc_lambda st what name v t =
   let name = match name with
      | C.Anonymous -> H.mk_fresh_name st.context anonymous_premise
      | name        -> name
   in
   let entry = Some (name, C.Decl (H.cic v)) in
   let intro = get_intro name in
   let script = proc_proof (add st entry) t in
   let script = T.Intros (Some 1, [intro], "") :: script in
   mk_preamble st what script

and proc_letin st what name v w t =
   let intro = get_intro name in
   let proceed, dtext = test_depth st in
   let script = if proceed then 
      let st, hyp, rqv = match get_inner_types st v with
         | Some (ity, _) ->
	    let st, rqv = match v with
               | C.AAppl (_, hd :: tl) when is_fwd_rewrite_right hd tl ->
	          mk_fwd_rewrite st dtext intro tl true v t ity
	       | C.AAppl (_, hd :: tl) when is_fwd_rewrite_left hd tl  ->
	          mk_fwd_rewrite st dtext intro tl false v t ity
	       | v                                                     ->
	          assert (Ut.is_sober st.context (H.cic ity));
		  let ity = H.acic_bc st.context ity in
	          let qs = [proc_proof (next st) v; [T.Id ""]] in
		  st, [T.Branch (qs, ""); T.Cut (intro, ity, dtext)]
	    in
	    st, C.Decl (H.cic ity), rqv
	 | None          ->
	    st, C.Def (H.cic v, H.cic w), [T.LetIn (intro, v, dtext)]
      in
      let entry = Some (name, hyp) in
      let qt = proc_proof (next (add st entry)) t in
      List.rev_append rqv qt      
   else
      [T.Apply (what, dtext)]
   in
   mk_preamble st what script

and proc_rel st what = 
   let _, dtext = test_depth st in
   let text = "assumption" in
   let script = [T.Apply (what, dtext ^ text)] in 
   mk_preamble st what script

and proc_mutconstruct st what = 
   let _, dtext = test_depth st in
   let script = [T.Apply (what, dtext)] in 
   mk_preamble st what script

and proc_const st what = 
   let _, dtext = test_depth st in
   let script = [T.Apply (what, dtext)] in 
   mk_preamble st what script

and proc_appl st what hd tl =
   let proceed, dtext = test_depth st in
   let script = if proceed then
      let ty = get_type "TC2" st hd in
      let classes, rc = Cl.classify st.context ty in
      let goal_arity, goal = match get_inner_types st what with
         | None            -> 0, None
	 | Some (ity, ety) -> 
	   snd (PEH.split_with_whd (st.context, H.cic ity)), Some (H.cic ety)
      in
      let parsno, argsno = List.length classes, List.length tl in
      let decurry = parsno - argsno in
      let diff = goal_arity - decurry in
      if diff < 0 then failwith (Printf.sprintf "NOT TOTAL: %i %s |--- %s" diff (Pp.ppcontext st.context) (Pp.ppterm (H.cic hd)));
      let classes = Cl.adjust st.context tl ?goal classes in
      let rec mk_synth a n =
         if n < 0 then a else mk_synth (I.S.add n a) (pred n)
      in
      let synth = mk_synth I.S.empty decurry in
      let text = "" (* Printf.sprintf "%u %s" parsno (Cl.to_string h) *) in
      let script = List.rev (mk_arg st hd) in
      match rc with
         | Some (i, j, uri, tyno) ->
	    let classes2, tl2, _, where = split2_last classes tl in
	    let script2 = List.rev (mk_arg st where) @ script in
	    let synth2 = I.S.add 1 synth in
	    let names = get_ind_names uri tyno in
	    let qs = proc_bkd_proofs (next st) synth2 names classes2 tl2 in
            if List.length qs <> List.length names then
	       let qs = proc_bkd_proofs (next st) synth [] classes tl in
	       let hd = mk_exp_args hd tl classes synth in
	       script @ [T.Apply (hd, dtext ^ text); T.Branch (qs, "")]
	    else if is_rewrite_right hd then 
	       script2 @ mk_rewrite st dtext where qs tl2 false what
	    else if is_rewrite_left hd then 
	       script2 @ mk_rewrite st dtext where qs tl2 true what
	    else
	       let predicate = List.nth tl2 (parsno - i) in
               let e = Cn.mk_pattern j predicate in
	       let using = Some hd in
	       (* convert_elim st what what e @ *) script2 @ 
	       [T.Elim (where, using, e, dtext ^ text); T.Branch (qs, "")]
	 | None        ->
	    let names = get_sub_names hd tl in
	    let qs = proc_bkd_proofs (next st) synth names classes tl in
	    let hd = mk_exp_args hd tl classes synth in
	    script @ [T.Apply (hd, dtext ^ text); T.Branch (qs, "")]
   else
      [T.Apply (what, dtext)]
   in
   mk_preamble st what script

and proc_other st what =
   let _, dtext = test_depth st in
   let text = Printf.sprintf "%s: %s" "UNEXPANDED" (string_of_head what) in
   let script = [T.Apply (what, dtext ^ text)] in 
   mk_preamble st what script

and proc_proof st t = 
   let f st =
      let xtypes, note = match get_inner_types st t with
         | Some (it, et) -> Some (H.cic it, H.cic et), 
	  (Printf.sprintf "\nInferred: %s\nExpected: %s"
	  (Pp.ppterm (H.cic it)) (Pp.ppterm (H.cic et))) 
         | None          -> None, "\nNo types"
      in
      let context, _clears = Cn.get_clears st.context (H.cic t) xtypes in
      {st with context = context}
   in
   match t with
      | C.ALambda (_, name, w, t) as what   -> proc_lambda (f st) what name w t
      | C.ALetIn (_, name, v, w, t) as what -> proc_letin (f st) what name v w t
      | C.ARel _ as what                    -> proc_rel (f st) what
      | C.AMutConstruct _ as what           -> proc_mutconstruct (f st) what
      | C.AConst _ as what                  -> proc_const (f st) what
      | C.AAppl (_, hd :: tl) as what       -> proc_appl (f st) what hd tl
      | what                                -> proc_other (f st) what

and proc_bkd_proofs st synth names classes ts =
try 
   let get_note =
      let names = ref (names, push st) in
      fun f -> 
         match !names with 
	    | [], st       -> fun _ -> f st
	    | "" :: tl, st -> names := tl, st; fun _ -> f st
	    | hd :: tl, st -> 
	       let note = case st hd in
	       names := tl, inc st; 
	       fun b -> if b then T.Note note :: f st else f st
   in
   let _, dtext = test_depth st in   
   let aux (inv, _) v =
      if I.overlaps synth inv then None else
      if I.S.is_empty inv then Some (get_note (fun st -> proc_proof st v)) else
      Some (get_note (fun _ -> [T.Apply (v, dtext ^ "dependent")]))
   in	
   let ps = T.list_map2_filter aux classes ts in
   let b = List.length ps > 1 in
   List.rev_map (fun f -> f b) ps

with Invalid_argument s -> failwith ("A2P.proc_bkd_proofs: " ^ s)

(* object costruction *******************************************************)

let is_theorem pars =
   pars = [] ||
   List.mem (`Flavour `Theorem) pars || List.mem (`Flavour `Fact) pars || 
   List.mem (`Flavour `Remark) pars || List.mem (`Flavour `Lemma) pars

let is_definition pars =
   List.mem (`Flavour `Definition) pars

let proc_obj st = function
   | C.AConstant (_, _, s, Some v, t, [], pars) when is_theorem pars    ->
      let ast = proc_proof st v in
      let steps, nodes = T.count_steps 0 ast, T.count_nodes 0 ast in
      let text = Printf.sprintf "tactics: %u\nnodes: %u" steps nodes in
      T.Statement (`Theorem, Some s, t, None, "") :: ast @ [T.Qed text]
   | C.AConstant (_, _, s, Some v, t, [], pars) when is_definition pars ->
      [T.Statement (`Definition, Some s, t, Some v, "")]
   | C.AConstant (_, _, s, None, t, [], pars)                           ->
      [T.Statement (`Axiom, Some s, t, None, "")]
   | _                                                                  ->
      failwith "not a theorem, definition, axiom"

(* interface functions ******************************************************)

let procedural_of_acic_object ~ids_to_inner_sorts ~ids_to_inner_types ?depth
   prefix anobj = 
   let st = {
      sorts       = ids_to_inner_sorts;
      types       = ids_to_inner_types;
      max_depth   = depth;
      depth       = 0;
      context     = [];
      case        = []
   } in
   HLog.debug "Procedural: level 2 transformation";
   let steps = proc_obj st anobj in
   HLog.debug "Procedural: grafite rendering";
   List.rev (T.render_steps [] steps)

let procedural_of_acic_term ~ids_to_inner_sorts ~ids_to_inner_types ?depth
   prefix context annterm = 
   let st = {
      sorts       = ids_to_inner_sorts;
      types       = ids_to_inner_types;
      max_depth   = depth;
      depth       = 0;
      context     = context;
      case        = []
   } in
   HLog.debug "Procedural: level 2 transformation";
   let steps = proc_proof st annterm in
   HLog.debug "Procedural: grafite rendering";
   List.rev (T.render_steps [] steps)
