/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file Symbol.h
    \brief Definition of Symbol graphics class.
    
    Magics Team - ECMWF 2004
    
    Started: Jan 2004
    
    Changes:
    
*/
#ifndef Symbol_H
#define Symbol_H

#include "magics.h"

#include "BasicGraphicsObject.h"
#include "BaseDriver.h"
#include "SymbolItem.h"
#include "PaperPoint.h"

namespace magics {


class Symbol: public BasicGraphicsObject, public vector<PaperPoint> {

public:
	enum TextPosition { M_NONE, M_BELOW , M_ABOVE , M_LEFT , M_RIGHT};
	Symbol();
	virtual ~Symbol();

	void redisplay(const BaseDriver& driver) const { driver.redisplay(*this); }

	void setMarker(int m)
	{ 
		marker_ = m;
		symbol_ = convert(m);
	}
	
	static string convert(int m)
	{
		ostringstream symbol;
		symbol << "magics_" << m; 
		return symbol.str();
	}

	virtual void push_back(const PaperPoint& point,const string& ="") {
		 vector<PaperPoint>::push_back(point);
	}
	
	int getMarker() const {return marker_;} 

	void   setSymbol(const string& symbol) { symbol_ = symbol; }
	const string& getSymbol() const  {return symbol_;} 

	void   setColour(const Colour& colour) { colour_ = colour; }
	const Colour& getColour() const  {return colour_;} 

	void   setHeight(double h){height_ = h;}
	double getHeight() const  {return height_;}
	
	void   setDistanceApart(double a){apart_ = a;}
	double getDistanceApart() const  {return apart_;}

//private:
// No copy allowed
	Symbol(const Symbol&);
	Symbol& operator=(const Symbol&);

protected:
	virtual void print(ostream&) const; 

private:
	int	marker_;
	double	height_;
	string	symbol_;
	Colour colour_;
	double apart_;

// -- Friends
	friend ostream& operator<<(ostream& s,const Symbol& p)
		{ p.print(s); return s; }
};


class TextSymbol : public Symbol
{
public:
	TextSymbol() : position_(M_BELOW) {}
	~TextSymbol() {}

	void push_back(const PaperPoint& point, const string& text) {
		Symbol::push_back(point);
		texts_.push_back(text);
	}

	void text(const vector<string>& text) { texts_ = text; }
	void redisplay(const BaseDriver& driver) const { driver.redisplay(*this); }

	vector<string>::const_iterator textBegin() const { return texts_.begin(); }
	vector<string>::const_iterator textEnd() const { return texts_.end(); }
	const vector<string>& text() const {return texts_; } 	 

	void font(const MagFont& font) { font_ = font; }
	const MagFont& font() const { return font_; }

	void position(TextPosition position) { position_ = position; }
	TextPosition position() const { return position_; }

protected:
	vector<string> texts_;
	MagFont font_;
	TextPosition position_;
};


class SimpleTextSymbol : public TextSymbol
{
public:
	SimpleTextSymbol(const string& text) : text_(text) {}
	~SimpleTextSymbol() {}
	
	void push_back(const PaperPoint& point) {
		TextSymbol::push_back(point, text_);
	}

protected:
	string text_;
};


class ComplexSymbol : public Symbol
{
public:
	ComplexSymbol(): rows_(1), columns_(1) {}
	ComplexSymbol(int rows, int columns): rows_(rows), columns_(columns) {}
	~ComplexSymbol() {}
	
	void redisplay(const BaseDriver& driver) const { Log::dev() << "Redisplay -->" << *this << endl; driver.redisplay(*this); }
	void add(GraphicsItem* item) { items_.push_back(item);}
	vector<GraphicsItem*>::const_iterator itemBegin() const { return items_.begin(); }
	vector<GraphicsItem*>::const_iterator itemEnd() const { return items_.end(); }
	const vector<GraphicsItem*>& items() const { return items_; }

protected:
	virtual void print(ostream&) const; 
	VectorOfPointers<vector<GraphicsItem*> > items_;
	int rows_;
	int columns_;
};

struct SymbolProperties
{
	Colour    colour_;
	double    height_;
	string    marker_;
	string    label_;
	MagFont   font_;
	Symbol::TextPosition position_;
	vector<string> text_;

    SymbolProperties(Colour colour, double height, const string& marker, const string& label="");
    SymbolProperties(Colour colour, double height, int marker, const string& label=""); 
    SymbolProperties();

    void setSymbol(const string& symbol, int marker);
    void position(Symbol::TextPosition position) { position_ = position; }

    virtual ~SymbolProperties() {}
    bool  operator<(const SymbolProperties& other) const
    {
        if ( other.colour_ == colour_ ) 
            if ( other.height_ == height_ )
                return other.marker_ < marker_;
            else return other.height_ < height_;
        else 
            return other.colour_ < colour_;
    }
    
    void label(const string& label) { label_ = label; }
    
    virtual void print(ostream& out) const {
        out << "SymbolProperty[";
        out << colour_ << "(colour),";
        out << height_ << "(height),";
        out << marker_ << "(marker)";
        out << "]";
    }
    
    Symbol* symbol(const string&) const;
//    
// -- Friends
	//! Overloaded << operator to call print().
	//! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const SymbolProperties& p)
		{ p.print(s); return s; }
};

} // namespace magics
#endif
