/*  GRAPHITE2 LICENSING

    Copyright 2010, SIL International
    All rights reserved.

    This library is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published
    by the Free Software Foundation; either version 2.1 of License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should also have received a copy of the GNU Lesser General Public
    License along with this library in the file named "LICENSE".
    If not, write to the Free Software Foundation, Inc., 59 Temple Place, 
    Suite 330, Boston, MA 02111-1307, USA or visit their web page on the 
    internet at http://www.fsf.org/licenses/lgpl.html.
*/
#include "graphite2/Font.h"
#include "Face.h"
#include "CachedFace.h"
#include "XmlTraceLog.h"


using namespace graphite2;

extern "C" {


gr_face* gr_make_face(const void* appFaceHandle/*non-NULL*/, gr_get_table_fn getTable, unsigned int faceOptions)
                  //the appFaceHandle must stay alive all the time when the gr_face is alive. When finished with the gr_face, call destroy_face    
{
    Face *res = new Face(appFaceHandle, getTable);

#ifndef DISABLE_TRACING
    XmlTraceLog::get().openElement(ElementFace);
#endif
    bool valid = true;
    valid &= res->readGlyphs(faceOptions);
    if (!valid) {
        delete res;
        return 0;
    }
    valid &= res->readFeatures();
    valid &= res->readGraphite();
#ifndef DISABLE_TRACING
    XmlTraceLog::get().closeElement(ElementFace);
#endif
    
    if (!(faceOptions & gr_face_dumbRendering) && !valid) {
        delete res;
        return 0;
    }
    return static_cast<gr_face *>(res);
}

gr_face* gr_make_face_with_seg_cache(const void* appFaceHandle/*non-NULL*/, gr_get_table_fn getTable, unsigned int cacheSize, unsigned int faceOptions)
                  //the appFaceHandle must stay alive all the time when the GrFace is alive. When finished with the GrFace, call destroy_face
{
    CachedFace *res = new CachedFace(appFaceHandle, getTable);
#ifndef DISABLE_TRACING
    XmlTraceLog::get().openElement(ElementFace);
#endif
    bool valid = true;
    valid &= res->readGlyphs(faceOptions);
    if (!valid) {
        delete res;
        return 0;
    }
    valid &= res->readFeatures();
    valid &= res->readGraphite();
    valid &= res->setupCache(cacheSize);

#ifndef DISABLE_TRACING
    XmlTraceLog::get().closeElement(ElementFace);
#endif

    if (!(faceOptions & gr_face_dumbRendering) && !valid) {
        delete res;
        return 0;
    }
    return static_cast<gr_face *>(static_cast<Face *>(res));
}

gr_uint32 gr_str_to_tag(const char *str)
{
    uint32 res = 0;
    int i = strlen(str);
    if (i > 4) i = 4;
    while (--i >= 0)
        res = (res >> 8) + (str[i] << 24);
    return res;
}

void gr_tag_to_str(gr_uint32 tag, char *str)
{
    int i = 4;
    while (--i >= 0)
    {
        str[i] = tag & 0xFF;
        tag >>= 8;
    }
}
        

gr_feature_val* gr_face_featureval_for_lang(const gr_face* pFace, gr_uint32 langname/*0 means clone default*/) //clones the features. if none for language, clones the default
{
    assert(pFace);
    return static_cast<gr_feature_val *>(pFace->theSill().cloneFeatures(langname));
}


const gr_feature_ref* gr_face_find_fref(const gr_face* pFace, gr_uint32 featId)  //When finished with the FeatureRef, call destroy_FeatureRef
{
    assert(pFace);
    const FeatureRef* pRef = pFace->featureById(featId);
    return static_cast<const gr_feature_ref*>(pRef);
}

unsigned short gr_face_n_fref(const gr_face* pFace)
{
    assert(pFace);
    return pFace->numFeatures();
}

const gr_feature_ref* gr_face_fref(const gr_face* pFace, gr_uint16 i) //When finished with the FeatureRef, call destroy_FeatureRef
{
    assert(pFace);
    const FeatureRef* pRef = pFace->feature(i);
    return static_cast<const gr_feature_ref*>(pRef);
}

unsigned short gr_face_n_languages(const gr_face* pFace)
{
    assert(pFace);
    return pFace->theSill().numLanguages();
}

gr_uint32 gr_face_lang_by_index(const gr_face* pFace, gr_uint16 i)
{
    assert(pFace);
    return pFace->theSill().getLangName(i);
}


void gr_face_destroy(gr_face *face)
{
    delete face;
}


gr_uint16 gr_face_name_lang_for_locale(gr_face *face, const char * locale)
{
    if (face)
    {
        return face->languageForLocale(locale);
    }
    return 0;
}

#if 0      //hidden since no way to release atm.

uint16 *face_name(const gr_face * pFace, uint16 nameid, uint16 lid)
{
    size_t nLen = 0, lOffset = 0, lSize = 0;
    const void *pName = pFace->getTable(tagName, &nLen);
    uint16 *res;
    if (!pName || !TtfUtil::GetNameInfo(pName, 3, 0, lid, nameid, lOffset, lSize))
        return NULL;
    lSize >>= 1;
    res = gralloc<uint16>(lSize + 1);
    for (size_t i = 0; i < lSize; ++i)
        res[i] = swap16(*(uint16 *)((char *)pName + lOffset));
    res[lSize] = 0;
    return res;
}
#endif

unsigned short gr_face_n_glyphs(const gr_face* pFace)
{
    return pFace->getGlyphFaceCache()->numGlyphs();
}


#ifndef DISABLE_FILE_FACE
gr_face* gr_make_file_face(const char *filename, unsigned int faceOptions)
{
    FileFace* pFileFace = new FileFace(filename);
    if (pFileFace->m_pTableDir)
    {
      gr_face* pRes =gr_make_face(pFileFace, &FileFace::table_fn, faceOptions);
      if (pRes)
      {
        pRes->takeFileFace(pFileFace);        //takes ownership
        return pRes;
      }
    }
    
    //error when loading

    delete pFileFace;
    return NULL;
}


gr_face* gr_make_file_face_with_seg_cache(const char* filename, unsigned int segCacheMaxSize, unsigned int faceOptions)   //returns NULL on failure. //TBD better error handling
                  //when finished with, call destroy_face
{
    FileFace* pFileFace = new FileFace(filename);
    if (pFileFace->m_pTableDir)
    {
      gr_face* pRes = gr_make_face_with_seg_cache(pFileFace, &FileFace::table_fn, segCacheMaxSize, faceOptions);
      if (pRes)
      {
        pRes->takeFileFace(pFileFace);        //takes ownership
        return pRes;
      }
    }

    //error when loading

    delete pFileFace;
    return NULL;
}

#endif      //!DISABLE_FILE_FACE


} // extern "C"


