package net.sf.saxon.expr;

import net.sf.saxon.om.EmptyIterator;
import net.sf.saxon.om.Item;
import net.sf.saxon.om.NodeInfo;
import net.sf.saxon.om.SequenceIterator;
import net.sf.saxon.sort.DocumentOrderIterator;
import net.sf.saxon.sort.GlobalOrderComparer;
import net.sf.saxon.trace.ExpressionPresenter;
import net.sf.saxon.trace.Location;
import net.sf.saxon.trans.XPathException;
import net.sf.saxon.type.ItemType;
import net.sf.saxon.type.TypeHierarchy;
import net.sf.saxon.value.AtomicValue;
import net.sf.saxon.value.Cardinality;

import javax.xml.transform.SourceLocator;
import java.util.Iterator;

/**
 * A simple mapping expression is an expression A/B where B has a static type that is an atomic type.
 * For example, * / name().
 */

public final class SimpleMappingExpression extends Expression
        implements ContextMappingFunction {

    private Expression start;
    private Expression step;
    private boolean isHybrid;

    /**
     * Constructor
     * @param start A node-set expression denoting the absolute or relative set of nodes from which the
     * navigation path should start.
     * @param step The step to be followed from each node in the start expression to yield a new
     * node-set
     * @param isHybrid if true, indicates that we don't know statically whether the step expression will
     * return nodes or atomic values. If false, we know it will return atomic values.
     */

    public SimpleMappingExpression(Expression start, Expression step, boolean isHybrid) {
        this.start = start;
        this.step = step;
        this.isHybrid = isHybrid;
        adoptChildExpression(start);
        adoptChildExpression(step);

    }

    /**
     * Get the start expression (the left-hand operand)
     * @return the first operand
     */

    public Expression getStartExpression() {
        return start;
    }

    /**
     * Get the step expression (the right-hand operand)
     * @return the second operand
     */

    public Expression getStepExpression() {
        return step;
    }


    /**
     * Determine whether this expression is capable (as far as static analysis is concerned)
     * of returning a mixture of nodes and atomic values. If so, this needs to be prevented
     * at run time
     * @return true if the static type allows both nodes and atomic values
     */

    public boolean isHybrid() {
        return isHybrid;
    }

    /**
     * Determine the data type of the items returned by this exprssion
     * @return the type of the step
     * @param th the type hierarchy cache
     */

    public final ItemType getItemType(TypeHierarchy th) {
        return step.getItemType(th);
    }

    /**
     * Type-check the expression
     */

    public Expression typeCheck(ExpressionVisitor visitor, ItemType contextItemType) {
        // rely on the fact that the original path expression has already been type-checked
        return this;
    }

    public Expression optimize(ExpressionVisitor visitor, ItemType contextItemType) throws XPathException {
        Optimizer opt = visitor.getConfiguration().getOptimizer();
        TypeHierarchy th = visitor.getConfiguration().getTypeHierarchy();


//        Expression k = opt.convertPathExpressionToKey(this, env);
//        if (k != null) {
//            ComputedExpression.setParentExpression(k, getParentExpression());
//            return k;
//        }

        Expression start2 = visitor.optimize(start, contextItemType);
        if (start2 != start) {
            adoptChildExpression(start2);
            start = start2;
        }
        Expression step2 = step.optimize(visitor, start.getItemType(th));
        if (step2 != step) {
            adoptChildExpression(step2);
            step = step2;
        }

        // If any subexpressions within the step are not dependent on the focus,
        // and if they are not "creative" expressions (expressions that can create new nodes), then
        // promote them: this causes them to be evaluated once, outside the path expression

        PromotionOffer offer = new PromotionOffer(opt);
        offer.action = PromotionOffer.FOCUS_INDEPENDENT;
        offer.promoteDocumentDependent = (start.getSpecialProperties() & StaticProperty.CONTEXT_DOCUMENT_NODESET) != 0;
        offer.containingExpression = this;

        step = doPromotion(step, offer);
        visitor.resetStaticProperties();
        if (offer.containingExpression != this) {
            offer.containingExpression =
                    visitor.optimize(visitor.typeCheck(offer.containingExpression, contextItemType), contextItemType);
            return offer.containingExpression;
        }
        return this;

    }

    /**
     * Promote this expression if possible
     */

    public Expression promote(PromotionOffer offer) throws XPathException {
        Expression exp = offer.accept(this);
        if (exp != null) {
            return exp;
        } else {
            start = doPromotion(start, offer);
            if (offer.action == PromotionOffer.INLINE_VARIABLE_REFERENCES ||
                    offer.action == PromotionOffer.REPLACE_CURRENT) {
                // Don't pass on other requests. We could pass them on, but only after augmenting
                // them to say we are interested in subexpressions that don't depend on either the
                // outer context or the inner context.
                step = doPromotion(step, offer);
            }
            //step.resetStaticProperties();
            return this;
        }
    }

    /**
     * Get the immediate subexpressions of this expression
     */

    public Iterator iterateSubExpressions() {
        return new PairIterator(start, step);
    }

    /**
     * Given an expression that is an immediate child of this expression, test whether
     * the evaluation of the parent expression causes the child expression to be
     * evaluated repeatedly
     * @param child the immediate subexpression
     * @return true if the child expression is evaluated repeatedly
     */

    public boolean hasLoopingSubexpression(Expression child) {
        return child == step;
    }

   /**
     * Replace one subexpression by a replacement subexpression
     * @param original the original subexpression
     * @param replacement the replacement subexpression
     * @return true if the original subexpression is found
     */

    public boolean replaceSubExpression(Expression original, Expression replacement) {
        boolean found = false;
        if (start == original) {
            start = replacement;
            found = true;
        }
        if (step == original) {
            step = replacement;
            found = true;
        }
                return found;
    }
    /**
     * Determine which aspects of the context the expression depends on. The result is
     * a bitwise-or'ed value composed from constants such as XPathContext.VARIABLES and
     * XPathContext.CURRENT_NODE
     */

    public int computeDependencies() {
        return start.getDependencies() |
                // not all dependencies in the step matter, because the context node, etc,
                // are not those of the outer expression
                (step.getDependencies() & StaticProperty.DEPENDS_ON_XSLT_CONTEXT);
    }

    /**
     * Copy an expression. This makes a deep copy.
     *
     * @return the copy of the original expression
     */

    public Expression copy() {
        return new SimpleMappingExpression(start.copy(), step.copy(), isHybrid);
    }

    /**
     * Get the static properties of this expression (other than its type). The result is
     * bit-signficant. These properties are used for optimizations. In general, if
     * property bit is set, it is true, but if it is unset, the value is unknown.
     */

    public int computeSpecialProperties() {
        int p = super.computeSpecialProperties();
        if ((start.getSpecialProperties() & step.getSpecialProperties() & StaticProperty.NON_CREATIVE) != 0) {
            p |= StaticProperty.NON_CREATIVE;
        }
        return p;
    }

    /**
     * Determine the static cardinality of the expression
     */

    public int computeCardinality() {
        int c1 = start.getCardinality();
        int c2 = step.getCardinality();
        return Cardinality.multiply(c1, c2);
    }

    /**
     * Is this expression the same as another expression?
     */

    public boolean equals(Object other) {
        if (!(other instanceof SimpleMappingExpression)) {
            return false;
        }
        SimpleMappingExpression p = (SimpleMappingExpression) other;
        return (start.equals(p.start) && step.equals(p.step));
    }

    /**
     * get HashCode for comparing two expressions
     */

    public int hashCode() {
        return "SimpleMappingExpression".hashCode() + start.hashCode() + step.hashCode();
    }

    /**
     * Iterate the path-expression in a given context
     * @param context the evaluation context
     */

    public SequenceIterator iterate(final XPathContext context) throws XPathException {

        // This class delivers the result of the path expression in unsorted order,
        // without removal of duplicates. If sorting and deduplication are needed,
        // this is achieved by wrapping the path expression in a DocumentSorter

        SequenceIterator result = start.iterate(context);
        XPathContext context2 = context.newMinorContext();
        context2.setCurrentIterator(result);
        context2.setOriginatingConstructType(Location.PATH_EXPRESSION);

        result = new ContextMappingIterator(this, context2);
        if (isHybrid) {
            // This case is rare so we don't worry too much about performance
            // Peek at the first node, and depending on its type, check that all the items
            // are atomic values or that all are nodes.
            final SourceLocator loc = this;
            Item first = result.next();
            if (first == null) {
                return EmptyIterator.getInstance();
            } else if (first instanceof AtomicValue) {
                ItemMappingFunction atomicValueChecker = new ItemMappingFunction() {
                    public Item map(Item item) throws XPathException {
                        if (item instanceof AtomicValue) {
                            return item;
                        } else {
                            throw reportMixedItems(loc, context);
                        }
                    }
                };
                return new ItemMappingIterator(result.getAnother(), atomicValueChecker);
            } else {
                ItemMappingFunction nodeChecker = new ItemMappingFunction() {
                    public Item map(Item item) throws XPathException {
                        if (item instanceof NodeInfo) {
                            return item;
                        } else {
                            throw reportMixedItems(loc, context);
                        }
                    }
                };
                return new DocumentOrderIterator(
                    new ItemMappingIterator(result.getAnother(), nodeChecker),
                    GlobalOrderComparer.getInstance());
            }
        } else {
            return result;
        }
    }

    private XPathException reportMixedItems(SourceLocator loc, XPathContext context) {
        XPathException err = new XPathException("Cannot mix nodes and atomic values in the result of a path expression");
        err.setErrorCode("XPTY0018");
        err.setLocator(loc);
        err.setXPathContext(context);
        return err;
    }

    /**
     * Mapping function, from a node returned by the start iteration, to a sequence
     * returned by the child.
     */

    public SequenceIterator map(XPathContext context) throws XPathException {
        return step.iterate(context);
    }

    /**
     * Diagnostic print of expression structure. The abstract expression tree
     * is written to the supplied output destination.
     */

    public void explain(ExpressionPresenter destination) {
        destination.startElement("atomicMap");
        start.explain(destination);
        step.explain(destination);
        destination.endElement();
    }
}


//
// The contents of this file are subject to the Mozilla Public License Version 1.0 (the "License");
// you may not use this file except in compliance with the License. You may obtain a copy of the
// License at http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied.
// See the License for the specific language governing rights and limitations under the License.
//
// The Original Code is: all this file.
//
// The Initial Developer of the Original Code is Michael H. Kay.
//
// Portions created by (your name) are Copyright (C) (your legal entity). All Rights Reserved.
//
// Contributor(s): none.
//
