/*
 * nss_sshsock: NSS module for providing NSS services from a remote
 * ssh server over a SSH socket.
 *
 * Copyright (C) 2011 Scott Balneaves <sbalneav@ltsp.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <nss.h>
#include <grp.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include "nss_sshsock.h"

/*
 * Needed for getgrent() statefulness.
 */

static FILE *gproc = NULL;

/*
 * buffer_to_grstruct:
 *
 * Given a buffer containing a single group(5) line, populate a
 * group struct.  Note that the ':' in the buffer are converted
 * to '\0' by the call to split.
 */

static enum nss_status
buffer_to_grstruct (struct group *grstruct, char *newbuf, char *buffer, size_t buflen, int *errnop)
{
  char **array;
  size_t count;
  char *c;
  char *bufstart;
  char *groupmembers;
  char **arraystart = (char **) buffer;
  size_t offset;

  if (!grstruct)
    {
      return NSS_STATUS_UNAVAIL;
    }

  /*
   * Here's how we'll lay out our buffer:
   *
   * | array of char | grname\0 | pwd\0 | gid\0 | group members terminated by \0 |
   * | pointers with |          |       |       |                                |
   * | null term.    |          |       |       |                                |
   *
   * The array of pointers are the pointers into the group members part of the
   * buffer.
   */

  /*
   * First, count # of ','s in the alloc'd buffer, so we know how
   * much space for the char * array we need to reserve. Start off
   * with a minimum size of 2, since if we find no ','s, it may mean
   * we have one group member, and we need a NULL for a terminator.
   */

  count = 2;

  for (c = newbuf; *c != '\0'; c++)
    {
      if (*c == ',')
        {
          count++;
        }
    }

  /*
   * So, we need to reserve (size of char ptr) * count
   * bytes at the beginning of the buffer.
   */

  offset = sizeof (char *) * count;
  bufstart = (buffer + offset);

  /*
   * Not enough space?
   */

  if ((strlen (newbuf) + 1 + offset) > buflen)
    {
      *errnop = ERANGE;
      return NSS_STATUS_TRYAGAIN;
    }

  /*
   * copy buffer into new location
   */

  strcpy (bufstart, newbuf);

  /*
   * split on ':'
   */

  array = split (bufstart, ":");

  if (!array)
    {
      /* couldn't split on the buffer */
      return NSS_STATUS_UNAVAIL;
    }

  /*
   * sanity check: we should have 4 fields
   */

  for (count = 0; array[count] != NULL; count++)
    {
    }

  if (count != 4)
    {
      free (array);
      return NSS_STATUS_UNAVAIL;
    }

  /*
   * Populate the grstruct
   */

  grstruct->gr_name   = array[0];
  grstruct->gr_passwd = array[1];
  grstruct->gr_gid    = (gid_t) atoi (array[2]);
  grstruct->gr_mem    = arraystart;
  groupmembers        = array[3];

  free (array);

  /*
   * groupmembers now contains the list
   * of comma separated group members.
   * Re split on ','
   */

  array = split (groupmembers, ",");

  if (!array)
    {
      /* couldn't split on the buffer */
      return NSS_STATUS_UNAVAIL;
    }

  /*
   * Copy array into the space reserved in the
   * buffer.  Make sure to NULL terminate at the end.
   */

  for (count = 0; array[count] != NULL; count++)
    {
      arraystart[count] = array[count];
    }

  arraystart[count] = NULL;

  free (array);

  return NSS_STATUS_SUCCESS;
}

/*
 * getprocline:
 *
 * Passed a FILE *, gets a line and does some sanity.
 */

static enum nss_status
getprocline (FILE *proc, struct group *result, char *buffer, size_t buflen, int *errnop)
{
  int linenl;
  char *newbuf;
  enum nss_status status;

  /*
   * Allocate a new buffer
   */

  if (!(newbuf = malloc (buflen)))
    {
      return NSS_STATUS_UNAVAIL;
    }

  /*
   * Grab our output line.
   */

  linenl = fgets_nonl (newbuf, buflen, proc);

  /*
   * Did we get any result back?
   */

  if (*newbuf == '\0')
    {
      free (newbuf);
      return NSS_STATUS_NOTFOUND;
    }

  /*
   * Do we have enough space in the buffer we're given to hold the result?
   */

  if (!linenl)
    {
      free (newbuf);
      *errnop = ERANGE;
      return NSS_STATUS_TRYAGAIN;
    }

  status = buffer_to_grstruct (result, newbuf, buffer, buflen, errnop);
  free (newbuf);
  return status;
}

/*
 * search:
 *
 * When passed a command, get the result and populate.
 */

static enum nss_status
search (char *command, struct group *result, char *buffer, size_t buflen, int *errnop)
{
  FILE *proc;
  enum nss_status status;

  *errnop = 0;

  proc = sshopen (command);

  if (!proc)
    {
      return NSS_STATUS_UNAVAIL;
    }

  status = getprocline (proc, result, buffer, buflen, errnop);

  sshclose (proc);

  return status;
}

/*
 * _nss_sshsock_getgrgid_r
 *
 * Implement getgrgid() functionality.
 */

enum nss_status
_nss_sshsock_getgrgid_r (gid_t gid, struct group *result, char *buffer, size_t buflen, int *errnop)
{
  char command[CHUNKSIZ];

  if (gid < MINGID)
    {
      return NSS_STATUS_NOTFOUND;
    }

  snprintf (command, sizeof command, "getent group %u", gid);

  return search (command, result, buffer, buflen, errnop);
}

/*
 * _nss_sshsock_getgrnam_r
 *
 * Implement getgrnam() functionality.
 */

enum nss_status
_nss_sshsock_getgrnam_r(const char *name, struct group *result, char *buffer, size_t buflen, int *errnop)
{
  char command[CHUNKSIZ];
  enum nss_status status;

  snprintf (command, sizeof command, "getent group '%s'", name);

  status = search (command, result, buffer, buflen, errnop);

  if (status == NSS_STATUS_SUCCESS)
    {
      if (result->gr_gid < MINGID)
	{
	  return NSS_STATUS_NOTFOUND;
	}
    }

  return status;
}


/*
 * _nss_sshsock_setgrent
 *
 * Implements setgrent() functionality.
 */

enum nss_status
_nss_sshsock_setgrent (void)
{
  if (gproc)
    {
      sshclose (gproc);
      gproc = NULL;
    }

  gproc = sshopen ("getent group");

  if (!gproc)
    {
      return NSS_STATUS_UNAVAIL;
    }

  return NSS_STATUS_SUCCESS;
}

/*
 * _nss_sshsock_getgrent_r
 *
 * Implements getgrent() functionality
 */

enum nss_status
_nss_sshsock_getgrent_r (struct group *result, char *buffer, size_t buflen, int *errnop)
{
  enum nss_status status;

  *errnop = 0;

  if (!gproc) /* Got a file to work on? */
    {
      return NSS_STATUS_UNAVAIL;
    }

  while (1)
    {
      status = getprocline (gproc, result, buffer, buflen, errnop);

      if ((status != NSS_STATUS_SUCCESS) ||
	  (result->gr_gid >= MINGID))
        {
          return status;
        }
    }
}

/*
 * _nss_sshsock_endgrent
 *
 * Implements the endgrent() functionality.
 */

enum nss_status
_nss_sshsock_endgrent (void)
{
  if (gproc)
    {
      sshclose (gproc);
      gproc = NULL;
    }

  return NSS_STATUS_SUCCESS;
}
