/*
 *  Copyright 1994-2011 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#include "erlang.h"

/*
 * Protocol v7:
 *
 * Input:
 * Output:
 *  [C][To, Cmd, Arg]        // keyboard command
 *  [U][C|P|S][uint32_t id]  // update event
 */

extern int write_exact(const u_char *, const int);


extern uint32_t last_colormap;
extern uint32_t last_picture;
extern uint32_t last_sequence;

extern char send_colormap_update;
extern char send_picture_update;
extern char send_sequence_update;


static void
v7_send_event(const Event_t *e)
{
  uint32_t proto = 7, proto2;
  uint32_t total, total2;
  Pixel_t *src;
  u_char val;
  int ret;

  /* send packet size */
  total = sizeof(uint32_t)+4*sizeof(u_char);
  total2 = htonl(total);
  ret = write_exact((const u_char *)&total2, sizeof(uint32_t));

  /* protocol version */
  proto2 = htonl(proto);
  ret = write_exact((const u_char *)&proto2, sizeof(uint32_t));

  val = 'C';
  ret = write_exact(&val, sizeof(u_char));
  val = e->to;
  ret = write_exact(&val, sizeof(u_char));
  val = e->cmd;
  ret = write_exact(&val, sizeof(u_char));
  val = e->arg0;
  ret = write_exact(&val, sizeof(u_char));
}


static void
v7_send_events(Context_t *ctx)
{
  GList *t;

  for (t = ctx->events; t != NULL; t = g_list_next(t)) {
    Event_t *e = (Event_t *)t->data;
    v7_send_event(e);
  }
}


static void
v7_send_id(const u_char what, const uint32_t id)
{
  uint32_t proto = 7, proto2;
  uint32_t total, total2;
  uint32_t id2;
  Pixel_t *src;
  u_char val;
  int ret;

  /* send packet size */
  total = sizeof(uint32_t)+2*sizeof(u_char)+sizeof(uint32_t);
  total2 = htonl(total);
  ret = write_exact((const u_char *)&total2, sizeof(uint32_t));

  /* protocol version */
  proto2 = htonl(proto);
  ret = write_exact((const u_char *)&proto2, sizeof(uint32_t));

  val = UPDATE_CHAR;
  ret = write_exact(&val, sizeof(u_char));
  val = what;
  ret = write_exact(&val, sizeof(u_char));

  id2 = htonl(id);
  ret = write_exact((const u_char *)&id2, sizeof(uint32_t));
}


static void
v7_send_sequence(const uint32_t id)
{
  uint32_t proto = 7, proto2;
  uint32_t total, total2;
  uint32_t id2;
  Pixel_t *src;
  u_char val;
  int ret;
  Sequence_t *seq;
  GList *tmp;
  uint8_t seq_size, i;

  seq = Sequences_find(id);
  assert(NULL != seq);
  seq_size = Sequence_size(seq);

  /* send packet size */
  total = sizeof(uint32_t)           /* proto */
    + 2*sizeof(u_char)               /* 'US' */
    + sizeof(uint32_t)               /* sequence id */
    + sizeof(uint8_t)                /* sequence size */
    + (seq_size * sizeof(uint32_t)); /* N plugins */ /* TODO plugin_id -> uint32_t */
  total2 = htonl(total);
  ret = write_exact((const u_char *)&total2, sizeof(uint32_t));

  /* protocol version */
  proto2 = htonl(proto);
  ret = write_exact((const u_char *)&proto2, sizeof(uint32_t));

  val = UPDATE_CHAR;
  ret = write_exact(&val, sizeof(u_char));
  val = UPDATE_SEQUENCE_CHAR;
  ret = write_exact(&val, sizeof(u_char));

  /* sequence id */
  id2 = htonl(id);
  ret = write_exact((const u_char *)&id2, sizeof(uint32_t));

  /* the sequence */
  ret = write_exact((const u_char *)&seq_size, sizeof(uint8_t));
  tmp = seq->layers;
  for (i = 0; i < seq_size; i++) {
    Layer_t *l = (Layer_t *)tmp->data;

    id2 = htonl(l->plugin->id);
    ret = write_exact((const u_char *)&id2, sizeof(uint32_t));
    tmp = g_list_next(tmp);
  }
}


void
v7(Context_t *ctx)
{
  v7_send_events(ctx);

  if (send_colormap_update)
    if (ctx->sm->cur->cmap_id != last_colormap) {
      v7_send_id(UPDATE_COLORMAP_CHAR, ctx->sm->cur->cmap_id);
      last_colormap = ctx->sm->cur->cmap_id;
    }

  if (send_picture_update)
    if (ctx->sm->cur->picture_id != last_picture) {
      v7_send_id(UPDATE_PICTURE_CHAR, ctx->sm->cur->picture_id);
      last_picture = ctx->sm->cur->picture_id;
    }

  if (send_sequence_update)
    if (ctx->sm->cur->id != last_sequence) {
      v7_send_sequence(ctx->sm->cur->id);
      last_sequence = ctx->sm->cur->id;
    }
}
