/***************************************************************************
 *   Copyright (C) 2008 by Jean-Baptiste Mardelle (jb@kdenlive.org)        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA          *
 ***************************************************************************/

/**
 * @class ClipManager
 * @brief Manages the list of clips in a document.
 * @author Jean-Baptiste Mardelle
 */

#ifndef CLIPMANAGER_H
#define CLIPMANAGER_H

#include <qdom.h>
#include <QPixmap>
#include <QObject>
#include <QTimer>
#include <QMutex>
#include <QFuture>

#include <KUrl>
#include <KUndoStack>
#include <KDirWatch>
#include <klocale.h>
#include <kdeversion.h>

#if KDE_IS_VERSION(4,5,0)
#include <KImageCache>
#endif


#include "gentime.h"
#include "definitions.h"


class KdenliveDoc;
class DocClipBase;
class AbstractGroupItem;

namespace Mlt
{
class Producer;
};

class ClipManager: public QObject
{
Q_OBJECT public:

    ClipManager(KdenliveDoc *doc);
    virtual ~ ClipManager();
    void addClip(DocClipBase *clip);
    DocClipBase *getClipAt(int pos);
    void deleteClip(const QString &clipId);

    /** @brief Add a file to the project.
     * @ref slotAddClipList
     * @param url file to add
     * @param group name of the group to insert the file in (can be empty)
     * @param groupId id of the group (if any) */
    void slotAddClipFile(const KUrl &url, const QString &group, const QString &groupId);

    /** @brief Adds a list of files to the project.
     * @param urls files to add
     * @param group name of the group to insert the files in (can be empty)
     * @param groupId id of the group (if any)
     * It checks for duplicated items and asks to the user for instructions. */
    void slotAddClipList(const KUrl::List urls, const QString &group, const QString &groupId);
    void slotAddTextClipFile(const QString &titleName, int out, const QString &xml, const QString &group, const QString &groupId);
    void slotAddTextTemplateClip(QString titleName, const KUrl &path, const QString &group, const QString &groupId);
    void slotAddXmlClipFile(const QString &name, const QDomElement &xml, const QString &group, const QString &groupId);
    void slotAddColorClipFile(const QString &name, const QString &color, QString duration, const QString &group, const QString &groupId);
    void slotAddSlideshowClipFile(const QString &name, const QString &path, int count, const QString &duration,
                                  const bool loop, const bool crop,const bool fade,
                                  const QString &luma_duration, const QString &luma_file, const int softness,
                                  const QString &animation, const QString &group, const QString &groupId);
    DocClipBase *getClipById(QString clipId);
    const QList <DocClipBase *> getClipByResource(QString resource);
    void slotDeleteClips(QStringList ids);
    void setThumbsProgress(const QString &message, int progress);
    void checkAudioThumbs();
    QList <DocClipBase*> documentClipList() const;
    QMap <QString, QString> documentFolderList() const;
    int getFreeClipId();
    int getFreeFolderId();
    int lastClipId() const;
    void startAudioThumbsGeneration();
    void endAudioThumbsGeneration(const QString &requestedId);
    void askForAudioThumb(const QString &id);
    QString projectFolder() const;
    void clearUnusedProducers();
    void resetProducersList(const QList <Mlt::Producer *> prods, bool displayRatioChanged, bool fpsChanged);
    void addFolder(const QString&, const QString&);
    void deleteFolder(const QString&);
    void clear();
    void clearCache();
    AbstractGroupItem *createGroup();
    void removeGroup(AbstractGroupItem *group);
    QDomElement groupsXml() const;
    int clipsCount() const;
    /** @brief Request creation of a clip thumbnail for specified frames. */
    void requestThumbs(const QString id, QList <int> frames);
    /** @brief remove a clip id from the queue list. */
    void stopThumbs(const QString &id);

#if KDE_IS_VERSION(4,5,0)
    KImageCache* pixmapCache;
#endif

private slots:
    /** A clip was externally modified, monitor for more changes and prepare for reload */
    void slotClipModified(const QString &path);
    void slotClipMissing(const QString &path);
    void slotClipAvailable(const QString &path);
    /** Check the list of externally modified clips, and process them if they were not modified in the last 1500 milliseconds */
    void slotProcessModifiedClips();
    void slotGetThumbs();

private:   // Private attributes
    /** the list of clips in the document */
    QList <DocClipBase*> m_clipList;
    /** the list of groups in the document */
    QList <AbstractGroupItem *> m_groupsList;
    QMap <QString, QString> m_folderList;
    QList <QString> m_audioThumbsQueue;
    /** the document undo stack*/
    KdenliveDoc *m_doc;
    int m_clipIdCounter;
    int m_folderIdCounter;
    QString m_generatingAudioId;
    KDirWatch m_fileWatcher;
    /** Timer used to reload clips when they have been externally modified */
    QTimer m_modifiedTimer;
    /** List of the clip IDs that need to be reloaded after being externally modified */
    QMap <QString, QTime> m_modifiedClips;
    /** Struct containing the list of clip thumbnails to request (clip id and frames) */
    QMap <QString, int> m_requestedThumbs;
    QMutex m_thumbsMutex;
    QFuture<void> m_thumbsThread;
    /** @brief If true, abort processing of clip thumbs before removing a clip. */
    bool m_abortThumb;
    /** @brief We are about to delete the clip producer, stop processing thumbs. */
    bool m_closing;

signals:
    void reloadClip(const QString &);
    void modifiedClip(const QString &);
    void missingClip(const QString &);
    void availableClip(const QString &);
    void checkAllClips(bool displayRatioChanged, bool fpsChanged);
};

#endif
