# Copyright (C) 2007 Samuel Abels <http://debain.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2, as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
import goocanvas
import gobject
import calendar
import pango

class TimelineItem(goocanvas.Group):
    """
    A canvas item representing a timeline.
    """
    def __init__(self, cal, **kwargs):
        """
        Constructor.
        """
        goocanvas.Group.__init__(self)

        self.cal        = cal
        self.x          = kwargs.get('x')
        self.y          = kwargs.get('y')
        self.width      = kwargs.get('width')
        self.height     = kwargs.get('height')
        self.line_color = kwargs.get('line_color')
        self.bg_color   = kwargs.get('bg_color')
        self.text_color = kwargs.get('text_color')

        # Create canvas items.
        self.timeline_rect = {}
        self.timeline_text = {}
        for n in range(0, 24):
            caption = '%s:00' % n
            self.timeline_rect[n] = goocanvas.Rect(parent = self)
            self.timeline_text[n] = goocanvas.Text(parent = self,
                                                   text   = caption)

        if self.x is not None:
            self.update()


    def update(self):
        text_padding = max(self.width / 20, 4)
        text_width   = self.width - 2 * text_padding
        text_height  = text_width / 5
        line_height  = self.height / 24
        style        = self.cal.get_style()
        font_descr   = style.font_desc.copy()
        font_descr.set_absolute_size(text_height * pango.SCALE)
        font         = font_descr.to_string()

        # Draw the timeline.
        for n in range(0, 24):
            rect    = self.timeline_rect[n]
            text    = self.timeline_text[n]
            y       = self.y + n * line_height

            rect.set_property('x',            self.x)
            rect.set_property('y',            y)
            rect.set_property('width',        self.width)
            rect.set_property('height',       line_height)
            rect.set_property('stroke_color', self.line_color)
            rect.set_property('fill_color',   self.bg_color)

            text.set_property('x',          self.x + text_padding)
            text.set_property('y',          y + text_padding)
            text.set_property('font',       font)
            text.set_property('fill_color', self.text_color)
