// ****************************************************************************
//  Project:        GUYMAGER
// ****************************************************************************
//  Programmer:     Guy Voncken
//                  Police Grand-Ducale
//                  Service de Police Judiciaire
//                  Section Nouvelles Technologies
// ****************************************************************************
//  Module:         The table widget (central widget of the application)
// ****************************************************************************

#include <QtGui>

#include "toolconstants.h"
#include "common.h"
#include "config.h"
#include "threadread.h"
#include "threadwrite.h"
#include "threadhash.h"
#include "threadcompress.h"
#include "dlgmessage.h"
#include "dlgacquire.h"
#include "itemdelegate.h"
#include "table.h"
#include "compileinfo.h"
#include "dlgabort.h"
#include "file.h"


class t_TableLocal
{
   public:
      t_pDeviceList  pDeviceList;
      QAction       *pActionAcquire;
      QAction       *pActionAbort;
      QAction       *pActionDeviceInfo;
      bool            SlowDownAcquisitions;
};

t_Table::t_Table ()
{
   CHK_EXIT (ERROR_TABLE_CONSTRUCTOR_NOT_SUPPORTED)
} //lint !e1401 not initialised

t_Table::t_Table (QWidget *pParent, t_pDeviceList pDeviceList)
   :QTableView (pParent)
{
   CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_TABLE_INVALID_ACTION                ))
   CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_TABLE_THREADREAD_ALREADY_RUNNING    ))
   CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_TABLE_THREADHASH_ALREADY_RUNNING    ))
   CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_TABLE_THREADWRITE_ALREADY_RUNNING   ))
   CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_TABLE_THREADCOMPRESS_ALREADY_RUNNING))
   CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_TABLE_THREADREAD_DOESNT_EXIT        ))
   CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_TABLE_FIFO_EXISTS                   ))
   CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_TABLE_CONSTRUCTOR_NOT_SUPPORTED     ))
   CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_TABLE_INVALID_FORMAT                ))

   setItemDelegate      (new t_ItemDelegate (this));
   setSelectionBehavior (QAbstractItemView::SelectRows     );
   setSelectionMode     (QAbstractItemView::SingleSelection);

   pOwn = new t_TableLocal;
   pOwn->pDeviceList          = pDeviceList;
   pOwn->SlowDownAcquisitions = false;

   verticalHeader  ()->hide                  ();
   horizontalHeader()->setClickable          (true);
   horizontalHeader()->setSortIndicatorShown (true);

   pOwn->pActionAcquire    = new QAction(tr("Acquire"          , "Context menu"), this);  addAction (pOwn->pActionAcquire   );
   pOwn->pActionAbort      = new QAction(tr("Abort acquisition", "Context menu"), this);  addAction (pOwn->pActionAbort     );
   pOwn->pActionDeviceInfo = new QAction(tr("Info"             , "Context menu"), this);  addAction (pOwn->pActionDeviceInfo);

   CHK_QT_EXIT (connect (this, SIGNAL(clicked (const QModelIndex &)), this, SLOT(SlotMouseClick(const QModelIndex &))))


//   pOwn->pTable->setContextMenuPolicy (Qt::ActionsContextMenu);
}

t_Table::~t_Table ()
{
   delete pOwn;
}

APIRET t_Table::GetDeviceUnderCursor (t_pDevice &pDevice)
{
   QModelIndex Index;
   int         Row;
   int         Count;

   pDevice = NULL;
   QModelIndexList IndexList = selectionModel()->selectedRows();
   Count = IndexList.count();
   switch (Count)
   {
      case 0:  break;
      case 1:  Index = IndexList.first();
               Row = model()->data(Index, t_ItemDelegate::RowNrRole).toInt();
               pDevice = pOwn->pDeviceList->at(Row);
               break;
      default: LOG_ERROR ("Strange, several rows seem to be selected.")
   }

   return NO_ERROR;
}

void t_Table::contextMenuEvent (QContextMenuEvent *pEvent)
{
   QAction     *pAction;
   QAction       LocalIndicator (tr("Local Device - cannot be acquired"), this);
   QMenu         Menu;
   QPoint        Pos;
   t_pDevice    pDevice;
   int           CorrectedY;
   int           Row;
   QModelIndex   Index;
   bool          Running;

   Pos = mapFromGlobal (pEvent->globalPos());           // The event coordinates are absolute screen coordinates, map them withb this widget
   CorrectedY = Pos.y() - horizontalHeader()->height(); // QTableView::rowAt seems to have a bug: It doesn't include the header height in the calculations. So, we correct it here.
   Pos.setY (CorrectedY);

   Index = indexAt (Pos);
   selectRow (Index.row());
   Row = model()->data(Index, t_ItemDelegate::RowNrRole).toInt();

   if ((Row < 0) || (Row >= pOwn->pDeviceList->count())) // User clicked in the empty area
      return;

   pDevice = pOwn->pDeviceList->at(Row);
//   LOG_INFO ("Context global %d -- widget %d -- corrected %d -- index %d", pEvent->globalPos().y(), Pos.y(), CorrectedY, RowIndex);

   Running = pDevice->pThreadRead || pDevice->pThreadWrite;

   pOwn->pActionAcquire->setEnabled (!Running);
   pOwn->pActionAbort  ->setEnabled ( Running);

   if (pDevice->Local)
   {
      LocalIndicator.setEnabled (false);
      Menu.addAction (&LocalIndicator);
      Menu.addSeparator ();
   }
   else
   {
      Menu.addAction (pOwn->pActionAcquire);
      Menu.addAction (pOwn->pActionAbort  );
   }
   Menu.addAction (pOwn->pActionDeviceInfo);

   pAction = Menu.exec (pEvent->globalPos());

//   pAction = QMenu::exec (actions(), pEvent->globalPos());
   if      (pAction == pOwn->pActionAcquire   ) CHK_EXIT (StartAcquisition  (pDevice))
   if      (pAction == pOwn->pActionAbort     ) CHK_EXIT (AbortAcquisition  (pDevice))
   else if (pAction == pOwn->pActionDeviceInfo) CHK_EXIT (ShowDeviceInfo    (pDevice))
   else
   {
//      CHK_EXIT (ERROR_TABLE_INVALID_ACTION)
   }
}

void t_Table::SlotMouseClick (const QModelIndex & Index)
{
   int Row;

   selectRow (Index.row());
   Row = model()->data (Index, t_ItemDelegate::RowNrRole).toInt();
   emit (SignalDeviceSelected (pOwn->pDeviceList->at(Row)));
}


APIRET t_Table::ShowDeviceInfo (t_pcDevice pDevice)
{
   QString Info;

   LOG_INFO ("Info for %s requested", QSTR_TO_PSZ(pDevice->LinuxDevice))
   pOwn->SlowDownAcquisitions = true;
   CHK (t_Info::GetDeviceInfo (pDevice, true, Info))
   pOwn->SlowDownAcquisitions = false;
   CHK (t_DlgMessage::Show (tr("Device info", "Dialog title"), Info, true))

   return NO_ERROR;
}

// ----------------------------------------------------------------------------------------------------------------
//
//                                                     Acquistion start
//
// ----------------------------------------------------------------------------------------------------------------



APIRET t_Table::InfoAcquisitionStart (t_pDevice pDevice)
{
   const char *pLibGuyToolsVersionInstalled;
   const char *pLibEwfVersionInstalled;
   QString      FormatDescription;
   QString      FormatExtension;

   CHK (pDevice->Info.Create())
   CHK (pDevice->Info.Title   (tr("GUYMAGER ACQUISITION INFO FILE", "Info file")))

   t_Log::GetLibGuyToolsVersion (&pLibGuyToolsVersionInstalled);
   pLibEwfVersionInstalled = libewf_get_version ();

   CHK (pDevice->Info.WriteLn ())
   CHK (pDevice->Info.Title   (tr("Guymager", "Info file")))
   CHK (pDevice->Info.WriteLn ())
   CHK (pDevice->Info.AddRow (tr("Version:: %1"              , "Info file") .arg(pCompileInfoVersion  )))
   CHK (pDevice->Info.AddRow (tr("Compilation timestamp:: %1", "Info file") .arg(pCompileInfoTimestamp)))
   CHK (pDevice->Info.AddRow (tr("Compiled with:: %1 %2"     , "Info file") .arg("gcc") .arg(__VERSION__)))
   CHK (pDevice->Info.AddRow (tr("libewf version:: %1"       , "Info file") .arg(pLibEwfVersionInstalled     )))
   CHK (pDevice->Info.AddRow (tr("libguytools version:: %1"  , "Info file") .arg(pLibGuyToolsVersionInstalled)))
   CHK (pDevice->Info.WriteTable ());

   CHK (pDevice->Info.WriteLn ())
   CHK (pDevice->Info.WriteLn ())
   CHK (pDevice->Info.Title   (tr("Device information", "Info file")))
   CHK (pDevice->Info.WriteLn ())
   CHK (pDevice->Info.WriteDeviceInfo ())

   CHK (pDevice->Info.WriteLn ())
   CHK (pDevice->Info.WriteLn ())
   CHK (pDevice->Info.Title   (tr("Image", "Info file")))
   CHK (pDevice->Info.WriteLn ())
   CHK (pDevice->Info.AddRow (tr("Linux device:: %1"    , "Info file").arg(pDevice->LinuxDevice) ))
   CHK (pDevice->Info.AddRow (tr("Device size:: %1 (%2)", "Info file").arg(pDevice->Size) .arg(t_Device::GetSizeHuman(pDevice).toString())))
   CHK (pDevice->Info.AddRow (tr("Image path and file name:: %1 %2" , "Info file").arg(pDevice->Acquisition.ImagePath) .arg(pDevice->Acquisition.ImageFilename)))
   CHK (pDevice->Info.AddRow (tr("Info  path and file name:: %1 %2" , "Info file").arg(pDevice->Acquisition.InfoPath ) .arg(pDevice->Acquisition.InfoFilename )))

   CHK (t_File::GetFormatDescription (pDevice->Acquisition.Format, FormatDescription))
   CHK (t_File::GetFormatExtension   (pDevice->Acquisition.Format, NULL, &FormatExtension))

   CHK (pDevice->Info.AddRow (tr("Image format:: %1 - file extension is %2", "Info file")  .arg(FormatDescription) .arg(FormatExtension)))

   if (pDevice->Acquisition.CalcHashes)
        CHK (pDevice->Info.AddRow (tr("Hash calculation:: on" , "Info file")))
   else CHK (pDevice->Info.AddRow (tr("Hash calculation:: off", "Info file")))

   if (pDevice->Acquisition.VerifySource)
        CHK (pDevice->Info.AddRow (tr("Source verification:: on" , "Info file")))
   else CHK (pDevice->Info.AddRow (tr("Source verification:: off", "Info file")))
   CHK (pDevice->Info.WriteTable ());

   return NO_ERROR;
}

APIRET t_Table::StartAcquisition (t_pDevice pDevice)
{
   t_DlgAcquire Dlg (pDevice, this);
   QString      Format;
   int          Fifos=0;

   if (Dlg.exec() == QDialog::Accepted)
   {
      pOwn->SlowDownAcquisitions = true;
      LOG_INFO ("Starting acquisition for %s", QSTR_TO_PSZ(pDevice->LinuxDevice))
      CHK (Dlg.GetParameters (pDevice->Acquisition))
      CHK (t_File::GetFormatExtension   (pDevice->Acquisition.Format, NULL, &Format))
      LOG_INFO ("Image %s%s%s", QSTR_TO_PSZ(pDevice->Acquisition.ImagePath), QSTR_TO_PSZ(pDevice->Acquisition.ImageFilename), QSTR_TO_PSZ(Format))
      LOG_INFO ("Info  %s%s%s" , QSTR_TO_PSZ(pDevice->Acquisition.InfoPath ), QSTR_TO_PSZ(pDevice->Acquisition.InfoFilename ), t_File::pExtensionInfo)

      CHK (pDevice->SetMessage (QString()))
      pDevice->State               = t_Device::Acquire;
      pDevice->AbortReason         = t_Device::None;
      pDevice->AbortRequest        = false;
      pDevice->DeleteAfterAbort    = false;
      pDevice->StartTimestampVerify= QDateTime();
      pDevice->StopTimestamp       = QDateTime();
      pDevice->PrevTimestamp       = QTime();
      pDevice->PrevSpeed           = 0.0;
      pDevice->PrevPos             = 0;
      pDevice->SetCurrentWritePos (0LL);
      pDevice->SetCurrentVerifyPos(0LL);
      CHK (pDevice->ClearBadSectors ())

      memset (&pDevice->MD5Digest         , 0, sizeof(pDevice->MD5Digest         ));
      memset (&pDevice->MD5DigestVerify   , 0, sizeof(pDevice->MD5DigestVerify   ));
      memset (&pDevice->SHA256Digest      , 0, sizeof(pDevice->SHA256Digest      ));
      memset (&pDevice->SHA256DigestVerify, 0, sizeof(pDevice->SHA256DigestVerify));
      pDevice->StartTimestamp = QDateTime::currentDateTime();
      CHK (InfoAcquisitionStart (pDevice))

      if (pDevice->pThreadRead  != NULL) CHK_EXIT (ERROR_TABLE_THREADREAD_ALREADY_RUNNING)
      if (pDevice->pThreadWrite != NULL) CHK_EXIT (ERROR_TABLE_THREADWRITE_ALREADY_RUNNING)
      if (pDevice->pThreadHash  != NULL) CHK_EXIT (ERROR_TABLE_THREADHASH_ALREADY_RUNNING)
      if (!pDevice->ThreadCompressList.isEmpty()) CHK_EXIT (ERROR_TABLE_THREADCOMPRESS_ALREADY_RUNNING)
      if ((pDevice->pFifoRead        != NULL) ||
          (pDevice->pFifoHashIn      != NULL) ||
          (pDevice->pFifoHashOut     != NULL) ||
          (pDevice->pFifoCompressIn  != NULL) ||
          (pDevice->pFifoCompressOut != NULL))
         CHK_EXIT (ERROR_TABLE_FIFO_EXISTS)

      // Create threads
      // --------------
      pDevice->pThreadRead  = new t_ThreadRead  (pDevice, &pOwn->SlowDownAcquisitions);
      pDevice->pThreadWrite = new t_ThreadWrite (pDevice, &pOwn->SlowDownAcquisitions);
      CHK_QT_EXIT (connect (pDevice->pThreadRead , SIGNAL(SignalEnded(t_pDevice)), this, SLOT(SlotThreadReadFinished (t_pDevice))))
      CHK_QT_EXIT (connect (pDevice->pThreadWrite, SIGNAL(SignalEnded(t_pDevice)), this, SLOT(SlotThreadWriteFinished(t_pDevice))))

      if (pDevice->HasHashThread())
      {
         pDevice->pThreadHash = new t_ThreadHash (pDevice);
         CHK_QT_EXIT (connect (pDevice->pThreadHash, SIGNAL(SignalEnded(t_pDevice)), this, SLOT(SlotThreadHashFinished (t_pDevice))))
      }

      if (pDevice->HasCompressionThreads())
      {
         for (int i=0; i < CONFIG(CompressionThreads); i++)
         {
            t_pThreadCompress pThread;

            pThread = new t_ThreadCompress (pDevice, i);
            pDevice->ThreadCompressList.append (pThread);
            CHK_QT_EXIT (connect (pThread, SIGNAL(SignalEnded(t_pDevice,int)), this, SLOT(SlotThreadCompressFinished (t_pDevice,int))))
         }
      }

      // Create fifos and assign threads
      // -------------------------------

      switch (pDevice->Acquisition.Format)
      {
         case t_File::DD : pDevice->FifoBlockSize = CONFIG(FifoBlockSizeDD );   break;
         case t_File::AFF: pDevice->FifoBlockSize = CONFIG(FifoBlockSizeAFF);   break;
         case t_File::EWF:
            if (pDevice->HasCompressionThreads())
                 pDevice->FifoBlockSize = EWF_MULTITHREADED_COMPRESSION_CHUNK_SIZE;
            else pDevice->FifoBlockSize = CONFIG(FifoBlockSizeEWF);
            if (pDevice->FifoBlockSize != (unsigned int) CONFIG(FifoBlockSizeEWF))
               LOG_INFO ("Running with FifoBlockSize of %d (no other size supported when running with multithreaded EWF compression)", pDevice->FifoBlockSize)
            break;
         default:  CHK (ERROR_TABLE_INVALID_FORMAT)
      }

      #define FIFO_MAX_BLOCKS(Fifos0)   std::max ((int)((((long long)CONFIG(FifoMaxMem))*BYTES_PER_MEGABYTE) / ((long long)pDevice->FifoBlockSize * (Fifos0))), 1)

      if      (!pDevice->HasHashThread() && !pDevice->HasCompressionThreads())
      {
         Fifos = 1;
         pDevice->FifoMaxBlocks = FIFO_MAX_BLOCKS (Fifos);
         pDevice->pFifoRead     = new t_FifoStd (pDevice->FifoMaxBlocks);
         pDevice->pFifoWrite    = pDevice->pFifoRead;
      }
      else if ( pDevice->HasHashThread() && !pDevice->HasCompressionThreads())
      {
         Fifos = 2;
         pDevice->FifoMaxBlocks = FIFO_MAX_BLOCKS (Fifos);
         pDevice->pFifoRead     = new t_FifoStd (pDevice->FifoMaxBlocks);
         pDevice->pFifoHashIn   = pDevice->pFifoRead;
         pDevice->pFifoHashOut  = new t_FifoStd (pDevice->FifoMaxBlocks);
         pDevice->pFifoWrite    = pDevice->pFifoHashOut;
      }
      else if (!pDevice->HasHashThread() &&  pDevice->HasCompressionThreads())
      {
         Fifos = 2 * CONFIG(CompressionThreads);
         pDevice->FifoMaxBlocks    = FIFO_MAX_BLOCKS (Fifos);
         pDevice->pFifoCompressIn  = new t_FifoCompressIn  (CONFIG(CompressionThreads), pDevice->FifoMaxBlocks);
         pDevice->pFifoRead        = pDevice->pFifoCompressIn;
         pDevice->pFifoCompressOut = new t_FifoCompressOut (CONFIG(CompressionThreads), pDevice->FifoMaxBlocks);
         pDevice->pFifoWrite       = pDevice->pFifoCompressOut;
      }
      else if ( pDevice->HasHashThread() &&  pDevice->HasCompressionThreads())
      {
         Fifos = 1 + 2 * CONFIG(CompressionThreads);
         pDevice->FifoMaxBlocks    = FIFO_MAX_BLOCKS (Fifos);
         pDevice->pFifoRead        = new t_FifoStd (pDevice->FifoMaxBlocks);
         pDevice->pFifoHashIn      = pDevice->pFifoRead;
         pDevice->pFifoCompressIn  = new t_FifoCompressIn  (CONFIG(CompressionThreads), pDevice->FifoMaxBlocks);
         pDevice->pFifoHashOut     = pDevice->pFifoCompressIn;
         pDevice->pFifoCompressOut = new t_FifoCompressOut (CONFIG(CompressionThreads), pDevice->FifoMaxBlocks);
         pDevice->pFifoWrite       = pDevice->pFifoCompressOut;
      }
      #undef FIFO_MAX_BLOCKS
      LOG_INFO ("Acquisition runs with %d fifos, each one with space for %d blocks of %d bytes (%0.1f MB per acquisition)",
                Fifos, pDevice->FifoMaxBlocks, pDevice->FifoBlockSize, (double)(Fifos * pDevice->FifoMaxBlocks * pDevice->FifoBlockSize) / BYTES_PER_MEGABYTE)

      // Start threads
      // -------------
      pDevice->pThreadRead ->start(QThread::LowPriority);
      if (pDevice->HasHashThread())
         pDevice->pThreadHash->start(QThread::LowPriority);
      for (int i=0; i < pDevice->ThreadCompressList.count(); i++)
         pDevice->ThreadCompressList[i]->start(QThread::LowPriority);
      pDevice->pThreadWrite->start(QThread::LowPriority);

      pOwn->SlowDownAcquisitions = false;
   }

   return NO_ERROR;
}


// ----------------------------------------------------------------------------------------------------------------
//
//                                                     Acquistion end
//
// ----------------------------------------------------------------------------------------------------------------

APIRET t_Table::InfoAcquisitionBadSectors (t_pDevice pDevice, bool Verify)
{
   QList<quint64> BadSectors;
   quint64        Count, i;
   quint64        From, To, Next;
   int            LineEntries    =  0;
   const int      MaxLineEntries = 10;
   bool           First          = true;

   CHK (pDevice->GetBadSectors (BadSectors, Verify))
   Count = BadSectors.count();
   if (Count)
   {
      if (Verify)
           LOG_INFO ("During verification, %lld bad sectors have been encountered", Count)
      else LOG_INFO ("During acquisition, %lld bad sectors have been encountered", Count)

      if (Verify)
           CHK (pDevice->Info.WriteLn (tr("During verification, %1 bad sectors have been encountered. The sector numbers are:", "Info file") .arg(Count)))
      else CHK (pDevice->Info.WriteLn (tr("During acquisition, %1 bad sectors have been encountered. They have been replaced by zeroed sectors. The sector numbers are:", "Info file") .arg(Count)))
      CHK (pDevice->Info.WriteLn ("   "))
      i = 0;
      while (i<Count)
      {
         From = BadSectors.at(i++);
         To   = From;
         while (i < Count)
         {
            Next = BadSectors.at(i);
            if (Next != To+1)
               break;
            To = Next;
            i++;
         }
         if (!First)
         {
            if (LineEntries >= MaxLineEntries)
            {
               CHK (pDevice->Info.Write   (","))
               CHK (pDevice->Info.WriteLn ("   "))
               LineEntries = 0;
            }
            else
            {
               CHK (pDevice->Info.Write (", "))
            }
         }
         First = false;
         if (From == To)
         {
            CHK (pDevice->Info.Write ("%Lu", From))
            LineEntries++;
         }
         else
         {
            CHK (pDevice->Info.Write ("%Lu-%Lu", From, To))
            LineEntries += 2;
         }
      }
   }
   else
   {
      if (Verify)
      {
         LOG_INFO ("No bad sectors encountered during verification.")
         CHK (pDevice->Info.WriteLn (tr("No bad sectors encountered during verification.", "Info file")))
      }
      else
      {
         LOG_INFO ("No bad sectors encountered during acquisition.")
         CHK (pDevice->Info.WriteLn (tr("No bad sectors encountered during acquisition.", "Info file")))
      }
   }

   return NO_ERROR;
}

static APIRET TableTimestampToISO (const QDateTime &Timestamp, QString &Str)
{
   Str = Timestamp.toString (Qt::ISODate);
   Str.replace ("T", " ");
   return NO_ERROR;
}

APIRET t_Table::InfoAcquisitionEnd (t_pDevice pDevice)
{
   QString    MD5;
   QString    MD5Verify;
   QString    SHA256;
   QString    SHA256Verify;
   QString    StateStr;
   quint64    BadSectors;
   QString    StartStr, StopStr, VerifyStr;
   int        Hours, Minutes, Seconds;
   double     Speed;

   CHK (pDevice->Info.WriteLn ())
   CHK (InfoAcquisitionBadSectors (pDevice, false))
   if (!pDevice->StartTimestampVerify.isNull())
      CHK (InfoAcquisitionBadSectors (pDevice, true))
   MD5 = tr("--", "Info file");
   MD5Verify    = MD5;
   SHA256       = MD5;
   SHA256Verify = MD5;

   switch (pDevice->State)
   {
      case t_Device::Finished:
         StateStr = tr("Finished successfully", "Info file");
         BadSectors = t_Device::GetBadSectorCount(pDevice).toULongLong();
         if (BadSectors)
            StateStr += " " + tr("(with %1 bad sectors)", "Info file, may be appended to 'finished successfully' message") .arg(BadSectors);
         if (pDevice->Acquisition.CalcHashes)
         {
            CHK (HashMD5DigestStr    (&pDevice->MD5Digest   , MD5   ))
            CHK (HashSHA256DigestStr (&pDevice->SHA256Digest, SHA256))
         }
         if (pDevice->Acquisition.VerifySource)
         {
            CHK (HashMD5DigestStr    (&pDevice->MD5DigestVerify   , MD5Verify   ))
            CHK (HashSHA256DigestStr (&pDevice->SHA256DigestVerify, SHA256Verify))
         }
         break;

      case t_Device::Aborted:
         switch (pDevice->AbortReason)
         {
            case t_Device::UserRequest:
               StateStr = tr("Aborted by user", "Info file") + " ";
               if (pDevice->StartTimestampVerify.isNull())
               {
                  StateStr += tr("(during acquisition)"     , "Info file");
               }
               else
               {
                  StateStr += tr("(during source verification)", "Info file");
                  if (pDevice->Acquisition.CalcHashes)                            // If the user interrupted during verification (i.e. after
                  {                                                               // acquisition), we can calculate the acquisition hash.
                     CHK (HashMD5DigestStr    (&pDevice->MD5Digest   , MD5   ))
                     CHK (HashSHA256DigestStr (&pDevice->SHA256Digest, SHA256))
                  }
               }
               break;
            case t_Device::ThreadWriteFileError: StateStr = tr("Aborted because of image write error", "Info file"); break;
            default:                             StateStr = tr("Aborted, strange reason (%1)"        , "Info file") .arg(pDevice->AbortReason); break;
         }
         break;

      default:
         StateStr = tr("Strange state (%1)", "Info file") .arg(pDevice->State);
   }
   CHK (pDevice->Info.WriteLn (tr("State: %1", "Info file") .arg(StateStr)))
   CHK (pDevice->Info.WriteLn ())

   CHK (pDevice->Info.AddRow (tr("MD5 hash:: %1"            , "Info file") .arg(MD5         )))
   CHK (pDevice->Info.AddRow (tr("MD5 hash verified:: %1"   , "Info file") .arg(MD5Verify   )))
   CHK (pDevice->Info.AddRow (tr("SHA256 hash:: %1"         , "Info file") .arg(SHA256      )))
   CHK (pDevice->Info.AddRow (tr("SHA256 hash verified:: %1", "Info file") .arg(SHA256Verify)))
   CHK (pDevice->Info.WriteTable ());
   if ((pDevice->State == t_Device::Finished) && pDevice->Acquisition.CalcHashes &&
                                                 pDevice->Acquisition.VerifySource)
   {
      if (HashMD5Match   (&pDevice->MD5Digest   , &pDevice->MD5DigestVerify   ) &&
          HashSHA256Match(&pDevice->SHA256Digest, &pDevice->SHA256DigestVerify))
           CHK (pDevice->Info.WriteLn (tr ("The hash values are identical. The device delivered the same data during acquisition and verification.")))
      else CHK (pDevice->Info.WriteLn (tr ("The hash values differ. The device didn't deliver the same data during acquisition and verification. "
                                           "Maybe you try to acquire the device once again. Check as well if the defect sector list was "
                                           "the same during acquisition and verification (see above).")))
   }
   CHK (pDevice->Info.WriteLn ())

   // Performance
   // -----------
   CHK (TableTimestampToISO (pDevice->StartTimestamp, StartStr))
   CHK (TableTimestampToISO (pDevice->StopTimestamp , StopStr ))
   StartStr += " " + tr("(ISO format YYYY-MM-DD HH:MM:SS)");
   Seconds  = pDevice->StartTimestamp.secsTo (pDevice->StopTimestamp);
   Hours    = Seconds / SECONDS_PER_HOUR  ; Seconds -= Hours   * SECONDS_PER_HOUR;
   Minutes  = Seconds / SECONDS_PER_MINUTE; Seconds -= Minutes * SECONDS_PER_MINUTE;

   if (pDevice->Acquisition.VerifySource)
   {
      if (!pDevice->StartTimestampVerify.isNull())
           CHK (TableTimestampToISO (pDevice->StartTimestampVerify , VerifyStr))
      else VerifyStr = tr ("Acquisition aborted before start of verification");

      CHK (pDevice->Info.AddRow (tr("Acquisition started:: %1" , "Info file") .arg(StartStr )))
      CHK (pDevice->Info.AddRow (tr("Verification started:: %1", "Info file") .arg(VerifyStr)))
      CHK (pDevice->Info.AddRow (tr("Ended:: %1 (%2 hours, %3 minutes and %4 seconds)", "Info file").arg(StopStr ) .arg(Hours) .arg(Minutes) .arg(Seconds)))

      if (pDevice->StartTimestampVerify.isNull())
           Seconds = pDevice->StartTimestamp.secsTo (pDevice->StopTimestamp);
      else Seconds = pDevice->StartTimestamp.secsTo (pDevice->StartTimestampVerify);
      Speed = (double) pDevice->CurrentReadPos / ((double)BYTES_PER_MEGABYTE * (double)Seconds);
      Hours   = Seconds / SECONDS_PER_HOUR  ; Seconds -= Hours   * SECONDS_PER_HOUR;
      Minutes = Seconds / SECONDS_PER_MINUTE; Seconds -= Minutes * SECONDS_PER_MINUTE;
      CHK (pDevice->Info.AddRow (tr("Acquisition speed:: %1 MByte/s (%2 hours, %3 minutes and %4 seconds)", "Info file").arg(Speed, 0, 'f', 2) .arg(Hours) .arg(Minutes) .arg(Seconds)))

      if (!pDevice->StartTimestampVerify.isNull())
      {
         Seconds = pDevice->StartTimestampVerify.secsTo (pDevice->StopTimestamp);
         Speed = (double) pDevice->CurrentReadPos / ((double)BYTES_PER_MEGABYTE * (double)Seconds);
         Hours   = Seconds / SECONDS_PER_HOUR  ; Seconds -= Hours   * SECONDS_PER_HOUR;
         Minutes = Seconds / SECONDS_PER_MINUTE; Seconds -= Minutes * SECONDS_PER_MINUTE;
         CHK (pDevice->Info.AddRow (tr("Verification speed:: %1 MByte/s (%2 hours, %3 minutes and %4 seconds)", "Info file").arg(Speed, 0, 'f', 2) .arg(Hours) .arg(Minutes) .arg(Seconds)))
      }
      CHK (pDevice->Info.WriteTable ());
   }
   else
   {
      CHK (pDevice->Info.AddRow (tr("Acquisition started:: %1"                        , "Info file").arg(StartStr)))
      CHK (pDevice->Info.AddRow (tr("Ended:: %1 (%2 hours, %3 minutes and %4 seconds)", "Info file").arg(StopStr ) .arg(Hours) .arg(Minutes) .arg(Seconds)))

      Seconds = pDevice->StartTimestamp.secsTo (pDevice->StopTimestamp);
      Speed = (double) pDevice->CurrentReadPos / ((double)BYTES_PER_MEGABYTE * (double)Seconds);
      Hours   = Seconds / SECONDS_PER_HOUR  ; Seconds -= Hours   * SECONDS_PER_HOUR;
      Minutes = Seconds / SECONDS_PER_MINUTE; Seconds -= Minutes * SECONDS_PER_MINUTE;
      CHK (pDevice->Info.AddRow (tr("Acquisition speed:: %1 MByte/s (%2 hours, %3 minutes and %4 seconds)", "Info file").arg(Speed, 0, 'f', 2) .arg(Hours) .arg(Minutes) .arg(Seconds)))
      CHK (pDevice->Info.WriteTable ());
   }
   CHK (pDevice->Info.WriteLn ());
   CHK (pDevice->Info.WriteLn ());

   return NO_ERROR;
}

APIRET t_Table::FinaliseThreadStructs (t_pDevice pDevice)
{
   if ((pDevice->pThreadRead  == NULL) &&        // TableCleanupThreadStructs is called upon every thread's end,
       (pDevice->pThreadWrite == NULL) &&        // but its core only is executed after the last thread ended (as
       (pDevice->pThreadHash  == NULL) &&        // we must not kill vital structures as long as any threads are
       (pDevice->ThreadCompressList.isEmpty()))  // still running).
   {
      if      (!pDevice->HasHashThread() && !pDevice->HasCompressionThreads())
      {
         delete pDevice->pFifoRead;
      }
      else if ( pDevice->HasHashThread() && !pDevice->HasCompressionThreads())
      {
         delete pDevice->pFifoRead;
         delete pDevice->pFifoHashOut;
      }
      else if (!pDevice->HasHashThread() &&  pDevice->HasCompressionThreads())
      {
         delete pDevice->pFifoCompressIn;
         delete pDevice->pFifoCompressOut;
      }
      else if ( pDevice->HasHashThread() &&  pDevice->HasCompressionThreads())
      {
         delete pDevice->pFifoRead;
         delete pDevice->pFifoCompressIn;
         delete pDevice->pFifoCompressOut;
      }

      pDevice->pFifoRead        = NULL;
      pDevice->pFifoHashIn      = NULL;
      pDevice->pFifoHashOut     = NULL;
      pDevice->pFifoWrite       = NULL;
      pDevice->pFifoCompressIn  = NULL;
      pDevice->pFifoCompressOut = NULL;

//      LibEwfGetMemStats (&LibEwfAllocs, &LibEwfFrees);
//      LOG_INFO ("LIBEWF mem  statistics: %d allocated - %d freed = %d remaining", LibEwfAllocs, LibEwfFrees, LibEwfAllocs - LibEwfFrees)

      LOG_INFO ("Acquisition of %s: All structures cleaned.", QSTR_TO_PSZ (pDevice->LinuxDevice))

      if (pDevice->AbortRequest)
           pDevice->State = t_Device::Aborted;
      else pDevice->State = t_Device::Finished;

      if (!pDevice->DeleteAfterAbort)
         CHK_EXIT (InfoAcquisitionEnd (pDevice))
   }

   return NO_ERROR;
}

void t_Table::SlotThreadReadFinished (t_pDevice pDevice)
{
   LOG_INFO ("Acquisition of %s: Reading thread finished", QSTR_TO_PSZ (pDevice->LinuxDevice))
   pDevice->pThreadRead->deleteLater();
   pDevice->pThreadRead = NULL;

   if (pDevice->AbortRequest)
        CHK_EXIT (WakeWaitingThreads(pDevice))
   else CHK_EXIT (pDevice->pFifoRead->InsertDummy ())
   CHK_EXIT (FinaliseThreadStructs (pDevice))
}

void t_Table::SlotThreadHashFinished (t_pDevice pDevice)
{
   LOG_INFO ("Acquisition of %s: Hash thread finished", QSTR_TO_PSZ (pDevice->LinuxDevice))
   pDevice->pThreadHash->deleteLater();
   pDevice->pThreadHash = NULL;

   if (pDevice->AbortRequest)
        CHK_EXIT (WakeWaitingThreads(pDevice))
   else CHK_EXIT (pDevice->pFifoHashOut->InsertDummy ())
   CHK_EXIT (FinaliseThreadStructs (pDevice))
}

void t_Table::SlotThreadCompressFinished (t_pDevice pDevice, int ThreadNr)
{
   bool NoMoreThreads = true;

   LOG_INFO ("Acquisition of %s: Compression thread #%d finished", QSTR_TO_PSZ (pDevice->LinuxDevice), ThreadNr)

   pDevice->ThreadCompressList[ThreadNr]->deleteLater();
   pDevice->ThreadCompressList[ThreadNr] = NULL;

   for (int i=0;
        (i < pDevice->ThreadCompressList.count()) && NoMoreThreads;
        i++)
      NoMoreThreads = (pDevice->ThreadCompressList[i] == NULL);

   if (NoMoreThreads)
   {
       LOG_INFO ("All %d compression threads finished", pDevice->ThreadCompressList.count())
       pDevice->ThreadCompressList.clear();
   }

   if (pDevice->AbortRequest)
        CHK_EXIT (WakeWaitingThreads(pDevice))
   else CHK_EXIT (pDevice->pFifoCompressOut->InsertDummy (ThreadNr))

   CHK_EXIT (FinaliseThreadStructs (pDevice))
}

void t_Table::SlotThreadWriteFinished (t_pDevice pDevice)
{
   LOG_INFO ("Acquisition of %s: Writing thread finished", QSTR_TO_PSZ (pDevice->LinuxDevice))

   pDevice->pThreadWrite->deleteLater();
   pDevice->pThreadWrite = NULL;

   if (pDevice->AbortRequest)
      CHK_EXIT (WakeWaitingThreads(pDevice))

   CHK_EXIT (FinaliseThreadStructs (pDevice))
}

APIRET t_Table::AbortAcquisition0 (t_pDevice pDevice)
{
   if (pDevice->pThreadRead == NULL)
   {
      LOG_INFO ("User pressed abort, but acquisition on %s is no longer running (it probably ended between the user's click and this message)", QSTR_TO_PSZ(pDevice->LinuxDevice))
      return NO_ERROR;
   }

   LOG_INFO ("User aborts acquisition on %s", QSTR_TO_PSZ(pDevice->LinuxDevice))
   pDevice->AbortReason  = t_Device::UserRequest;
   pDevice->AbortRequest = true;

   CHK (WakeWaitingThreads (pDevice))

   return NO_ERROR;
}

APIRET t_Table::AbortAcquisition (t_pDevice pDevice)
{
   bool Abort;

   CHK (t_DlgAbort::Show (pDevice, Abort, pDevice->DeleteAfterAbort))
   if (Abort)
      CHK (AbortAcquisition0 (pDevice))

   return NO_ERROR;
}

APIRET t_Table::AbortAllAcquisitions (void)
{
   t_pDevice pDevice;
   int        i;

   for (i=0; i<pOwn->pDeviceList->count(); i++)
   {
      pDevice = pOwn->pDeviceList->at(i);
      if ((pDevice->State == t_Device::Acquire)       ||
          (pDevice->State == t_Device::AcquirePaused) ||
          (pDevice->State == t_Device::Verify)        ||
          (pDevice->State == t_Device::VerifyPaused))
      {
         CHK (AbortAcquisition0 (pDevice))
      }
   }

   return NO_ERROR;
}

APIRET t_Table::WakeWaitingThreads (t_pDevice pDevice)
{
   if      (!pDevice->HasHashThread() && !pDevice->HasCompressionThreads())
   {
      CHK (pDevice->pFifoRead->WakeWaitingThreads())
   }
   else if ( pDevice->HasHashThread() && !pDevice->HasCompressionThreads())
   {
      CHK (pDevice->pFifoRead   ->WakeWaitingThreads())
      CHK (pDevice->pFifoHashOut->WakeWaitingThreads())
   }
   else if (!pDevice->HasHashThread() &&  pDevice->HasCompressionThreads())
   {
      CHK (pDevice->pFifoCompressIn ->WakeWaitingThreads())
      CHK (pDevice->pFifoCompressOut->WakeWaitingThreads())
   }
   else if ( pDevice->HasHashThread() &&  pDevice->HasCompressionThreads())
   {
      CHK (pDevice->pFifoRead       ->WakeWaitingThreads())
      CHK (pDevice->pFifoCompressIn ->WakeWaitingThreads())
      CHK (pDevice->pFifoCompressOut->WakeWaitingThreads())
   }

   return NO_ERROR;
}

