// ****************************************************************************
//  Project:        GUYMAGER
// ****************************************************************************
//  Programmer:     Guy Voncken
//                  Police Grand-Ducale
//                  Service de Police Judiciaire
//                  Section Nouvelles Technologies
// ****************************************************************************
//  Module:         The main application window including menu and central
//                  widget
// ****************************************************************************


#include <QtGui>

#include "common.h"

#include "libewf.h"

#include "config.h"
#include "compileinfo.h"
#include "devicelistmodel.h"
#include "itemdelegate.h"
#include "mainwindow.h"
#include "threadscan.h"
#include "qtutil.h"
#include "infofield.h"
#include "table.h"
#include "dlgmessage.h"


// -----------------------------
//            Constants
// -----------------------------

const int MAINWINDOW_WAIT_FOR_THREADS_TO_ABORT = 5000;


// -----------------------------
//  Table model for device list
// -----------------------------

t_DeviceListModel::t_DeviceListModel ()
{
   CHK_EXIT (ERROR_MAINWINDOW_CONSTRUCTOR_NOT_SUPPORTED)
} //lint !e1401 Not initialised

t_DeviceListModel::t_DeviceListModel (t_pDeviceList pDeviceList)
         : QAbstractTableModel()
{
   t_ColAssoc ColAssoc;

   poDeviceList = pDeviceList;

   #define COL_ASSOC(Str, pDataFn, Display, Align, MinDx)  \
      ColAssoc.Name        = Str;            \
      ColAssoc.pGetDataFn  = pDataFn;        \
      ColAssoc.DisplayType = Display;        \
      ColAssoc.Alignment   = Align;          \
      ColAssoc.MinWidth    = MinDx;          \
      ColAssocList.append (ColAssoc);
   #define LEFT   (Qt::AlignLeft    | Qt::AlignVCenter)
   #define RIGHT  (Qt::AlignRight   | Qt::AlignVCenter)
   #define CENTER (Qt::AlignHCenter | Qt::AlignVCenter)

      COL_ASSOC (tr("Serial\nnr."            , "Column on main screen"), t_Device::GetSerialNumber  , t_ItemDelegate::DISPLAYTYPE_STANDARD, LEFT  ,   0)
      COL_ASSOC (tr("Linux\ndevice"          , "Column on main screen"), t_Device::GetLinuxDevice   , t_ItemDelegate::DISPLAYTYPE_STANDARD, LEFT  ,   0)
      COL_ASSOC (tr("Model"                  , "Column on main screen"), t_Device::GetModel         , t_ItemDelegate::DISPLAYTYPE_STANDARD, LEFT  ,   0)
      COL_ASSOC (tr("State"                  , "Column on main screen"), t_Device::GetState         , t_ItemDelegate::DISPLAYTYPE_STATE   , LEFT  , 200)
      COL_ASSOC (tr("Size"                   , "Column on main screen"), t_Device::GetSizeHuman     , t_ItemDelegate::DISPLAYTYPE_STANDARD, RIGHT ,   0)
      COL_ASSOC (tr("Bad\nsectors"           , "Column on main screen"), t_Device::GetBadSectorCount, t_ItemDelegate::DISPLAYTYPE_STANDARD, RIGHT ,   0)
      COL_ASSOC (tr("Progress"               , "Column on main screen"), t_Device::GetProgress      , t_ItemDelegate::DISPLAYTYPE_PROGRESS, LEFT  ,   0)
      COL_ASSOC (tr("Average\nSpeed\n[MB/s]" , "Column on main screen"), t_Device::GetAverageSpeed  , t_ItemDelegate::DISPLAYTYPE_STANDARD, RIGHT ,   0)
      COL_ASSOC (tr("Time\nremaining"        , "Column on main screen"), t_Device::GetRemaining     , t_ItemDelegate::DISPLAYTYPE_STANDARD, CENTER,   0)
      COL_ASSOC (tr("FIFO queues\nusage\n[%]", "Column on main screen"), t_Device::GetFifoStatus    , t_ItemDelegate::DISPLAYTYPE_STANDARD, LEFT  ,   0)
//      COL_ASSOC (tr("Sector\nsize\nlog."    , "Column on main screen"), t_Device::GetSectorSize    , t_ItemDelegate::DISPLAYTYPE_STANDARD, LEFT, 0      )
//      COL_ASSOC (tr("Sector\nsize\nphys."   , "Column on main screen"), t_Device::GetSectorSizePhys, t_ItemDelegate::DISPLAYTYPE_STANDARD, LEFT, 0      )
//      COL_ASSOC (tr("Current\nSpeed\n[MB/s]", "Column on main screen"), t_Device::GetCurrentSpeed  , t_ItemDelegate::DISPLAYTYPE_STANDARD, LEFT, 0      )
   #undef COL_ASSOC
   #undef LEFT
   #undef RIGHT
   #undef CENTER
}

int t_DeviceListModel::rowCount (const QModelIndex & /*parent*/) const
{
   return poDeviceList->count();
}

int t_DeviceListModel::columnCount (const QModelIndex & /*parent*/) const
{
   return ColAssocList.count();
}

void t_DeviceListModel::SlotRefresh (void)
{
   static int LastCount=0;

   if (poDeviceList->count() != LastCount)
        reset();
   else emit dataChanged (index(0,0), index(rowCount()-1, columnCount()-1) );
   LastCount = poDeviceList->count();
}

//void t_DeviceListModel::SlotUpdate (void)
//{
//   emit dataChanged (index(0,0), index(rowCount()-1, columnCount()-1) );
//}

QVariant t_DeviceListModel::data(const QModelIndex &Index, int Role) const
{
   t_DeviceListModel::t_pGetDataFn pGetDataFn;
   t_pDevice                       pDev;
   QString                          Ret;
   QString                          Text;
   QVariant                         Value;

   if (!Index.isValid() && (Role == t_ItemDelegate::RowNrRole))  // For correct handling of the click events in the table
      return -1;

   if (Index.isValid() && (Index.column() < columnCount()) &&
                          (Index.row   () < rowCount   ()))
   {
      if (Index.column() >= ColAssocList.count())
         CHK_EXIT (ERROR_MAINWINDOW_INVALID_COLUMN)
      switch (Role)
      {
         case t_ItemDelegate::DisplayTypeRole: Value = ColAssocList[Index.column()].DisplayType; break;
         case t_ItemDelegate::MinColWidthRole: Value = ColAssocList[Index.column()].MinWidth;    break;
         case Qt::TextAlignmentRole          : Value = ColAssocList[Index.column()].Alignment;   break;

         case t_ItemDelegate::RowNrRole:
            Value = Index.row();
            break;
         case t_ItemDelegate::DeviceRole:
            pDev = poDeviceList->at(Index.row());
            Value = qVariantFromValue ((void *)pDev);
            break;
         case Qt::BackgroundRole:
            pDev = poDeviceList->at(Index.row());
            if (pDev->Local)
                 Value = QBrush (CONFIG_COLOR(COLOR_LOCALDEVICES));
            else Value = QVariant();
            break;
         case Qt::DisplayRole:
            if (Index.column() >= ColAssocList.count())
               CHK_EXIT (ERROR_MAINWINDOW_INVALID_COLUMN)
            pDev = poDeviceList->at(Index.row());
            pGetDataFn = ColAssocList[Index.column()].pGetDataFn;
            Value = (*pGetDataFn) (pDev);
            break;
         case Qt::SizeHintRole:
            break;  // See t_ItemDelegate::sizeHint for column width calculation
         default:
            break;
      }
   }
   return Value;
}

QVariant t_DeviceListModel::headerData(int section, Qt::Orientation orientation, int role) const
{
   if (role != Qt::DisplayRole)
      return QVariant();

   if (orientation == Qt::Horizontal)
      return ColAssocList[section].Name;
   else
      return QString("Row %1").arg(section);
}

// -----------------------------
//          Main window
// -----------------------------

class t_MainWindowLocal
{
   public:
      t_DeviceList          *pDeviceList;
      t_DeviceListModel     *pDeviceListModel;
      QSortFilterProxyModel *pProxyModel;   // Inserted between pDeviceListModel and pTable, provides sorting functionality
      QWidget               *pCentralWidget;
      t_pTable               pTable;
      t_pInfoField           pInfoField;
      t_ThreadScan          *pThreadScan;
      QAction               *pActionRescan;
      t_ThreadScanWorker    *pScanWorker;
      QTimer                *pTimerRefresh;
};

static APIRET MainWindowRegisterErrorCodes (void)
{
   CHK (TOOL_ERROR_REGISTER_CODE (ERROR_MAINWINDOW_CONSTRUCTOR_NOT_SUPPORTED))
   CHK (TOOL_ERROR_REGISTER_CODE (ERROR_MAINWINDOW_INVALID_COLUMN))
   CHK (TOOL_ERROR_REGISTER_CODE (ERROR_MAINWINDOW_INVALID_DATATYPE))

   return NO_ERROR;
}

APIRET t_MainWindow::CreateMenu (void)
{
   QMenuBar *pMenuBar = menuBar();
   QToolBar *pToolBar;
   QMenu    *pMenu;

   // Actions
   // -------
   pOwn->pActionRescan = new QAction (tr("&Rescan"), this);
   pOwn->pActionRescan->setShortcut  (tr("F5"));
   pOwn->pActionRescan->setToolTip   (tr("Rescan devices and update list"));

   // Menu
   // ----
   pMenu = pMenuBar->addMenu (tr("&Devices"));
   pMenu->addAction (pOwn->pActionRescan);

   pMenu = pMenuBar->addMenu (tr("&Misc", "Menu entry"));
   pMenu->addAction (tr("Debug", "Menu entry"), this, SLOT(SlotDebug()));

   pMenu = pMenuBar->addMenu (tr("&Help", "Menu entry"));
   pMenu->addAction (tr("About &GUYMAGER", "Menu entry"), this, SLOT(SlotAboutGuymager()));  //lint !e64 !e119 Lint reports about type mismatches and
   pMenu->addAction (tr("About &Qt"      , "Menu entry"), this, SLOT(SlotAboutQt      ()));  //lint !e64 !e119 too many arguments, Lint is probably wrong


   // Toolbar
   // -------
   pToolBar = addToolBar (QString());
   pToolBar->addAction (pOwn->pActionRescan);

   return NO_ERROR;
}


t_MainWindow::t_MainWindow(void)
{
   CHK_EXIT (ERROR_MAINWINDOW_CONSTRUCTOR_NOT_SUPPORTED)
} //lint !e1401 pOwn not initialised

t_MainWindow::t_MainWindow (t_pDeviceList pDeviceList, QWidget *pParent, Qt::WFlags Flags)
   :QMainWindow (pParent, Flags)
{
   static bool Initialised = false;

   if (!Initialised)
   {
      CHK_EXIT (MainWindowRegisterErrorCodes())
      Initialised = true;
   }

   pOwn = new t_MainWindowLocal;
   pOwn->pDeviceList = pDeviceList;

   CHK_EXIT (CreateMenu ())
   setWindowTitle ("GUYMAGER");

   // Create models and view according do the following layering:
   //    TableView        -> GUI
   //    ProxyModel       -> Provides sorting
   //    DeviceListModel  -> Describes data access
   //    DeviceList       -> Contains data
   // -----------------------------------------------------------
   pOwn->pCentralWidget = new QWidget (this);
   QVBoxLayout *pLayout = new QVBoxLayout (pOwn->pCentralWidget);

   pOwn->pTable                      = new t_Table (pOwn->pCentralWidget, pDeviceList);
   pOwn->pDeviceListModel            = new t_DeviceListModel     (pDeviceList);
   pOwn->pProxyModel                 = new QSortFilterProxyModel (pOwn->pCentralWidget);
   pOwn->pProxyModel->setSourceModel (pOwn->pDeviceListModel);
   pOwn->pTable     ->setModel       (pOwn->pProxyModel);

   pOwn->pInfoField = new t_InfoField (pOwn->pCentralWidget);

   pLayout->addWidget (pOwn->pTable);
   pLayout->addWidget (pOwn->pInfoField);

   CHK_QT_EXIT (connect (pOwn->pTable->horizontalHeader(), SIGNAL(sectionClicked      (int      )), pOwn->pTable    , SLOT(sortByColumn  (int      ))))
   CHK_QT_EXIT (connect (pOwn->pTable                    , SIGNAL(SignalDeviceSelected(t_pDevice)), pOwn->pInfoField, SLOT (SlotShowInfo (t_pDevice))))

   setCentralWidget (pOwn->pCentralWidget);

   // Start the device scan thread
   // ----------------------------

   pOwn->pThreadScan = new t_ThreadScan ();
   CHK_EXIT (pOwn->pThreadScan->Start (&pOwn->pScanWorker))
   CHK_QT_EXIT (connect (pOwn->pActionRescan, SIGNAL (triggered ()),          pOwn->pScanWorker, SLOT (SlotRescan ())))
   CHK_QT_EXIT (connect (pOwn->pScanWorker  , SIGNAL (SignalScanFinished (t_pDeviceList)), this, SLOT (SlotScanFinished (t_pDeviceList))))
   QTimer::singleShot (300, pOwn->pScanWorker, SLOT(SlotRescan()));  // pOwn->pScanWorker->SlotRescan must not be called directly as it would be called from the
                                                                     // wrong thread! Using the signal/slot connection makes it behave correctly (see
                                                                     // Qt_AutoConnection, which is the default for connecting signals and slots).

   // Screen refresh timer
   // --------------------
   pOwn->pTimerRefresh = new QTimer(this);
   CHK_QT_EXIT (connect (pOwn->pTimerRefresh, SIGNAL(timeout()), this, SLOT(SlotRefresh())))
   pOwn->pTimerRefresh->start (CONFIG(ScreenRefreshInterval));
}

void t_MainWindow::SlotRefresh (void)
{
   t_pDevice pDevice;

   pOwn->pDeviceListModel->SlotRefresh();
   CHK_EXIT (pOwn->pTable->GetDeviceUnderCursor (pDevice))
   pOwn->pInfoField->SlotShowInfo (pDevice);
}

// SlotScanFinished provides us with the new device list. Now, our current device list needs to be updated. The rules are:
//  (1) For devices, who are in both lists:
//        Set the state of the device in the current list to Connected. Thus, devices that were temporarly disconnected
//        will become connected again.
//  (2) For devices, that only exist in the current list (but no longer in the new list):
//        a) delete from the current list if there was no acquisition running
//        b) switch to disconnected if there was an acquisition running, thus giving the user a chance to continue to acquistion
//           on another device (for instance, if the hard disk had been unpligged from firewrie and plugged to USB)
//  (3) For devices, that only exist in the new list:
//        Add to the current list
// Remark: Is it quite tricky to compare the devices from both lists, as there devices without a serial number. See
// t_DeviceList::MatchDevice for details.

void t_MainWindow::SlotScanFinished (t_pDeviceList pNewDeviceList)
{
   t_pDeviceList pDeviceList;
   t_pDevice     pDev, pNewDev;
   int            i;

   pDeviceList = pOwn->pDeviceList;
   for (i=0; i<pDeviceList->count(); i++)
   {
      pDev = pDeviceList->at (i);
      pDev->Checked = false;  // Checked is used to remember which devices we have seen and which ones not
   }

   for (i=0; i<pNewDeviceList->count(); i++)
   {
      pNewDev = pNewDeviceList->at (i);
      CHK_EXIT (pDeviceList->MatchDevice (pNewDev, pDev))
//      LOG_INFO ("Result of match for %s: %s", QSTR_TO_PSZ (pNewDev->LinuxDevice), pDev ? "Match" : "NoMatch")

      if (pDev)
      {
         pDev->Checked = true;
         if ((pDev->State == t_Device::AcquirePaused) ||
             (pDev->State == t_Device::VerifyPaused))
         {
            LOG_INFO ("Switching %s to connected again", QSTR_TO_PSZ(pNewDev->LinuxDevice))
            pDev->LinuxDevice = pNewDev->LinuxDevice;    // Linux may choose a new path when reconnecting the device
            if (pDev->State == t_Device::AcquirePaused)  // (1)
                 pDev->State = t_Device::Acquire;
            else pDev->State = t_Device::Verify;
         }
      }
      else
      {
         pNewDev->Checked = true;
         pNewDeviceList->removeAt (i--);                 // (3)
         pDeviceList->append (pNewDev);
      }
   }

   for (i=0; i<pDeviceList->count(); i++)
   {
      pDev = pDeviceList->at (i);
      if (!pDev->Checked)
      {
         switch (pDev->State)
         {
            case t_Device::Acquire      : pDev->State = t_Device::AcquirePaused; break;  // (2b)
            case t_Device::Verify       : pDev->State = t_Device::VerifyPaused ; break;  // (2b)
            case t_Device::AcquirePaused: break;
            case t_Device::VerifyPaused : break;
            default                     : pDeviceList->removeAt (i--);                   // (2a)
                                          delete pDev;
         }
      }
   }

   delete pNewDeviceList;

   pOwn->pDeviceListModel->SlotRefresh();
   pOwn->pTable->resizeColumnsToContents();
}

void t_MainWindow::closeEvent (QCloseEvent *pEvent)
{
   QMessageBox::StandardButton  Button;
   t_pDevice                   pDevice;
   bool                         AcquisitionsActive=false;
   int                          i;

   LOG_INFO ("User exits application")
   for (i=0; (i<pOwn->pDeviceList->count()) && (!AcquisitionsActive); i++)
   {
      pDevice = pOwn->pDeviceList->at(i);
      AcquisitionsActive = (pDevice->State == t_Device::Acquire)       ||
                           (pDevice->State == t_Device::Verify )       ||
                           (pDevice->State == t_Device::Cleanup)       ||
                           (pDevice->State == t_Device::AcquirePaused) ||
                           (pDevice->State == t_Device::VerifyPaused );
   }

   if (AcquisitionsActive)
   {
      LOG_INFO ("Some acquisitions still active - ask user if he really wants to quit")
      Button = QMessageBox::question (this, tr ("Exit GUYMAGER", "Dialog title"), tr("There are active acquisitions. Do you really want to abort them and quit?"),
                                      QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
      if (Button == QMessageBox::Yes)
      {  // stop acquisitions
         LOG_INFO ("User confirms abortion in order to quit")
         CHK_EXIT (pOwn->pTable->AbortAllAcquisitions ())
         CHK_EXIT (QtUtilSleep (MAINWINDOW_WAIT_FOR_THREADS_TO_ABORT))
      }
      else
      {
         LOG_INFO ("User cancels exit request")
         pEvent->ignore();
      }
   }
}

void t_MainWindow::SlotDebug (void)
{
   QString DebugInfo;
   quint64 FifoAllocs, FifoFrees;
   qint64  FifoRemaining;
   qint64  FifoAllocated;

   CHK_EXIT (FifoGetStatistics (FifoAllocs, FifoFrees, FifoAllocated))
   FifoRemaining = FifoAllocs - FifoFrees;
   DebugInfo  = "FIFO Buffers";
   DebugInfo += QString ("\n   %1 allocated") .arg(FifoAllocs   , 10);
   DebugInfo += QString ("\n   %1 released" ) .arg(FifoFrees    , 10);
   DebugInfo += QString ("\n   %1 remaining") .arg(FifoRemaining, 10);
   DebugInfo += QString ("\n   %1 MB"       ) .arg((double)FifoAllocated / (1024.0*1024.0), 10, 'f', 1);

   CHK_EXIT (t_DlgMessage::Show (tr("Debug information"), DebugInfo, true))
}

void t_MainWindow::SlotAboutGuymager (void)
{
   const char *pLibGuyToolsVersionInstalled;
   const char *pLibEwfVersionInstalled;

   t_Log::GetLibGuyToolsVersion (&pLibGuyToolsVersionInstalled);

   pLibEwfVersionInstalled = libewf_get_version ();

   QMessageBox::about(this, tr("About GUYMAGER", "Dialog title"),
                            tr("GUYMAGER is a Linux-based forensic imaging tool\n\n"
                               "Version: %1\n"
                               "Compilation timestamp: %2\n"
                               "Compiled with gcc %3\n"
                               "Using libewf %4\n"
                               "Using libguytools %5")
                               .arg(pCompileInfoVersion)
                               .arg(pCompileInfoTimestamp)
                               .arg(__VERSION__)
                               .arg(pLibEwfVersionInstalled)
                               .arg(pLibGuyToolsVersionInstalled));
}

void t_MainWindow::SlotAboutQt (void)
{
   QMessageBox::aboutQt (this, tr("About Qt", "Dialog title"));
}

t_MainWindow::~t_MainWindow ()
{
   CHK_EXIT (pOwn->pThreadScan->Stop ())
   delete pOwn->pTable;
   delete pOwn->pProxyModel;
   delete pOwn->pDeviceListModel;
   delete pOwn;
}

