#
# RBox - gDesklets control for Rhythmbox
# Copyright (c) 2005 Christoph Lenggenhager
#
# based on Rhythmlet and rbscobbler by Alex Revo
#
# ----------------------------------------------------------------------
#  __init__.py - RBox control
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
########################################################################


from libdesklets.controls import Control

from IRBox import IRBox

try:
    import bonobo
    import CORBA
    import types
    import os
    import urllib

except ImportError, error:
    print("*** ERROR ***")
    print("Unable to import required modules: %s" % error)
    print("*** ERROR ***")
    print("")
except Exception, error:
    print("*** ERROR ***")
    print("Unknown error: %s" % error)
    print("*** ERROR ***")
#end try


# constants
EVENTSOURCE_IID         = "OAFIID:Bonobo_Activation_EventSource"
EVENTSOURCE_IDL         = "IDL:Bonobo/EventSource:1.0"
EVENT_REGISTER          = "Bonobo/ObjectDirectory:activation:register"
EVENT_UNREGISTER        = "Bonobo/ObjectDirectory:activation:unregister"
RHYTHMBOX_IID           = "OAFIID:GNOME_Rhythmbox_Factory"
RHYTHMBOX_IDL           = "IDL:GNOME/Rhythmbox:1.0"
RB_CHANGE_PLAYING       = "Bonobo/Property:change:playing"
RB_CHANGE_PLAY_ORDER    = "Bonobo/Property:change:play-order"
RB_CHANGE_REPEAT        = "Bonobo/Property:change:repeat"
RB_CHANGE_SHUFFLE       = "Bonobo/Property:change:shuffle"
RB_CHANGE_SONG          = "Bonobo/Property:change:song"
BOOLEAN_IDL             = "IDL:omg.org/CORBA/boolean:1.0"


#
# Control for Rhythmbox.
#
class RBox(Control, IRBox):


  def __init__(self):
  #-------------------------------------------------------------------
    self.__rb = None
    self.__rbProps = None
    self.__songProbs = None
    self.__song_properties = None
    self.__playtimetimer = None
    self.__perc_played = None
    self.__inittimer = None
    Control.__init__(self)
    
    
    # delay the loading of information to be sure
    # they are displayed
    self.__inittimer = self._add_timer(1000,self.__delayed_init)
    
    event_source = bonobo.get_object(EVENTSOURCE_IID, EVENTSOURCE_IDL)
    bonobo.event_source_client_add_listener(event_source, self.__rbox_link, EVENT_REGISTER)
    bonobo.event_source_client_add_listener(event_source, self.__rbox_link, EVENT_UNREGISTER)
  #-------------------------------------------------------------------


  def __delayed_init(self):
  #-------------------------------------------------------------------
    self.__rbox_link()
    self._remove_timer(self.__inittimer)
    self.__inittimer = None
  #-------------------------------------------------------------------


  def __rbox_link(self, listener=None, event=None, value=None, error=None):
  #-------------------------------------------------------------------
    
    if not (self.__is_active()):
      
      self.__rb = None
      self.__rbProps = None
      self.__songProbs = None
      self.__song_properties = None
      self.__playing = False
      if (self.__playtimetimer != None):
        self._remove_timer(self.__playtimetimer)
        self.__playtimetimer = None
                
    else:
      try:
        self.__rb = bonobo.activation.activate_from_id(RHYTHMBOX_IID, 0, None)
        self.__rbProps = self.__rb.getPlayerProperties()
      
        bonobo.event_source_client_add_listener(self.__rbProps, self.__song_change, RB_CHANGE_SONG)
        bonobo.event_source_client_add_listener(self.__rbProps, self.__playing_change, RB_CHANGE_PLAYING)
        bonobo.event_source_client_add_listener(self.__rbProps, self.__repeat_change, RB_CHANGE_REPEAT)
        bonobo.event_source_client_add_listener(self.__rbProps, self.__shuffle_change, RB_CHANGE_SHUFFLE)
        bonobo.event_source_client_add_listener(self.__rbProps, self.__play_order_change, RB_CHANGE_PLAY_ORDER)
      
        self.__playing = self.__rbProps.getValue("playing").value()
      
        self.__playtimetimer=self._add_timer(1000,self.__playtime_change)
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
    self.__song_change()
    self._update("playing")
    self._update("play_order")
    self._update("shuffle")
    self._update("repeat")
    self._update("active")
    self._update("volume")
    return True
    
  #-------------------------------------------------------------------

  def __shuffle_change(self, listener=None, event=None, value=None, error=None):
  #-------------------------------------------------------------------
    self._update("shuffle")
    self._update("play_order")
  #-------------------------------------------------------------------
  
  
  def __repeat_change(self, listener=None, event=None, value=None, error=None):
  #-------------------------------------------------------------------
    self._update("repeat")
    self._update("play_order")
  #-------------------------------------------------------------------
  
  
  def __play_order_change(self, listener=None, event=None, value=None, error=None):
  #-------------------------------------------------------------------
    self._update("play_order")
  #-------------------------------------------------------------------
  

  def __playtime_change(self):
  #-------------------------------------------------------------------
    _perc_played = self.__get_perc_played()
    if (self.__perc_played != _perc_played):
      self._update("elapsed_perc")
      self.__perc_played = _perc_played
    
    self._update("elapsed")
    
    return True
  #-------------------------------------------------------------------


  def __playing_change(self, listener=None, event=None, value=None, error=None):
  #-------------------------------------------------------------------
    try:
      
      self.__playing = self.__rbProps.getValue("playing").value()
      self._update("playing")
      
    except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
  #-------------------------------------------------------------------


  def __song_change(self, listener=None, event=None, value=None, error=None):
  #-------------------------------------------------------------------
    if (self.__rbProps != None):
      try:
      
        self.__songProbs = self.__rbProps.getValue("song").value()
        if (self.__songProbs != None):
          self.__song_properties = {}
          self.__song_properties["artist"] = self.__songProbs.artist
          self.__song_properties["title"] = self.__songProbs.title
          self.__song_properties["album"] = self.__songProbs.album
          self.__song_properties["genre"] = self.__songProbs.genre
          self.__song_properties["path"] = self.__songProbs.path
          self.__song_properties["track_number"] = self.__songProbs.track_number
          self.__song_properties["duration"] = self.__songProbs.duration
          self.__song_properties["bitrate"] = self.__songProbs.bitrate
          self.__song_properties["filesize"] = self.__songProbs.filesize
          self.__song_properties["rating"] = self.__songProbs.rating
          self.__song_properties["play_count"] = self.__songProbs.play_count
          self.__song_properties["last_played"] = self.__songProbs.last_played
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
    self._update("songinfo")
    
  #-------------------------------------------------------------------


  ## Song Info Read Functions

  def __get_song_properties(self):
  #-------------------------------------------------------------------
    if (self.__song_properties != None):
      return self.__song_properties
    
    return {}
    
  #-------------------------------------------------------------------

###  Status Functions Read Only Part


  def __get_play_order(self):
  #-------------------------------------------------------------------
    if (self.__rbProps != None):
      try:
        return self.__rbProps.getValue("play-order").value()
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
    return 0
    
  #-------------------------------------------------------------------


  def __is_playing(self):
  #-------------------------------------------------------------------
    return self.__playing
    
  #-------------------------------------------------------------------


  def __is_active(self):
  #-------------------------------------------------------------------
    try:
      if (bonobo.activation.query("(repo_ids.has('%s')) AND (_active == TRUE)" % RHYTHMBOX_IDL)):
        return True
    except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
    return False
	#-------------------------------------------------------------------


  def __get_perc_played(self):
  #-------------------------------------------------------------------
    if (self.__songProbs != None) and (self.__rb != None):
      try:
        _play_time = self.__rb.getPlayingTime()
        if (_play_time != -1):
          return int((_play_time*1.0/self.__songProbs.duration)*100)
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
    
    return 0
    
  #-------------------------------------------------------------------


### Status Functions Read/Write

  def __get_timeplayed(self):
  #-------------------------------------------------------------------
    if (self.__rb != None and self.__rb.getPlayingTime() >=0):
      try:
        return self.__rb.getPlayingTime()
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
    
    return 0
    
  #-------------------------------------------------------------------

  def __set_timeplayed(self, value):
  #-------------------------------------------------------------------
    if (self.__rb != None):
      try:
        self.__rb.setPlayingTime(value)
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
  #-------------------------------------------------------------------


  def __get_volume(self):
  #-------------------------------------------------------------------
    if (self.__rb != None):
      try:
        return self.__rb.getVolume()
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
    
    return 0
  #-------------------------------------------------------------------
  
  def __set_volume(self, value):
  #-------------------------------------------------------------------
    if (self.__rb != None):
      try:
        self.__rb.setVolume(value)
        self._update("volume")
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
      
  #-------------------------------------------------------------------
  
  def __get_repeat(self):
  #-------------------------------------------------------------------
    if (self.__rbProps != None):
      try:
        return self.__rbProps.getValue("repeat").value()
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
        
    return False
  #-------------------------------------------------------------------


  def __set_repeat(self, value):
  #-------------------------------------------------------------------
    if (self.__rbProps != None):
      try:
        new_val = CORBA.Any(CORBA.TypeCode(BOOLEAN_IDL), CORBA.FALSE)
        if (value):
          new_val = CORBA.Any(CORBA.TypeCode(BOOLEAN_IDL), CORBA.TRUE)
        self.__rbProps.setValue("repeat", new_val)
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
      
  #-------------------------------------------------------------------
  

  def __get_shuffle(self):
  #-------------------------------------------------------------------
    if (self.__rbProps != None):
      try:
        return self.__rbProps.getValue("shuffle").value()
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
    
    return False
  #-------------------------------------------------------------------


  def __set_shuffle(self, value):
  #-------------------------------------------------------------------
    if (self.__rbProps != None):
      try:
        new_val = CORBA.Any(CORBA.TypeCode(BOOLEAN_IDL), CORBA.FALSE)
        if (value):
          new_val = CORBA.Any(CORBA.TypeCode(BOOLEAN_IDL), CORBA.TRUE)
      
        self.__rbProps.setValue("shuffle", new_val)
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
  #-------------------------------------------------------------------
  
###  write-only commands  activation by writing true to it
  
  def __toggle_hide(self, value):
	#-------------------------------------------------------------------
    if (self.__is_active() and self.__rb != None and value):
      try:
        self.__rb.toggleHide()
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
 
    
  #-------------------------------------------------------------------
  
  
  def __toggle_mute(self, value):
	#-------------------------------------------------------------------
    if (self.__is_active() and self.__rb != None and value):
      try:
        self.__rb.toggleMute()
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
    
  #-------------------------------------------------------------------
   
  
  def __volume_up(self, value):
	#-------------------------------------------------------------------
    if (self.__is_active() and self.__rb != None and value):
      try:
        self.__rb.volumeUp()
        self._update("volume")
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
    
  #-------------------------------------------------------------------
  
  
  def __volume_down(self, value):
	#-------------------------------------------------------------------
    if (self.__is_active() and self.__rb != None and value):
      try:
        self.__rb.volumeDown()
        self._update("volume")
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
  #-------------------------------------------------------------------
  
  
  def __play_pause(self, value):
	#-------------------------------------------------------------------
    if (self.__is_active() and self.__rb != None and value):
      try:
        self.__rb.playPause()
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
    
  #-------------------------------------------------------------------
  
  
  def __next(self, value):
	#-------------------------------------------------------------------
    if (self.__is_active() and self.__rb != None and value):
      try:
        self.__rb.next()
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
    
  #-------------------------------------------------------------------
  
  
  def __previous(self, value):
	#-------------------------------------------------------------------
    if (self.__is_active() and self.__rb != None and value):
      try:
        self.__rb.previous()
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
    
  #-------------------------------------------------------------------
  
  def __play_uri(self, value):
  #-------------------------------------------------------------------
    if (self.__is_active() and self.__rb != None):
      try:
        self.__rb.handleFile(value)
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
  #-------------------------------------------------------------------
  
  def __set_rating(self, value):
  #-------------------------------------------------------------------
    if (self.__is_active() and self.__rb != None):
      try:
        if (value > 5):
          value = 5
        elif (value < 0):
          value = 0
        self.__rb.setRating(value)
        self.__refresh(True)
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
  #-------------------------------------------------------------------

  def __quit(self, value):
  #-------------------------------------------------------------------
    if (self.__is_active() and self.__rb != None and value):
      try:
        self.__rb.quit()
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
  #-------------------------------------------------------------------
  
  def __refresh(self, value):
  #-------------------------------------------------------------------
    if (self.__is_active() and self.__rb != None and value):
      self.__shuffle_change()
      self.__repeat_change()
      self.__play_order_change()
      self.__playtime_change()
      self.__playing_change()
      self.__song_change()
      self._update("volume")
  #-------------------------------------------------------------------
  
  
  def __grab_focus(self, value):
  #-------------------------------------------------------------------
    if (self.__is_active() and self.__rb != None and value):
      try:
        self.__rb.grabFocus()
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
  #-------------------------------------------------------------------
  
  
  def __skip(self, value):
  #-------------------------------------------------------------------
    if (self.__is_active() and self.__rb != None):
      try:
        value = long(value)
        self.__rb.skip(value)
      except Exception, error:
        print("*** ERROR ***")
        print("An Error occured: %s" % error)
        print("*** ERROR ***")
        pass
  #-------------------------------------------------------------------
  
###   Properties definitions

  # Song Info
  songinfo = property(__get_song_properties, doc="Dictionary with the song properties: artist, title, album, genre, path, track_number, duration, bitrate, filesize, rating, play_count, last_played")
  
  # Status
  play_order = property(__get_play_order, doc = "Play order of the player, empty if inactive")
  playing = property(__is_playing, doc = "boolean flag to show if rb is playing a song")
  active = property(__is_active, doc = "boolean flag to show if rb active")
  elapsed_perc = property(__get_perc_played, doc = "number between 0 and 100 reprecenting the percentage of the played time")
  
  elapsed = property(fget=__get_timeplayed, fset=__set_timeplayed, doc = "time elapsed (secs), 0 if inactive")
  volume = property(fget=__get_volume, fset=__set_volume, doc = "current volume of the player")
  repeat = property(fget=__get_repeat, fset=__set_repeat, doc = "boolean flag of repeat state")
  shuffle = property(fget=__get_shuffle, fset=__set_shuffle, doc = "boolean flag of shuffle state")
  
  skip = property(fset=__skip, doc ="Skips forward the number of seconds written to the property")

  
  # commands (write only, activated by writing True to it)
  toggle_hide = property(fset=__toggle_hide, doc = "Toggles the visibility between hidden and shown")
  toggle_mute = property(fset=__toggle_mute, doc = "Toggles the mute state of the player")
  volume_up = property(fset=__volume_up, doc = "Increases the volume")
  volume_down = property(fset=__volume_down, doc ="Decreases the volume")
  play_pause = property(fset=__play_pause, doc = "Switches between play and pause")
  next = property(fset=__next, doc="Switches to the next song")
  previous = property(fset=__previous, doc ="Switches to the previous song")
  play_uri = property(fset=__play_uri, doc="Plays the uri written to the property")
  set_rating = property(fset=__set_rating, doc="Sets the rating of the current song")
  quit = property(fset=__quit, doc="Quits RB")
  refresh = property(fset=__refresh, doc="Refreshes Display")
  grab_focus = property(fset=__grab_focus, doc="Grabs the focus")


def get_class(): return RBox

