/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * fische-3.1
 * Copyright (C) Marcel Ebmer 2009 <marcel@26elf.at>
 * 
 * fische-3.1 is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * fische-3.1 is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <iostream> 
#include <fstream>
#include <cstring>
#include <cstdlib>
#include <stdint.h>
#include <malloc.h>
#include "blurengine.h"
#include "vectorfield.h"
#include "sdlscreen.h"
extern "C"
{
	#include "traditional.h"
}

using namespace std;

BlurEngine::BlurEngine(SdlScreen* screen, VectorField* vfield, bool mt)
{
	__sdlscreen = screen;
	__vectorfield = vfield;
	__x0 = (__sdlscreen->width() - __vectorfield->width()) / 2;
	__y0 = (__sdlscreen->height() - __vectorfield->height()) / 2;
	__xRes = __vectorfield->width();
	__yRes = __vectorfield->height();
	__buffer = malloc(__xRes * __yRes * 4);
	if(!__buffer)
	{
		cerr << "ERROR: unable to allocate memory for graphics buffer." << endl;
		exit(EXIT_FAILURE);
	}

	ifstream cpuinfo("/proc/cpuinfo");
	if(!cpuinfo.is_open()) cout << "- could not open /proc/cpuinfo - using single threaded blur algorithm" << endl;

	string str;
	__cpus = 0;
	if(mt && cpuinfo.is_open())
	{
		while(cpuinfo >> str)
		{
			if(str.find("processor") != string::npos) __cpus ++;
		}
		cpuinfo.close();
		if(__cpus == 1) cout << "* detected one CPU - using single threaded blur algorithm" << endl;
		else cout << "* detected " << __cpus << " CPUs - using multi threaded blur algorithm" << endl;
	}
	else __cpus = 1;
}

BlurEngine::~BlurEngine()
{
	if(__buffer) free(__buffer);
	__buffer = NULL;
}

void BlurEngine::blur()
{
	emptyBuffer();
	__sdlscreen->lock();

	if(__cpus == 1) blurTrad();
	else blurTrad_MT();

	writeBuffer();
	__sdlscreen->unlock();
}

void BlurEngine::blurTrad()
{
	static param_t p;
	p.pixels = __sdlscreen->pixels();
	p.xRes = __xRes;
	p.y0 = __y0;
	p.x0 = __x0;
	p.yS = 0;
	p.yE = __yRes;
	p.pitch = __sdlscreen->width();
	p.vectors = __vectorfield->get();
	p.buffer = __buffer;
	traditional(&p);
}

void BlurEngine::blurTrad_MT()
{
	int n = __cpus;
	if(n > 8) n = 8;
	if(n % 2 != 0) n--;
	
	pthread_t t[n];
	param_t p[n];
	for(int i = 0; i < n; i ++)
	{
		p[i].pixels = __sdlscreen->pixels();
		p[i].xRes = __xRes;
		p[i].y0 = __y0;
		p[i].x0 = __x0;
		p[i].pitch = __sdlscreen->width();
		p[i].vectors = __vectorfield->get();
		p[i].buffer = __buffer;
		p[i].yS = (i * __yRes ) / n;
		p[i].yE = ((i + 1) * __yRes ) / n;
		pthread_create(&t[i], NULL, traditional, &p[i]);
	}
	for(int i = 0; i < n; i ++)
	{
		pthread_join(t[i], NULL);
	}
}

void BlurEngine::emptyBuffer()
{
	bzero(__buffer, __xRes * __yRes * 4);
}

void BlurEngine::writeBuffer()
{
	uint32_t* pixels = (uint32_t*)__sdlscreen->pixels();
	uint32_t* src = (uint32_t*)__buffer;
	int pitch = __sdlscreen->width();
	for(int y = 0; y < __yRes; y ++)
	{
		memcpy(pixels + (y + __y0) * pitch + __x0, src + y * __xRes, __xRes * 4);
	}
}
