/* -*- mode: c++; c-basic-offset: 2; indent-tabs-mode: nil; -*-
 *  vim:expandtab:shiftwidth=2:tabstop=2:smarttab:
 *
 *  Copyright (C) 2008 Sun Microsystems
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


#ifndef DRIZZLED_SELECT_SEND_H
#define DRIZZLED_SELECT_SEND_H

#include <drizzled/plugin/client.h>
#include <drizzled/plugin/transactional_storage_engine.h>

namespace drizzled
{

class select_send :public select_result {
  /**
    True if we have sent result set metadata to the client.
    In this case the client always expects us to end the result
    set with an eof or error packet
  */
  bool is_result_set_started;
public:
  select_send() :is_result_set_started(false) {}
  bool send_eof()
  {
    /*
      We may be passing the control from mysqld to the client: release the
      InnoDB adaptive hash S-latch to avoid thread deadlocks if it was reserved
      by session
    */
    plugin::TransactionalStorageEngine::releaseTemporaryLatches(session);

    /* Unlock tables before sending packet to gain some speed */
    if (session->lock)
    {
      mysql_unlock_tables(session, session->lock);
      session->lock= 0;
    }
    session->my_eof();
    is_result_set_started= 0;
    return false;
  }

  bool send_fields(List<Item> &list)
  {
    bool res;
    if (! (res= session->client->sendFields(&list)))
      is_result_set_started= 1;
    return res;
  }

  void abort()
  {
    return;
  }


  /**
    Cleanup an instance of this class for re-use
    at next execution of a prepared statement/
    stored procedure statement.
  */

  virtual void cleanup()
  {
    is_result_set_started= false;
  }

  /* Send data to client. Returns 0 if ok */

  bool send_data(List<Item> &items)
  {
    if (unit->offset_limit_cnt)
    {						// using limit offset,count
      unit->offset_limit_cnt--;
      return false;
    }

    /*
      We may be passing the control from mysqld to the client: release the
      InnoDB adaptive hash S-latch to avoid thread deadlocks if it was reserved
      by session
    */
    plugin::TransactionalStorageEngine::releaseTemporaryLatches(session);

    List_iterator_fast<Item> li(items);
    char buff[MAX_FIELD_WIDTH];
    String buffer(buff, sizeof(buff), &my_charset_bin);

    Item *item;
    while ((item=li++))
    {
      if (item->send(session->client, &buffer))
      {
        my_message(ER_OUT_OF_RESOURCES, ER(ER_OUT_OF_RESOURCES), MYF(0));
        break;
      }
    }
    session->sent_row_count++;
    if (session->is_error())
      return true;
    return session->client->flush();
  }
};

} /* namespace drizzled */

#endif /* DRIZZLED_SELECT_SEND_H */
