# -*- coding: utf-8 -*-
"""This module defines some special functions
(originally defined in SpecialFunctions.h)."""

# Copyright (C) 2008-2009 Anders Logg
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN. If not, see <http://www.gnu.org/licenses/>.
#
# Modified by Johan Hake, 2008-2009
# Modified by Garth N. Wells, 2010
#
# First added:  2008-12-08
# Last changed: 2011-04-18

__all__ = ["MeshCoordinates", "FacetArea", "FacetNormal", "CellSize"]

# Import UFL and SWIG-generated extension module (DOLFIN C++)
import ufl
import dolfin.cpp as cpp

# Local imports
from dolfin.functions.functionspace import dim2domain
from dolfin.functions.expression import Expression

class MeshCoordinates(Expression, ufl.Coefficient, cpp.MeshCoordinates):
    def __init__(self, mesh):
        "Create function that evaluates to the mesh coordinates at each vertex."

        domain = dim2domain[mesh.topology().dim()]
        self._ufl_element = ufl.VectorElement("Lagrange", domain, 1)

        ufl.Coefficient.__init__(self, self._ufl_element)
        cpp.MeshCoordinates.__init__(self, mesh)

MeshCoordinates.__doc__ = cpp.MeshCoordinates.__doc__

class FacetArea(Expression, ufl.Coefficient, cpp.FacetArea):

    def __init__(self, mesh):
        """
        Create function that evaluates to the facet area/length on each facet.

        *Arguments*
            mesh
                a :py:class:`Mesh <dolfin.cpp.Mesh>`.

        *Example of usage*

            .. code-block:: python

                mesh = UnitSquare(4,4)
                fa = FacetArea(mesh)

        """

        domain = dim2domain[mesh.topology().dim()]
        self._ufl_element = ufl.FiniteElement("Discontinuous Lagrange", domain, 0)

        ufl.Coefficient.__init__(self, self._ufl_element)
        cpp.FacetArea.__init__(self, mesh)

FacetArea.__doc__ = cpp.FacetArea.__doc__

# Simple definition of FacetNormal via UFL
def FacetNormal(mesh):
    """
    Return facet normal function for given mesh.

    *Arguments*
        mesh
            a :py:class:`Mesh <dolfin.cpp.Mesh>`.


    *Example of usage*

        .. code-block:: python

            mesh = UnitSquare(4,4)
            n = FacetNormal(mesh)

    """
    if mesh.topology().dim() == 1:
        return ufl.interval.n
    elif mesh.topology().dim() == 2:
        return ufl.triangle.n
    elif mesh.topology().dim() == 3:
        return ufl.tetrahedron.n
    else:
        cpp.error("Unable to create facet normal, unhandled topological dimension %s." % str(mesh.topology().dim()))

# Simple definition of CellSize via UFL
def CellSize(mesh):
    """
    Return cell size function for given mesh.

    *Arguments*
        mesh
            a :py:class:`Mesh <dolfin.cpp.Mesh>`.

    *Example of usage*

        .. code-block:: python

            mesh = UnitSquare(4,4)
            h = CellSize(mesh)

    """
    if mesh.topology().dim() == 1:
        return 2.0*ufl.interval.circumradius
    elif mesh.topology().dim() == 2:
        return 2.0*ufl.triangle.circumradius
    elif mesh.topology().dim() == 3:
        return 2.0*ufl.tetrahedron.circumradius
    else:
        cpp.error("Unable to create cell size, unhandled topological dimension %s." % str(mesh.topology().dim()))
