import ufl
from dolfin import FunctionSpace, MixedFunctionSpace, TrialFunction

__all__ = ["derivative", "increase_order", "tear",
           "assign_domains"]

def assign_domains(fro=None, to=None):
    # Remove me if this gets added to UFL instead.
    to.cell_domains = fro.cell_domains
    to.interior_facet_domains = fro.interior_facet_domains
    to.exterior_facet_domains = fro.exterior_facet_domains

def derivative(form, u, du=None):
    if du is None:
        du = TrialFunction(u.function_space())
    a = ufl.derivative(form, u, du)
    assign_domains(fro=form, to=a)
    return ufl.derivative(form, u, du)

def increase_order(V):
    """
    For a given function space, return the same space, but with a
    higher polynomial degree
    """

    n = V.num_sub_spaces()
    if n > 0:
        return MixedFunctionSpace([increase_order(V.sub(i)) for i in range(n)])

    if V.ufl_element().family() == "Real":
        return FunctionSpace(V.mesh(), "Real", 0)

    return FunctionSpace(V.mesh(), V.ufl_element().family(),
                         V.ufl_element().degree() + 1)

def change_regularity(V, family):
    """
    For a given function space, return the corresponding space with
    the finite elements specified by 'family'. Possible families
    are the families supported by the form compiler
    """

    n = V.num_sub_spaces()
    if n > 0:
        return MixedFunctionSpace([change_regularity(V.sub(i), family)
                                   for i in range(n)])

    element = V.ufl_element()
    shape = element.value_shape()
    if not shape:
        return FunctionSpace(V.mesh(), family, element.degree())

    return MixedFunctionSpace([FunctionSpace(V.mesh(), family, element.degree())
                               for i in range(shape[0])])

def tear(V_h):
    """
    For a given function space, return the corresponding discontinuous
    space
    """
    W = change_regularity(V_h, "DG")
    return W

