/* -*- C -*- */
// Copyright (C) 2007-2009 Ola Skavhaug
//
// This file is part of DOLFIN.
//
// DOLFIN is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// DOLFIN is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with DOLFIN.  If not, see <http://www.gnu.org/licenses/>.
//
// Modified by Johan Hake, 2008-2009.
// Modified by Anders logg, 2009.
//
// First added:  2007-12-16
// Last changed: 2009-09-24

//-----------------------------------------------------------------------------
// Add numpy typemaps and macro for numpy typemaps
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Macro for defining an unsafe in-typemap for NumPy arrays -> c arrays 
// 
// The typmaps defined by this macro just passes the pointer to the C array, 
// contained in the NumPy array to the function. The knowledge of the length
// of the incomming array is not used.
//
// TYPE       : The pointer type
// TYPE_UPPER : The SWIG specific name of the type used in the array type checks values
//              SWIG use: INT32 for integer, DOUBLE for double aso.
// NUMPY_TYPE : The NumPy type that is going to be checked for
// TYPE_NAME  : The name of the pointer type, 'double' for 'double', 'uint' for
//              'dolfin::uint'
// DESCR      : The char descriptor of the NumPy type
//-----------------------------------------------------------------------------
%define UNSAFE_NUMPY_TYPEMAPS(TYPE,TYPE_UPPER,NUMPY_TYPE,TYPE_NAME,DESCR)
%{
//-----------------------------------------------------------------------------
// Typemap function (Reducing wrapper code size)
//-----------------------------------------------------------------------------
SWIGINTERN bool convert_numpy_to_ ## TYPE_NAME ## _array_no_check(PyObject* input, TYPE*& ret)
{
  if (PyArray_Check(input)) 
  {
    PyArrayObject *xa = reinterpret_cast<PyArrayObject*>(input);
    if ( PyArray_TYPE(xa) == NUMPY_TYPE )
    {
      ret  = static_cast<TYPE*>(PyArray_DATA(xa));
      return true;
    }
  }
  PyErr_SetString(PyExc_TypeError,"numpy array of 'TYPE_NAME' expected. Make sure that the numpy array use dtype='DESCR'.");
  return false;
}
%}

//-----------------------------------------------------------------------------
// The typecheck
//-----------------------------------------------------------------------------
%typecheck(SWIG_TYPECHECK_ ## TYPE_UPPER ## _ARRAY) TYPE *
{
    $1 = PyArray_Check($input) ? 1 : 0;
}

//-----------------------------------------------------------------------------
// The typemap
//-----------------------------------------------------------------------------
%typemap(in) TYPE *
{
if (!convert_numpy_to_ ## TYPE_NAME ## _array_no_check($input,$1))
    return NULL;
}

//-----------------------------------------------------------------------------
// Apply the typemap on the TYPE* _array argument
//-----------------------------------------------------------------------------
%apply TYPE* {TYPE* _array}

%enddef

//-----------------------------------------------------------------------------
// Macro for defining an safe in-typemap for NumPy arrays -> c arrays 
// 
// Type       : The pointer type
// TYPE_UPPER : The SWIG specific name of the type used in the array type checks values
//              SWIG use: INT32 for integer, DOUBLE for double aso.
// NUMPY_TYPE : The NumPy type that is going to be checked for
// TYPE_NAME  : The name of the pointer type, 'double' for 'double', 'uint' for
//              'dolfin::uint'
// DESCR      : The char descriptor of the NumPy type
//-----------------------------------------------------------------------------
%define SAFE_NUMPY_TYPEMAPS(TYPE,TYPE_UPPER,NUMPY_TYPE,TYPE_NAME,DESCR)
%{
//-----------------------------------------------------------------------------
// Typemap function (Reducing wrapper code size)
//-----------------------------------------------------------------------------
SWIGINTERN bool convert_numpy_to_ ## TYPE_NAME ## _array_with_check(PyObject* input, dolfin::uint& _array_dim, TYPE*& _array)
{
  if (PyArray_Check(input)) 
  {
    PyArrayObject *xa = reinterpret_cast<PyArrayObject*>(input);
    if ( PyArray_TYPE(xa) == NUMPY_TYPE )
    {
      _array  = static_cast<TYPE*>(PyArray_DATA(xa));
      _array_dim = static_cast<dolfin::uint>(PyArray_DIM(xa,0));
      return true;
    }
  }
  PyErr_SetString(PyExc_TypeError,"numpy array of 'TYPE_NAME' expected. Make sure that the numpy array use dtype='DESCR'.");
  return false;
}
%}

//-----------------------------------------------------------------------------
// The typecheck
//-----------------------------------------------------------------------------
%typecheck(SWIG_TYPECHECK_ ## TYPE_UPPER ## _ARRAY) (dolfin::uint _array_dim, TYPE* _array)
{
  $1 = PyArray_Check($input) ? 1 : 0;
}

//-----------------------------------------------------------------------------
// The typemap
//-----------------------------------------------------------------------------
%typemap(in) (dolfin::uint _array_dim, TYPE* _array)
{
  if (!convert_numpy_to_ ## TYPE_NAME ## _array_with_check($input,$1,$2))
    return NULL;
}
%enddef

//-----------------------------------------------------------------------------
// Run the different macros and instantiate the typemaps
// NOTE: If a typemap is not used an error will be issued as the generated 
//       typemap function will not be used 
//-----------------------------------------------------------------------------
UNSAFE_NUMPY_TYPEMAPS(dolfin::uint,INT32,NPY_UINT,uint,I)
UNSAFE_NUMPY_TYPEMAPS(double,DOUBLE,NPY_DOUBLE,double,d)
//UNSAFE_NUMPY_TYPEMAPS(int,INT,NPY_INT,int,i)

SAFE_NUMPY_TYPEMAPS(dolfin::uint,INT32,NPY_UINT,uint,I)
//SAFE_NUMPY_TYPEMAPS(double,DOUBLE,NPY_DOUBLE,double,d)
//SAFE_NUMPY_TYPEMAPS(int,INT32,NPY_INT,int,i)

//-----------------------------------------------------------------------------
// Typecheck for function expecting two-dimensional NumPy arrays of double
//-----------------------------------------------------------------------------
%typecheck(SWIG_TYPECHECK_DOUBLE_ARRAY) (int _array_dim_0, int _array_dim_1, double* _array) 
{
    $1 = PyArray_Check($input) ? 1 : 0;
}

//-----------------------------------------------------------------------------
// Generic typemap to expand a two-dimensional NumPy arrays into three
// C++ arguments: _array_dim_0, _array_dim_1, _array
//-----------------------------------------------------------------------------
%typemap(in) (int _array_dim_0, int _array_dim_1, double* _array) 
{
  if (PyArray_Check($input)) 
  {
    PyArrayObject *xa = reinterpret_cast<PyArrayObject*>($input);
    if ( PyArray_TYPE(xa) == NPY_DOUBLE ) 
    {
      if ( PyArray_NDIM(xa) == 2 ) 
      {
        $1 = PyArray_DIM(xa,0);
        $2 = PyArray_DIM(xa,1);
        $3  = static_cast<double*>(PyArray_DATA(xa));
      } 
      else 
      {
        SWIG_exception(SWIG_ValueError, "2d Array expected");
      }
    } 
    else 
    {
      SWIG_exception(SWIG_TypeError, "Array of doubles expected");
    }
  } 
  else 
  {
    SWIG_exception(SWIG_TypeError, "Array expected");
  }
}

//-----------------------------------------------------------------------------
// Typecheck for function expecting two-dimensional NumPy arrays of int
//-----------------------------------------------------------------------------
%typecheck(SWIG_TYPECHECK_DOUBLE_ARRAY) (int _array_dim_0, int _array_dim_1, int* _array) 
{
    $1 = PyArray_Check($input) ? 1 : 0;
}

//-----------------------------------------------------------------------------
// Generic typemap to expand a two-dimensional NumPy arrays into three
// C++ arguments: _array_dim_0, _array_dim_1, _array
//-----------------------------------------------------------------------------
%typemap(in) (int _array_dim_0, int _array_dim_1, int* _array) 
{
    if PyArray_Check($input) {
        PyArrayObject *xa = reinterpret_cast<PyArrayObject*>($input);
        if ( PyArray_TYPE(xa) == NPY_INT ) {
            if ( PyArray_NDIM(xa) == 2 ) {
                $1 = PyArray_DIM(xa,0);
                $2 = PyArray_DIM(xa,1);
                $3  = static_cast<int*>(PyArray_DATA(xa));
            } else {
                SWIG_exception(SWIG_ValueError, "2d Array expected");
            }
        } else {
            SWIG_exception(SWIG_TypeError, "Array of integers expected");
        }
    } else {
        SWIG_exception(SWIG_TypeError, "Array expected");
    }
}

//-----------------------------------------------------------------------------
// Cleaner of temporary data when passing 2D NumPy arrays to C++ functions 
// expecting double **
//-----------------------------------------------------------------------------
%{
namespace __private {
  class DppDeleter {
  public:
    double** amat;
    DppDeleter () {amat = 0;}
    ~DppDeleter ()
    {
      delete[] amat;
      //free(amat);
      amat = 0;
    }
  };
}
%}

//-----------------------------------------------------------------------------
// Typemap for 2D NumPy arrays to C++ functions expecting double **
//-----------------------------------------------------------------------------
%typemap(in) double**
{
    if PyArray_Check($input) {
        PyArrayObject *xa = reinterpret_cast<PyArrayObject*>($input);
        if ( PyArray_TYPE(xa) == NPY_DOUBLE ) {
            if ( PyArray_NDIM(xa) == 2 ) {
	        const int m = PyArray_DIM(xa,0);
	        const int n = PyArray_DIM(xa,1);
                $1 = new double*[m];
                double *data = reinterpret_cast<double*>((*xa).data);
                for (int i=0;i<m;++i)
                    $1[i] = &data[i*n];
            } else {
                SWIG_exception(SWIG_ValueError, "2d Array expected");
            }
        } else {
            SWIG_exception(SWIG_TypeError, "Array of doubles expected");
        }
    } else {
        SWIG_exception(SWIG_TypeError, "Array expected");
    }
}

//-----------------------------------------------------------------------------
// Delete temporary data
//-----------------------------------------------------------------------------
%typemap(freearg) double**
{
  delete[] $1;
}

//-----------------------------------------------------------------------------
// Typemap for 2D NumPy arrays to C++ functions expecting double **
//-----------------------------------------------------------------------------
%typemap(in) (int _matrix_dim_0, int _matrix_dim_1, double** _matrix) (__private::DppDeleter tmp)
{
  if PyArray_Check($input) 
  {
    PyArrayObject *xa = reinterpret_cast<PyArrayObject *>($input);
    if ( PyArray_TYPE(xa) == NPY_DOUBLE ) 
    {
      if ( PyArray_NDIM(xa) == 2 ) 
      {
        int n = PyArray_DIM(xa,0);
        int m = PyArray_DIM(xa,1);
        $1 = n;
        $2 = m;
        double **amat = static_cast<double **>(malloc(n*sizeof*amat));
        double *data = reinterpret_cast<double *>(PyArray_DATA(xa));
        for (int i=0;i<n;++i)
            amat[i] = data + i*n;
        $3 = amat;
        tmp.amat = amat;
      } 
      else 
      {
        SWIG_exception(SWIG_ValueError, "2d Array expected");
      }
    } 
    else 
    {
      SWIG_exception(SWIG_TypeError, "Array of doubles expected");
    }
  } 
  else 
  {
    SWIG_exception(SWIG_TypeError, "Array expected");
  }
}

// vim:ft=cpp:
