# Copyright 2010 Canonical Ltd.
#
# This file is part of desktopcouch.
#
# desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Authors: Eric Casteleijn <eric.casteleijn@canonical.com>
#          Vincenzo Di Somma <vincenzo.di.somma@canonical.com>

"""Tests for migration infrastructure"""

#pylint: disable=W0212
#tests can access private attributes

from couchdb.http import ResourceNotFound
from uuid import uuid4

import desktopcouch.application.tests as test_environment

from desktopcouch.application import migration
from desktopcouch.application.tests import TestCase
from desktopcouch.application.server import DesktopDatabase
from desktopcouch.records import Record, NoRecordTypeSpecified
from desktopcouch.records.database import DCTRASH, base_n
from desktopcouch.application.stop_local_couchdb import stop_couchdb


def get_test_context():
    """Return test context."""
    return test_environment.test_context


def fake_migration():
    """fake migration code"""
    pass


class MigrationBase(TestCase):
    """Tests the registration of a migration script"""

    def setUp(self):
        """setup each test"""
        self.old_registry = migration.MIGRATIONS_REGISTRY[:]
        super(MigrationBase, self).setUp()
        self.ctx = get_test_context()
        self.dbname = self._testMethodName
        self.database = DesktopDatabase(self.dbname, create=True, ctx=self.ctx)
        self.server = self.database._server

    def tearDown(self):
        """tear down each test"""
        del self.database._server[self.database._database_name]
        try:
            del self.database._server[DCTRASH]
        except ResourceNotFound:
            pass
        this_context = get_test_context()
        if this_context != test_environment.test_context:
            stop_couchdb(ctx=this_context)
        super(MigrationBase, self).tearDown()
        migration.MIGRATIONS_REGISTRY = self.old_registry[:]


class TestRegistration(MigrationBase):
    """Tests the registration of a migration script"""

    def setUp(self):
        """setup each test"""
        super(TestRegistration, self).setUp()
        self.one_more_database = DesktopDatabase(
            'one_more_database', create=True, ctx=self.ctx)

    def tearDown(self):
        """tear down each test"""
        del self.database._server[self.one_more_database._database_name]
        try:
            del self.database._server[DCTRASH]
        except ResourceNotFound:
            pass
        super(TestRegistration, self).tearDown()

    def test_register_migration_is_added_to_the_registry(self):
        """Test that the migration script is correctly registered."""
        migration.MIGRATIONS_REGISTRY = []
        migration.register(migration_method=fake_migration, dbs=[self.dbname])
        self.assertEqual([{'method': fake_migration, 'dbs': [self.dbname]}],
            migration.MIGRATIONS_REGISTRY)
        self.assertEqual(1, len(migration.MIGRATIONS_REGISTRY))


class TestMigration(MigrationBase):
    """Tests the actual migration script"""

    def setUp(self):
        super(TestMigration, self).setUp()
        try:
            del self.database._server[DCTRASH]
        except ResourceNotFound:
            pass
        self.trash = DesktopDatabase(DCTRASH, create=True, ctx=self.ctx)

        for document_id in self.trash.db:
            # remove documents added from other tests.
            del self.trash.db[document_id]

    def tearDown(self):
        """tear down each test"""
        super(TestMigration, self).tearDown()
        try:
            del self.database._server[DCTRASH]
        except ResourceNotFound:
            pass

    def test_migration_script_is_run(self):
        """Test that the migration script is run."""

        def simple_migration(database):
            """simple migration code"""
            # just asserting something to check this code is run
            # and get the right db as argument
            self.assertEqual(self.dbname, database.db.name)
        migration.register(
            migration_method=simple_migration, dbs=[self.dbname])
        migration.run_needed_migrations(ctx=self.ctx)

    def test_migration_script_is_run_and_can_access_view(self):
        """Test that the migration script is run."""

        simple_view_code = 'function(doc){emit(doc._id, doc.record_type)}'

        def simple_migration(database):
            """simple migration code"""
            database.add_view(
                'simple_view', map_js=simple_view_code,
                design_doc=migration.MIGRATION_DESIGN_DOCUMENT)
            results = database.execute_view(
                'simple_view',
                design_doc='dc_migration')
            self.assertEqual(3, len(results))

        self.database.put_record(Record({
            'key1_1': 'val1_1', 'record_type': 'test.com'}))
        self.database.put_record(Record({
            'key2_1': 'val2_1', 'record_type': 'test.com'}))
        self.database.put_record(Record({
            'key13_1': 'va31_1', 'record_type': 'test.com'}))

        migration.register(
            migration_method=simple_migration, dbs=[self.dbname])
        migration.run_needed_migrations(ctx=self.ctx)

    def test_migration_deleted_flag_to_trash(self):
        """Test that the migration script is run."""
        record_id = self.database.put_record(
            Record({
                'key1_1': 'val1_1',
                'record_type': 'test.com',
                'application_annotations': {
                    'Ubuntu One': {
                        'private_application_annotations': {
                            'deleted': True}}}}))
        undeleted_record_id1 = self.database.put_record(
            Record({
                'key1_1': 'val1_1',
                'record_type': 'test.com',
                'application_annotations': {
                    'Ubuntu One': {
                        'private_application_annotations': {
                            'deleted': False}}}}))
        undeleted_record_id2 = self.database.put_record(
            Record({
                'key1_1': 'val1_1',
                'record_type': 'test.com'}))

        migration.run_needed_migrations(ctx=self.ctx)
        # Record no longer exists in database
        self.assertIs(None, self.database.get_record(record_id))
        # Undeleted records still exist
        self.assertIsNot(None, self.database.get_record(undeleted_record_id1))
        self.assertIsNot(None, self.database.get_record(undeleted_record_id2))

        # Known deleted record is only record in trash
        for document_id in self.trash.db:
            try:
                record = self.trash.get_record(document_id)
            except NoRecordTypeSpecified:
                continue
            private = record.application_annotations['desktopcouch'][
                'private_application_annotations']
            self.assertEqual(self.dbname, private['original_database_name'])
            self.assertEqual(record_id, private['original_id'])

    def test_migration_in_face_of_broken_records(self):
        """Test that the migration does not break when we have a
        'record' without a record_type.

        """
        # pylint: disable=E1101
        record_id = base_n(uuid4().int, 62)
        self.database.db[record_id] = {
            "results": [],
            "last_seq": 1973,
            "application_annotations": {
                "Ubuntu One": {
                    "private_application_annotations": {
                        "deleted": True}}}}
        undeleted_record_id1 = base_n(uuid4().int, 62)
        self.database.db[undeleted_record_id1] = {
            'key1_1': 'val1_1',
            'application_annotations': {
                'Ubuntu One': {
                    'private_application_annotations': {
                        'deleted': False}}}}
        undeleted_record_id2 = base_n(uuid4().int, 62)
        self.database.db[undeleted_record_id2] = {
            'key1_1': 'val1_1'}
        # pylint: enable=E1101
        migration.run_needed_migrations(ctx=self.ctx)
        # None of them are deleted, since they are not records
        self.assertIn(record_id, self.database.db)
        self.assertIn(undeleted_record_id1, self.database.db)
        self.assertIn(undeleted_record_id2, self.database.db)
        self.assertNotIn(record_id, self.trash.db)
        self.assertNotIn(undeleted_record_id1, self.trash.db)
        self.assertNotIn(undeleted_record_id2, self.trash.db)
