/*
Copyright 2010 Google, Inc.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <sysexits.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

#include "subprocess.h"

int childpid = -1;  /* default to a bogus pid */
volatile sig_atomic_t killed_by_us = 0;
volatile sig_atomic_t fatal_error_in_progress = 0;
unsigned int next_kill_signal = 0;


static void rekill_alarm_handler(int signum);

/* http://partmaps.org/era/unix/award.html#uuk9letter
   http://www.catonmat.net/blog/musical-geek-friday-kill-dash-nine/
*/
void kill_process_group() {
  int pgid;
  const int signals[] = {SIGTERM, SIGINT, SIGHUP};

  killed_by_us = 1;
  pgid = getpgid(childpid);
  syslog(LOG_DEBUG, "sending signal %d to process group %d\n",
         signals[next_kill_signal], pgid);
  if (killpg(pgid, signals[next_kill_signal]) < 0) {
    if (errno == ESRCH) {
      /* The process group no longer exists, so we can return! */
      return;
    } else {
      perror("killpg");
      exit(EX_OSERR);
    }
  }
  if (next_kill_signal < (sizeof(signals) / sizeof(int))) {
    syslog(LOG_DEBUG, "triggering rekill handler in 1s\n");
    next_kill_signal++;
    signal(SIGALRM, rekill_alarm_handler);
    alarm(1);
  }
}

static void rekill_alarm_handler(int signum) {
  int old_errno;

  (void) signum; /* suppress unused parameter warnings */
  old_errno = errno;
  kill_process_group();
  errno = old_errno;
}


static void termination_handler(int sig);
static void termination_handler(int sig) {
  int old_errno;

  if (fatal_error_in_progress) {
    raise(sig);
  }
  fatal_error_in_progress = 1;

  if (childpid > 0) {
    old_errno = errno;
    /* we were killed (SIGTERM), so make sure child dies too */
    kill_process_group();
    errno = old_errno;
  }

  signal(sig, SIG_DFL);
  raise(sig);
}

void install_termination_handler(void);
void install_termination_handler(void) {
  struct sigaction sa, old_sa;
  sa.sa_handler = termination_handler;
  sigemptyset(&sa.sa_mask);
  sigaddset(&sa.sa_mask, SIGINT);
  sigaddset(&sa.sa_mask, SIGHUP);
  sigaddset(&sa.sa_mask, SIGTERM);
  sigaddset(&sa.sa_mask, SIGQUIT);
  sa.sa_flags = 0;
  sigaction(SIGINT, NULL, &old_sa);
  if (old_sa.sa_handler != SIG_IGN)
    sigaction(SIGINT, &sa, NULL);
  sigaction(SIGHUP, NULL, &old_sa);
  if (old_sa.sa_handler != SIG_IGN)
    sigaction(SIGHUP, &sa, NULL);
  sigaction(SIGTERM, NULL, &old_sa);
  if (old_sa.sa_handler != SIG_IGN)
    sigaction(SIGTERM, &sa, NULL);
}

int run_subprocess(char * command, char ** args, void (*pre_wait_function)(void)) {
  int pid;
  int status;

  childpid = fork();
  if (childpid == 0) {
    /* try to detach from parent's process group */
    if (setsid() == -1) {
      syslog(LOG_ERR, "Unable to detach child.  Aborting\n");
      return -1;
    }
    if (execvp(command, args)) {
      perror("execvp");
      exit(EX_NOINPUT);
    } else {
      /* Control almost certainly will not get to this point, ever. If
       * the call to execvp returned, instead of switching to a new memory
       * image, there was a problem. This exit will be collected by the
       * parent's call to waitpid() below.
       */
      exit(EX_UNAVAILABLE);
    }
  } else if (childpid < 0) {
    perror("fork");
    exit(EX_OSERR);
  } else {
    /* Make sure the child dies if we get killed. */
    /* Only the parent should do this, of course! */
    install_termination_handler();

    if (pre_wait_function != NULL) {
      pre_wait_function();
    }

    /* blocking wait on the child */
    if ((pid = waitpid(childpid, &status, 0)) < 0) {
      alarm(0);
      if (errno == EINTR && killed_by_us) {
        /* TODO(jaq): this is only true if we are runalarm... */
        /* status = EX_PROTOCOL; */
      } else {
        perror("waitpid");
      }
    } else {
      alarm(0);
      if (pid != childpid) {
        syslog(LOG_ERR, "childpid %d not retured by waitpid! instead %d\n",
               childpid, pid);
        kill_process_group();
        exit(EX_OSERR);
      }

      /* exited normally? */
      if (WIFEXITED(status)) {
        /* decode and return exit sttus */
        status = WEXITSTATUS(status);
        syslog(LOG_DEBUG, "child exited with status %d\n", status);
      } else {
        /* This formula is a Unix shell convention */
        status = 128 + WTERMSIG(status);
        syslog(LOG_DEBUG, "child exited via signal %d\n",
               WTERMSIG(status));
      }
      childpid = -1;
    }
  }
  return status;
}
