/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/**
 * \file thread.c
 * \brief Code creating threads and processes used in communication
 *        with external tools
 *
 * Main function in this file is run_command(): its main tasks are:
 * \li creating pairs of sockets for communication with child process
 *     (run_command() calls socketpair())
 * \li creating child process using fork()
 * \li replacing child process with external tool process using exec() call
 *     (this happens in child process)
 * \li creating two threads (with thread functions) that communicate with
 *     child process (this happens in parent process)
 * \li waiting for thread functions to return
 *
 * run_command() hides threads and exec() from rest of project code.
 *
 * Thread functions are read_child_stdout() and read_child_stderr() (and
 * write_to_md5sum_stdin() - see below). They are executed in parent
 * process, reading data from stdout and stderr of child process, using
 * stdout_pipe[PARENTS_END] and stderr_pipe[PARENTS_END] file descriptors.
 * Child process is an external tool that creates iso image, burns data to
 * optical disc or does some other useful things. Purpose of those two
 * thread functions is to intercept output of child process: either data
 * printed by child process to its stdout or stderr. This output (charactr
 * strings) is placed by read_child_stdout() and read_child_stderr() in two
 * buffers: stdout_pipe_buffer[] and stderr_pipe_buffer[].
 *
 * Information read from pipes is then processed by parent process using
 * regexp functions called in thread functions.See pipe_regexp.c and
 * other *_pipe_regexp.c files for details on processing data
 * in *_pipe_buffer[] with regexp functions.
 *
 * File defines two generic thread functions that can be used for almost
 * any external command: read_child_stdout() and read_child_stderr().
 * There is also one special thread function: write_to_md5sum_stdin():
 * its task is to _write_ to stdin of child process, and it calls functions
 * related to one specific tool (md5sum). Description of read_child_*()
 * thread functions can be applied to this function if you replace 'read
 * stdout/stderr of child process' with 'write to stdin of child process'.
 */

#define _BSD_SOURCE /* usleep(), setenv() */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <errno.h>

#include "gettext.h"
#include "cdw_logging.h"
#include "cdw_regex_dispatch.h"
#include "cdw_thread.h" /* PIPE_BUFFER_SIZE */
#include "cdw_utils.h"
#include "cdw_md5sum.h"
#include "cdw_debug.h"
#include "cdw_ext_tools.h"

/* choosing which descriptor in table (first or second) should belong
   to parent process and which belongs to child process seems to be
   purely arbitrary, but once set - it has to be used consistently;
   perhaps network applications using sockets already use industry-wide
   convention for this, where parent would be server and child would be
   client. I don't know nothing about such convention */
#define PARENTS_END 0
#define CHILDS_END 1

static const int cdw_stdout = 1;
static const int cdw_stderr = 2;


/* these are pairs of desciptors; their names seem to suggest that this
   is how _child_ process sees them: these are desciptors that will be
   connected to _child_ stdin, _child_ stdout and _child_ stderr;
   parent process will be looking at them and saying, "so I have listen
   on stdout_pipe[PARENTS_END] if I want to read stdout data of child
   process" */
int stdin_pipe[2];
int stdout_pipe[2];
int stderr_pipe[2];

cdw_task_t *thread_task;

/* time captured at the beginning of process */
time_t time0;

/* buffers for data read/written through pair of UNIX sockets */
char stdout_pipe_buffer[PIPE_BUFFER_SIZE + 1];
char stderr_pipe_buffer[PIPE_BUFFER_SIZE + 1];


/* this is THREAD.c, so avoid conflicts */
static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;


/* two generic thread functions */
static void *read_child_stdout(void *dummy);
static void *read_child_stderr(void *dummy);
static void cdw_thread_set_child_env(void);

#define CDW_THREAD_EVAL_READ(m_char_buf, m_i, m_pipe_buf, m_new_lines, m_f) \
	cdw_assert (m_f == cdw_stdout || m_f == cdw_stderr, "ERROR: incorrect file %d\n", m_f);	\
	if (m_char_buf[0] == 0x00) {					\
		;							\
	} else if (m_char_buf[0] == '\n'				\
		   || m_char_buf[0] == 0x0D				\
		   || m_char_buf[0] == '\b' /* ^H */			\
		   || m_i == PIPE_BUFFER_SIZE) {			\
									\
		if (m_i != 0) { /* non-empty line */			\
			m_pipe_buf[m_i] = '\0';				\
			cdw_logging_write("%s\n", m_pipe_buf);		\
			/* weird thing with 2 "ifs", but otherwise gcc issues some warning */ \
			if (m_f == cdw_stdout) {			\
				cdw_regex_stdout_execute(thread_task);	\
			}						\
			if (m_f == cdw_stderr) {			\
				cdw_regex_stderr_execute(thread_task);	\
			}						\
			m_i = 0;					\
			m_new_lines = 0;				\
		} else { /* empty line */				\
			if (m_new_lines == 0) {				\
				m_pipe_buf[m_i] = '\0';			\
				cdw_logging_write("%s\n", m_pipe_buf);	\
				m_new_lines++;				\
			} else {					\
				; /* do nothing - avoid printing multiple empty lines */ \
			}						\
		}							\
	} else {							\
		m_pipe_buf[m_i] = m_char_buf[0];			\
		m_i++;							\
	}



/* original developer comment:
 * I see CDrecorder version 0.1 source, and I write this code after this!!!
 * THX to Sharad Mittal */
void *read_child_stdout(__attribute__((unused)) void *dummy)
{
	cdw_sdm ("INFO: thread %s() awaits for data\n", __func__);

	/* initialize regex variables */
	cdw_regex_stdout_prepare(thread_task);

	ssize_t r = 1;
	while (r > 0) {
		/* memset(stdout_pipe_buffer, '\0', sizeof(stdout_pipe_buffer)); */

		int i = 0;
		int newlines = 0;

		/* get some data from pipe */
		while (r > 0) {
			char buf[1];
			buf[0] = '\0';
			r = read(stdout_pipe[0], buf, 1);
			int e = errno;
			pthread_mutex_lock(&mutex);

			if (r == 0) { /* EOF */
				; /* don't break inside mutex */
			} else if (r == -1) {
				cdw_vdm ("ERROR: read() returns -1, error = \"%s\"\n", strerror(e));
				/* don't break inside mutex */
			} else {
				CDW_THREAD_EVAL_READ(buf, i, stdout_pipe_buffer, newlines, cdw_stdout);
			}
			pthread_mutex_unlock(&mutex);
		}
	}

        cdw_regex_stdout_destroy(thread_task);

	return NULL;
}







/*
 * see comments before read_child_stdout()
 */
void *read_child_stderr(__attribute__((unused)) void *dummy)
{
	cdw_sdm ("INFO: thread %s() awaits for data\n", __func__);

	/* initialize regexp variables */
	cdw_regex_stderr_prepare(thread_task);

	ssize_t er = 1;
	while (er > 0) {
		/* memset(stderr_pipe_buffer, '\0', sizeof(stderr_pipe_buffer)); */

		int ei = 0;
		int enewlines = 0;

		/* get some data from pipe */
		while (er > 0) {
			char ebuf[1];
			ebuf[0] = '\0';
			er = read(stderr_pipe[0], ebuf, 1);
			int e = errno;
			pthread_mutex_lock(&mutex);

			if (er == 0) { /* EOF */
				; /* don't break inside mutex */
			} else if (er == -1) {
				cdw_vdm ("ERROR: read() returns -1, error = \"%s\"\n", strerror(e));
				/* don't break inside mutex */
			} else {
				CDW_THREAD_EVAL_READ(ebuf, ei, stderr_pipe_buffer, enewlines, cdw_stderr);
			}
			pthread_mutex_unlock(&mutex);
		}
	}

	cdw_regex_stderr_destroy(thread_task);

	return NULL;
}





/**
 * \brief Run external command
 *
 * Create pairs of sockets to new process (using socketpair()) and launch
 * external program/tool (using provided 'command' and call to exec()) as
 * new child process. The process will inherit created sockets.
 *
 * Function makes 'task' available as 'thread_task' so that thread functions
 * (working in parent code) and regexp functions can set task->tool_status.
 *
 * \param command - name of program (tool) that you want to run + it's arguments
 * \param task - variable describing current task
 *
 * \return 1 on success
 */
int run_command(const char *command, cdw_task_t *task)
{
	thread_task = task;
	cdw_sdm ("non-masked tool_status = 0x%04x\n", task->tool_status);

	cdw_logging_write_separator();

	time0 = time(NULL);
	/* 2TRANS: this is time stamp (current time) printed to log file;
	   %d is an integer representing UNIX time */
	cdw_logging_write(_("Time stamp: %d\n"), (int) time0);
	/* 2TRANS: this is message printed in log file; this is current
	   time (in readable form, like "Wed Jun 30 21:49:08 1993");
	   %s is a string with time in readable, probably localized, form */
	cdw_logging_write(_("Time: %s\n"), ctime(&time0));
	/* 2TRANS: this is message printed in log file; "current command"
	   is a call to external command that will be performed */
	cdw_logging_write(_("Full command issued to shell: \"%s\"\n"), command);
	/* 2TRANS: this is message printed in log file; 'external tools' are
	   programs like cdrecord or mkisofs */
	cdw_logging_write(_("Output from external tool(s) follow:\n\n"));

	if ((socketpair(AF_UNIX, SOCK_STREAM, 0, stdin_pipe) == 0)
		    && (socketpair(AF_UNIX, SOCK_STREAM, 0, stdout_pipe) == 0)
		    && (socketpair(AF_UNIX, SOCK_STREAM, 0, stderr_pipe) == 0)) {

		int fork_result;

		/* I'm using blocking read in thread code to minimize
		   idle looping, hence no O_NONBLOCK here */
		fcntl(stdin_pipe[CHILDS_END], F_SETFL, O_ASYNC);
		fcntl(stdin_pipe[PARENTS_END], F_SETFL, O_ASYNC);

		fcntl(stdout_pipe[CHILDS_END], F_SETFL, O_ASYNC);
		fcntl(stdout_pipe[PARENTS_END], F_SETFL, O_ASYNC);

		fcntl(stderr_pipe[CHILDS_END], F_SETFL, O_ASYNC);
		fcntl(stderr_pipe[PARENTS_END], F_SETFL, O_ASYNC);

		fork_result = fork();

		if (fork_result == -1) {
			/* 2TRANS: this is string displayed in console when
			   something went very badly and cdw cannot create child
			   process - this is serious error and cdw will exit */
			fprintf(stderr, _("Fork failure\n"));
			exit(-1);
		} else if (fork_result == 0) {
			/* we are child process - don't write to stdout and stderr;
			   instead we should write to socked inherited from parent process,
			   so parent process can know what we (child process) are doing

			   dup() will copy it's argument descriptors to lowest available
			   descriptors; let's make stdin, stdout and stderr
			   'lowest available' by closing them before calling dup();

			   let's also close original descriptor after copying it:
			   we have copy, original is no longer needed; we don't need
			   the other end of pipe here either (that is why we close both
			   *_pipe[0] and *_pipe[1]), but parent will keep the other ends
			   open to read from it as we write to our new stdout and stderr
			*/



			/* now some work on switchboard */

			/* close our original stdin so we can replace it
			   with stdin descriptor from pipe */
			close(0); /* WARNING: closing it may have strange influence on mkisofs */
			/* this makes copy of pipe stdout descriptor
			   and assigns '0' to this copy */
			dup(stdin_pipe[CHILDS_END]);
			/* close unused _copy_ of descriptor from pipe */
			close(stdin_pipe[CHILDS_END]);

			/* close our original stdout so we can replace it
			   with stdout descriptor from pipe */
			close(1);
			/* this makes copy of pipe stdout descriptor
			   and assigns '1' to this copy */
			dup(stdout_pipe[CHILDS_END]);
			/* close unused _copy_ of descriptor from pipe */
			close(stdout_pipe[CHILDS_END]);

			/* close our original stderr so we can replace
			   it with stderr descriptor from pipe */
			close(2);
			/* this makes copy of pipe stderr descriptor and
			   assigns '2' to this copy */
			dup(stderr_pipe[CHILDS_END]);
			/* close unused _copy_ of descriptor from pipe */
			close(stderr_pipe[CHILDS_END]);

			/* at this point child process still has descriptors
			   '0', '1' and '2' (stdin, stdout and stderr) open,
			   but they aren't connected to terminal (keyboard
			   and screen) anymore: they are now connected to
			   pipe, so any data printed by child application
			   to stdout or stderr (e.g. using
			   fprintf(stderr, ...) ) will be sent via pipe to
			   parent. Parent app will be able to read child
			   app data sent via pipe, as long as parent keeps
			   his ends of stdout and stderr pipe open. */

			/* close parent's ends of pipes: they are
			   to be operated by parent only, we shouldn't
			   read/write to them so let's close them and
			   don't bother with them anymore */
			close(stdin_pipe[PARENTS_END]);
			close(stdout_pipe[PARENTS_END]);
			close(stderr_pipe[PARENTS_END]);

			fcntl(stdin_pipe[CHILDS_END], F_SETFL, O_ASYNC);
			/* fcntl(stdin_pipe[PARENTS_END], F_SETFL, O_ASYNC); */

			/* below a copy of cdw turns into e.g. mkisofs */


#if 0
			/* I'm not using execve anymore because it seems
			   that it didn't have access to extended PATH
			   variable, which made me unable to search in
			   user's bin catalogs (using "which"); I could
			   pass PATH to new process by env[], but building
			   PATH for new process would require using concat() */

			char *cmd[] = { "sh" , "-c", command, (char *) 0 };
			char *env[] = { "LC_ALL=C", "TERM=xterm", (char *) 0 };

			/* I'm using execve, because it allows me to set
			   LC_ALL variable. After setting LC_ALL to C
			   I am sure that any command will be running with
			   English locale and will produce English messages
			   passed via pipe to regexp code. This way I don't
			   have to worry that some day cdrecord or mkisofs
			   developers will localize their software. */
			int exec_ret = execve ("/bin/sh", cmd, env);
#endif
			cdw_thread_set_child_env();

			int exec_ret = execl("/bin/sh", "sh", "-c", command, NULL);

			/* exec*() functions don't return */

			/* 2TRANS: this is message displayed in console when
			   something went very badly and exec() function
			   returned after a call */
			perror(_("Error: return of exec() function\n"));
			/* 2TRANS: this is second part of message displayed
			   in console when something went very badly and
			   exec() function returned after a call; %d is
			   value returned by exec(); cdw will exit after
			   displaying this message */
			fprintf(stderr, _("exec() returned with value %d\n"), exec_ret);
			exit(0);
		} else { /* we are still in parent process */

			/* now close childs ends of pipes: they are to be
			   operated by child process only, we shouldn't
			   read/write to them so let's close them and
			   don't bother with them anymore */
			close(stdin_pipe[CHILDS_END]);
			close(stderr_pipe[CHILDS_END]);
			close(stdout_pipe[CHILDS_END]);

			/* we will need two threads:
			   - one that reads stderr data of child thread
			     (read_child_stderr)
			   - one that reads stdout data of child thread
			     (read_child_stdout)

			   there is one special case when we create md5sum
			   process as child process: in such situation we need
			   to write data to process' stdin, and we will need
			   separate thread for this task; since md5sum does
			   not produce any data on stderr (I think) we can
			   disregard any data that would show on stderr, and
			   we don't have to create thread with read_child_stderr
			   function; instead create thread with function that
			   will write data to md5sum's stdin; */
			int rv;
			pthread_t thread_1, thread_2;

			cdw_sdm ("INFO: creating threads\n");

			// if (task->id == CDW_TASK_CHECK_MD5SUM && task->verify.from_cd) {
			if (task->id == CDW_TASK_CHECK_MD5SUM) {
				/* cdw_verify_write_to_md5sum_stdin is a special thread function
				   just with purpose of configuring and performing read of data
				   from data source and writing the data to stdin of md5sum process */
				thread_task->verify.md5sum_stdin_fd = stdin_pipe[PARENTS_END];
				rv = pthread_create(&thread_1, NULL,
						    thread_task->verify.read_and_write,
						    (void *) thread_task);
				cdw_sdm ("INFO: created thread with function cdw_verify_write_to_md5sum_stdin\n");
			} else {
				/* "regular" thread with function for reading child's stderr */
				rv = pthread_create(&thread_1, NULL, &read_child_stderr, NULL);
				cdw_sdm ("INFO: created thread with function read_child_stderr\n");
			}

			if (rv != 0) {
				cdw_vdm ("ERROR: failed to create thread 1\n");
				/* 2TRANS: this is debug message displayed in
				   console when thread cannot be created */
				fprintf(stderr, _("Thread 1 creation error"));
			}
			rv = pthread_create(&thread_2, NULL, &read_child_stdout, NULL);
			cdw_sdm ("INFO: created thread read_child_stdout\n");
			if (rv != 0) {
				cdw_vdm ("ERROR: failed to create thread 2\n");
				/* 2TRANS: this is debug message displayed in
				   console when thread cannot be created */
				fprintf(stderr, _("Thread 2 creation error"));
			}

			/* threads are created; now we must wait for them to complete
			   their job and return (both of them);
			   when thread functions end we can execute pthread_join()s */
			pthread_join(thread_2, NULL);
			pthread_join(thread_1, NULL);

			/* closing file equals marking end of input (eof) for
			   external processes (like md5sum) waiting for more data */
			close(stdin_pipe[PARENTS_END]);

			close(stderr_pipe[PARENTS_END]);
			close(stdout_pipe[PARENTS_END]);
		}
	} else { /* creating socket pairs failed */
		/* TODO: change this function to return CDW_OK / CDW_GEN_ERROR
		   and make all callers check its return value */
		exit(-1);
	}
	cdw_sdm ("non-masked tool_status = 0x%04x\n", task->tool_status);

	return 1;
}





/* this function modifies child process' environment, which is a "throw-away"
   environment - no need to remember initial state of environment in order
   to restore it later */
void cdw_thread_set_child_env(void)
{
	/* since this function is called in child process, after modifying
	   stderr, debug messages (cdw_vdm()) will be printed to cdw.log file,
	   not to stderr; keep that in mind when debugging */
	int rv = setenv("LC_ALL", "C", 1);
	if (rv != 0) {
		int e = errno;
		cdw_vdm ("ERROR: setenv(\"LC_ALL\", \"C\", 1) returns !0, error = \"%s\"\n", strerror(e));
		/* don't return, try to execute rest of code anyway */
	}


	if (thread_task->id == CDW_TASK_BURN_FROM_FILES
	    && thread_task->burn.tool.id == CDW_TOOL_GROWISOFS) {

		if (cdw_ext_tools_is_tool_available(CDW_TOOL_MKISOFS)) {
			const char *fullpath = cdw_ext_tools_get_tool_fullpath(CDW_TOOL_MKISOFS);
			cdw_assert (fullpath != (char *) NULL, "ERROR: ext tools module returned NULL pointer\n");
			rv = setenv("MKISOFS", fullpath, 1);
			/* growisofs from Debain repository uses GENISOIMAGE instead of MKISOFS :) */
			rv = setenv("GENISOIMAGE", fullpath, 1);
			if (rv != 0) {
				int e = errno;
				cdw_vdm ("ERROR: setenv(\"MKISOFS\" ...) returns !0, error = \"%s\"\n", strerror(e));
			} else {
#ifndef NDEBUG
				const char *m = getenv("MKISOFS");
				cdw_assert (!strcmp(m, fullpath),
					    "ERROR: failed to set MKISOFS env variable to fullpath, env = \"%s\", fullpath = \"%s\"\n",
					    m, fullpath);
				cdw_vdm ("INFO: MKISOFS env variable set to \"%s\" - \"%s\"\n", m, fullpath);
#endif
			}
		} else {
			cdw_vdm ("ERROR: burning files with growisofs, but no mkisofs available\n");
		}
	} else {
#ifndef NDEBUG
		cdw_vdm ("INFO: no conditions to set mkisofs path\n");
		if (thread_task->id != CDW_TASK_BURN_FROM_FILES) {
			cdw_vdm ("INFO: not burning from files\n");
		}
		if (thread_task->burn.tool.id == CDW_TOOL_GROWISOFS) {
			cdw_vdm ("INFO: not burning with growisofs\n");
		}
#endif
	}

	return;
}





cdw_rv_t cdw_thread_send_key_to_child_process(int key)
{
	unsigned char buf[1];
	buf[0] = (unsigned char) key;
	cdw_vdm ("INFO: sending key %d to child process\n", key);
	ssize_t w = write(stdin_pipe[PARENTS_END], buf, 1);
	if (w == -1) {
		cdw_vdm ("ERROR: write returns -1\n");
		return CDW_GEN_ERROR;
	} else if (w == 0) {
		cdw_vdm ("ERROR: write returns 0\n");
		return CDW_OK;
	} else {
		return CDW_OK;
	}
}
