/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define _GNU_SOURCE /* asprintf() */
#include <stdio.h>

#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <unistd.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <ncursesw/ncurses.h>

#include "gettext.h"
#include "cdw_string.h"
#include "cdw_processwin.h"
#include "cdw_thread.h"
#include "cdw_widgets.h"
#include "cdw_ncurses.h"
#include "cdw_md5sum.h"
#include "cdw_logging.h"
#include "cdw_debug.h"
#include "cdw_main_window.h"
#include "cdw_read_disc.h"
#include "cdw_ext_tools.h"
#include "cdw_task.h"
#include "cdw_processwin.h"



/* md5sum regexp code is called twice, this is counter allowing
   the code to recognize in which table store digest */
int digest_index;

/** \brief Buffer for MD5 digest of optical disc track */
char md5sum_digest_disc[CDW_MD5SUM_DIGEST_LEN + 1];
/** \brief Buffer for MD5 digest of iso image */
char md5sum_digest_image[CDW_MD5SUM_DIGEST_LEN + 1];


static bool using_local_processwin = false;

static cdw_rv_t cdw_verify_set_up_reading_disc(cdw_task_t *task);
static void    *cdw_verify_read_and_write_disc(void *thread_task);
static void     cdw_verify_clean_up_reading_disc(cdw_task_t *task);

static cdw_rv_t cdw_verify_set_up_reading_image(cdw_task_t *task);
static void    *cdw_verify_read_and_write_image(void *thread_task);
static void     cdw_verify_clean_up_reading_image(cdw_task_t *task);

static cdw_rv_t cdw_verify_read_disc(cdw_task_t *task);
static cdw_rv_t cdw_verify_read_image(cdw_task_t *task);
static cdw_rv_t cdw_verify_display_summary_for_disc_and_image(void);


/**
 * \brief Prepare and run commands for checking md5 sums of optical disc track and iso image
 *
 * Function checks if it is possible to read data from optical disc, and
 * calls md5sum tool (using run_command()) to check md5sum of first track
 * of optical disc. Then it calls md5sum again to calculate md5 sum of iso image
 * used to burn track. Both sums are then compared and appropriate message
 * is displayed.
 *
 * Function sets and checks task->tool_status.
 *
 * \param image_fullpath - full path to iso image for which you wan to calculate md5 sum
 * \param task - this task iformation (task id and tool are required)
 *
 * \return CDW_OK on success
 * \return CDW_MEM_ERROR on malloc errors
 */
cdw_rv_t cdw_verify(cdw_task_t *task)
{
	md5sum_digest_disc[0] = '\0';
	md5sum_digest_image[0] = '\0';

	digest_index = CDW_MD5SUM_DIGEST_4_DISC;

	if (cdw_processwin_is_active()) {
		using_local_processwin = false;
	} else {
		using_local_processwin = true;
		/* 2TRANS: this is title of process window,
		   "data" is data written on optical disc */
		cdw_processwin_create(_("Verify data"),
				      /* 2TRANS: this is message in process
					 window */
				      _("Verifying result of burning"), true);
		cdw_vdm ("INFO: created processwin\n");
	}

	cdw_rv_t crv = cdw_verify_read_disc(task);
	cdw_vdm ("INFO: md5sum_digest_disc = %s\n", md5sum_digest_disc);

	/* when cdw is run in login console and there are some
	   disc read errors, OS prints some error messages to the
	   console that overwrite cdw ui; this line fixes it */
	cdw_main_ui_main_window_wrefresh();
	cdw_processwin_wrefresh();

	/* no progress information can be displayed during calculation
	   of md5sum if iso image */
	cdw_processwin_delete_progress_bar();
	cdw_processwin_force_refresh();

	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to check md5 sum of track on disc\n");
		if (using_local_processwin) {
			cdw_processwin_destroy("", false);
		}
		return CDW_GEN_ERROR;
	}
	cdw_task_check_tool_status(task);
	if (! task->tool_status.ok) {
		cdw_vdm ("ERROR: tool_status.ok flag was set to \"false\" after checking track on disc\n");
		if (using_local_processwin) {
			cdw_processwin_destroy("", false);
		}
		return CDW_GEN_ERROR;
	} else {
		cdw_task_reset_tool_status(task);
	}

	digest_index = CDW_MD5SUM_DIGEST_4_IMAGE;

	crv = cdw_verify_read_image(task);
	cdw_vdm ("INFO: md5sum_digest_image = %s\n", md5sum_digest_image);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to check md5 sum of iso image\n");
		if (using_local_processwin) {
			cdw_processwin_destroy("", false);
		}
		return CDW_GEN_ERROR;
	}
	cdw_task_check_tool_status(task);
	if (! task->tool_status.ok) {
		cdw_vdm ("ERROR: tool_status.ok flag was set to \"false\" after checking iso image\n");
		if (using_local_processwin) {
			cdw_processwin_destroy("", false);
		}
		return CDW_GEN_ERROR;
	}
	if (using_local_processwin) {
		/* 2TRANS: this is message in progress window: 'verification'
		   refers to verifying correctness of burning; no reference to
		   success/failure of verification */
		cdw_processwin_destroy(_("Verification finished"), true);
	} else {
		cdw_processwin_display_main_info("");
	}

	cdw_verify_display_summary_for_disc_and_image();

	return CDW_OK;
}





cdw_rv_t cdw_verify_read_disc(cdw_task_t *task)
{
	cdw_assert (cdw_processwin_is_active(), "ERROR: inactive processwin\n");

	cdw_rv_t crv = cdw_read_disc_check_preconditions(task);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to get disc\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window:
				      some preconditions of reading a CD were not met */
				   _("Can't read disc for purposes of verification."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_GEN_ERROR;
	}

	crv = cdw_verify_set_up_reading_disc((void *) NULL);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set up reading optical disc\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window:
				      it describes result of verification */
				   _("Failed to open optical disc for verification."),
				   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);
		return CDW_GEN_ERROR;
	} else {
		cdw_assert (task->verify.tool.fullpath != (char *) NULL, "ERROR: tool fullpath is NULL\n");
		char *command = cdw_string_concat(task->verify.tool.fullpath, " - ", (char *) NULL);
		if (command == (char *) NULL) {
			cdw_vdm ("ERROR: failed to allocate memory for command (1)\n");
			return CDW_GEN_ERROR;
		}

		/* 2TRANS: this is message in progress window:
		   description of task currently performed */
		cdw_processwin_display_main_info(_("Calculating md5 sum of burned track"));
		/* this erases stale message in line 2 */
		cdw_processwin_display_sub_info("");
		cdw_processwin_wrefresh();

		task->verify.read_and_write = cdw_verify_read_and_write_disc;
		run_command(command, task);
		free(command);
		command = (char *) NULL;

		cdw_verify_clean_up_reading_disc((cdw_task_t *) NULL);

		/* when cdw is run in login console and there are some
		   disc read errors, OS prints some error messages to the
		   console that overwrite cdw ui; this line fixes it */
		cdw_main_ui_main_window_wrefresh();
		cdw_processwin_wrefresh();

		return CDW_OK;
	}
}





cdw_rv_t cdw_verify_read_image(cdw_task_t *task)
{
	cdw_assert (cdw_processwin_is_active(), "ERROR: inactive processwin\n");

	cdw_rv_t crv = cdw_verify_set_up_reading_image(task);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set up reading of iso image\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window:
				      it describes result of verification */
				   _("Failed to open ISO9660 image for verification."),
				   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);
		return CDW_GEN_ERROR;
	} else {
		cdw_assert (task->verify.tool.fullpath != (char *) NULL, "ERROR: tool fullpath is NULL\n");
		char *command = cdw_string_concat(task->verify.tool.fullpath, " - ", (char *) NULL);
		if (command == (char *) NULL) {
			cdw_vdm ("ERROR: failed to allocate memory for command (2)\n");
			return CDW_GEN_ERROR;
		}

		/* 2TRANS: this is message in process window */
		cdw_processwin_display_main_info(_("Calculating md5 sum of source ISO9660 image"));
		/* 2TRANS: this is message in progress window */
		cdw_processwin_display_sub_info(_("No progress information will be displayed"));
		task->verify.read_and_write = cdw_verify_read_and_write_image;

		run_command(command, task);
		free(command);
		command = (char *) NULL;

		cdw_verify_clean_up_reading_image(task);

		/* when cdw is run in login console and there are some
		   disc read errors, OS prints some error messages to the
		   console that overwrite cdw ui; this line fixes it */
		cdw_main_ui_main_window_wrefresh();
		cdw_processwin_wrefresh();

		return CDW_OK;
	}
}





cdw_rv_t cdw_verify_display_summary_for_disc_and_image(void)
{
	bool success = !strcmp(md5sum_digest_disc, md5sum_digest_image);

	/* 2TRANS: this is message in dialog window: it describes result of verification */
	const char *message_the_same = _("ISO9660 image and burned track are the same.");
	/* 2TRANS: this is message in dialog window: it describes result of verification */
	const char *message_different = _("ISO9660 image and burned track are different.");
	char *msg = (char *) NULL;
	/* 2TRANS: this is message in dialog window. First "%s" is
	   "ISO image and burned track are {the same | different}",
	   second and third "%s" are md5sum values of burned track
	   and of source iso image file */
	int a = asprintf(&msg, _("Verification finished. %s\n\n md5 sum of track:\n\"%s\"\n md5 sum of source ISO9660 image:\n\"%s\""),
			 success ? message_the_same : message_different,
			 md5sum_digest_disc, md5sum_digest_image);
	if (a == -1) {
		if (success) {
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Message"), message_the_same,
					   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);
		} else {
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Error"), message_different,
					   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
		}
	} else {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(success ? _("Success") : _("Error"), msg,
				   CDW_BUTTONS_OK, success ? CDW_COLORS_DIALOG : CDW_COLORS_ERROR);
		free(msg);
		msg = (char *) NULL;
	}
	cdw_main_ui_main_window_wrefresh();

	/* 2TRANS: this is text written to log file: a header printed before
	   two md5 sums for track and for ISO image are printed */
	cdw_logging_write(_("\n\nmd5 sums:\n"));
	/* 2TRANS: this is text written to log file: %s is a md5 digest of
	   data burned to first track of optical disc;
	   use spaces to make sure that '%s' in this string and in
	   "md5 sum of iso image:   %s\n" is at the same position */
	cdw_logging_write(_("md5 sum of burned data: %s\n"), md5sum_digest_disc);
	/* 2TRANS: this is text written to log file: %s is md5 digest of
	   data from iso image;
	   use spaces to make sure that '%s' in this string and in
	   "md5 sum of burned data: %s\n" is at the same position */
	cdw_logging_write(_("md5 sum of source ISO9660 image:   %s\n"), md5sum_digest_image);
	cdw_logging_write("\n\n");

	return CDW_OK;
}





/**
 * \brief Initialize cdio disc for reading of data track
 *
 * Open cdio disc and get its metadata needed for further reading of
 * sectors from first track. Function will return CDW_GEN_ERROR if
 * disc cannot be open, if function cannot read metainformation from that
 * disc, if there is more than one track or if the track is of wrong kind.
 *
 * \return CDW_OK on success
 * \return CDW_NO if disc has more than one track or the track is of wrong kind
 * \return CDW_GEN_ERROR on other problems
 */
cdw_rv_t cdw_verify_set_up_reading_disc(cdw_task_t *task)
{
	cdw_rv_t crv = cdw_cdio_open_disc();

	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to open cdio disc\n");
		task->tool_status.libcdio |= CDW_TOOL_STATUS_LIBCDIO_SETUP_FAIL;
		cdw_vdm ("ERROR: task->tool_status |= CDW_TOOL_STATUS_LIBCDIO_SETUP_FAIL\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window: disc
				      is in drive, but cdw cannot open it using a
				      library API */
				   _("Cannot open disc."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_GEN_ERROR;
	}

	/* we can verify only first track of type supported by cdw */
	track_t n_tracks = cdw_cdio_get_number_of_tracks();
	track_t first_track = cdw_cdio_get_first_track();

	if (n_tracks > 1) {
		cdw_vdm ("WARNING: the disc has more than one track, cdw will check only first track\n");
	}
	if (cdw_cdio_is_data_track(first_track)) {
		return CDW_OK;
	} else {
		cdw_vdm ("ERROR: first track on disc is not a data track\n");
		return CDW_GEN_ERROR;
	}
}





/**
 * \brief Read first track of optical disc
 *
 * Read first track from disc opened with md5sum_set_up_reading_track(),
 * write sectors that you read to given file descriptor. The file
 * descriptor must be associated with file that is already open for
 * writing.
 *
 * Function sets reading_success flag according to result of reading.
 *
 * \param file - target file descriptor
 *
 * \return CDW_OK on read success
 * \return CDW_GEN_ERROR on read failure
 */
void *cdw_verify_read_and_write_disc(void *thread_task)
{
	cdw_task_t *task = (cdw_task_t *) thread_task;

	track_t i = cdw_cdio_get_first_track_number();
	cdw_sdm ("reading track %d\n", (int) i);
	int rv = cdw_cdio_copy_sectors_to_file(i, task->verify.md5sum_stdin_fd);

	/* closing file equals marking end of input for md5sum */
	close(task->verify.md5sum_stdin_fd);

	if (rv == 0) {
		cdw_vdm ("INFO: read and write optical disc sectors ok, end\n");
		return (void *) NULL;
	} else {
		task->tool_status.libcdio |= CDW_TOOL_STATUS_LIBCDIO_READ_FAIL;
		cdw_vdm ("ERROR: tool_status.libcdio |= CDW_TOOL_STATUS_LIBCDIO_READ_FAIL\n");
		cdw_cdio_copy_print_debug_on_error(rv);
		return (void *) NULL;
	}
}





/**
 * \brief Close disc that you have opened with md5sum_set_up_reading_track()
 *
 * Function closes disc using proper cdio call.
 */
void cdw_verify_clean_up_reading_disc(__attribute__((unused)) cdw_task_t *task)
{
	cdw_vdm ("INFO: closing cdio disc after read\n");
	cdw_cdio_close_disc();

	return;
}





cdw_rv_t cdw_verify_set_up_reading_image(cdw_task_t *task)
{
	cdw_vdm ("INFO: attempting to open iso file with fullpath %s\n", task->verify.iso_fullpath);
	task->verify.data_source_fd = open(task->verify.iso_fullpath, O_RDONLY);
	if (task->verify.data_source_fd == -1) {
		task->tool_status.verify |= CDW_TOOL_STATUS_VERIFY_SETUP_FAIL;
		cdw_vdm ("ERROR: tool_status.verify |= CDW_TOOL_STATUS_VERIFY_SETUP_FAIL\n");
		cdw_vdm ("ERROR: failed to open iso image, reason: %s\n", strerror(errno));
		return CDW_GEN_ERROR;
	} else {
		cdw_vdm ("INFO: iso image open ok\n");
		return CDW_OK;
	}
}





void *cdw_verify_read_and_write_image(void *thread_task)
{
	cdw_task_t *task = (cdw_task_t *) thread_task;

	/* WARNING: the buffer can't be too large; there were some
	   problems when n was 100k, the code could read that much,
	   but was unable to write that much in single function call
	   (e.g. w = 31744 r = 100000); this problem happened for
	   reading iso image one in two times, but never for reading
	   optical disc; so it's better to keep the value low,
	   e.g. 10k, and do *multiple* verifications to be sure
	   that the buffer is not too large;

	   EDIT: this problem may occur only when running cdw in gdb,
	   but still...

	*/

	const size_t n = 10000;
	char read_buffer[n];

	ssize_t r = read(task->verify.data_source_fd, read_buffer, n);
	while (r > 0) {
		ssize_t w = write(task->verify.md5sum_stdin_fd, read_buffer, (size_t) r);
		if (w != r) {
			r = -1;
			break;
		}
		cdw_sdm ("INFO: read %zd, write %zd\n", r, w);
		r = read(task->verify.data_source_fd, read_buffer, n);
	}

	/* closing file equals marking end of input for md5sum */
	close(task->verify.md5sum_stdin_fd);

	if (r == 0) {
		cdw_vdm ("INFO: image read ok, EOF\n");
		return (void *) NULL;
	} else {
		task->tool_status.verify |= CDW_TOOL_STATUS_VERIFY_READ_FAIL;
		cdw_vdm ("ERROR: tool_status.verify |= CDW_TOOL_STATUS_VERIFY_READ_FAIL\n");
		cdw_vdm ("ERROR: read() returns -1, reason: %s\n", strerror(errno));
		return (void *) NULL;
	}

}





void cdw_verify_clean_up_reading_image(cdw_task_t *task)
{
	cdw_vdm ("INFO: closing source file descriptor\n");
	close(task->verify.data_source_fd);

	return;
}

