/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * Content of file:
 * - main()
 * - cdw_main_loop(): main events/keys handling loop
 * - clean_before_cdw_exit(): clean up resources before exiting cdw
 */

#include <stdio.h>
#include <stdlib.h>

#include <ncursesw/ncurses.h>
#include <ncursesw/menu.h>

#include <fcntl.h>
#include <libintl.h>
#include <locale.h>
#include <mcheck.h>
#include <string.h>
#include <unistd.h>

#include "help.h"
#include "options.h"
#include "log.h"
#include "config.h"
#include "gettext.h"
#include "commands.h"
#include "cdda2wav.h"
#include "cdw_widgets.h" /* dialogbox() */
#include "main.h"
#include "utils.h"
#include "cdw_tmp_files.h"
#include "cdrom_ctrl.h"
#include "cdw_ui.h" /* delete_files () */


#define COLOR_GRAY


void clean_before_cdw_exit(void);
int cdw_main_loop(void);


/* global across files */
int num;
extern struct conf config; /* main cdw configuration variable */
extern WINDOW *mainwin;    /* main application window */
extern MENU *menu;



int DEBUGMSG = 0;



int main(int argc, char *argv[])
{
	int rv; /* generic return value variable */

	mtrace();

	cdw_locale_init();

	if (argc > 1) {
		process_commandline_args(argc, argv);
	}

	/* initialize ncurses - do this as soon as possible,
	 * some error situations may require use of dialog windows */
	rv = cdw_curses_init();
	if (rv == -1) {
		clean_before_cdw_exit();
		/* 2TRANS: this message displayed in console when program
		   detects that some requirements are not met, and exits;
		   please keep '\n' chars */
		printf(_("\nCannot start cdw: needed min 80x25 terminal!\n"));
		exit(EXIT_FAILURE);
	}


	/* setup of cdw options module */
	rv = preferences_init();

	/* creating new processes in threads code may create some zombie processes */
	zombie_handler_init();

	/* list of files to be written to image/disc */
	filelist_init();

	/* cdw uses some temporary files */
	tmpfiles_init();

	/* application name string for main app window - yes, it's trivial */
	cdw_title_init();

	/* main app window */
	mainform_init();

	select_window(true);

	/*
	 * initially cursor is on first left-hand menu item, but tooltips are
	 * displayed only after cursor movement; show tooltip for initial
	 * cursor position
	 */
	display_tooltip(0);

	/* main program loop - top level keys handler */
	cdw_main_loop();

	/*
	 * now let's unwind (almost) all setup code
	 * called before cdw_main_loop(); we will do this in
	 * one nice function: clean_before_cdw_exit() - wrapper function user
	 * in many other places
	 */
	clean_before_cdw_exit();

	return 0;
}




/**
 * Deallocate all previously allocated resources, terminate ncurses
 *
 * Prepare application for closing:close all files, deallocate memory,
 * destroy custom and ncurses widgets. Exit ncurses. Try to do all this
 * in reverse order to the one found in main().
 */
void clean_before_cdw_exit(void)
{
	title_clean();

	filelist_clean(); /* list of files to be written to image/disc */

	preferences_clean(); /* cdw options module */

	mainform_clean(); /* main application window */

	endwin(); /* exit ncurses */

	return;
}




/**
 * Main ncurses events loop - main keys handling loop
 *
 * Function that handles keys when cdw is started and user sits in front of
 * main cdw window: menu on the left, list of files in main area and
 * information area at the bottom. User can select items from letf-hand menu
 * or use hotkeys.
 *
 * \returns 0
 */
int cdw_main_loop(void)
{
	int c;      /* input from user */
	int rv = 0; /* generic return value */

	while ((c = wgetch(mainwin))) {
		switch (c) {

			/* -------- HOT KEYS -------- */
#if 0 /* not in this release */
#ifdef HAVE_LIBMYSQLCLIENT
		case 'c':
		case 'C':
			/* CDDB */
			cddb_window();
			main_win_refresh(false);
			break;
#endif
#ifdef HAVE_LIBSQLITE
		case 'c':
		case 'C':
			/* CDDB */
			cddb_window();
			main_win_refresh(false);
			break;
#endif
#endif
		case 'e':
		case 'E':
			/* eject tray */
			rv = cdrom_status();
			if (rv == CDS_TRAY_OPEN) {
				close_tray(config.cdrwdevice);
			} else {
				eject_tray(config.cdrwdevice);
			}
			break;
		case 'g':
		case 'G':
			/* cdda2wav */
			poor_man_cdda2wav();
			/* cdda2wav(); */
			main_win_refresh(true);
			break;
		case 'q':
		case 'Q':
			/* this should be the same code as for menu item 'Quit' below */
			/* 2TRANS: this is title of dialog window */
			rv = dialogbox(_("Please confirm"),
				       /* 2TRANS: this is message in dialog window */
				       _("Do you really want to quit cdw?"), DIALOG_OK_CANCEL);
			if (rv == BUTTON_OK) {
				clean_before_cdw_exit();
				exit(EXIT_SUCCESS);
			}
			break;
		case KEY_F(1):
			/*  Show readme */
			/* sprintf(command, "%s/README.gz", DOCDIR);
			 * log_window(_("Help"), command, 1); */
			show_help_main();
			main_win_refresh(false);
			break;
		case KEY_F(2):
			/* Edit volume ID */
			/* 2TRANS: this is title of dialog window */
			input_box(_("Volume ID"),
				  /* 2TRANS: this is message in dialog window,
				     below is input field for user to enter
				     CD label */
				  _("Enter volume label:"), config.volumeid, 0);
			break;
		case KEY_F(3):
			/* Edit other parameters */
			/* 2TRANS: this is title of dialog window */
			input_box(_("CD record parameters"),
				  /* 2TRANS: this is message in dialog window,
				     user can enter (in input filed below)
				     some parameters that will be passed
				     directly to program called by cdw */
				  _("Enter parameters:"), config.other, 0);
			break;
		case KEY_F(4):
			/* CD size select */
			size_selector();
			break;
		case 'L':
		case 'l':
		case KEY_F(5):
			/* Show last log */
			/* 2TRANS: this is title of window showing content
			   of log file with latest messages */
			log_window(_("Last log"), config.logfile, 0);
			break;
		case KEY_F(6):
			/* Show CD info */
			run_command_get_cd_info();
			/* 2TRANS: this is title of dialog window with some
			   meta-data about device and about disc currently
			   in drive */
			log_window(_("CD info"), config.logfile, 0);
			break;
/*	    case KEY_F(9):
		dialogbox("Dialog box", "This is a sample message. Show in a simple\ndialog box\n\nOk?", 3);
		break;*/

		case KEY_F(10):
			{ /* Show GPL */
				char command[555];

				sprintf(command, "%s/LICENSE", DOCDIR);
				/* 2TRANS: this is title of dialog window showing cdw license */
				log_window(_("License"), command, 1);
			}
			break;
#if 0
#ifdef HAVE_LIBMYSQLCLIENT
		case 'A':
		case 'a':
			/* Add disk to dic database */
			add_to_dic();
			main_win_refresh(true);
			break;
#endif
#ifdef HAVE_LIBSQLITE
		case 'A':
		case 'a':
			/* Add disk to dic database */
			add_to_dic();
			main_win_refresh(true);
			break;
#endif
#endif

		/* ******************************
	         * ******  LEFT-HAND MENU  ******
		 * ******************************/

		case KEY_HOME:
			menu_driver(menu, REQ_FIRST_ITEM);
			display_tooltip(item_index(current_item(menu)));
			break;
		case KEY_END:
			menu_driver(menu, REQ_LAST_ITEM);
			display_tooltip(item_index(current_item(menu)));
			break;
		case KEY_DOWN:
			menu_driver(menu, REQ_DOWN_ITEM);
			display_tooltip(item_index(current_item(menu)));
			break;
		case KEY_UP:
			menu_driver(menu, REQ_UP_ITEM);
			display_tooltip(item_index(current_item(menu)));
			break;
		case CDW_ENTER:
			{
			int ii = item_index(current_item(menu));
			/* UI_MENU* elements are numbered from 0, so they neatly correspond with menu item numbers */

			if (ii == UI_MENU_ADD_FILES) { /* Add files to list of files to be written to cd/image */
				put_addwin(getmaxy(mainwin) - 10, getmaxx(mainwin) - 10, (LINES - (getmaxy(mainwin) - 10)) / 2, (COLS - (getmaxx(mainwin) - 10)) / 2);
				main_win_refresh(true);
			} else if (ii == UI_MENU_DELETE_FILES) { /* Delete files from list of files to be written to cd/image */
				/* if cd files list not empty */
				if (num > 0) {
					set_menu_fore(menu, COLOR_PAIR(4));
					wrefresh(mainwin);
					delete_files();
					set_menu_fore(menu, COLOR_PAIR(4) | A_REVERSE);
				} else {
					/* 2TRANS: this is title of dialog window */
					dialogbox(_("No files to delete"),
							/* 2TRANS: this is message in dialog window. User wants to deselect files from list, but there are no files at this time */
							_("Nothing to delete - no files selected."), DIALOG_OK);
				}
			} else if (ii == UI_MENU_CREATE_IMAGE) { /* Create iso image */
				if (num > 0) { /* there are some files on 'selected files' list */
					rv = conditional_volume_label_dialog();
					if (rv == CDW_ESCAPE) {
						/* 2TRANS: this is title of dialog window */
						dialogbox(_("Operation canceled"),
								/* 2TRANS: this is message in dialog window. User canceled creating iso image from selected files */
								_("You pressed escape - writing cancelled."), DIALOG_OK);
						break;
					}
					rv = run_command_create_image();
					if ((rv == 0) || (rv == -2)) { /* success or failure (but not cancellation) */
						/* 2TRANS: this is title of dialog window, showing messages from program writing selected files to iso image (creating iso image from selected files) */
						after_event("\"Write to image\" log", 1);
					}
				} else {
					/* 2TRANS: this is title of dialog window */
					dialogbox(_("No files to write"),
							/* 2TRANS: this is message in dialog window: user wants to write files to CD, but there are no selected files */
							_("No files selected. Please use 'Add files'"), DIALOG_OK);
				}
			} else if (ii == UI_MENU_COPY_DATA_DISC_TO_IMAGE) { /* Copy content of optical data disc to image file on your hard disc */
				if (disc_in_drive()) {
					/* checking if file exists and if can be overwritten
					is made inside copy_cd_to_image(); FIXME: extract this feature */
					/* 2TRANS: this is title of dialog window */
					copy_cd_to_image(_("Copy CD to image"),
							/* 2TRANS: this is message dialog window, copying data CD to iso file on hdd is in progress */
							_("Copying disc to iso image..."));
				}
			} else if (ii == UI_MENU_RIP_AUDIO_DISC) { /* Copy tracks from optical audio disc to track files on your hard disc */
				if (disc_in_drive()) {
					poor_man_cdda2wav();
				}
			} else if (ii == UI_MENU_WRITE_TO_DISC) { /* Write selected files or iso image to optical disk */
				if (!disc_in_drive()) {
					break;
				}

				if (strcmp(config.fromimage, "1") == 0) {
					if (access(config.tempdir, R_OK | W_OK) == 0) {
						/* Write from image */
						run_command_write_from_image();
						eject_tray(config.cdrom);
						close_tray(config.cdrom);
						/* 2TRANS: this is title of dialog window, displaying messages from program writing iso image to CD disc */
						after_event("\"Write image\" log", 1);
					} else {
						/* 2TRANS: this is title of dialog window */
						dialogbox(_("No image file"),
								/* 2TRANS: this is message in dialog window, user wants to write image file to cd, but there is no such file selected, it can be selected in cdw options window */
								_("No image file. Please check your preferences."), DIALOG_OK);
					}
				} else if ((strcmp(config.fromimage, "0") == 0)) {
					rv = conditional_volume_label_dialog();
					if (rv == CDW_ESCAPE) {
						/* 2TRANS: this is title of dialog window */
						dialogbox(_("Operation canceled"),
								/* 2TRANS: this is message in dialog window, writing to CD cancelled by user before it even started */
								_("You pressed escape - writing cancelled."), DIALOG_OK);
						break;
					}
					if ( ! filelist_empty() ){
						/* Write directly from filesystem, based on list of selected files */
						run_command_write_direct();
						eject_tray(config.cdrom);
						close_tray(config.cdrom);
						/* 2TRANS: this is title of dialog window showing output messages from program writing to CD*/
						after_event("\"Write direct\" log", 1);
					} else {
						/* 2TRANS: this is title of dialog window */
						dialogbox(_("No files to write"),
								/* 2TRANS: this is message in dialog window, user wants to write files, but no files from hdd are selected yet */
								_("No files selected. Please use 'Add files'"), DIALOG_OK);
					}
				} else { /* is it really necessary? */
					;
				}
			}

#if 0 /* not in ths release */
			/* Write Audio */
			if ((item_index(current_item(menu)) == 5)) {
				run_command_write_audio();
			}
			/* Copy data CD */
			if ((item_index(current_item(menu)) == 6) && (config.cdrom != config.cdrwdevice)) {
				run_command_copy_disk();
			}
#endif


			else if (ii == UI_MENU_BLANK_CDRW) { /* Blank CD-RW */
				if (!disc_in_drive()) {
					break;
				}

				/* prompt user for blank method */
				rv =  blank_method_selector();

				if (rv == CDW_ESCAPE) { /* selection of blanking method was cancelled by user */
					break;
				}

				/* check if disc is erasable;
				 * possible answers:
				 * yes (rv = 0), don't know (rv = 0), no (rv != 0) */
				rv = run_command_check_erasable();
				if (rv == 0) {
					rv = run_command_blank_cd();
					if ((rv == 0) && (!strcmp(config.eject, "1")) ) { /* blanking successful */
						eject_tray(config.cdrom);
						close_tray(config.cdrom);
					}
				} else {
					/* 2TRANS: this is title of dialog window */
					dialogbox(_("Media not erasable"),
							/* 2TRANS: this is message dialog window, user tries to erase disc that is not erasable */
							_("Disc in drive is not erasable. Cannot blank it."), DIALOG_OK);
				}

			} else if (ii == UI_MENU_EDIT_OPTIONS) { /* Edit cdw options */
				/* preferences_window() reads current configuration
				 * (saved in disk file), shows user a window
				 * that allows him modify the options, save
				 * options to disk file and to global variable
				 * struct conf config */
				preferences_window();
				fill_info();
			} else if (ii == UI_MENU_EXIT_CDW) { /* Quit - Exit cdw */
				/* this should be the same code as for q/Q hotkey above */
				/* 2TRANS: this is title of dialog window asking user to confirm action */
				rv = dialogbox(_("Please confirm"),
					       /* 2TRANS: this is message in dialog window, user can press OK or Cancel */
					       _("Do you really want to quit cdw?"), DIALOG_OK_CANCEL);
				if (rv == BUTTON_OK) {
					clean_before_cdw_exit();
					exit(EXIT_SUCCESS);
				}
			} else {
				;
			}
			}
			break;
		} /* switch(c) */
		wrefresh(mainwin);
	} /* while (c = wgetch(mainwin)) */

	return 0;
}


