/***************************************************************************
                  firebird.cpp  -  Firebird Class Implementation
                             -------------------
    begin                : Mon Aug 18 2003
    copyright            : (C) 2003 by Ken Schenke
    email                : kschenke at users dot sourceforge dot net
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU      *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the Free Software           *
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA         *
 *   02110-1301, USA                                                       *
 *                                                                         *
 ***************************************************************************/

#include "firebird.h"
#include "browserlist.h"

#include <QFileDialog>

#ifdef _WIN32
#include <shlobj.h>
#endif

/***************************************************************************
 *                                                                         *
 *   Firebird::BrowseForBookmarks()                                        *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BridgeCfg &cfg (WIN32 Only)                                  *
 *      QString &bookmarks                                                 *
 *      QWidget *parent                                                    *
 *   Return:                                                               *
 *      true if the user did not select a bookmark file, false otherwise   *
 *   Description:                                                          *
 *      This function is called to present the user with a file dialog box *
 *      allowing them to select the location of their bookmark file.       *
 *                                                                         *
 ***************************************************************************/

#if defined(Q_WS_WIN)
bool Firebird::BrowseForBookmarks(const BridgeCfg &cfg, QString &bookmarks, QWidget *parent)
{
	if(bookmarks==QString::null || bookmarks=="")
		bookmarks = cfg.m_AppDataDir + "/Mozilla/Firefox/Profiles/Default";
	bookmarks = QFileDialog::getOpenFileName(parent, "Select Your "
											 "Bookmarks File", bookmarks,
											 "Bookmark File (bookmarks.html)" );

	return (bookmarks == QString::null);
}
#elif defined(Q_WS_X11)
bool Firebird::BrowseForBookmarks(const BridgeCfg &, QString &bookmarks, QWidget *parent)
{
	if(bookmarks==QString::null || bookmarks=="")
		bookmarks = QDir::homePath() + "/.firefox/default";
	bookmarks = QFileDialog::getOpenFileName(parent, "Select Your Bookmarks File",
		bookmarks, "Bookmark File (bookmarks.html)");

	return (bookmarks == QString::null);
}
#else
#error "Must define platform-dependent Firebird::BrowseForBookmarks() Function"
#endif

/***************************************************************************
 *                                                                         *
 *   Firebird::classFactory()                                              *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      the newly allocated class                                          *
 *   Description:                                                          *
 *      This is a static member function of the Mozilla class.  Its job is *
 *      to allocate an instance of the Mozilla class for the caller.       *
 *                                                                         *
 ***************************************************************************/

BrowserBk *Firebird::classFactory(void)
{
	return new Firebird;
}

/***************************************************************************
 *                                                                         *
 *   Firebird::DetectBrowser()                                             *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BridgeCfg &cfg (WIN32 Only)                                  *
 *      QStringList &path                                                  *
 *   Return:                                                               *
 *      true if an installation of Mozilla was found, false otherwise      *
 *   Description:                                                          *
 *      This function attempts to locate Firebird's bookmark database on   *
 *      system.  It will only work if the user has just one profile.       *
 *                                                                         *
 ***************************************************************************/

#if defined(Q_WS_WIN)
bool Firebird::DetectBrowser(const BridgeCfg &cfg, QStringList &paths)
{
	QString basePath;

	paths.clear();

	// Start off with the base path.  On Unix it would be something like:
	// /home/ken/.mozilla/firefox
	// On Windows it would be something like:
	// C:\Documents and Settings\Ken\Application Data\Mozilla\FireFox\Profiles

	basePath = cfg.m_AppDataDir + "/Mozilla/FireFox/Profiles";

	// With the base path defined, see if it exists

	QDir base(basePath);
	if(!base.exists())
		return false;

	// Now the fun part.  For security reasons, Firebird obfuscates the
	// profile directory.  It always begins with "default", so we'll just
	// search for anything matching this pattern.

	QStringList profileList = base.entryList(QStringList("*.default"),
											 QDir::Dirs);

	// Did we find any?

	if(profileList.count() < 1)
		return false;

	// Was there only one?

	if(profileList.count() == 1)
	{
		QString	path;
		path = basePath + "/" + profileList[0] + "/bookmarks.html";

		if(QFile::exists(path))
		{
			paths.append(path);
			return true;
		}
	}

	// There was more than one match.  Maybe someone can write some code
	// to list the profiles, allowing the user to pick one.

	return false;
}
#elif defined(Q_WS_X11)
bool Firebird::DetectBrowser(const BridgeCfg &, QStringList &paths)
{
	QString basePath;

	paths.clear();

	// Start off with the base path.  On Unix it would be something like:
	// /home/ken/.mozilla/firefox
	// On Windows it would be something like:
	// C:\Documents and Settings\Ken\Application Data\Mozilla\FireFox\Profiles

	basePath = QDir::homePath() + "/.mozilla/firefox";

	// With the base path defined, see if it exists

	QDir base(basePath);
	if(!base.exists())
		return false;

	// Now the fun part.  For security reasons, Firebird obfuscates the
	// profile directory.  It always ends in ".slt", so we'll just search
	// for anything matching this pattern.

	QStringList profileList = base.entryList(QStringList("*.default"), QDir::Dirs);

	// Did we find any?

	if(profileList.count() < 1)
		return false;

	// Was there only one?

	if(profileList.count() == 1)
	{
		QString	path;
		path = basePath + "/" + profileList[0] + "/bookmarks.html";

		if(QFile::exists(path))
		{
			paths.append(path);
			return true;
		}
	}

	// There was more than one match.  Maybe someone can write some code
	// to list the profiles, allowing the user to pick one.

	return false;
}
#else
#error "Must define platform-dependent Firebird::DetectBrowser() Function"
#endif
