/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "NetworkPrefs.h"
#include "CFUtils.h"
#include "StringLess.h"
#include "OutputFunction.h"
#include "AssertException.h"
#include <map>
#include <vector>
#include <iterator>
#include <algorithm>

using namespace std;

class NetworkPrefs::StringMapper
{
public:
	typedef std::map<StringRef, StringRef, StringLess> Map;
	
	StringMapper(Map const& map);
	
	~StringMapper();
	
	StringRef operator()(StringRef const& id) const;
private:
	Map const& m_rMap;
};

	
NetworkPrefs::NetworkPrefs(CFStringRef client_name)
:	m_ptrPrefs(SCPreferencesCreate(NULL, client_name, NULL), NORETAIN),
	m_isLocked(false)
{
	if (!m_ptrPrefs) {
		throw AssertException(__FILE__, __LINE__);
	}
}

NetworkPrefs::~NetworkPrefs()
{
	if (m_isLocked) {
		SCPreferencesUnlock(m_ptrPrefs);
	}
}

bool
NetworkPrefs::lock()
{
	return doLock(true);
}

bool
NetworkPrefs::tryLock()
{
	return doLock(false);
}

bool
NetworkPrefs::doLock(bool wait)
{
	if (!m_isLocked) {
		if (SCPreferencesLock(m_ptrPrefs, wait)) {
			m_isLocked = true;
		}
	}
	return m_isLocked;
}

bool
NetworkPrefs::applyChanges()
{
	if (!SCPreferencesCommitChanges(m_ptrPrefs)) {
		return false;
	}
	SCPreferencesApplyChanges(m_ptrPrefs);
	return true;
}

StringRef
NetworkPrefs::getCurrentSetId() const
{
	StringRef path(
		(CFStringRef)SCPreferencesGetValue(m_ptrPrefs, kSCPrefCurrentSet),
		RETAIN
	);
	if (!path || CFGetTypeID(path) != CFStringGetTypeID()) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	return CFUtils::extractLastPathComponent(path);
}

void
NetworkPrefs::setCurrentSetId(StringRef const& set_id)
{
	StringRef path = CFUtils::composePath(kSCPrefSets, set_id.get(), 0);
	if (!SCPreferencesSetValue(m_ptrPrefs, kSCPrefCurrentSet, path)) {
		throw AssertException(__FILE__, __LINE__);
	}
}

StringRef
NetworkPrefs::getSetName(StringRef const& set_id) const
{
	StringRef set_path = CFUtils::composePath(
		kSCPrefSets, set_id.get(), 0
	);
	DictRef set(
		SCPreferencesPathGetValue(m_ptrPrefs, set_path),
		RETAIN
	);
	if (!set) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	StringRef name(
		(CFStringRef)CFDictionaryGetValue(set, kSCPropUserDefinedName),
		RETAIN
	);
	if (!name || CFGetTypeID(name) != CFStringGetTypeID()) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	return name;
}

void
NetworkPrefs::enumSets(OutputFunction<StringPair>& output) const
{
	DictRef sets(
		(CFDictionaryRef)SCPreferencesGetValue(m_ptrPrefs, kSCPrefSets),
		RETAIN
	);
	if (!sets || CFGetTypeID(sets) != CFDictionaryGetTypeID()) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	CFIndex count = CFDictionaryGetCount(sets);
	vector<void const*> keys(count, 0);
	vector<void const*> values(count, 0);
	CFDictionaryGetKeysAndValues(sets, &keys[0], &values[0]);
	
	for (CFIndex i = 0; i < count; ++i) {
		CFStringRef key = (CFStringRef)keys[i];
		if (!key || CFGetTypeID(key) != CFStringGetTypeID()) {
			throw AssertException(__FILE__, __LINE__);
		}
		
		CFDictionaryRef value = (CFDictionaryRef)values[i];
		if (!value || CFGetTypeID(value) != CFDictionaryGetTypeID()) {
			throw AssertException(__FILE__, __LINE__);
		}
		
		CFStringRef name = (CFStringRef)CFDictionaryGetValue(value, kSCPropUserDefinedName);
		if (!name || CFGetTypeID(name) != CFStringGetTypeID()) {
			throw AssertException(__FILE__, __LINE__);
		}
		
		output(std::make_pair(
			RefPtr<CFStringRef>(key, RETAIN), RefPtr<CFStringRef>(name, RETAIN)
		));
	}
}

void
NetworkPrefs::enumGlobalServices(OutputFunction<StringRef>& output) const
{
	RefPtr<CFDictionaryRef> services(
		(CFDictionaryRef)SCPreferencesGetValue(m_ptrPrefs, kSCPrefNetworkServices),
		RETAIN
	);
	if (!services || CFGetTypeID(services) != CFDictionaryGetTypeID()) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	CFIndex count = CFDictionaryGetCount(services);
	vector<void const*> keys(count, 0);
	CFDictionaryGetKeysAndValues(services, &keys[0], NULL);
	
	for (CFIndex i = 0; i < count; ++i) {
		CFStringRef key = (CFStringRef)keys[i];
		if (!key || CFGetTypeID(key) != CFStringGetTypeID()) {
			throw AssertException(__FILE__, __LINE__);
		}
		StringRef path = CFUtils::composePath(
			kSCPrefNetworkServices, key, 0
		);
		output(path);
	}
}

void
NetworkPrefs::enumServicesReferencedBySet(
	StringRef const& set_id, OutputFunction<StringPair>& output) const
{
	StringRef services_path = CFUtils::composePath(
		kSCPrefSets, set_id.get(), kSCCompNetwork, kSCCompService, 0
	);
	DictRef services(
		SCPreferencesPathGetValue(m_ptrPrefs, services_path),
		RETAIN
	);
	if (!services) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	CFIndex count = CFDictionaryGetCount(services);
	vector<void const*> keys(count, 0);
	CFDictionaryGetKeysAndValues(services, &keys[0], NULL);
	
	for (CFIndex i = 0; i < count; ++i) {
		StringRef service_id((CFStringRef)keys[i], RETAIN);
		if (!service_id || CFGetTypeID(service_id) != CFStringGetTypeID()) {
			throw AssertException(__FILE__, __LINE__);
		}
		
		StringRef link_path = CFUtils::composePath(
			services_path, service_id.get(), 0
		);
		
		StringRef service_path(
			SCPreferencesPathGetLink(m_ptrPrefs, link_path),
			RETAIN
		);
		if (!service_path) {
			throw AssertException(__FILE__, __LINE__);
		}
		
		output(std::make_pair(service_id, service_path));
	}
}

void
NetworkPrefs::enumSetsReferencingService(
	StringRef const& service_path, OutputFunction<StringPair>& output) const
{
	typedef vector<StringPair> Sets;
	Sets sets;
	DECLARE_INSERTER(SetInserter, StringPair, Sets, c.push_back(val));
	SetInserter set_inserter(sets);
	enumSets(set_inserter);
	
	class MatchFinder : public OutputFunction<StringPair>
	{
	public:
		MatchFinder(StringRef const& subject)
		: m_subject(subject), m_found(false) {}
		
		virtual void operator()(StringPair const& val) {
			if (!m_found && CFEqual(val.second, m_subject)) {
				m_found = true;
			}
		}
		
		bool matchFound() const { return m_found; }
		
		void reset() { m_found = false; }
	private:
		StringRef m_subject;
		bool m_found;
	};
	MatchFinder match_finder(service_path);
	
	Sets::iterator it = sets.begin();
	Sets::iterator const end = sets.end();
	for (; it != end; ++it, match_finder.reset()) {
		enumServicesReferencedBySet(it->first, match_finder);
		if (match_finder.matchFound()) {
			output(*it);
		}
	}
}

StringRef
NetworkPrefs::duplicateSet(StringRef const& set_id,
	StringRef const& new_set_name, StringRef const& new_set_id_hint)
{
	StringRef sets_path = CFUtils::composePath(kSCPrefSets, 0);
	StringRef orig_set_path = CFUtils::composePath(sets_path, set_id.get(), 0);
	
	DictRef orig_set(
		SCPreferencesPathGetValue(m_ptrPrefs, orig_set_path),
		RETAIN
	);
	if (!orig_set) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	MutableDictRef new_set(
		CFDictionaryCreateMutableCopy(NULL, 0, orig_set),
		NORETAIN
	);
	CFDictionarySetValue(new_set, kSCPropUserDefinedName, new_set_name);
	
	StringRef new_set_id;
	StringRef new_set_path;
	if (new_set_id_hint) {
		new_set_id = new_set_id_hint;
		new_set_path = CFUtils::composePath(sets_path, new_set_id.get(), 0);
	} else {
		new_set_path = StringRef(
			SCPreferencesPathCreateUniqueChild(m_ptrPrefs, sets_path),
			NORETAIN
		);
		if (!new_set_path) {
			throw AssertException(__FILE__, __LINE__);
		}
		new_set_id = CFUtils::extractLastPathComponent(new_set_path);
	}
	
	if (!SCPreferencesPathSetValue(m_ptrPrefs, new_set_path, new_set)) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	duplicateSetServices(new_set_id);
	
	return new_set_id;
}

void
NetworkPrefs::duplicateSetServices(StringRef const& set_id)
{
	typedef vector<StringPair> Services;
	Services services;
	DECLARE_INSERTER(ServiceInserter, StringPair, Services, c.push_back(val));
	ServiceInserter service_inserter(services);
	enumServicesReferencedBySet(set_id, service_inserter);
	
	typedef map<StringRef, StringRef, StringLess> ServiceIdMap;
	ServiceIdMap service_id_map; // old_local_service_id => new_local_service_id
	
	MutableDictRef new_service_refs(
		CFDictionaryCreateMutable(
			NULL, 0, &kCFTypeDictionaryKeyCallBacks,
			&kCFTypeDictionaryValueCallBacks
		),
		NORETAIN
	);
	
	StringRef services_path = CFUtils::composePath(kSCPrefNetworkServices, 0);
	
	Services::iterator it = services.begin();
	Services::iterator const end = services.end();
	for (; it != end; ++it) {
		StringRef const& orig_service_id = it->first;
		StringRef const& orig_service_path = it->second;
		
		DictRef service(
			SCPreferencesPathGetValue(m_ptrPrefs, orig_service_path),
			RETAIN
		);
		if (!service) {
			throw AssertException(__FILE__, __LINE__);
		}
		
		StringRef new_service_path(
			SCPreferencesPathCreateUniqueChild(m_ptrPrefs, services_path),
			NORETAIN
		);
		if (!new_service_path) {
			throw AssertException(__FILE__, __LINE__);
		}
		
		if (!SCPreferencesPathSetValue(m_ptrPrefs, new_service_path, service)) {
			throw AssertException(__FILE__, __LINE__);
		}
		
		StringRef new_service_id = CFUtils::extractLastPathComponent(new_service_path);
		service_id_map[orig_service_id] = new_service_id;
		addServiceReference(new_service_refs, new_service_id, new_service_path);
	}
	
	setServiceReferences(set_id, new_service_refs);
	StringMapper service_mapper(service_id_map);
	updateServiceOrder(set_id, service_mapper);
}

void
NetworkPrefs::deleteSet(StringRef const& bf_set_id)
{
	StringRef set_path = CFUtils::composePath(kSCPrefSets, bf_set_id.get(), 0);
	if (!SCPreferencesPathRemoveValue(m_ptrPrefs, set_path)) {
		throw AssertException(__FILE__, __LINE__);
	}
	deleteUnreferencedServices();
}

void
NetworkPrefs::deleteUnreferencedServices()
{
	typedef vector<StringRef> StringVector;
	StringVector all_services;
	StringVector referenced_services;
	StringVector sets;
	
	DECLARE_INSERTER(AllServInserter, StringRef, StringVector, c.push_back(val));
	AllServInserter all_serv_inserter(all_services);
	enumGlobalServices(all_serv_inserter);
	
	DECLARE_INSERTER(SetInserter, StringPair, StringVector, c.push_back(val.first));
	SetInserter set_inserter(sets);
	enumSets(set_inserter);
	
	DECLARE_INSERTER(RefServInserter, StringPair, StringVector, c.push_back(val.second));
	RefServInserter ref_serv_inserter(referenced_services);
	
	StringVector::const_iterator it = sets.begin();
	StringVector::const_iterator end = sets.end();
	for (; it != end; ++it) {
		enumServicesReferencedBySet(*it, ref_serv_inserter);
	}
	
	StringLess comp;
	std::sort(all_services.begin(), all_services.end(), comp);
	std::sort(referenced_services.begin(), referenced_services.end(), comp);
	
	// referenced_services may contain duplicates, that's not a problem.
	
	StringVector unreferenced_services;
	std::set_difference(
		all_services.begin(), all_services.end(),
		referenced_services.begin(), referenced_services.end(),
		std::back_inserter(unreferenced_services), comp
	);
	
	it = unreferenced_services.begin();
	end = unreferenced_services.end();
	for (; it != end; ++it) {
		if (!SCPreferencesPathRemoveValue(m_ptrPrefs, *it)) {
			throw AssertException(__FILE__, __LINE__);
		}
	}
}

void
NetworkPrefs::switchCurrentSetIfRemoved()
{
	StringRef cur_set_id = getCurrentSetId();
	StringRef cur_set_path = CFUtils::composePath(
	    kSCPrefSets, cur_set_id.get(), 0
	);
	if (SCPreferencesPathGetValue(m_ptrPrefs, cur_set_path)) {
		// The current set exists -- no action is necessary.
		return;
	}
	
	DictRef sets(
		(CFDictionaryRef)SCPreferencesGetValue(m_ptrPrefs, kSCPrefSets),
		RETAIN
	);
	if (!sets || CFGetTypeID(sets) != CFDictionaryGetTypeID()) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	CFStringRef set_id = NULL;
	
	if (CFDictionaryContainsKey(sets, CFSTR("0"))) {
		// The set with id "0" is the one that is created during the installation.
		set_id = CFSTR("0");
	} else {
		CFIndex count = CFDictionaryGetCount(sets);
		if (count != 0) {
			vector<void const*> keys(count, 0);
			CFDictionaryGetKeysAndValues(sets, &keys[0], NULL);
			set_id = (CFStringRef)keys[0];
			if (!set_id || CFGetTypeID(set_id) != CFStringGetTypeID()) {
				throw AssertException(__FILE__, __LINE__);
			}
		}
	}
	
	StringRef set_path;
	
	if (set_id) {
		set_path = CFUtils::composePath(kSCPrefSets, set_id, 0);
	} else {
		// There are no sets.
		set_path = StringRef(CFSTR(""), RETAIN);
	}
	
	if (!SCPreferencesSetValue(m_ptrPrefs, kSCPrefCurrentSet, set_path)) {
		throw AssertException(__FILE__, __LINE__);
	}
}

void
NetworkPrefs::setServiceReferences(StringRef const& set_id, DictRef const& refs)
{
	StringRef path = CFUtils::composePath(
		kSCPrefSets, set_id.get(), kSCCompNetwork, kSCCompService, 0
	);
	if (!SCPreferencesPathSetValue(m_ptrPrefs, path, refs)) {
		throw AssertException(__FILE__, __LINE__);
	}
}

void
NetworkPrefs::updateServiceOrder(
	StringRef const& set_id, StringMapper const& mapper)
{
	// SCPreferencesPathGetValue() only returns dictionaries, while
	// ServiceOrder is an array, so we work with it through its parent.
	
	StringRef parent_path = CFUtils::composePath(
		kSCPrefSets, set_id.get(), kSCCompNetwork,
		kSCCompGlobal, kSCEntNetIPv4, 0
	);
	
	DictRef orig_parent(
		SCPreferencesPathGetValue(m_ptrPrefs, parent_path),
		RETAIN
	);
	if (!orig_parent) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	ArrayRef orig_order(
		(CFArrayRef)CFDictionaryGetValue(orig_parent, kSCPropNetServiceOrder),
		RETAIN
	);
	if (!orig_order || CFGetTypeID(orig_order) != CFArrayGetTypeID()) {
		throw AssertException(__FILE__, __LINE__);
	}
	
	MutableDictRef new_parent(
		CFDictionaryCreateMutableCopy(NULL, 0, orig_parent),
		NORETAIN
	);
	MutableArrayRef new_order(
		CFArrayCreateMutable(NULL, 0, &kCFTypeArrayCallBacks),
		NORETAIN
	);
	CFDictionarySetValue(new_parent, kSCPropNetServiceOrder, new_order);
	
	CFIndex const count = CFArrayGetCount(orig_order);
	for (CFIndex i = 0; i < count; ++i) {
		StringRef val(
			(CFStringRef)CFArrayGetValueAtIndex(orig_order, i),
			RETAIN
		);
		if (!val || CFGetTypeID(val) != CFStringGetTypeID()) {
			throw AssertException(__FILE__, __LINE__);
		}
		
		StringRef new_val = mapper(val);
		if (new_val) {
			CFArrayAppendValue(new_order, new_val);
		}
	}
	
	if (!SCPreferencesPathSetValue(m_ptrPrefs, parent_path, new_parent)) {
		throw AssertException(__FILE__, __LINE__);
	}
}

void
NetworkPrefs::addServiceReference(MutableDictRef const& dict,
	StringRef const& local_service_id, StringRef const& global_service_path)
{
	DictRef link = createLink(global_service_path);
	CFDictionarySetValue(dict, local_service_id, link);
}

DictRef
NetworkPrefs::createLink(StringRef const& target)
{
	CFTypeRef keys[] = { kSCResvLink };
	CFTypeRef values[] = { target.get() };
	
	DictRef dict(
		CFDictionaryCreate(
			NULL, keys, values, 1,
			&kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks
		),
		NORETAIN
	);
	return dict;
}
		

/*====================== NetworkPrefs::StringMapper ===================*/

NetworkPrefs::StringMapper::StringMapper(Map const& map)
:	m_rMap(map)
{
}

NetworkPrefs::StringMapper::~StringMapper()
{
}

StringRef
NetworkPrefs::StringMapper::operator()(StringRef const& id) const
{
	Map::const_iterator it = m_rMap.find(id);
	if (it == m_rMap.end()) {
		return StringRef();
	} else {
		return it->second;
	}
}
