/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "DebugMessage.h"
#include "DebugLayout.h"
#include "DebugConnection.h"
#include "ClientRequestDescriptor.h"
#include <gtkmm/window.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/textbuffer.h>
#include <gtkmm/textview.h>
#include <glibmm/convert.h>

using namespace std;

namespace GtkGUI
{

class DebugMessage::ContentWindow : public Gtk::Window
{
public:
	ContentWindow(std::string const& content);
	
	virtual ~ContentWindow();
	
	ContentWindow* access() { return this; }
protected:
	void on_hide();
private:
	Gtk::ScrolledWindow m_scrolledWindow;
	Gtk::TextView m_textView;
};


DebugMessage::DebugMessage(
	DebugConnection* conn, int top, std::string const& headers,
	Type type, IntrusivePtr<ClientRequestDescriptor> const& client_request)
:	m_pConnection(conn),
	m_top(top),
	m_height(DebugLayout::MSG_INITIAL_OUTER_HEIGHT),
	m_state(ST_OPEN),
	m_isSelected(false),
	m_type(type),
	m_headers(headers),
	m_ptrClientRequest(client_request),
	m_pPrev(0),
	m_pNext(0)
{
	typedef DebugLayout DL;
	
	add(m_drawingArea);
	m_drawingArea.set_size_request(DebugLayout::MSG_OUTER_WIDTH, m_height);
	m_drawingArea.show();
	
	signal_button_press_event().connect(
		sigc::mem_fun(*this, &DebugMessage::onButtonPress)
	);
	m_drawingArea.signal_expose_event().connect(
		sigc::mem_fun(*this, &DebugMessage::onExpose)
	);
	
	if (m_ptrClientRequest) {
		m_ptrClientRequest->registerMessage(this);
	}
	
	show();
}

DebugMessage::~DebugMessage()
{
	if (m_ptrClientRequest) {
		m_ptrClientRequest->unregisterMessage(this);
	}
}

void
DebugMessage::select()
{
	if (!m_isSelected) {
		m_isSelected = true;
		m_drawingArea.queue_draw();
	}
}

void
DebugMessage::unselect()
{
	if (m_isSelected) {
		m_isSelected = false;
		m_drawingArea.queue_draw();
	}
}

void
DebugMessage::growBy(int increment)
{
	typedef DebugLayout DL;
	m_height += increment;
	m_drawingArea.set_size_request(DL::MSG_OUTER_WIDTH, m_height);
	//m_drawingArea.queue_draw();
}

void
DebugMessage::finish(bool error)
{
	if (m_state == ST_OPEN) {
		m_state = error ? ST_ERROR : ST_FINISHED;
		m_drawingArea.queue_draw();
	}
}

void
DebugMessage::ensureClosed()
{
	finish(true);
}

Gdk::Color
DebugMessage::getColorFor(Type type)
{
	typedef AbstractDebugAgent ADB;
	switch (type) {
		case ADB::INCOMING_REQUEST:
		return  Gdk::Color("#4fa81f");
		case ADB::OUTGOING_REQUEST:
		return Gdk::Color("#63e24d");
		case ADB::SCRIPT_REQUEST:
		return Gdk::Color("#fcba6a");
		case ADB::SCRIPT_RESPONSE:
		return Gdk::Color("#bcb220");
		case ADB::INCOMING_RESPONSE:
		return Gdk::Color("#4a52e8");
		case ADB::OUTGOING_RESPONSE:
		case ADB::UNMODIFIED_OUTGOING_RESPONSE:
		return Gdk::Color("#4a52e8");
		case ADB::FILTERED_OUTGOING_RESPONSE:
		return Gdk::Color("#5abddd");
		case ADB::CRAFTED_OUTGOING_RESPONSE:
		return Gdk::Color("#f7e11b");
		case ADB::ERROR_OUTGOING_RESPONSE:
		default:
		return Gdk::Color("#e84c4c");
	}
}

void
DebugMessage::setupGraphicContexts()
{
	// This method can only be called after m_drawingArea,
	// has been realized, that is not from DebugMessage CTOR.
	
	typedef DebugLayout DL;
	
	if (!m_ptrMainGC) {
		m_ptrMainGC = Gdk::GC::create(m_drawingArea.get_window());
		m_ptrMainGC->set_rgb_fg_color(getColorFor(m_type));
	}
	if (!m_ptrSelectionGC) {
		m_ptrSelectionGC = Gdk::GC::create(m_drawingArea.get_window());
		m_ptrSelectionGC->set_rgb_fg_color(Gdk::Color("#000000"));
	}
}

bool
DebugMessage::onButtonPress(GdkEventButton* event)
{
	if (event->button != 1) {
		return true;
	}
	if (event->type == GDK_BUTTON_PRESS) {
		if (m_ptrClientRequest) {
			m_ptrClientRequest->onMessageSelected();
		}
	} else if (event->type == GDK_2BUTTON_PRESS) {
		showContentWindow();
	}
	return true;
}

bool
DebugMessage::onExpose(GdkEventExpose* event)
{
	setupGraphicContexts();
	drawMainArea();
	if (m_isSelected) {
		drawSelectionBorder();
	}
	return true; // we've handled this event
}

void
DebugMessage::drawMainArea()
{
	typedef DebugLayout DL;
	
	Gdk::Rectangle allocation = m_drawingArea.get_allocation();
	Glib::RefPtr<Gdk::Window> wnd = m_drawingArea.get_window();
	int x = allocation.get_x() + DL::MSG_MAIN_BORDER_OFFSET;
	int y = allocation.get_y() + DL::MSG_MAIN_BORDER_OFFSET;
	int width = allocation.get_width() - DL::MSG_MAIN_BORDER_OFFSET*2;
	int height = allocation.get_height() - DL::MSG_MAIN_BORDER_OFFSET*2;
	
	if (m_state == ST_FINISHED) {
		wnd->draw_rectangle(
			m_ptrMainGC, true,
			x, y, width, height
		);
	} else {
		wnd->draw_rectangle(
			m_ptrMainGC, true,
			x, y,
			DL::MSG_MAIN_BORDER_WIDTH, height
		);
		wnd->draw_rectangle(
			m_ptrMainGC, true,
			x + width - DL::MSG_MAIN_BORDER_WIDTH, y,
			DL::MSG_MAIN_BORDER_WIDTH, height
		);
		wnd->draw_rectangle(
			m_ptrMainGC, true,
			x + DL::MSG_MAIN_BORDER_WIDTH, y, 
			width - DL::MSG_MAIN_BORDER_WIDTH*2,
			DL::MSG_MAIN_BORDER_WIDTH
		);
	}
}

void
DebugMessage::drawSelectionBorder()
{
	typedef DebugLayout DL;
	
	Gdk::Rectangle allocation = m_drawingArea.get_allocation();
	Glib::RefPtr<Gdk::Window> wnd = m_drawingArea.get_window();
	int x = allocation.get_x();
	int y = allocation.get_y();
	int width = allocation.get_width();
	int height = allocation.get_height();
	
	if (DL::MSG_SEL_BORDER_WIDTH == 1) {
		wnd->draw_rectangle(
			m_ptrSelectionGC, false,
			x, y,
			width - DL::MSG_SEL_BORDER_WIDTH,
			height - DL::MSG_SEL_BORDER_WIDTH
		);
	} else {
		wnd->draw_rectangle(
			m_ptrSelectionGC, true,
			x, y,
			DL::MSG_SEL_BORDER_WIDTH, height
		);
		wnd->draw_rectangle(
			m_ptrSelectionGC, true,
			x + width - DL::MSG_SEL_BORDER_WIDTH, y,
			DL::MSG_SEL_BORDER_WIDTH, height
		);
		wnd->draw_rectangle(
			m_ptrSelectionGC, true,
			x + DL::MSG_SEL_BORDER_WIDTH, y, 
			width - DL::MSG_SEL_BORDER_WIDTH*2,
			DL::MSG_SEL_BORDER_WIDTH
		);
		wnd->draw_rectangle(
			m_ptrSelectionGC, true,
			x + DL::MSG_SEL_BORDER_WIDTH,
			y + height - DL::MSG_SEL_BORDER_WIDTH,
			width - DL::MSG_SEL_BORDER_WIDTH*2,
			DL::MSG_SEL_BORDER_WIDTH
		);
	}
}

void
DebugMessage::showContentWindow()
{
	if (ContentWindow* wnd = m_contentWindowAccessor()) {
		wnd->present();
	} else {
		auto_ptr<ContentWindow> wnd(new ContentWindow(m_headers));
		m_contentWindowAccessor = sigc::mem_fun(
			*wnd, &DebugMessage::ContentWindow::access
		);
		wnd->show();
		wnd.release();
	}
}


/*===================== DebugMessage::ContentWindow ======================*/

DebugMessage::ContentWindow::ContentWindow(std::string const& content)
{
	set_title("HTTP Message");
	set_size_request(500, 250);
	
	add(m_scrolledWindow);
	m_scrolledWindow.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	m_scrolledWindow.set_shadow_type(Gtk::SHADOW_IN);
	
	m_scrolledWindow.add(m_textView);
	m_textView.set_editable(false);
	m_textView.set_cursor_visible(false);
	try {
		m_textView.get_buffer()->set_text(Glib::convert(content, "utf-8", "iso-8859-1"));
	} catch (Glib::ConvertError&) {}
	
	show_all_children();
}

DebugMessage::ContentWindow::~ContentWindow()
{
}

void
DebugMessage::ContentWindow::on_hide()
{
	delete this;
}

} // namespace GtkGUI
