/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "FileInstance.h"
#include "FileId.h"
#include "AbstractFileIO.h"
#include "AbstractFileStorage.h"
#include "AtomicOps.h"
#include "AtomicCounter.h"
#include "types.h"

namespace HttpCache
{

class FileInstance::Rep
{
	DECLARE_NON_COPYABLE(Rep)
public:
	Rep(IntrusivePtr<AbstractFileStorage> const& storage,
		FileId const& file_id, bool unlink_on_no_refs);
	
	~Rep();
	
	void ref() { ++m_numRefs; }
	
	void unref() { if (--m_numRefs == 0) delete this; }
	
	void setUnlinkOnNoRefs(bool val) {
		AtomicOps::set(&m_unlinkOnNoRefs, val ? 1 : 0);
	}
private:
	IntrusivePtr<AbstractFileStorage> const m_ptrStorage;
	FileId const m_fileId;
	AtomicCounter<ACE_MT_SYNCH> m_numRefs;
	int32_t m_unlinkOnNoRefs;
};


FileInstance::FileInstance(
	IntrusivePtr<AbstractFileStorage> const& storage,
	FileId const& file_id, std::auto_ptr<AbstractFileIO> file_io,
	bool const unlink_on_no_refs)
:	m_ptrRep(new Rep(storage, file_id, unlink_on_no_refs)),
	m_ptrFileIO(file_io)
{
}

FileInstance::FileInstance(
	FileInstance const& other_instance, std::auto_ptr<AbstractFileIO> file_io)
:	m_ptrRep(other_instance.m_ptrRep),
	m_ptrFileIO(file_io)
{
}

FileInstance::~FileInstance()
{
}

void
FileInstance::unlinkOnNoRefs(bool const val)
{
	m_ptrRep->setUnlinkOnNoRefs(val);
}


/*=========================== FileInstance::Rep =========================*/

FileInstance::Rep::Rep(
	IntrusivePtr<AbstractFileStorage> const& storage,
	FileId const& file_id, bool unlink_on_no_refs)
:	m_ptrStorage(storage),
	m_fileId(file_id),
	m_numRefs(),
	m_unlinkOnNoRefs(unlink_on_no_refs ? 1 : 0)
{
}

FileInstance::Rep::~Rep()
{
	if (AtomicOps::add(&m_unlinkOnNoRefs, 0)) {
		m_ptrStorage->unlink(m_fileId);
	}
}

} // namespace HttpCache
