/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "JpegImageInspector.h"
#include "JpegContentIdentifier.h"
#include "ScopedDecInc.h"
#include "types.h"

using namespace std;

JpegImageInspector::JpegImageInspector()
:	m_state(ST_SIGNATURE),
	m_status(IN_PROGRESS),
	m_dataSize(0),
	m_curPos(m_data.begin()),
	m_curOffset(0),
	m_width(0),
	m_height(0)
{
}

JpegImageInspector::~JpegImageInspector()
{
}

void
JpegImageInspector::reset()
{
	m_state = ST_SIGNATURE;
	m_status = IN_PROGRESS;
	m_data.clear();
	m_dataSize = 0;
	m_curPos = m_data.begin();
	m_curOffset = 0;
	m_width = 0;
	m_height = 0;
}

void
JpegImageInspector::consumeDataChunk(SplittableBuffer& data, bool eof)
{
	{
		ScopedDecInc<SplittableBuffer::ByteIterator> di(m_curPos);
		// if m_curPos was at end(), it would remain there
		m_dataSize += data.size();
		m_data.appendDestructive(data);
	}
	if (m_status == IN_PROGRESS) {
		processNewData();
		if (eof && m_status == IN_PROGRESS) {
			m_status = FAILED;
		}
	}
}

void
JpegImageInspector::processNewData()
{
	switch (m_state) {
		case ST_SIGNATURE: {
			if (!checkSignature()) {
				return;
			}
			advance(3);
			m_state = ST_MARKER_OR_PADDING;
			// fall through
		}
		case ST_MARKER_OR_PADDING:
		case_ST_MARKER_OR_PADDING: {
			uint8_t byte;
			do {
				if (m_dataSize <= m_curOffset) {
					return;
				}
				byte = *m_curPos;
				advance();
			} while (byte == 0xff);
			if (byte >= 0xc0 && byte <= 0xcf && byte != 0xc4 && byte != 0xc8 && byte != 0xcc) {
				m_state = ST_SOF;
				goto case_ST_SOF;
			} else if (byte == 0xd9 || byte == 0xda) {
				m_status = FAILED;
				return;
			}
			m_state = ST_SKIP_VAR;
			// fall through
		}
		case ST_SKIP_VAR: {
			if (m_dataSize < m_curOffset + 2) {
				return;
			}
			m_curBlockSize = unsigned(uint8_t(*m_curPos)) << 8;
			advance();
			m_curBlockSize += unsigned(uint8_t(*m_curPos));
			advance();
			if (m_curBlockSize < 2) {
				m_status = FAILED;
				return;
			}
			m_curBlockSize -= 2;
			m_state = ST_SKIP_VAR_DATA;
			// fall through
		}
		case ST_SKIP_VAR_DATA: {
			if (m_dataSize < m_curOffset + m_curBlockSize) {
				return;
			}
			advance(m_curBlockSize);
			m_state = ST_MARKER_OR_PADDING;
			goto case_ST_MARKER_OR_PADDING;
		}
		case ST_SOF:
		case_ST_SOF: {
			if (m_dataSize < m_curOffset + 2) {
				return;
			}
			m_curBlockSize = unsigned(uint8_t(*m_curPos)) << 8;
			advance();
			m_curBlockSize += unsigned(uint8_t(*m_curPos));
			advance();
			if (m_curBlockSize < 8) {
				m_status = FAILED;
				return;
			}
			m_curBlockSize -= 2;
			m_state = ST_SOF_DATA;
			// fall through
		}
		case ST_SOF_DATA: {
			if (m_dataSize < m_curOffset + m_curBlockSize) {
				return;
			}
			advance();
			m_height = unsigned(uint8_t(*m_curPos)) << 8;
			advance();
			m_height += unsigned(uint8_t(*m_curPos));
			advance();
			m_width = unsigned(uint8_t(*m_curPos)) << 8;
			advance();
			m_width += unsigned(uint8_t(*m_curPos));
			m_status = COMPLETE;
			return;
		}
	}
}

bool
JpegImageInspector::checkSignature()
{
	typedef JpegContentIdentifier CI;
	SplittableBuffer::ByteIterator it(m_curPos);
	char const* sigp = CI::SIGNATURE;
	char const* const sige = CI::SIGNATURE + sizeof(CI::SIGNATURE);
	for (;; ++it, ++sigp) {
		if (sigp == sige) {
			return true;
		} else if (it.isAtRightBorder()) {
			return false;
		} else if (*sigp != char(*it)) {
			m_status = FAILED;
			return false;
		}
	}
}
