/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "HttpStateStatusLine.h"
#include "HttpResponseParser.h"
#include "SplittableBuffer.h"
#include "BString.h"
#include "HttpRequestMetadata.h"
#include "HttpResponseMetadata.h"
#include "HttpStatusLine.h"
#include "HttpVersion.h"
#include "StringUtils.h"
#include <memory>

using namespace std;

static bool checkPrefix(SplittableBuffer& data)
{
	static char const prefix[] = {'H','T','T','P','/'};
	char const* pos = prefix;
	SplittableBuffer::ByteIterator it = data.begin();
	for (; pos != prefix + sizeof(prefix) && !it.isAtRightBorder(); ++pos, ++it) {
		if (*pos != char(*it)) {
			return false;
		}
	}
	return true;
}

HttpStateStatusLine::HttpStateStatusLine(HttpResponseParser& parser)
:	m_rParser(parser)
{
}

HttpStateStatusLine::~HttpStateStatusLine()
{
}

void
HttpStateStatusLine::activate(ConstRequestPtr const& req)
{
	m_ptrRequest = req;
}

HttpState*
HttpStateStatusLine::processNewData(
	SplittableBuffer& input, bool eof,
	SplittableBuffer& body_data, bool& body_eof)
{	
	if (!checkPrefix(input)) {
		return switchToSimpleResponse();
	}
	
	SplittableBuffer::ByteIterator begin = input.begin();
	SplittableBuffer::ByteIterator const end = input.end();
	SplittableBuffer::ByteIterator nl = SplittableBuffer::find(begin, end, '\n');
	if (nl == end) {
		return this;
	}
	
	BString line(SplittableBuffer::toBString(begin, nl));
	HttpState* next_state = processStatusLine(line);
	if (next_state) {
		input.trimFront(++nl);
	} else {
		next_state = m_rParser.activateStateError("error parsing status line");
	}
	return next_state;
}

HttpState*
HttpStateStatusLine::processStatusLine(BString const& line)
{
	char const* begin = line.begin();
	char const* end = line.end();
	if (end != begin && *(end-1) == '\r') {
		--end; // end now points to '\r'
	}
	
	char const* pos = StringUtils::find(begin, end, ' ');
	pos = StringUtils::find(begin, pos, '\t');
	if (pos == end) {
		return 0;
	}
	
	HttpVersion http_version = HttpVersion::HTTP_0_9;
	if (!http_version.parse(begin, pos)) {
		return 0;
	}
	begin = pos;
	for (; begin != end && (*begin == ' ' || *begin == '\t'); ++begin) {}
	
	pos = end;
	int code = StringUtils::parseUnsigned<int>(begin, pos);
	if (pos != begin + 3) {
		return 0;
	}
	
	pos = StringUtils::ltrim(pos, end);
	end = StringUtils::rtrim(pos, end);
	BString message(line, pos, end);
	return m_rParser.activateStateHeaders(m_ptrRequest, HttpStatusLine(http_version, code, message));
}

HttpState*
HttpStateStatusLine::switchToSimpleResponse()
{
	if (m_ptrRequest->requestLine().getMethod() == BString("HEAD")) {
		// responses to HEAD requests can't contain a body
		return m_rParser.activateStateError("error parsing the status line");
	}
	auto_ptr<HttpResponseMetadata> metadata(new HttpResponseMetadata(HttpStatusLine(HttpVersion::HTTP_0_9, 200)));
	metadata->setBodyStatus(HttpResponseMetadata::BODY_SIZE_UNKNOWN);
	m_rParser.eventGotMetadata(metadata, false);
	return m_rParser.activateStateUnsizedFlatBody();
}

void
HttpStateStatusLine::cleanup()
{
	m_ptrRequest.reset(0);
}
