/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "HttpStateHeaders.h"
#include "HttpParserBase.h"
#include "HttpHeader.h"
#include "HttpHeadersCollection.h"
#include "HttpHeaderStructure.h"
#include "HttpHeaderElement.h"
#include "HttpVersion.h"
#include "SplittableBuffer.h"
#include "StringUtils.h"
#include "InsensitiveEqual.h"
#include <cassert>

using namespace std;

HttpStateHeaders::HttpStateHeaders(HttpParserBase& parser)
:	m_rParser(parser),
	m_state(ST_CHECKING_FOR_END),
	m_headerValueStream(400)
{
}

HttpStateHeaders::~HttpStateHeaders()
{
}

HttpState*
HttpStateHeaders::processNewData(SplittableBuffer& input, bool eof,
                                 SplittableBuffer& body_data, bool& body_eof)
{
	SplittableBuffer::ByteIterator begin(input.begin());
	HttpState* next_state = processData(begin, input.end(), body_eof);
	input.trimFront(begin);
	if (eof && next_state == this) {
		return m_rParser.activateStateError("unexpected end of data (in HttpStateHeaders)");
	}
	return next_state;
}

void
HttpStateHeaders::reset()
{
	m_state = ST_CHECKING_FOR_END;
	m_headerName.clear();
	m_headerValueStream.clear();
}

HttpState*
HttpStateHeaders::processData(
	SplittableBuffer::ByteIterator& begin,
	SplittableBuffer::ByteIterator const& end, bool& body_eof)
{
	HttpState* next_state = this;
	
	switch (m_state) {
		case ST_CHECKING_FOR_END:
		case_ST_CHECKING_FOR_END: {
			if (begin == end) {
				break;	
			} else if (*begin == '\r') {
				++begin;
				m_state = ST_CHECKING_FOR_END_2;
				// fall through
			} else if (*begin == '\n') {
				++begin;
				next_state = headersReceived(body_eof);
				break;
			} else {
				m_state = ST_LOOKING_FOR_COLON;
				goto case_ST_LOOKING_FOR_COLON;
			}
		}
		case ST_CHECKING_FOR_END_2: {
			if (begin == end) {
				break;
			} else if (*begin == '\n') {
				++begin;
				next_state = headersReceived(body_eof);
				break;
			}
			// fall through
		}
		case ST_LOOKING_FOR_COLON:
		case_ST_LOOKING_FOR_COLON: {
			SplittableBuffer::ByteIterator cpos(
				SplittableBuffer::find(begin, end, ':')
			);
			SplittableBuffer::ByteIterator nlpos(
				SplittableBuffer::find(begin, cpos, '\n')
			);
			if (nlpos != cpos) { // broken header
#if 0
				next_state = m_rParser.activateStateError("error parsing header");
				break;
#else
				// It looks like every browser silently ignores this problem,
				// and web sites manage to get away with serving broken headers.
				// This leaves us no choise but to ignore them as well.
				begin = nlpos;
				++begin;
				m_state = ST_CHECKING_FOR_END;
				goto case_ST_CHECKING_FOR_END;
#endif
			} else if (cpos == end) {
				// need more data
				break;
			}
			
			m_headerName = trim(begin, cpos);
			begin = cpos;
			++begin;
			m_state = ST_LOOKING_FOR_NEWLINE;
			// fall through
		}
		case ST_LOOKING_FOR_NEWLINE:
		case_ST_LOOKING_FOR_NEWLINE: {
			SplittableBuffer::ByteIterator nlpos(
				SplittableBuffer::find(begin, end, '\n')
			);
			if (nlpos == end) {
				break;
			}
			SplittableBuffer::ByteIterator after_nlpos(nlpos);
			++after_nlpos;
			if (after_nlpos == end) {
				break;
			}
			bool const lws = (*after_nlpos == ' ' || *after_nlpos == '\t');
			
			if (!lws && m_headerValueStream.data().empty()) {
				addHeader(m_headerName, trim(begin, nlpos));
			} else {
				begin = StringUtils::ltrim(begin, nlpos);
				nlpos = StringUtils::rtrim(begin, nlpos);
				m_headerValueStream.data().append(begin, nlpos);
				if (lws) {
					m_headerValueStream << ' ';
				} else {
					SplittableBuffer data;
					m_headerValueStream.swapData(data);
					addHeader(m_headerName, data.toBString());
				}
			}
			begin = after_nlpos;
			if (lws) {
				m_state = ST_LOOKING_FOR_NEWLINE;
				goto case_ST_LOOKING_FOR_NEWLINE;
			} else {
				m_state = ST_CHECKING_FOR_END;
				goto case_ST_CHECKING_FOR_END;
			}
			break;
		}
	}
	
	return next_state;
}

BString
HttpStateHeaders::trim(
	SplittableBuffer::ByteIterator begin,
	SplittableBuffer::ByteIterator end)
{
	begin = StringUtils::ltrim(begin, end);
	end = StringUtils::rtrim(begin, end);
	return SplittableBuffer::toBString(begin, end);
}

bool
HttpStateHeaders::isPersistentConnection(
	HttpHeadersCollection const& headers,
	HttpVersion const& http_version)
{
	BString const connection("Connection");
	HttpHeader const* conn_hdr = headers.getHeaderPtr(connection);
	if (http_version >= HttpVersion::HTTP_1_1) {
		BString const close("close");
		if (conn_hdr) {
			HttpHeaderStructure structure(*conn_hdr);
			if (structure.hasElement(close)) {
				return false;
			}
		}
		return true;
	} else {
		BString const keep_alive("keep-alive");
		if (conn_hdr) {
			HttpHeaderStructure structure(*conn_hdr);
			if (structure.hasElement(keep_alive)) {
				return true;
			}
		}
		BString const proxy_connection("Proxy-Connection");
		HttpHeader const* prconn_hdr = headers.getHeaderPtr(proxy_connection);
		if (prconn_hdr) {
			HttpHeaderStructure structure(*prconn_hdr);
			if (structure.hasElement(keep_alive)) {
				return true;
			}
		}
		return false;
	}
}

bool
HttpStateHeaders::isSingularHeader(BString const& header)
{
	InsensitiveEqual ieq;
	return ieq(header, BString("Content-Length"))
		|| ieq(header, BString("Content-Type"));
	// Here we only check for headers that cause problems to us
	// when enountered in multiple instances.
}
