/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "HttpStateChunkHeader.h"
#include "HttpParserBase.h"
#include "SplittableBuffer.h"
#include <cctype>
#include <string>

using namespace std;

HttpStateChunkHeader::HttpStateChunkHeader(HttpParserBase& parser)
:	m_rParser(parser)
{
}

HttpStateChunkHeader::~HttpStateChunkHeader()
{
}

void
HttpStateChunkHeader::activate()
{
	m_state = ST_LEFT_PADDING;
	m_chunkSize = 0;
	m_chunkSizeDigits = 0;
}

HttpState*
HttpStateChunkHeader::processNewData(SplittableBuffer& input, bool eof,
                                     SplittableBuffer& body_data, bool& body_eof)
{
	SplittableBuffer::ByteIterator begin(input.begin());
	HttpState* next_state = processData(begin, input.end(), body_eof);
	input.trimFront(begin);
	if (eof && next_state == this) {
		return m_rParser.activateStateError("unexpected end of data (in HttpStateChunkHeader)");
	}
	return next_state;
}

HttpState*
HttpStateChunkHeader::processData(
	SplittableBuffer::ByteIterator& begin,
	SplittableBuffer::ByteIterator const& end, bool& body_eof)
{
	HttpState* next_state = this;
	uint8_t ch;
	
	switch (m_state) {
		case ST_LEFT_PADDING: {
			// here we handle the \r\n following the chunk's body
			for (; begin != end; ++begin) {
				ch = *begin;
				if (!isspace(ch)) {
					m_state = ST_CHUNK_SIZE;
					goto case_ST_CHUNK_SIZE;
				}
			}
			break;
		}
		case ST_CHUNK_SIZE:
		case_ST_CHUNK_SIZE: {
			while (begin != end) {
				ch = *begin;
				if (ch >= uint8_t('0') && ch <= uint8_t('9')) {
					m_chunkSize = (m_chunkSize << 4) + (ch - uint8_t('0'));
				} else if (ch >= uint8_t('a') && ch <= uint8_t('f')) {
					m_chunkSize = (m_chunkSize << 4) + 10 + (ch - uint8_t('a'));
				} else if (ch >= uint8_t('A') && ch <= uint8_t('F')) {
					m_chunkSize = (m_chunkSize << 4) + 10 + (ch - uint8_t('A'));
				} else {
					m_state = ST_RIGHT_PADDING;
					goto case_ST_RIGHT_PADDING;
				}
				++begin;
				++m_chunkSizeDigits;
			}
			break;
		}
		case ST_RIGHT_PADDING:
		case_ST_RIGHT_PADDING: {
			for (; begin != end; ++begin) {
				ch = *begin;
				if (ch == '\n') {
					next_state = processChunkSize(body_eof);
					++begin;
					break;
				} else if (ch == ';') {
					++begin;
					m_state = ST_ARGUMENTS;
					goto case_ST_ARGUMENTS;
				} else if (isspace(ch)) {
					continue;
				} else {
					next_state = m_rParser.activateStateError("error parsing chunk header");
					break;
				}
			}
			break;
		}
		case ST_ARGUMENTS:
		case_ST_ARGUMENTS: {
			begin = SplittableBuffer::find(begin, end, '\n');
			if (begin != end) {
				next_state = processChunkSize(body_eof);
				++begin;
			}
			break;
		}
	}
	
	return next_state;
}

HttpState*
HttpStateChunkHeader::processChunkSize(bool& body_eof)
{
	if (m_chunkSizeDigits == 0) {
		return m_rParser.activateStateError("error parsing chunk header");
	} else if (m_chunkSize == 0) {
		body_eof = true;
		return m_rParser.activateStateFooters();
	}
	return m_rParser.activateStateChunkBody(m_chunkSize);
}
