/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "HttpFetcher.h"
#include "NonCopyable.h"
#include "AbstractRequestHandler.h"
#include "AbstractResponseHandler.h"
#include "FilterTryList.h"
#include "ErrorDescriptor.h"
#include "ErrorFactory.h"
#include "ErrorCodes.h"
#include "DownloadSizeLimiter.h"
#include "AbstractServer.h"
#include "HttpRequestMetadata.h"
#include "HttpResponseMetadata.h"
#include "HttpStatusLine.h"
#include "ResponseFilterChain.h"
#include "RequestTag.h"
#include "IntrusivePtr.h"
#include "WeakPtr.h"
#include "State.h"
#include "GlobalState.h"
#include "AvailableFiltersOrdered.h"
#include "URI.h"
#include "Debug.h"
#include <sigc++/sigc++.h>
#include <memory>
#include <stddef.h>

class DownloadProgress;
class RequestStatus;

using namespace std;

class HttpFetcher::ResponseHandler : public AbstractResponseHandler
{
	DECLARE_NON_COPYABLE(ResponseHandler)
public:
	typedef WeakPtr<ACE_NULL_SYNCH, HttpFetcher, HttpFetcher*> OwnerPtr;
	
	ResponseHandler(OwnerPtr const& owner) : m_ptrOwner(owner) {}
private:
	virtual void processProvisionalResponse(
		RequestStatus& status,
		std::auto_ptr<HttpResponseMetadata> metadata);
	
	virtual void processResponseMetadata(
		RequestStatus& status,
		std::auto_ptr<HttpResponseMetadata> metadata);
	
	// postcondition: data is empty
	virtual void processBodyData(
		RequestStatus& status, SplittableBuffer& data, bool eof);
	
	virtual void processError(
		RequestStatus& status, std::auto_ptr<ErrorDescriptor> edesc);
	
	OwnerPtr m_ptrOwner;
};


/*======================= HttpFetcher::ResponseHandler ====================*/

void
HttpFetcher::ResponseHandler::processProvisionalResponse(
	RequestStatus&, auto_ptr<HttpResponseMetadata>)
{
}

void
HttpFetcher::ResponseHandler::processResponseMetadata(
	RequestStatus&, std::auto_ptr<HttpResponseMetadata> metadata)
{
	HttpFetcher* const owner = m_ptrOwner.get();
	if (!owner) {
		return;
	}
	
	if (metadata->getBodyStatus() == HttpResponseMetadata::BODY_SIZE_KNOWN
	    && metadata->getBodySize() > owner->m_maxBodySize) {
		owner->handleError(
			ErrorFactory::errDummyError(
				ErrorCodes::RESPONSE_BODY_TOO_BIG,
				"response body is too big"
			)
		);
	}
	owner->m_ptrResponseMetadata = metadata;
}

void
HttpFetcher::ResponseHandler::processBodyData(
	RequestStatus&, SplittableBuffer& data, bool eof)
{
	HttpFetcher* const owner = m_ptrOwner.get();
	if (owner) {
		if (owner->getStatus() == IN_PROGRESS) {
			owner->handleData(data, eof);
		} else {
			data.clear();
		}
	}
}

void
HttpFetcher::ResponseHandler::processError(
	RequestStatus&, std::auto_ptr<ErrorDescriptor> edesc)
{
	HttpFetcher* const owner = m_ptrOwner.get();
	if (owner) {
		if (owner->getStatus() == IN_PROGRESS) {
			owner->handleError(edesc);
		}
	}
}


/*============================= HttpFetcher ===============================*/

HttpFetcher::HttpFetcher(
	ServiceContext& context, AbstractServer& server,
	ConstRequestPtr const& request, RequestTag const& request_tag,
	size_t max_body_size, size_t max_fetch_size)
:	m_ptrSelf(this),
	m_rContext(context),
	m_rServer(server),
	m_ptrRequest(request),
	m_status(IN_PROGRESS),
	m_maxBodySize(max_body_size),
	m_maxFetchSize(max_fetch_size),
	m_receivedBodySize(0)
{
	auto_ptr<FilterTryList> filters(new FilterTryList);
	if (max_fetch_size != std::numeric_limits<size_t>::max()) {
		filters->append(
			FilterTryList::FilterConstructorPtr(
				new FilterTryList::FilterConstructor(
					sigc::bind(
						sigc::ptr_fun(&HttpFetcher::createSizeLimiterFilter),
						m_maxFetchSize
					)
				)
			)
		);
	}
	
	BString const url_str(request->requestLine().getURI().toBString());
	
	// It's a good idea to allow regex filters to run,
	// as they may be used to de-obfuscate scripts.
	applyRegexFilters(*filters, url_str);
	
	IntrusivePtr<AbstractResponseHandler> const response_handler(
		new ResponseHandler(m_ptrSelf)
	);
	IntrusivePtr<AbstractResponseHandler> const filter_chain(
		new ResponseFilterChain(
			m_rContext, request, request_tag,
			filters, response_handler
		)
	);
	m_ptrRequestHandler = m_rServer.submitRequest(
		request, request_tag, filter_chain
	);
	
	SplittableBuffer body;
	m_ptrRequestHandler->appendBodyData(body, true);
}

HttpFetcher::~HttpFetcher()
{
	m_ptrSelf.reset(0); // prevent possible callbacks through the weak ptr
	m_ptrRequestHandler->cancel(ErrorFactory::errDummyError());
}

void
HttpFetcher::applyRegexFilters(FilterTryList& filters, BString const& url_str)
{
	GlobalState::ReadAccessor state;
	
	AvailableFiltersOrdered const& avail_filters = state->contentFilters();
	AvailableFiltersOrdered::iterator it(avail_filters.begin());
	AvailableFiltersOrdered::iterator const end(avail_filters.end());
	for (; it != end; ++it) {
		filters.tryAppendRegexFilter(url_str, *it);
	}
}

void
HttpFetcher::handleData(SplittableBuffer& data, bool eof)
{
	m_receivedBodySize += data.size();
	m_responseBody.appendDestructive(data);
	if (eof) {
		m_status = COMPLETE;
	} else if (m_receivedBodySize > m_maxBodySize) {
		handleError(
			ErrorFactory::errDummyError(
				ErrorCodes::RESPONSE_BODY_TOO_BIG,
				"response body is too big"
			)
		);
	}
}

void
HttpFetcher::handleError(std::auto_ptr<ErrorDescriptor> edesc)
{
	DEBUGLOG(
		"HttpFetcher::handleError(): "
		<< edesc->getDebugMessage() << "\r\nURL: "
		<< m_ptrRequest->requestLine().getURI()
	);
	m_status = FAILED;
	m_ptrErrDesc = edesc;
}

void
HttpFetcher::createSizeLimiterFilter(
	ResponseFilterChain& chain,
	HttpResponseMetadata const& metadata, size_t limit)
{
	chain.appendFilter(IntrusivePtr<AbstractResponseFilter>(
		new DownloadSizeLimiter(chain, limit)
	));
}
