/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Date.h"
#include "SBOutStream.h"
#include "SplittableBuffer.h"
#include "BString.h"
#include "StringUtils.h"
#include "ArraySize.h"
#include "types.h"
#include <ace/config-lite.h>
#include <ace/OS_NS_time.h>
#include <stdio.h>
#include <string.h>

int const Date::m_daysByMonth[12] = {
	0,   // jan 31
	31,  // feb 28
	59,  // mar 31
	90,  // apr 30
	120, // may 31
	151, // jun 30
	181, // jul 31
	212, // aug 31
	243, // sep 30
	273, // oct 31
	304, // nov 30
	334  // dec 31
};

char const Date::m_wkdays[][4] = {
	"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"
};

char const Date::m_weekdays[][10] = {
	"Sunday", "Monday", "Tuesday", "Wednesday",
        "Thursday", "Friday", "Saturday"
};

char const Date::m_months[][4] = {
	"Jan", "Feb", "Mar", "Apr", "May", "Jun",
	"Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};

BString
Date::format(time_t time, Style style)
{
	if (time > INT64_C(0x793406fff)) {
		// Workaround a bug in _gmtime64_s on Windows, which leads to a crash.
		return BString();
	}
	
	struct tm tmval = {};
	if (!ACE_OS::gmtime_r(&time, &tmval)) {
		return BString();
	}
	
	SBOutStream strm(33); // chunk size
	if (style == RFC822) {
		// Sun, 06 Nov 1994 08:49:37 GMT
		strm.fill('0');
		strm << m_wkdays[tmval.tm_wday] << ", ";
		strm.width(2);
		strm << tmval.tm_mday << ' ';
		strm << m_months[tmval.tm_mon] << ' ';
		strm << (1900 + tmval.tm_year) << ' ';
		strm.width(2);
		strm << tmval.tm_hour << ':';
		strm.width(2);
		strm << tmval.tm_min << ':';
		strm.width(2);
		strm << tmval.tm_sec << " GMT";
	} else if (style == RFC850) {
		// Sunday, 06-Nov-94 08:49:37 GMT
		strm.fill('0');
		strm << m_weekdays[tmval.tm_wday] << ", ";
		strm.width(2);
		strm << tmval.tm_mday << '-';
		strm << m_months[tmval.tm_mon] << '-';
		strm.width(2);
		strm << tmval.tm_year << ' ';
		strm.width(2);
		strm << tmval.tm_hour << ':';
		strm.width(2);
		strm << tmval.tm_min << ':';
		strm.width(2);
		strm << tmval.tm_sec << " GMT";
	} else if (style == ASCTIME) {
		// Sun Nov  6 08:49:37 1994
		strm << m_wkdays[tmval.tm_wday] << ' ';
		strm << m_months[tmval.tm_mon] << ' ';
		strm.fill(' ');
		strm.width(2);
		strm << tmval.tm_mday << ' ';
		strm.fill('0');
		strm.width(2);
		strm << tmval.tm_hour << ':';
		strm.width(2);
		strm << tmval.tm_min << ':';
		strm.width(2);
		strm << tmval.tm_sec << ' ';
		strm << (tmval.tm_year + 1900);
	}
	return strm.data().toBString();
}

BString
Date::formatCurrentTime(Style style)
{
	return format(time(NULL), style);
}

time_t
Date::parse(BString const& str)
{
	/*
	Supported formats are:
	Sun, 06 Nov 1994 08:49:37 GMT  // RFC 822, updated by RFC 1123
	Sunday, 06-Nov-94 08:49:37 GMT // RFC 850, obsoleted by RFC 1036
	Sun Nov  6 08:49:37 1994       // ANSI C's asctime() format
	*/
	
	time_t const err = static_cast<time_t>(-1);
	
	char const* const begin = str.begin();
	char const* const end = str.end();
	char const* p = begin;
	
	for (;; ++p) {
		if (p == end) {
			return err;
		}
		if (*p == ' ' || *p == ',') {
			break;
		}
	}
	
	if (*p == ' ') {
		if (p - begin == 3) {
			return parseAsctime(p + 1, end);
		} else {
			return err;
		}
	} else if (*p == ',') {
		if (p - begin == 3) {
			return parseRFC822(p + 1, end);
		} else {
			return parseRFC850(p + 1, end);
		}
	}
	return err;
}

time_t
Date::parseAsctime(char const* const begin, char const* const end)
{
	time_t const err = static_cast<time_t>(-1);
	
	char buf[sizeof("Nov  6 08:49:37 1994")];
	if (end - begin >= (int)sizeof(buf)) {
		return err;
	}
	memcpy(buf, begin, end - begin);
	buf[end - begin] = '\0';
	
	char month_str[4] = {};
	unsigned day = 0;
	unsigned hour = 0;
	unsigned min = 0;
	unsigned sec = 0;
	unsigned year = 0;
	
	int const num_scanned = sscanf(
		buf, "%3s %u %u:%u:%u %u",
		month_str, &day, &hour, &min, &sec, &year
	);
	if (num_scanned != 6) {
		return err;
	}
	
	int const month = matchMonth(month_str);
	if (month < 0) {
		return err;
	}
	
	return constructGmtTime(year, month + 1, day, hour, min, sec);
}

time_t
Date::parseRFC822(char const* const begin, char const* const end)
{
	time_t const err = static_cast<time_t>(-1);
	
	char buf[sizeof(" 06 Nov 1994 08:49:37 GMT")];
	if (end - begin >= (int)sizeof(buf)) {
		return err;
	}
	memcpy(buf, begin, end - begin);
	buf[end - begin] = '\0';
	
	char month_str[4] = {};
	unsigned day = 0;
	unsigned hour = 0;
	unsigned min = 0;
	unsigned sec = 0;
	unsigned year = 0;
	
	int const num_scanned = sscanf(
		buf, " %u %3s %u %u:%u:%u GMT",
		&day, month_str, &year, &hour, &min, &sec
	);
	if (num_scanned != 6) {
		return err;
	}
	
	int const month = matchMonth(month_str);
	if (month < 0) {
		return err;
	}
	
	return constructGmtTime(year, month + 1, day, hour, min, sec);
}

time_t
Date::parseRFC850(char const* const begin, char const* const end)
{
	time_t const err = static_cast<time_t>(-1);
	
	char buf[sizeof(" 06-Nov-94 08:49:37 GMT")];
	if (end - begin >= (int)sizeof(buf)) {
		return err;
	}
	memcpy(buf, begin, end - begin);
	buf[end - begin] = '\0';
	
	char month_str[4] = {};
	unsigned day = 0;
	unsigned hour = 0;
	unsigned min = 0;
	unsigned sec = 0;
	unsigned year = 0;
	
	int const num_scanned = sscanf(
		buf, " %u-%3s-%u %u:%u:%u GMT",
		&day, month_str, &year, &hour, &min, &sec
	);
	if (num_scanned != 6) {
		return err;
	}
	
	year += 1900;
	
	int const month = matchMonth(month_str);
	if (month < 0) {
		return err;
	}
	
	return constructGmtTime(year, month + 1, day, hour, min, sec);
}

int
Date::matchMonth(char const* month)
{
	for (unsigned i = 0; i < ARRAY_SIZE(m_months); ++i) {
		if (strcmp(month, m_months[i]) == 0) {
			return i;
		}
	}
	return -1;
}

inline bool
Date::isLeapYear(unsigned const year)
{
	return (year % 4 == 0 && year % 100 != 0) || year % 400 == 0;
}

inline int
Date::leapsThroughEndOf(unsigned const year)
{
	return year / 4 - year / 100 + year / 400;
}

time_t
Date::constructGmtTime(
	unsigned const year, unsigned const month, unsigned const day,
	unsigned const hour, unsigned const min, unsigned const sec)
{
	time_t const err = static_cast<time_t>(-1);
	
	if (month < 1 || month > 12 || day < 1 || day > 31
	    || hour > 23 || min > 59 || sec > 59) {
		return err;
	}
	
	time_t accum = year - 1970;
	accum *= 365;
	
	// add in leap day for all previous years
	accum += leapsThroughEndOf(year > 0 ? year - 1 : year);
	accum -= leapsThroughEndOf(1970);
	
	// add in leap day for this year
	if (month >= 3) { // march or later
		if (isLeapYear(year)) {
			accum += 1;
		}
	}
	
	accum += m_daysByMonth[month - 1];
	accum += day - 1;
	accum *= 24;
	accum += hour;
	accum *= 60;
	accum += min;
	accum *= 60;
	accum += sec;
	
	return accum;
}
