/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2004  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef SPLITTABLEBUFFER_H_
#define SPLITTABLEBUFFER_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "IntrusivePtr.h"
#include "BString.h"
#include <stddef.h>
#include <memory>
#include <iterator>
#include <string>

class DataChunk;

class SplittableBuffer
{
private:
	class Range;
public:
	class ByteIterator :
		public std::iterator<
			std::bidirectional_iterator_tag,
			char, size_t, char const*, char const&
		>
	{
		friend class SplittableBuffer;
	public:
		ByteIterator() : m_pRange(0), m_pSpan(0), m_pData(0) {}
		
		// Default copy constructor OK
		
		// Default assignment operator OK
	
		reference operator*() const { return *m_pData; }
		
		pointer operator->() const { return m_pData; }
		
		pointer spanBegin() const { return m_pSpan->begin(); }
		
		pointer spanEnd() const { return m_pSpan->end(); }
		
		void nextSpan();
		
		void prevSpan();
		
		ByteIterator& operator++();
		
		ByteIterator& operator--();
		
		ByteIterator operator++(int);
		
		ByteIterator operator--(int);
		
		ByteIterator& operator+=(difference_type amount);
		
		ByteIterator& operator-=(difference_type amount);
		
		ByteIterator operator+(difference_type rhs) const;
		
		ByteIterator operator-(difference_type rhs) const;
		
		difference_type operator-(ByteIterator const& rhs) const;
		
		bool operator==(ByteIterator const& rhs) const;
		
		bool operator!=(ByteIterator const& rhs) const { return !(*this == rhs); }
		
		bool operator>(ByteIterator const& rhs) const { return rhs.less(*this, false); }
		
		bool operator>=(ByteIterator const& rhs) const { return rhs.less(*this, true); }
		
		bool operator<(ByteIterator const& rhs) const { return less(rhs, false); }
		
		bool operator<=(ByteIterator const& rhs) const { return less(rhs, true); }
		
		bool isAtLeftBorder() const { return m_pRange->prev == m_pRange; }
		
		bool isAtRightBorder() const { return m_pRange->next == m_pRange; }
		
		bool isSameSpan(ByteIterator const& other) const { return m_pSpan == other.m_pSpan; }
	private:
		ByteIterator(Range const* range, BString const* span, char const* data)
		:   m_pRange(range), m_pSpan(span), m_pData(data) {}
		
		bool less(ByteIterator const& than, bool or_equal) const;
		
		Range const* m_pRange;
		BString const* m_pSpan;
		char const* m_pData;
	};
	
	class SpanIterator
	{
		friend class SplittableBuffer;
	public:
		// Default copy constructor OK
		
		// Default assignment operator OK
		
		BString const& operator*() { return *m_pSpan; }
		
		BString const* operator->() { return m_pSpan; }
		
		SpanIterator& operator++();
		
		SpanIterator& operator--();
		
		bool operator==(SpanIterator const& rhs) const { return m_pSpan == rhs.m_pSpan; }
		// since spans are located inside ranges, we don't need to compare ranges as well
		bool operator!=(SpanIterator const& rhs) const { return m_pSpan != rhs.m_pSpan; }
		
		bool isAtLeftBorder() const { return m_pRange->prev == m_pRange; }
		
		bool isAtRightBorder() const { return m_pRange->next == m_pRange; }
	private:
		SpanIterator(Range const* range, BString const* span) : m_pRange(range), m_pSpan(span) {}
		
		Range const* m_pRange;
		BString const* m_pSpan;
	};
public:
	SplittableBuffer();
	
	SplittableBuffer(SplittableBuffer const& other);
	
	SplittableBuffer(ByteIterator const& begin, ByteIterator const& end);
	
	~SplittableBuffer();
	
	SplittableBuffer& operator=(SplittableBuffer const& rhs);
	
	bool empty() const { return m_rightBorder.prev == &m_leftBorder; }
	
	size_t size() const;
	
	void clear();
	
	void append(SplittableBuffer const& buf);
	
	void append(std::auto_ptr<DataChunk> chunk, size_t trim_front = 0);
	
	void append(IntrusivePtr<DataChunk const> const& chunk,
		char const* begin, char const* end);
	
	void append(ByteIterator const& begin, ByteIterator const& end);
	
	void append(BString const& str);
	
	void prepend(SplittableBuffer const& buf);
	
	void prepend(std::auto_ptr<DataChunk> chunk, size_t trim_front = 0);
	
	void prepend(IntrusivePtr<DataChunk const> const& chunk,
		char const* begin, char const* end);
	
	void prepend(ByteIterator const& begin, ByteIterator const& end);
	
	void prepend(BString const& str);
	
	void appendDestructive(SplittableBuffer& buf); // buf becomes empty
	
	void prependDestructive(SplittableBuffer& buf); // buf becomes empty
	
	void trimFront(size_t size);
	
	void trimFront(ByteIterator const& pos); // same as trimFront(pos - begin())
	
	void trimBack(size_t size);
	
	void trimBack(ByteIterator const& pos); // same as trimBack(end() - pos); pos is invalidated
	
	void splitFrontAppendBack(size_t size, SplittableBuffer& append_target);
	
	void splitFrontAppendBack(ByteIterator const& pos, SplittableBuffer& append_target);
	
	void splitBackAppendFront(size_t size, SplittableBuffer& append_target);
	
	void splitBackAppendFront(ByteIterator const& pos, SplittableBuffer& append_target); // pos is invalidated
	
	BString* firstSpan();
	
	BString const* firstSpan() const;
	
	BString* lastSpan();
	
	BString const* lastSpan() const;
	
	ByteIterator begin() const;
	
	ByteIterator end() const;
	
	ByteIterator rbegin() const;
	
	ByteIterator rend() const;
	
	SpanIterator spanBegin() const;
	
	SpanIterator spanEnd() const;
	
	SpanIterator spanRBegin() const;
	
	SpanIterator spanREnd() const;
	
	static ByteIterator find(
		ByteIterator const& begin,
		ByteIterator const& end, char ch);
	
	static ByteIterator find(
		ByteIterator const& begin,
		ByteIterator const& end,
		char const* target_begin,
		char const* target_end);
	
	static ByteIterator find(
		ByteIterator const& begin,
		ByteIterator const& end, BString const& target);
	
	static ByteIterator find(
		ByteIterator const& begin,
		ByteIterator const& end, std::string const& target);
	
	static ByteIterator ciFind(
		ByteIterator const& begin,
		ByteIterator const& end, char ch);
	
	static ByteIterator ciFind(
		ByteIterator const& begin,
		ByteIterator const& end,
		char const* target_begin,
		char const* target_end);
	
	static ByteIterator ciFind(
		ByteIterator const& begin,
		ByteIterator const& end, BString const& target);
	
	static ByteIterator ciFind(
		ByteIterator const& begin,
		ByteIterator const& end, std::string const& target);
	
	ByteIterator find(char ch) const;
	
	ByteIterator find(char const* begin, char const* end) const;
	
	ByteIterator find(BString const& target) const;
	
	ByteIterator find(std::string const& target) const;
	
	ByteIterator ciFind(char ch) const;
	
	ByteIterator ciFind(char const* begin, char const* end) const;
	
	ByteIterator ciFind(BString const& target) const;
	
	ByteIterator ciFind(std::string const& target) const;
	
	static std::string toString(ByteIterator const& begin, ByteIterator const& end);
	
	std::string toString() const { return toString(begin(), end()); }
	
	static BString toBString(ByteIterator const& begin, ByteIterator const& end);
	
	BString toBString() const { return toBString(begin(), end()); }
private:
	enum { DEFAULT_RANGE_CAPACITY = 8 };
	
	class Range
	{
	public:
		enum GrowDirection { APPEND, PREPEND };
	protected:
		Range(GrowDirection direction, size_t capacity);
	public:
		~Range();
		
		static Range* create(GrowDirection direction, size_t capacity = DEFAULT_RANGE_CAPACITY) {
			return new(capacity * sizeof(BString)) Range(direction, capacity);
		}
		
		BString* spanStorageBegin() {
			return reinterpret_cast<BString*>(reinterpret_cast<char*>(this) + sizeof(*this));
		}
		
		BString const* spanStorageBegin() const {
			return reinterpret_cast<BString const*>(reinterpret_cast<char const*>(this) + sizeof(*this));
		}
		
		BString* spanStorageEnd() { return spanStorageEnd_; }
		
		BString const* spanStorageEnd() const { return spanStorageEnd_; }
		
		static void* operator new(size_t size, size_t extra_space);
		
		static void operator delete(void* addr);
		
		static void operator delete(void* addr, size_t) {
			operator delete(addr);
		}
		
		Range* prev;
		Range* next;
		BString* const spanStorageEnd_;
		BString* spanBegin;
		BString* spanEnd;
		// extra space for Span objects is allocated here
	};
	
	class Border : public Range
	{
	public:
		Border();
	private:
		BString m_borderSpan;
	};
	
	static void moveRanges(Range* begin, Range* end, Range* before_this);
	
	void setupBorders();
	
	Border m_leftBorder;
	Border m_rightBorder;
};

inline void
SplittableBuffer::append(BString const& str)
{
	append(str.chunk(), str.begin(), str.end());
}

inline void
SplittableBuffer::prepend(BString const& str)
{
	prepend(str.chunk(), str.begin(), str.end());
}

inline BString*
SplittableBuffer::firstSpan()
{
	Range* range = m_leftBorder.next;
	return (range == &m_rightBorder) ? 0 : range->spanBegin;
}

inline BString const*
SplittableBuffer::firstSpan() const
{
	Range const* range = m_leftBorder.next;
	return (range == &m_rightBorder) ? 0 : range->spanBegin;
}

inline BString*
SplittableBuffer::lastSpan()
{
	Range* range = m_rightBorder.prev;
	return (range == &m_leftBorder) ? 0 : range->spanEnd - 1;
}

inline BString const*
SplittableBuffer::lastSpan() const
{
	Range const* range = m_rightBorder.prev;
	return (range == &m_leftBorder) ? 0 : range->spanEnd - 1;
}

inline SplittableBuffer::ByteIterator
SplittableBuffer::begin() const
{
	BString const* span = m_leftBorder.next->spanBegin;
	return ByteIterator(m_leftBorder.next, span, span->begin());
}

inline SplittableBuffer::ByteIterator
SplittableBuffer::end() const
{
	return ByteIterator(&m_rightBorder, m_rightBorder.spanStorageBegin(), 0);
}

inline SplittableBuffer::ByteIterator
SplittableBuffer::rbegin() const
{
	BString const* span = m_rightBorder.prev->spanEnd - 1;
	return ByteIterator(m_rightBorder.prev, span, span->end() - 1);
}

inline SplittableBuffer::ByteIterator
SplittableBuffer::rend() const
{
	return ByteIterator(&m_leftBorder, m_leftBorder.spanStorageBegin(), 0);
}

inline SplittableBuffer::SpanIterator
SplittableBuffer::spanBegin() const
{
	return SpanIterator(m_leftBorder.next, m_leftBorder.next->spanBegin);
}

inline SplittableBuffer::SpanIterator
SplittableBuffer::spanEnd() const
{
	return SpanIterator(&m_rightBorder, m_rightBorder.spanStorageBegin());
}

inline SplittableBuffer::SpanIterator
SplittableBuffer::spanRBegin() const
{
	return SpanIterator(m_rightBorder.prev, m_rightBorder.prev->spanEnd - 1);
}

inline SplittableBuffer::SpanIterator
SplittableBuffer::spanREnd() const
{
	return SpanIterator(&m_leftBorder, m_leftBorder.spanStorageBegin());
}

inline
SplittableBuffer::ByteIterator
SplittableBuffer::find(
	ByteIterator const& begin, ByteIterator const& end, BString const& target)
{
	return find(begin, end, target.begin(), target.end());
}

inline
SplittableBuffer::ByteIterator
SplittableBuffer::find(
	ByteIterator const& begin, ByteIterator const& end, std::string const& target)
{
	return find(begin, end, target.c_str(), target.c_str()+target.size());
}

inline
SplittableBuffer::ByteIterator
SplittableBuffer::find(char ch) const
{
	return find(begin(), end(), ch);
}

inline
SplittableBuffer::ByteIterator
SplittableBuffer::find(char const* target_begin, char const* target_end) const
{
	return find(begin(), end(), target_begin, target_end);
}

inline
SplittableBuffer::ByteIterator
SplittableBuffer::find(BString const& target) const
{
	return find(begin(), end(), target);
}

inline
SplittableBuffer::ByteIterator
SplittableBuffer::find(std::string const& target) const
{
	return find(begin(), end(), target);
}

inline
SplittableBuffer::ByteIterator
SplittableBuffer::ciFind(
	ByteIterator const& begin, ByteIterator const& end, BString const& target)
{
	return ciFind(begin, end, target.begin(), target.end());
}

inline
SplittableBuffer::ByteIterator
SplittableBuffer::ciFind(
	ByteIterator const& begin, ByteIterator const& end, std::string const& target)
{
	return ciFind(
		begin, end,
		(char const*)target.c_str(),
		(char const*)target.c_str()+target.size()
	);
}

inline
SplittableBuffer::ByteIterator
SplittableBuffer::ciFind(char ch) const
{
	return ciFind(begin(), end(), ch);
}

inline
SplittableBuffer::ByteIterator
SplittableBuffer::ciFind(char const* target_begin, char const* target_end) const
{
	return ciFind(begin(), end(), target_begin, target_end);
}

inline
SplittableBuffer::ByteIterator
SplittableBuffer::ciFind(BString const& target) const
{
	return ciFind(begin(), end(), target);
}

inline
SplittableBuffer::ByteIterator
SplittableBuffer::ciFind(std::string const& target) const
{
	return ciFind(begin(), end(), target);
}

inline SplittableBuffer::ByteIterator
SplittableBuffer::ByteIterator::operator++(int)
{
	ByteIterator temp(*this);
	++*this;
	return temp;
}

inline SplittableBuffer::ByteIterator
SplittableBuffer::ByteIterator::operator--(int)
{
	ByteIterator temp(*this);
	--*this;
	return temp;
}

inline SplittableBuffer::ByteIterator
SplittableBuffer::ByteIterator::operator+(difference_type rhs) const
{
	ByteIterator temp(*this);
	temp += rhs;
	return temp;
}

inline SplittableBuffer::ByteIterator
SplittableBuffer::ByteIterator::operator-(difference_type rhs) const
{
	ByteIterator temp(*this);
	temp -= rhs;
	return temp;
}

inline bool
SplittableBuffer::ByteIterator::operator==(ByteIterator const& rhs) const
{
	return (m_pData == rhs.m_pData && m_pSpan == rhs.m_pSpan);
	// testing equality of ranges isn't necessary,
	// as spans are stored inside Range objects
}

inline
SplittableBuffer::Border::Border()
:	Range(Range::APPEND, 1),
	m_borderSpan(BString::ChunkPtr(), 0, (char const*)1)
{
	++spanEnd;
}

#endif
