/*
    AWFFull - A Webalizer Fork, Full o' features
    
    $Id: hashtab.c 440 2007-08-05 12:20:41Z steve $

    Copyright (C) 1997-2001  Bradford L. Barrett (brad@mrunix.net)
    Copyright (C) 2006 by Alexander Lazic (al-awffull@none.at)
    Copyright (C) 2006, 2007 by Stephen McInerney (spm@stedee.id.au)
    Copyright (C) 2006 by John Heaton (john@manchester.ac.uk)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

    This software uses the gd graphics library, which is copyright by
    Quest Protein Database Center, Cold Spring Harbor Labs.  Please
    see the documentation supplied with the library for additional
    information and license terms, or visit www.boutell.com/gd/ for the
    most recent version of the library and supporting documentation.
*/

/*********************************************/
/* STANDARD INCLUDES                         */
/*********************************************/

#include "awffull.h"                            /* main header              */

/* internal function prototypes */

HNODEPTR new_hnode(char *);                     /* new host node            */
UNODEPTR new_unode(char *);                     /* new url node             */
RNODEPTR new_rnode(char *);                     /* new referrer node        */
ANODEPTR new_anode(char *);                     /* new user agent node      */
SNODEPTR new_snode(char *);                     /* new search string..      */
INODEPTR new_inode(char *);                     /* new ident node           */
ENODEPTR new_enode(char *, char *);             /* new errorpage node       */

void update_entry(char *);                      /* update entry/exit        */
void update_exit(char *, bool);                 /* page totals              */

unsigned long hash(char *);                     /* hash function            */

/* local data */

HNODEPTR sm_htab[MAXHASH];                      /* hash tables              */
HNODEPTR sd_htab[MAXHASH];
UNODEPTR um_htab[MAXHASH];                      /* for hits, sites,         */
RNODEPTR rm_htab[MAXHASH];                      /* referrers and agents...  */
ANODEPTR am_htab[MAXHASH];
SNODEPTR sr_htab[MAXHASH];                      /* search string table      */
INODEPTR im_htab[MAXHASH];                      /* ident table (username)   */
ENODEPTR ep_htab[MAXHASH];                      /* errorpage table          */


/*********************************************/
/* DEL_HTABS - clear out our hash tables     */
/*********************************************/

void
del_htabs()
{
    del_hlist(sd_htab);                         /* Clear out our various    */
    del_ulist(um_htab);                         /* hash tables here by      */
    del_hlist(sm_htab);                         /* calling the appropriate  */
    del_rlist(rm_htab);                         /* del_* fuction for each   */
    del_alist(am_htab);
    del_slist(sr_htab);
    del_ilist(im_htab);
    del_elist(ep_htab);
}

/*********************************************/
/* NEW_HNODE - create host node              */
/*********************************************/

HNODEPTR
new_hnode(char *str)
{
    HNODEPTR newptr;
    size_t len = strlen(str) + 1;

    if (len > MAXHOST) {
        ERRVPRINT(VERBOSE1, "[new_hnode] %s (%d)\n", _("Warning: String exceeds storage size"), (int) len);
        ERRVPRINT(VERBOSE3, "  --> %s\n", str);
        str[MAXHOST] = 0;
        len = MAXHOST;
    }

    newptr = XMALLOC(struct hnode, 1);
    newptr->string = XMALLOC(char, len);

    strlcpy(newptr->string, str, len);
    newptr->lasturl = blank_str;
    return newptr;
}

/*********************************************
 * PUT_HNODE - insert/update host node       
 * 
 * if visit > 0; then this is a "load" event from preserve.c
 * OR is a CreateGroup event from output.c
 *********************************************/

int
put_hnode(char *str,                            /* Hostname  */
          int type,                             /* obj type  */
          unsigned long count,                  /* hit count */
          unsigned long file,                   /* File flag */
          unsigned long long xfer,              /* xfer size */
          unsigned long *ctr,                   /* counter   */
          unsigned long visit,                  /* visits    */
          unsigned long page,                   /* pages     */
          unsigned long tstamp,                 /* timestamp */
          char *url,                            /* Current URL */
          char *lasturl,                        /* lasturl */
          HNODEPTR * htab,                      /* Host Hash Table */
          bool lasturl_is_entry,                /* true if the last URL is an entry page. Only used for creating a new node from preserve/restore */
          bool isapage,                         /* Is a Page Y/N */
          bool * isnewsite)
{
    HNODEPTR cptr, nptr, prev_cptr;

    cptr = htab[hash(str)];
    prev_cptr = NULL;
    while (cptr != NULL) {
        if (strncmp(cptr->string, str, MAXHOST) == 0) {
            if ((type == cptr->flag) || ((type != OBJ_GRP) && (cptr->flag != OBJ_GRP))) {
                /* found... bump counters */
                cptr->count += count;
                cptr->files += file;
                cptr->xfer += xfer;
                cptr->pages += page;
                if (type == OBJ_GRP) {
                    /* Updating a group */
                    cptr->visit += visit;
                } else if (isapage) {
                    if ((tstamp - cptr->tstamp) >= g_settings.settings.visit_timeout) {
                        /* Is a new Visit */
                        cptr->visit++;
                        /* ONLY! Update entry/exit if working with Monthly table */
                        if (htab == sm_htab) {
                            update_exit(cptr->lasturl, cptr->lasturl_is_entry);
                            update_entry(url);
                            cptr->lasturl_is_entry = true;
                            cptr->lasturl = find_url(url);
                        }
                    } else {                    /* Still in same visit */
                        if (htab == sm_htab) {
                            cptr->lasturl_is_entry = false;
                            cptr->lasturl = find_url(url);
                        }
                    }
                    /* Either way, update the last timestamp for this visit - we have a page! */
                    cptr->tstamp = tstamp;
                }
                return 0;
            }
        }
        prev_cptr = cptr;
        cptr = cptr->next;
    }
    /* Failed to Hash, or doesn't already exist */
    if ((nptr = new_hnode(str)) != NULL) {
        nptr->flag = type;
        nptr->count = count;
        nptr->files = file;
        nptr->xfer = xfer;
        nptr->pages = page;
        nptr->next = NULL;
        nptr->lasturl_is_entry = lasturl_is_entry;
        if (prev_cptr != NULL) {
            prev_cptr->next = nptr;
        } else {
            htab[hash(str)] = nptr;
        }
        if (type == OBJ_GRP) {
            nptr->flag = OBJ_GRP;
        } else {
            (*ctr)++;
            if (!*isnewsite) {
                *isnewsite = true;
            }
        }
        if (visit) {
            nptr->visit = visit;
            if (type != OBJ_GRP) {
                nptr->lasturl = find_url(lasturl);
                nptr->tstamp = tstamp;
            }
            return 0;
        } else {
            if (isapage) {
                /* ONLY! Update entry/exit if working with Monthly table */
                if (htab == sm_htab) {
                    update_entry(url);
                    nptr->lasturl = find_url(url);
                    nptr->lasturl_is_entry = true;
                }
                nptr->tstamp = tstamp;
                nptr->visit = 1;
            }
        }
    }
    return nptr == NULL;
}

/*********************************************/
/* DEL_HLIST - delete host hash table        */
/*********************************************/

void
del_hlist(HNODEPTR * htab)
{
    /* free memory used by hash table */
    HNODEPTR aptr, temp;
    int i;

    for (i = 0; i < MAXHASH; i++) {
        if (htab[i] != NULL) {
            aptr = htab[i];
            while (aptr != NULL) {
                temp = aptr->next;
                XFREE(aptr->string);            /* free hostname string space */
                XFREE(aptr);                    /* free hostname structure    */
                aptr = temp;
            }
            htab[i] = NULL;
        }
    }
}

/*********************************************/
/* NEW_UNODE - URL node creation             */
/*********************************************/

UNODEPTR
new_unode(char *str)
{
    UNODEPTR newptr;
    size_t len = strlen(str) + 1;

    if (len > MAXURL) {
        ERRVPRINT(VERBOSE1, "[new_unode] %s (%d)\n", _("Warning: String exceeds storage size"), (int) len);
        ERRVPRINT(VERBOSE3, "  --> %s\n", str);
        str[MAXURL] = 0;
        len = MAXURL;
    }

    newptr = XMALLOC(struct unode, 1);
    newptr->string = XMALLOC(char, len);

    strlcpy(newptr->string, str, len);
    newptr->count = 0;
    newptr->flag = OBJ_REG;
    return newptr;
}

/*********************************************/
/* PUT_UNODE - insert/update URL node        */
/*********************************************/

int
put_unode(char *str,                            /* Hostname  */
          int type,                             /* obj type  */
          unsigned long count,                  /* hit count */
          unsigned long long xfer,              /* xfer size */
          unsigned long *ctr,                   /* counter   */
          unsigned long entry_page_ctr,         /* Entry Page Counter */
          unsigned long exit_page_ctr,          /* Exit Page Counter */
          unsigned long single_access,          /* Single Access Counter */
          int resp_code, UNODEPTR * htab)
{
    UNODEPTR cptr, nptr;

    if (str[0] == '-')
        return 0;

    /* check if hashed */
    if ((cptr = htab[hash(str)]) == NULL) {
        /* not hashed */
        if ((nptr = new_unode(str)) != NULL) {
            nptr->flag = type;
            if (resp_code == RC_PARTIALCONTENT) {
                nptr->count = 0;
                nptr->xfer = 0;
                nptr->pcount = count;
                nptr->pxfer = xfer;
                g_counters.month.partial_hit += count;
                g_counters.month.partial_vol += xfer;
            } else {
                nptr->count = count;
                nptr->xfer = xfer;
                nptr->pcount = 0;
                nptr->pxfer = 0;
            }
            nptr->next = NULL;
            nptr->entry = entry_page_ctr;
            nptr->exit = exit_page_ctr;
            nptr->single_access = single_access;
            htab[hash(str)] = nptr;
            if (type != OBJ_GRP)
                (*ctr)++;
        }
    } else {
        /* hashed */
        while (cptr != NULL) {
            if (strcmp(cptr->string, str) == 0) {
                if ((type == cptr->flag) || ((type != OBJ_GRP) && (cptr->flag != OBJ_GRP))) {
                    /* found... bump counter */
                    if (resp_code == RC_PARTIALCONTENT) {
                        cptr->pcount += count;
                        cptr->pxfer += xfer;
                        g_counters.month.partial_hit += count;
                        g_counters.month.partial_vol += xfer;
                    } else {
                        cptr->count += count;
                        cptr->xfer += xfer;
                    }
                    return 0;
                }
            }
            cptr = cptr->next;
        }
        /* not found... */
        if ((nptr = new_unode(str)) != NULL) {
            nptr->flag = type;
            if (resp_code == RC_PARTIALCONTENT) {
                nptr->count = 0;
                nptr->xfer = 0;
                nptr->pcount = count;
                nptr->pxfer = xfer;
                g_counters.month.partial_hit += count;
                g_counters.month.partial_vol += xfer;
            } else {
                nptr->count = count;
                nptr->xfer = xfer;
                nptr->pcount = 0;
                nptr->pxfer = 0;
            }
            nptr->next = htab[hash(str)];
            nptr->entry = entry_page_ctr;
            nptr->exit = exit_page_ctr;
            nptr->single_access = single_access;
            htab[hash(str)] = nptr;
            if (type != OBJ_GRP)
                (*ctr)++;
        }
    }
    if (nptr != NULL) {
        if (type == OBJ_GRP)
            nptr->flag = OBJ_GRP;
        else if (isinlist(hidden_urls, nptr->string) != NULL)
            nptr->flag = OBJ_HIDE;
    }
    return nptr == NULL;
}

/*********************************************/
/* DEL_ULIST - delete URL hash table         */
/*********************************************/

void
del_ulist(UNODEPTR * htab)
{
    /* free memory used by hash table */
    UNODEPTR aptr, temp;
    int i;

    for (i = 0; i < MAXHASH; i++) {
        if (htab[i] != NULL) {
            aptr = htab[i];
            while (aptr != NULL) {
                temp = aptr->next;
                XFREE(aptr->string);            /* free up URL string memory */
                XFREE(aptr);                    /* free up URL struct node   */
                aptr = temp;
            }
            htab[i] = NULL;
        }
    }
}

/*********************************************/
/* NEW_RNODE - Referrer node creation        */
/*********************************************/

RNODEPTR
new_rnode(char *str)
{
    RNODEPTR newptr;
    size_t len = strlen(str) + 1;

    if (len >= MAXREF) {
        ERRVPRINT(VERBOSE1, "[new_rnode] %s (%d)\n", _("Warning: String exceeds storage size"), (int) len);
        ERRVPRINT(VERBOSE3, "  --> %s\n", str);
        str[MAXREF] = 0;
        len = MAXREF;
    }

    newptr = XMALLOC(struct rnode, 1);
    newptr->string = XMALLOC(char, len);

    strlcpy(newptr->string, str, len);
    newptr->count = 1;
    newptr->flag = OBJ_REG;
    return newptr;
}

/*********************************************/
/* PUT_RNODE - insert/update referrer node   */
/*********************************************/

int
put_rnode(char *str, int type, unsigned long count, unsigned long *ctr, RNODEPTR * htab)
{
    RNODEPTR cptr, nptr;

    if (str[0] == '-')
        strcpy(str, "- (Direct Request)");

    /* check if hashed */
    if ((cptr = htab[hash(str)]) == NULL) {
        /* not hashed */
        if ((nptr = new_rnode(str)) != NULL) {
            nptr->flag = type;
            nptr->count = count;
            nptr->next = NULL;
            htab[hash(str)] = nptr;
            if (type != OBJ_GRP)
                (*ctr)++;
        }
    } else {
        /* hashed */
        while (cptr != NULL) {
            if (strcmp(cptr->string, str) == 0) {
                if ((type == cptr->flag) || ((type != OBJ_GRP) && (cptr->flag != OBJ_GRP))) {
                    /* found... bump counter */
                    cptr->count += count;
                    return 0;
                }
            }
            cptr = cptr->next;
        }
        /* not found... */
        if ((nptr = new_rnode(str)) != NULL) {
            nptr->flag = type;
            nptr->count = count;
            nptr->next = htab[hash(str)];
            htab[hash(str)] = nptr;
            if (type != OBJ_GRP)
                (*ctr)++;
        }
    }
    if (nptr != NULL) {
        if (type == OBJ_GRP)
            nptr->flag = OBJ_GRP;
        else if (isinlist(hidden_refs, nptr->string) != NULL)
            nptr->flag = OBJ_HIDE;
    }
    return nptr == NULL;
}

/*********************************************/
/* DEL_RLIST - delete referrer hash table    */
/*********************************************/

void
del_rlist(RNODEPTR * htab)
{
    /* free memory used by hash table */
    RNODEPTR aptr, temp;
    int i;

    for (i = 0; i < MAXHASH; i++) {
        if (htab[i] != NULL) {
            aptr = htab[i];
            while (aptr != NULL) {
                temp = aptr->next;
                XFREE(aptr->string);
                XFREE(aptr);
                aptr = temp;
            }
            htab[i] = NULL;
        }
    }
}

/*********************************************/
/* NEW_ANODE - User Agent node creation      */
/*********************************************/

ANODEPTR
new_anode(char *str)
{
    ANODEPTR newptr;
    size_t len = strlen(str) + 1;

    if (len > MAXAGENT) {
        ERRVPRINT(VERBOSE1, "[new_anode] %s (%d)\n", _("Warning: String exceeds storage size"), (int) len);
        ERRVPRINT(VERBOSE3, "  --> %s\n", str);
        str[MAXAGENT] = 0;
        len = MAXAGENT;
    }

    newptr = XMALLOC(struct anode, 1);
    newptr->string = XMALLOC(char, len);

    strlcpy(newptr->string, str, len);
    newptr->count = 1;
    newptr->flag = OBJ_REG;
    return newptr;
}

/*********************************************/
/* PUT_ANODE - insert/update user agent node */
/*********************************************/

int
put_anode(char *str, int type, unsigned long count, unsigned long *ctr, ANODEPTR * htab)
{
    ANODEPTR cptr, nptr;

    if (str[0] == '-')
        return 0;                               /* skip bad user agents */

    /* check if hashed */
    if ((cptr = htab[hash(str)]) == NULL) {
        /* not hashed */
        if ((nptr = new_anode(str)) != NULL) {
            nptr->flag = type;
            nptr->count = count;
            nptr->next = NULL;
            htab[hash(str)] = nptr;
            if (type != OBJ_GRP)
                (*ctr)++;
        }
    } else {
        /* hashed */
        while (cptr != NULL) {
            if (strcmp(cptr->string, str) == 0) {
                if ((type == cptr->flag) || ((type != OBJ_GRP) && (cptr->flag != OBJ_GRP))) {
                    /* found... bump counter */
                    cptr->count += count;
                    return 0;
                }
            }
            cptr = cptr->next;
        }
        /* not found... */
        if ((nptr = new_anode(str)) != NULL) {
            nptr->flag = type;
            nptr->count = count;
            nptr->next = htab[hash(str)];
            htab[hash(str)] = nptr;
            if (type != OBJ_GRP)
                (*ctr)++;
        }
    }
    if (type == OBJ_GRP)
        nptr->flag = OBJ_GRP;
    else if (isinlist(hidden_agents, nptr->string) != NULL)
        nptr->flag = OBJ_HIDE;
    return nptr == NULL;
}

/*********************************************/
/* DEL_ALIST - delete user agent hash table  */
/*********************************************/

void
del_alist(ANODEPTR * htab)
{
    /* free memory used by hash table */
    ANODEPTR aptr, temp;
    int i;

    for (i = 0; i < MAXHASH; i++) {
        if (htab[i] != NULL) {
            aptr = htab[i];
            while (aptr != NULL) {
                temp = aptr->next;
                XFREE(aptr->string);
                XFREE(aptr);
                aptr = temp;
            }
            htab[i] = NULL;
        }
    }
}

/*********************************************/
/* NEW_SNODE - Search str node creation      */
/*********************************************/

SNODEPTR
new_snode(char *str)
{
    SNODEPTR newptr;
    size_t len = strlen(str) + 1;

    if (len > MAXSRCHH) {
        ERRVPRINT(VERBOSE1, "[new_snode] %s (%d)\n", _("Warning: String exceeds storage size"), (int) len);
        ERRVPRINT(VERBOSE3, "  --> %s\n", str);
        str[MAXSRCHH] = 0;
        len = MAXSRCHH;
    }

    newptr = XMALLOC(struct snode, 1);
    newptr->string = XMALLOC(char, len);

    strlcpy(newptr->string, str, len);
    newptr->count = 1;
    return newptr;
}

/*********************************************/
/* PUT_SNODE - insert/update search str node */
/*********************************************/

int
put_snode(char *str, unsigned long count, SNODEPTR * htab)
{
    SNODEPTR cptr, nptr;

    if (str[0] == 0 || str[0] == ' ')
        return 0;                               /* skip bad search strs */

    /* check if hashed */
    if ((cptr = htab[hash(str)]) == NULL) {
        /* not hashed */
        if ((nptr = new_snode(str)) != NULL) {
            nptr->count = count;
            nptr->next = NULL;
            htab[hash(str)] = nptr;
        }
    } else {
        /* hashed */
        while (cptr != NULL) {
            if (strcmp(cptr->string, str) == 0) {
                /* found... bump counter */
                cptr->count += count;
                return 0;
            }
            cptr = cptr->next;
        }
        /* not found... */
        if ((nptr = new_snode(str)) != NULL) {
            nptr->count = count;
            nptr->next = htab[hash(str)];
            htab[hash(str)] = nptr;
        }
    }
    return nptr == NULL;
}

/*********************************************/
/* DEL_SLIST - delete search str hash table  */
/*********************************************/

void
del_slist(SNODEPTR * htab)
{
    /* free memory used by hash table */
    SNODEPTR aptr, temp;
    int i;

    for (i = 0; i < MAXHASH; i++) {
        if (htab[i] != NULL) {
            aptr = htab[i];
            while (aptr != NULL) {
                temp = aptr->next;
                XFREE(aptr->string);
                XFREE(aptr);
                aptr = temp;
            }
            htab[i] = NULL;
        }
    }
}

/*********************************************/
/* NEW_INODE - create ident (username) node  */
/*********************************************/

INODEPTR
new_inode(char *str)
{
    INODEPTR newptr;
    size_t len = strlen(str) + 1;

    if (len > MAXIDENT) {
        ERRVPRINT(VERBOSE1, "[new_inode] %s (%d)\n", _("Warning: String exceeds storage size"), (int) len);
        ERRVPRINT(VERBOSE1, "  --> %s\n", str);
        str[MAXIDENT] = 0;
        len = MAXIDENT;
    }

    newptr = XMALLOC(struct inode, 1);
    newptr->string = XMALLOC(char, len);

    strlcpy(newptr->string, str, len);
    newptr->visit = 1;
    newptr->tstamp = 0;
    return newptr;
}

/*********************************************/
/* PUT_INODE - insert/update ident node      */
/*********************************************/

int
put_inode(char *str,                            /* ident str */
          int type,                             /* obj type  */
          unsigned long count,                  /* hit count */
          unsigned long file,                   /* File flag */
          unsigned long long xfer,              /* xfer size */
          unsigned long *ctr,                   /* counter   */
          unsigned long visit,                  /* visits    */
          unsigned long tstamp,                 /* timestamp */
          INODEPTR * htab, bool isapage)
{                                               /* hashtable */
    INODEPTR cptr, nptr;

    if ((str[0] == '-') || (str[0] == 0))
        return 0;                               /* skip if no username */

    /* check if hashed */
    if ((cptr = htab[hash(str)]) == NULL) {
        /* not hashed */
        if ((nptr = new_inode(str)) != NULL) {
            nptr->flag = type;
            nptr->count = count;
            nptr->files = file;
            nptr->xfer = xfer;
            nptr->next = NULL;
            htab[hash(str)] = nptr;
            if (type != OBJ_GRP)
                (*ctr)++;

            if (visit) {
                nptr->visit = visit;
                nptr->tstamp = tstamp;
                return 0;
            } else {
                if (isapage) {
                    nptr->tstamp = tstamp;
                }
            }
        }
    } else {
        /* hashed */
        while (cptr != NULL) {
            if (strcmp(cptr->string, str) == 0) {
                if ((type == cptr->flag) || ((type != OBJ_GRP) && (cptr->flag != OBJ_GRP))) {
                    /* found... bump counter */
                    cptr->count += count;
                    cptr->files += file;
                    cptr->xfer += xfer;

                    if (isapage) {
                        if ((tstamp - cptr->tstamp) >= g_settings.settings.visit_timeout)
                            cptr->visit++;
                        cptr->tstamp = tstamp;
                    }
                    return 0;
                }
            }
            cptr = cptr->next;
        }
        /* not found... */
        if ((nptr = new_inode(str)) != NULL) {
            nptr->flag = type;
            nptr->count = count;
            nptr->files = file;
            nptr->xfer = xfer;
            nptr->next = htab[hash(str)];
            htab[hash(str)] = nptr;
            if (type != OBJ_GRP)
                (*ctr)++;

            if (visit) {
                nptr->visit = visit;
                nptr->tstamp = tstamp;
                return 0;
            } else {
                if (isapage) {
                    nptr->tstamp = tstamp;
                }
            }
        }
    }

    if (nptr != NULL) {
        /* set object type */
        if (type == OBJ_GRP)
            nptr->flag = OBJ_GRP;               /* is it a grouping? */
        else {
            /* check if it's a hidden object */
            if (isinlist(hidden_users, nptr->string) != NULL)
                nptr->flag = OBJ_HIDE;
        }
    }
    return nptr == NULL;
}

/*********************************************/
/* DEL_ILIST - delete ident hash table       */
/*********************************************/

void
del_ilist(INODEPTR * htab)
{
    /* free memory used by hash table */
    INODEPTR aptr, temp;
    int i;

    for (i = 0; i < MAXHASH; i++) {
        if (htab[i] != NULL) {
            aptr = htab[i];
            while (aptr != NULL) {
                temp = aptr->next;
                XFREE(aptr->string);            /* free ident string space */
                XFREE(aptr);                    /* free ident structure    */
                aptr = temp;
            }
            htab[i] = NULL;
        }
    }
}

/*********************************************/
/* NEW_ENODE - create errorpage node         */
/*********************************************/

ENODEPTR
new_enode(char *url,                            /* url str   */
          char *refer)
{                                               /* my referer */
    ENODEPTR newptr;
    char *url_ptr, *ref_ptr, *key_ptr;
    size_t len_url = strlen(url);
    size_t len_ref = strlen(refer);

    if (len_url >= MAXURL) {
        ERRVPRINT(VERBOSE1, "[new_enode] %s (%d)\n", _("Warning: String exceeds storage size"), (int) len_url);
        ERRVPRINT(VERBOSE3, "  --> %s\n", url);
        url[MAXURL - 1] = 0;
        len_url = MAXURL - 1;
    }

    url_ptr = XMALLOC(char, len_url + 1);

    memcpy(url_ptr, url, len_url);

    if (len_ref >= MAXURL) {
        ERRVPRINT(VERBOSE1, "[new_enode:referer] %s (%d)\n", _("Warning: String exceeds storage size"), (int) len_ref);
        ERRVPRINT(VERBOSE3, "  --> %s\n", refer);
        refer[MAXURL - 1] = 0;
        len_ref = MAXURL - 1;
    }

    ref_ptr = XMALLOC(char, len_ref + 1);

    memcpy(ref_ptr, refer, len_ref);

    /* Hash key can be sized MAXURL x 2 */
    key_ptr = XMALLOC(char, len_url + len_ref + 1);

    memcpy(key_ptr, url, len_url);
    strncat(key_ptr, refer, len_ref);

    newptr = XMALLOC(struct enode, 1);

    newptr->error_request = key_ptr;
    newptr->url = url_ptr;
    newptr->referer = ref_ptr;
    newptr->count = 1;
    newptr->next = NULL;
    return newptr;
}

/*********************************************/
/* PUT_ENODE - insert/update error node      */
/*********************************************/

int
put_enode(char *url,                            /* url str   */
          char *refer,                          /* my referer */
          int type,                             /* obj type  */
          unsigned long count,                  /* hit count */
          unsigned long *ctr,                   /* counter   */
          ENODEPTR * htab)
{                                               /* hashtable */
    ENODEPTR cptr, nptr;
    char *key_ptr;
    size_t len_url = strlen(url);
    size_t len_ref = strlen(refer);
    unsigned long hash_val;

    key_ptr = XMALLOC(char, len_url + len_ref + 1);

    memcpy(key_ptr, url, len_url);
    strncat(key_ptr, refer, len_ref);

    hash_val = hash(key_ptr);
    /* check if hashed */
    if ((cptr = htab[hash_val]) == NULL) {
        /* not hashed */
        if ((nptr = new_enode(url, refer)) != NULL) {
            nptr->next = NULL;
            htab[hash_val] = nptr;
            (*ctr)++;
            nptr->count = count;
            XFREE(key_ptr);
            return 0;
        }
    } else {
        /* hashed */
        while (cptr != NULL) {
            if (strcmp(cptr->error_request, key_ptr) == 0) {
                /* found... bump counter */
                cptr->count += count;
                XFREE(key_ptr);
                return 0;
            }
            cptr = cptr->next;
        }
        /* not found... */
        if ((nptr = new_enode(url, refer)) != NULL) {
            nptr->next = htab[hash_val];
            htab[hash_val] = nptr;
            (*ctr)++;
        }
    }

    XFREE(key_ptr);
    return nptr == NULL;
}

/*********************************************/
/* DEL_ELIST - delete errorpage hash table   */
/*********************************************/

void
del_elist(ENODEPTR * htab)
{
    /* free memory used by hash table */
    ENODEPTR aptr, temp;
    int i;

    for (i = 0; i < MAXHASH; i++) {
        if (htab[i] != NULL) {
            aptr = htab[i];
            while (aptr != NULL) {
                temp = aptr->next;
                if (aptr->error_request) {
                    XFREE(aptr->error_request); /* free errorpage string space */
                    aptr->error_request = NULL;
                }
                if (aptr->referer) {
                    XFREE(aptr->referer);       /* free referer string space */
                    aptr->referer = NULL;
                }
                if (aptr->url) {
                    XFREE(aptr->url);           /* free referer string space */
                    aptr->url = NULL;
                }
                XFREE(aptr);                    /* free errorpage structure    */
                aptr = temp;
            }
            htab[i] = NULL;
        }
    }
}

/*********************************************/
/* HASH - return hash value for string       */
/*********************************************/

unsigned long
hash(char *str)
{
    unsigned long hashval = MAXHASH;

    while (*str != '\0') {
        hashval = (hashval << 5) + hashval + *str;
        str++;
    }
    return hashval % MAXHASH;
}

/*********************************************/
/* FIND_URL - Find URL in hash table         */
/*********************************************/

char *
find_url(char *str)
{
    UNODEPTR cptr;

    if ((cptr = um_htab[hash(str)]) != NULL) {
        while (cptr != NULL) {
            if (strcmp(cptr->string, str) == 0)
                return cptr->string;
            cptr = cptr->next;
        }
    }
    return blank_str;                           /* shouldn't get here */
}

/*********************************************/
/* UPDATE_ENTRY - update entry page total    */
/*********************************************/

void
update_entry(char *str)
{
    UNODEPTR uptr;

    if (str == NULL)
        return;
    if ((uptr = um_htab[hash(str)]) == NULL)
        return;
    else {
        while (uptr != NULL) {
            if (strcmp(uptr->string, str) == 0) {
                if (uptr->flag != OBJ_GRP) {
                    uptr->entry++;
                    return;
                }
            }
            uptr = uptr->next;
        }
    }
}

/*********************************************/
/* UPDATE_EXIT  - update exit page total     */
/*********************************************/

void
update_exit(char *str, bool is_single_access)
{
    UNODEPTR uptr;

    if (str == NULL)
        return;
    if ((uptr = um_htab[hash(str)]) == NULL)
        return;
    else {
        while (uptr != NULL) {
            if (strcmp(uptr->string, str) == 0) {
                if (uptr->flag != OBJ_GRP) {
                    uptr->exit++;
                    if (is_single_access) {
                        uptr->single_access++;
                    }
                    return;
                }
            }
            uptr = uptr->next;
        }
    }
}

/*********************************************/
/* MONTH_UPDATE_EXIT  - eom exit page update */
/*********************************************/

void
month_update_exit(unsigned long tstamp)
{
    HNODEPTR nptr;
    int i;

    for (i = 0; i < MAXHASH; i++) {
        nptr = sm_htab[i];
        while (nptr != NULL) {
            if (nptr->flag != OBJ_GRP) {
                if ((tstamp - nptr->tstamp) >= g_settings.settings.visit_timeout) {
                    /* End of month, update all "still open" sessions; and count as closed sessions that would be expired */
                    update_exit(nptr->lasturl, nptr->lasturl_is_entry);
                }
            }
            nptr = nptr->next;
        }
    }
}

/*********************************************/
/* TOT_VISIT - calculate total visits        */
/*********************************************/

unsigned long
tot_visit(HNODEPTR * list)
{
    HNODEPTR hptr;
    unsigned long tot = 0;
    int i;

    for (i = 0; i < MAXHASH; i++) {
        hptr = list[i];
        while (hptr != NULL) {
            if (hptr->flag != OBJ_GRP) {
                tot += hptr->visit;
            }
            hptr = hptr->next;
        }
    }
    return tot;
}
