# -*- coding: utf-8 -*-

# ==============================================================================
# COPYRIGHT (C) 1991 - 2003  EDF R&D                  WWW.CODE-ASTER.ORG
# THIS PROGRAM IS FREE SOFTWARE; YOU CAN REDISTRIBUTE IT AND/OR MODIFY
# IT UNDER THE TERMS OF THE GNU GENERAL PUBLIC LICENSE AS PUBLISHED BY
# THE FREE SOFTWARE FOUNDATION; EITHER VERSION 2 OF THE LICENSE, OR
# (AT YOUR OPTION) ANY LATER VERSION.
#
# THIS PROGRAM IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL, BUT
# WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF
# MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. SEE THE GNU
# GENERAL PUBLIC LICENSE FOR MORE DETAILS.
#
# YOU SHOULD HAVE RECEIVED A COPY OF THE GNU GENERAL PUBLIC LICENSE
# ALONG WITH THIS PROGRAM; IF NOT, WRITE TO EDF R&D CODE_ASTER,
#    1 AVENUE DU GENERAL DE GAULLE, 92141 CLAMART CEDEX, FRANCE.
# ==============================================================================

"""
This module provides functions to modify a profile.
"""

import os
from asrun.installation import datadir
from asrun.common.i18n import _
from asrun.config       import ASTER_CONFIG
from asrun.common_func  import get_tmpname
from asrun.common.utils import YES

SEPAR = '%NEXT%'


class ProfileModifier(object):
    """;-) """
    service = 'service_name'
    
    def __init__(self, prof, run=None):
        """Init."""
        self.run       = run
        self.prof_orig = prof
        self.new_prof  = self.prof_orig.copy()
        self.special   = self.prof_orig['special'][0].split(SEPAR)

    def modify(self):
        """Modifications for all services.
        """
        # keep arguments
        # empty data and result
        self.new_prof.data = []
        self.new_prof.resu = []
        
        # set some parameters
        self.new_prof['nomjob']  = '%s_%s' % (self.prof_orig['nomjob'][0], self.service)
        self.new_prof['actions'] = 'make_etude'
        self.new_prof['ncpus']   = 1
        self.new_prof['consbtc'] = YES
        self.new_prof['soumbtc'] = YES
        
        # remove some parameters :
        for p in ('special',):
            del self.new_prof[p]

    def check_filename(self, name_in):
        """If we start execution on a remote node, we must add user@'here'."""
        node = self.prof_orig['noeud'][0].split('.')[0]  # without domain name
        user, host = self.run.system.getuser_host()
        name_out = name_in
        if node != host:
            name_out = '%s@%s:%s' % (user, host, name_in)
        return name_out
    
    def return_profile(self):
        """Return changed profile."""
        return self.new_prof


class ProfileModifierMeshtool(ProfileModifier):
    """Modifier for meshtool service.
    """
    service = 'meshtool'

    def modify(self):
        """Modifications for meshtool service.
        """
        super(ProfileModifierMeshtool, self).modify()
        
        # job name
        self.new_prof['nomjob']  = '%s_mesh' % self.prof_orig['nomjob'][0]
        
        # commands file
        fcomm = os.path.join(datadir, 'meshtool.comm')
        self.new_prof.Set('D',
            { 'path' : fcomm, 'ul' : 1, 'type' : 'comm',
           'isrep' : False, 'compr' : False})
        
        # special : mesh IN (unit 71), mesh OUT (unit 72)
        assert len(self.special) >= 3
        self.new_prof.parse("""%s 71\n%s 72""" % tuple(self.special[1:3]))

        # parameter file : field 4 and next
        fpara = get_tmpname(self.run, self.run['tmp_user'], basename='meshtool.para')
        self.run.Delete(fpara)
        txt = os.linesep.join(self.special[3:])
        open(fpara, 'w').write(txt)
        self.new_prof.Set('D',
            { 'path' : self.check_filename(fpara), 'ul' : 70, 'type' : 'libr',
           'isrep' : False, 'compr' : False})


class ProfileModifierConvbase(ProfileModifier):
    """Modifier for convbase service.
    """
    service = 'convbase'

    def modify(self):
        """Modifications for convbase service.
        """
        super(ProfileModifierConvbase, self).modify()
        
        # commands file
        fcomm = os.path.join(datadir, 'convbase.comm')
        self.new_prof.Set('D',
            { 'path' : fcomm, 'ul' : 1, 'type' : 'comm',
           'isrep' : False, 'compr' : False})

        # special : base IN, base OUT
        assert len(self.special) >= 3
        self.new_prof.parse("""%s\n%s""" % tuple(self.special[1:3]))


class ProfileModifierStanley(ProfileModifier):
    """Modifier for stanley service.
    """
    service = 'stanley'

    def modify(self):
        """Modifications for stanley service.
        """
        super(ProfileModifierStanley, self).modify()
        
        # job name
        self.new_prof['nomjob']  = '%s_post' % self.prof_orig['nomjob'][0]
        
        # commands file
        fcomm = os.path.join(datadir, 'stanley_post.comm')
        self.new_prof.Set('D',
            { 'path' : fcomm, 'ul' : 1, 'type' : 'comm', 'isrep' : False, 'compr' : False})

        # special : "R base repertoire flag 0"
        assert len(self.special) >= 2
        self.new_prof.parse(self.special[1])


class ProfileModifierDistribution(ProfileModifier):
    """Modifier for distributed calculations.
    """
    service = 'distribution'

    def modify(self):
        """Modifications for distribution service.
        """
        self.new_prof['actions'] = self.service
        del self.new_prof['distrib']


class ProfileModifierExecTool(ProfileModifier):
    """Modifier for extern tool.
    """
    service = 'exectool'

    def modify(self):
        """Modifications for extern tool.
        """
        # get executable
        dbg = self.prof_orig['debug'][0]
        if dbg == '':
            dbg = 'nodebug'
        if self.prof_orig.Get('D', typ='exec'):
            d_exe = self.prof_orig.Get('D', typ='exec')[0]
        else:
            d_exe = { 'path' : '?', 'type' : 'exec', 'isrep' : False, 'compr' : False, 'ul' : 0 }
            REPREF = self.prof_orig.get_version_path()
            lconf = self.prof_orig.Get('D', typ='conf')
            if not lconf:
                ficconf = os.path.join(REPREF, 'config.txt')
            else:
                ficconf = lconf[0]['path']
                if self.run.IsRemote(ficconf):
                    ficconf = get_tmpname(self.run, self.run['tmp_user'], basename='config.txt')
                    self.run.ToDelete(ficconf)
                    ibid = self.run.Copy(ficconf, lconf[0]['path'])
                else:
                    ficconf = self.run.PathOnly(ficconf)
            conf = ASTER_CONFIG(ficconf, self.run)
            if dbg == 'nodebug':
                d_exe['path'] = os.path.join(REPREF, conf['BIN_NODBG'][0])
            else:
                d_exe['path'] = os.path.join(REPREF, conf['BIN_DBG'][0])
        # add valgrind command
        if self.prof_orig['exectool'][0] == '':
            self.run.Mess(_(u'"exectool" is not defined !'), '<F>_PROGRAM_ERROR')
        cmd = self.prof_orig['exectool'][0]
        if self.run.get(cmd):
            cmd = self.run[cmd] + ' '
        else:
            cmd += ' '
        if self.run.IsRemote(d_exe['path']):
            self.run.Mess(_(u'"exectool" can not be used with a remote executable.'), "<F>_ERROR")
        cmd += d_exe['path'] + ' "$@" 2>&1'
        # write script
        exetmp = get_tmpname(self.run, self.run['shared_tmp'], basename='front_exec')
        cmd += '\n' + 'rm -f %s' % exetmp
        open(exetmp, 'w').write(cmd)
        os.chmod(exetmp, 0755)
        d_exe['path'] = exetmp
        # change profile
        del self.new_prof['exectool']
        self.new_prof.Del('D', typ='exec')
        self.new_prof.Set('D', d_exe)


def ModifierFactory(service, prof, run=None):
    if service == 'meshtool':
        return ProfileModifierMeshtool(prof, run)
    elif service == 'stanley':
        return ProfileModifierStanley(prof, run)
    elif service == 'convbase':
        return ProfileModifierConvbase(prof, run)
    elif service == 'distribution':
        return ProfileModifierDistribution(prof, run)
    elif service == 'exectool':
        return ProfileModifierExecTool(prof, run)
    return None

