// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __SysCall_h__
#define __SysCall_h__

// Encapsulate some system calls. 

#include <time.h>
#include <stdio.h>
#include <stddef.h>
#include <sys/stat.h>
#include <sys/types.h>
#ifdef _MSC_VER
#include <string.h>
#include <stdlib.h>
#else
#include <dirent.h>
#endif // _MSVC

#include "Puma/Filename.h"

namespace Puma {

class SysCall;

class FileInfo {
  friend class SysCall;
#ifndef _MSC_VER
  struct stat _info;
#else
  struct _stat _info;
#endif

public:
  bool is_dir () const {
#ifdef _MSC_VER
    return _S_IFDIR & _info.st_mode;
#else
    return S_ISDIR (_info.st_mode);
#endif // _MSC_VER
  }
  time_t modi_time () const { return _info.st_mtime; }
  int size () const { return _info.st_size; }
};

#ifdef _MSC_VER
struct DIR {
  long _handle;
  DIR (long h) : _handle (h) {}
};
#endif // _MSC_VER

typedef DIR *DirHandle;
typedef FILE *FileHandle;

class ErrorSink;

class SysCall {
  static void printerror (ErrorSink *, const char *, 
                          const char * = (const char *)0);
  static void printerror (ErrorSink *, const char *, int);

public:
  SysCall () {}
        
  // create temporary file name, needs to be freed by caller
  static const char* mktemp (const char *prefix, ErrorSink * = (ErrorSink *)0);

  // open/create/close file with exclusive file locking
  static int open_excl (const char *, int, ErrorSink * = (ErrorSink *)0);
  static int create_excl (const char *, int, ErrorSink * = (ErrorSink *)0);
  static bool close_excl (int, ErrorSink * = (ErrorSink *)0);
        
  // open/create/close file
  static int open (const char *, int, ErrorSink * = (ErrorSink *)0);
  static int create (const char *, int, ErrorSink * = (ErrorSink *)0);
  static bool close (int, ErrorSink * = (ErrorSink *)0);
        
  // read from file descriptor
  static long read (int, void *, size_t, ErrorSink * = (ErrorSink *)0);
        
  // get file status
  static bool stat (const char *, FileInfo &, ErrorSink * = (ErrorSink *)0);
  static bool fstat (int, FileInfo &, ErrorSink * = (ErrorSink *)0);
  static char* readlink (const char*, char*, size_t, ErrorSink * = (ErrorSink *)0);
        
  // open/close file stream
  static FileHandle fopen (const char *, const char *, ErrorSink * = (ErrorSink *)0);
  static bool fclose (FileHandle, ErrorSink * = (ErrorSink *)0);

  // open/close directory stream
  static DirHandle opendir (const char *, ErrorSink * = (ErrorSink *)0);
  static bool closedir (DirHandle, ErrorSink * = (ErrorSink *)0);

  // read from directory stream
  static const char *readdir (DirHandle, ErrorSink * = (ErrorSink *)0);
        
  // change directory 
  static bool chdir (const char *, ErrorSink * = (ErrorSink *)0);
        
  // create directory
  static bool mkdir (const char *, ErrorSink * = (ErrorSink *)0);
        
  // get current working directory
  static char *getcwd (char *, size_t, ErrorSink * = (ErrorSink *)0);
        
  // get current time
  static time_t time (time_t *, ErrorSink * = (ErrorSink *)0);

  // platform depended path handling
  static void MakeUnixPath (char*);
  static void MakeDosPath  (char*);
  
#ifdef WIN32
  // normalize an absolute Win32 filename of an exisiting file
  static bool normalize (Filename filename, Filename &norm, ErrorSink *err);
#endif // WIN32

  // return a canonical path name for an exisiting file
  static bool canonical (Filename filename, Filename &result, ErrorSink * = 0);
};


} // namespace Puma

#endif /* __SysCall_h__ */
