// This file is part of the AspectC++ compiler 'ac++'.
// Copyright (C) 1999-2003  The 'ac++' developers (see aspectc.org)
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __source_loc_h__
#define __source_loc_h__

#include "File.h"

#include <string>
using std::string;

enum SourceLocKind { SLK_DEF, SLK_DECL, SLK_NONE };

class SourceLoc {
  int _file_id;
  int _line;
  int _len;
  SourceLocKind _kind;
public:
  SourceLoc (int file_id, int line, int len, SourceLocKind kind) :
    _file_id (file_id), _line (line), _len (len), _kind (kind) {}

  SourceLoc (RepoXMLNode sn) {
    _file_id = sn.get_int_prop ("file");
    _line = sn.get_int_prop ("line");
    _len  = sn.get_int_prop ("len");
    _kind = SLK_NONE;
    if (sn.has_prop ("kind")) {
      string kind_str = sn.get_str_prop ("kind");
      if (kind_str == "decl")
        _kind = SLK_DECL;
      else if (kind_str == "def")
        _kind = SLK_DEF;
    }
  }
      
  SourceLocKind kind() const { return _kind; }
  void file (File *f) { _file_id = f->id (); }
  int file_id () const { return _file_id; }
  int line () const { return _line; }
  int len () const { return _len; }
  
  void print (int indent = 0) const {
    for (int i = 0; i < indent; i++)
      cout << "  ";
    cout << "source file " << _file_id << " line " << _line << " len " << _len;
    if (_kind != SLK_NONE)
      cout << " kind " << (int)_kind;
    cout << endl;
  }

  void make_xml (RepoXMLNode parent) const {
    RepoXMLNode fn = parent.make_child ("src");
    fn.set_int_prop ("file", _file_id);
    fn.set_int_prop ("line", _line);
    fn.set_int_prop ("len", _len);
    if (_kind != SLK_NONE)
      fn.set_str_prop ("kind", _kind == SLK_DECL ? "decl" : "def");
  }

  bool operator < (const SourceLoc &sl) const {
    return _kind < sl._kind ? true : (_line < sl._line ? true : (_file_id < sl._file_id));
  }
  bool operator == (const SourceLoc &sl) const {
    return _kind == sl._kind && _line == sl._line && _file_id == sl._file_id;
  }
};

#endif // __source_loc_h__
