/*
 * Argus Software.  Common library routines - Utilities
 * Copyright (c) 2000-2006 QoSient, LLC
 * All rights reserved.
 *
 * QoSIENT, LLC DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL QoSIENT, LLC BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
 * THIS SOFTWARE.
 *
 */

/* 
 * $Id: $
 * $DateTime: $
 * $Change: $
 */


#ifndef argus_util
#define argus_util
#endif

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/syslog.h>
#include <stdarg.h>


#include <string.h>
#include <compat.h>

#include <argus_os.h>
#include <argus_util.h>
#include <argus_parser.h>

#ifdef HAVE_ETHER_HOSTTON
/*
 * XXX - do we need any of this if <netinet/if_ether.h> doesn't declare
 * ether_hostton()?
 */
#ifdef HAVE_NETINET_IF_ETHER_H
struct mbuf;            /* Squelch compiler warnings on some platforms for */
struct rtentry;         /* declarations in <net/if.h> */
#include <net/if.h>     /* for "struct ifnet" in "struct arpcom" on Solaris */
#include <netinet/if_ether.h>
#endif /* HAVE_NETINET_IF_ETHER_H */
#ifdef NETINET_ETHER_H_DECLARES_ETHER_HOSTTON
#include <netinet/ether.h>
#endif /* NETINET_ETHER_H_DECLARES_ETHER_HOSTTON */
#endif /* HAVE_ETHER_HOSTTON */


extern char *ArgusProgramName;

void ArgusInitAddrtoname(struct ArgusParserStruct *, u_int, u_int);
void ArgusInitServarray(struct ArgusParserStruct *);
void ArgusInitEprotoarray(struct ArgusParserStruct *);
void ArgusInitProtoidarray(struct ArgusParserStruct *);
void ArgusInitEtherarray(struct ArgusParserStruct *);
void ArgusInitLlcsaparray(struct ArgusParserStruct *);

u_int ipaddrtonetmask(u_int);
u_int getnetnumber( u_int);

 
struct ArgusParserStruct *
ArgusNewParser(char *progname)
{
   struct ArgusParserStruct *retn = NULL;
 
   if ((retn  = (struct ArgusParserStruct *) ArgusCalloc(1, sizeof(*retn))) == NULL)
      ArgusLog (LOG_ERR, "ArgusNewParser(%s) ArgusCalloc error %s", progname, strerror(errno));
 
   retn->ArgusCIDRPtr = &retn->ArgusCIDRBuffer;
   ArgusInitAddrtoname (retn, 0L, 0L);
   return (retn);
}


void
ArgusAdjustGlobalTime (struct timeval *global, struct timeval *now)
{
   struct timeval ArgusTimeDelta;

   ArgusTimeDelta.tv_sec  = now->tv_sec  - global->tv_sec;
   ArgusTimeDelta.tv_usec = now->tv_usec - global->tv_usec;
 
   global->tv_sec  = now->tv_sec  - ArgusTimeDelta.tv_sec;
   global->tv_usec = now->tv_usec - ArgusTimeDelta.tv_usec;
 
   if (global->tv_usec < 0) {
      global->tv_sec--;
      global->tv_usec += 1000000;
   } else {
      if (global->tv_usec > 1000000) {
         global->tv_sec++;
         global->tv_usec -= 1000000;
      }
   }
}

extern char *print_time(struct timeval *);


#ifdef ARGUSDEBUG
void ArgusDebug (int d, char *fmt, ...);

#include <sys/time.h>

void
ArgusDebug (int d, char *fmt, ...)
{
   struct timeval now;
   extern int Argusdflag;
   va_list ap;
   char buf[1024], *ptr;

   if (d <= Argusdflag) {
      gettimeofday (&now, 0L);
      (void) snprintf (buf, 1024, "%s[%d]: %s ", ArgusProgramName, (int)getpid(), print_time(&now));
      ptr = &buf[strlen(buf)];

      va_start (ap, fmt);
      (void) vsnprintf (ptr, 1024, fmt, ap);
      ptr = &buf[strlen(buf)];
      va_end (ap);

      if (*fmt) {
         fmt += (int) strlen (fmt);
         if (fmt[-1] != '\n')
            snprintf (ptr, 2, "\n");
      }
      fprintf (stderr, "%s", buf);
   }
}
#endif

void ArgusPrintHex (const u_char *, u_int);

#if defined(_LITTLE_ENDIAN)
#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
#include <extract.h>
#if !defined(ntohll)
#define ntohll(x) EXTRACT_64BITS(&x)
#define htonll(x) EXTRACT_64BITS(&x)
#else
#define ntohll(x) x
#define htonll(x) x
#endif
#else
#include <byteswap.h>
#if !defined(ntohll)
#define ntohll(x) bswap_64(x)
#define htonll(x) bswap_64(x)
#else
#define ntohll(x) x
#define htonll(x) x
#endif
#endif
#endif

void ArgusNtoH (struct ArgusRecord *);
void ArgusHtoN (struct ArgusRecord *);

#include <argus_def.h>

void
ArgusNtoH (struct ArgusRecord *argus)
{
#if defined(_LITTLE_ENDIAN)
   struct ArgusRecordHeader *hdr = &argus->hdr;
   struct ArgusDSRHeader *dsr = (struct ArgusDSRHeader *) (hdr + 1);


   if (argus->hdr.type & ARGUS_MAR) {
      argus->hdr.len = ntohs(argus->hdr.len);
      argus->argus_mar.status            = ntohl(argus->argus_mar.status);
      argus->argus_mar.argusid            = ntohl(argus->argus_mar.argusid);
      argus->argus_mar.localnet          = ntohl(argus->argus_mar.localnet);
      argus->argus_mar.netmask           = ntohl(argus->argus_mar.netmask);
      argus->argus_mar.nextMrSequenceNum = ntohl(argus->argus_mar.nextMrSequenceNum);
      argus->argus_mar.startime.tv_sec   = ntohl(argus->argus_mar.startime.tv_sec);
      argus->argus_mar.startime.tv_usec  = ntohl(argus->argus_mar.startime.tv_usec);
      argus->argus_mar.now.tv_sec        = ntohl(argus->argus_mar.now.tv_sec);
      argus->argus_mar.now.tv_usec       = ntohl(argus->argus_mar.now.tv_usec);
      argus->argus_mar.reportInterval    = ntohs(argus->argus_mar.reportInterval);
      argus->argus_mar.argusMrInterval    = ntohs(argus->argus_mar.argusMrInterval);

      argus->argus_mar.pktsRcvd          = ntohll(argus->argus_mar.pktsRcvd);
      argus->argus_mar.bytesRcvd         = ntohll(argus->argus_mar.bytesRcvd);
      argus->argus_mar.drift             = ntohll(argus->argus_mar.drift);

      argus->argus_mar.records           = ntohl(argus->argus_mar.records);
      argus->argus_mar.flows             = ntohl(argus->argus_mar.flows);
      argus->argus_mar.dropped           = ntohl(argus->argus_mar.dropped);
      argus->argus_mar.queue             = ntohl(argus->argus_mar.queue);
      argus->argus_mar.output            = ntohl(argus->argus_mar.output);
      argus->argus_mar.clients           = ntohl(argus->argus_mar.clients);
      argus->argus_mar.bufs              = ntohl(argus->argus_mar.bufs);
      argus->argus_mar.bytes             = ntohl(argus->argus_mar.bytes);

      argus->argus_mar.thisid            = ntohl(argus->argus_mar.thisid);
      argus->argus_mar.record_len        = ntohl(argus->argus_mar.record_len);
      return;
   }

   hdr->len = ntohs(hdr->len);
   if (hdr->len > 1) {
      int cnt;
      while ((char *) dsr < ((char *) argus + (hdr->len * 4))) {
         switch (dsr->type & 0x7F) {
            case ARGUS_FLOW_DSR: {
               struct ArgusFlow *flow = (struct ArgusFlow *) dsr;

               switch (flow->hdr.subtype & 0x3F) {
                  case ARGUS_FLOW_CLASSIC5TUPLE: {
                     switch (flow->hdr.argus_dsrvl8.qual & 0x7F) {
                        case ARGUS_TYPE_IPV4:
                           flow->ip_flow.ip_src = ntohl(flow->ip_flow.ip_src);
                           flow->ip_flow.ip_dst = ntohl(flow->ip_flow.ip_dst);
                           switch (flow->ip_flow.ip_p) {
                              case IPPROTO_TCP:
                              case IPPROTO_UDP:
                                 flow->ip_flow.sport = ntohs(flow->ip_flow.sport);
                                 flow->ip_flow.dport = ntohs(flow->ip_flow.dport);
                                 break;
                              case IPPROTO_ESP:
                                 flow->esp_flow.spi = ntohl(flow->esp_flow.spi);
                                 break;
                              case IPPROTO_IGMP:
                                 flow->igmp_flow.ip_id = ntohs(flow->igmp_flow.ip_id);
                                 break;
                           }
                           break; 

                        case ARGUS_TYPE_IPV6: {
                           unsigned int *iptr = (unsigned int *)&flow->ipv6_flow;
                           iptr[8] = ntohl(iptr[8]);

                           if (flow->hdr.argus_dsrvl8.qual & ARGUS_FRAGMENT) {
                              flow->fragv6_flow.ip_id = ntohl(flow->fragv6_flow.ip_id);
                           } else {
                              switch (flow->ipv6_flow.ip_p) {
                                 case IPPROTO_TCP:
                                 case IPPROTO_UDP:
                                    flow->ipv6_flow.sport = ntohs(flow->ipv6_flow.sport);
                                    flow->ipv6_flow.dport = ntohs(flow->ipv6_flow.dport);
                                    break;
                                 case IPPROTO_ESP:
                                    flow->esp6_flow.spi = ntohl(flow->esp6_flow.spi);
                                    break;
                              }
                           }
                           break; 
                        }

                        case ARGUS_TYPE_ETHER: {
                           struct ArgusMacFlow *mac = (struct ArgusMacFlow *) &flow->mac_flow;
                           mac->ehdr.ether_type = ntohs(mac->ehdr.ether_type);
                           break;
                        }
                     
                        case ARGUS_TYPE_RARP: {
                           struct ArgusRarpFlow *rarp = (struct ArgusRarpFlow *) &flow->rarp_flow;
                           rarp->arp_tpa = ntohl(rarp->arp_tpa);
                           break;
                        }
                     
                        case ARGUS_TYPE_ARP: {
                           struct ArgusArpFlow *arp = (struct ArgusArpFlow *) &flow->arp_flow;
                           arp->arp_spa = ntohl(arp->arp_spa);
                           arp->arp_tpa = ntohl(arp->arp_tpa);
                           break;
                        } 
                     }
                     break; 
                  }
               }
               break;
            }

            case ARGUS_IPATTR_DSR: {
               struct ArgusIPAttrStruct *attr = (struct ArgusIPAttrStruct *) dsr;
               unsigned int *dsrptr = (unsigned int *)(dsr + 1);

               if (attr->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_SRC) {
                  struct ArgusIPAttrObject *aobj = (struct ArgusIPAttrObject *) dsrptr;
                  aobj->ip_id = ntohs(aobj->ip_id);
                  dsrptr++;
               }
               if (attr->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_SRC_OPTIONS) {
                  *dsrptr = ntohl(*dsrptr);
                  dsrptr++;
               }
               if (attr->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_DST) {
                  struct ArgusIPAttrObject *aobj = (struct ArgusIPAttrObject *) dsrptr;
                  aobj->ip_id = ntohs(aobj->ip_id);
                  dsrptr++;
               }
               if (attr->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_DST_OPTIONS) {
                  *dsrptr = ntohl(*dsrptr);
                  dsrptr++;
               }
               break;
            }

            case ARGUS_TRANSPORT_DSR: {
               struct ArgusTransportStruct *trans = (struct ArgusTransportStruct *) dsr;

               switch (trans->hdr.subtype) {
                  case ARGUS_SRCSEQ:
                     trans->seqnum = ntohl(trans->seqnum);
                  case ARGUS_SRCID:
                     trans->srcid.value = ntohl(trans->srcid.value);
                     break;
               }
               break;
            }

            case ARGUS_TIME_DSR: {
               unsigned int x, *dtime = (unsigned int *) dsr; 
 
               for (x = 1; x < dsr->argus_dsrvl8.len; x++)
                  dtime[x] = ntohl(dtime[x]);
               break; 
            }

            case ARGUS_METER_DSR: {
               if (dsr->subtype & ARGUS_METER_PKTS_BYTES) {
                  switch (dsr->argus_dsrvl8.qual & 0x0F) {
                     case ARGUS_SRCDST_BYTE:
                        break;
                     case ARGUS_SRCDST_SHORT:
                        ((unsigned short *)(dsr + 1))[0] = ntohs(((unsigned short *)(dsr + 1))[0]);
                        ((unsigned short *)(dsr + 1))[1] = ntohs(((unsigned short *)(dsr + 1))[1]);
                        ((unsigned short *)(dsr + 1))[2] = ntohs(((unsigned short *)(dsr + 1))[2]);
                        ((unsigned short *)(dsr + 1))[3] = ntohs(((unsigned short *)(dsr + 1))[3]);
                        break;
                     case ARGUS_SRCDST_INT:
                        ((unsigned int *)(dsr + 1))[0] = ntohl(((unsigned int *)(dsr + 1))[0]);
                        ((unsigned int *)(dsr + 1))[1] = ntohl(((unsigned int *)(dsr + 1))[1]);
                        ((unsigned int *)(dsr + 1))[2] = ntohl(((unsigned int *)(dsr + 1))[2]);
                        ((unsigned int *)(dsr + 1))[3] = ntohl(((unsigned int *)(dsr + 1))[3]);
                        break;
                     case ARGUS_SRC_SHORT:
                        ((unsigned short *)(dsr + 1))[0] = ntohs(((unsigned short *)(dsr + 1))[0]);
                        ((unsigned short *)(dsr + 1))[1] = ntohs(((unsigned short *)(dsr + 1))[1]);
                        break;
                     case ARGUS_SRC_INT:
                        ((unsigned int *)(dsr + 1))[0] = ntohl(((unsigned int *)(dsr + 1))[0]);
                        ((unsigned int *)(dsr + 1))[1] = ntohl(((unsigned int *)(dsr + 1))[1]);
                        break;
                     case ARGUS_SRC_LONGLONG:
                        break;
                     case ARGUS_DST_SHORT:
                        ((unsigned short *)(dsr + 1))[0] = ntohs(((unsigned short *)(dsr + 1))[0]);
                        ((unsigned short *)(dsr + 1))[1] = ntohs(((unsigned short *)(dsr + 1))[1]);
                        break;
                     case ARGUS_DST_INT:
                        ((unsigned int *)(dsr + 1))[0] = ntohl(((unsigned int *)(dsr + 1))[0]);
                        ((unsigned int *)(dsr + 1))[1] = ntohl(((unsigned int *)(dsr + 1))[1]);
                        break;
                  }

               } else 
               if (dsr->subtype & ARGUS_METER_PKTS_BYTES_APP) {
                  switch (dsr->argus_dsrvl8.qual & 0x0F) {
                     case ARGUS_SRCDST_BYTE:
                     case ARGUS_SRC_BYTE:
                     case ARGUS_DST_BYTE:
                        break;
                     case ARGUS_SRCDST_SHORT:
                        ((unsigned short *)(dsr + 1))[0] = ntohs(((unsigned short *)(dsr + 1))[0]);
                        ((unsigned short *)(dsr + 1))[1] = ntohs(((unsigned short *)(dsr + 1))[1]);
                        ((unsigned short *)(dsr + 1))[2] = ntohs(((unsigned short *)(dsr + 1))[2]);
                        ((unsigned short *)(dsr + 1))[3] = ntohs(((unsigned short *)(dsr + 1))[3]);
                        ((unsigned short *)(dsr + 1))[4] = ntohs(((unsigned short *)(dsr + 1))[4]);
                        ((unsigned short *)(dsr + 1))[5] = ntohs(((unsigned short *)(dsr + 1))[5]);
                        break;
                     case ARGUS_SRCDST_INT:
                        ((unsigned int *)(dsr + 1))[0] = ntohl(((unsigned int *)(dsr + 1))[0]);
                        ((unsigned int *)(dsr + 1))[1] = ntohl(((unsigned int *)(dsr + 1))[1]);
                        ((unsigned int *)(dsr + 1))[2] = ntohl(((unsigned int *)(dsr + 1))[2]);
                        ((unsigned int *)(dsr + 1))[3] = ntohl(((unsigned int *)(dsr + 1))[3]);
                        ((unsigned int *)(dsr + 1))[4] = ntohl(((unsigned int *)(dsr + 1))[4]);
                        ((unsigned int *)(dsr + 1))[5] = ntohl(((unsigned int *)(dsr + 1))[5]);
                        break;
                     case ARGUS_SRC_SHORT:
                        ((unsigned short *)(dsr + 1))[0] = ntohs(((unsigned short *)(dsr + 1))[0]);
                        ((unsigned short *)(dsr + 1))[1] = ntohs(((unsigned short *)(dsr + 1))[1]);
                        ((unsigned short *)(dsr + 1))[2] = ntohs(((unsigned short *)(dsr + 1))[2]);
                        break;
                     case ARGUS_SRC_INT:
                        ((unsigned int *)(dsr + 1))[0] = ntohl(((unsigned int *)(dsr + 1))[0]);
                        ((unsigned int *)(dsr + 1))[1] = ntohl(((unsigned int *)(dsr + 1))[1]);
                        ((unsigned int *)(dsr + 1))[2] = ntohl(((unsigned int *)(dsr + 1))[2]);
                        break;
                     case ARGUS_SRC_LONGLONG:
                        break;
                     case ARGUS_DST_SHORT:
                        ((unsigned short *)(dsr + 1))[0] = ntohs(((unsigned short *)(dsr + 1))[0]);
                        ((unsigned short *)(dsr + 1))[1] = ntohs(((unsigned short *)(dsr + 1))[1]);
                        ((unsigned short *)(dsr + 1))[2] = ntohs(((unsigned short *)(dsr + 1))[2]);
                        break;
                     case ARGUS_DST_INT:
                        ((unsigned int *)(dsr + 1))[0] = ntohl(((unsigned int *)(dsr + 1))[0]);
                        ((unsigned int *)(dsr + 1))[1] = ntohl(((unsigned int *)(dsr + 1))[1]);
                        ((unsigned int *)(dsr + 1))[2] = ntohl(((unsigned int *)(dsr + 1))[2]);
                        break;
                  }
               }
               break;
            }

            case ARGUS_NETWORK_DSR: {
               struct ArgusNetworkStruct *net = (struct ArgusNetworkStruct *)dsr;
               switch (net->hdr.subtype) {
                  case ARGUS_METER_TCP_PERF: {
                     struct ArgusTCPObject *tcp = (struct ArgusTCPObject *)&net->net_union.tcp;
                     tcp->status       = ntohl(tcp->status);
                     tcp->state        = ntohl(tcp->state);
                     tcp->options      = ntohl(tcp->options);
                     tcp->synAckuSecs  = ntohl(tcp->synAckuSecs);
                     tcp->ackDatauSecs = ntohl(tcp->ackDatauSecs);

                     tcp->src.lasttime.tv_sec  = ntohl(tcp->src.lasttime.tv_sec);
                     tcp->src.lasttime.tv_usec = ntohl(tcp->src.lasttime.tv_usec);
                     tcp->src.status = ntohl(tcp->src.status);
                     tcp->src.seqbase = ntohl(tcp->src.seqbase);
                     tcp->src.seq = ntohl(tcp->src.seq);
                     tcp->src.ack = ntohl(tcp->src.ack);
                     tcp->src.winnum = ntohl(tcp->src.winnum);
                     tcp->src.bytes = ntohl(tcp->src.bytes);
                     tcp->src.retrans = ntohl(tcp->src.retrans);
                     tcp->src.ackbytes = ntohl(tcp->src.ackbytes);
                     tcp->src.state = ntohs(tcp->src.state);
                     tcp->src.win = ntohs(tcp->src.win);
                     tcp->src.winbytes = ntohs(tcp->src.winbytes);

                     if (dsr->argus_dsrvl8.len > (((sizeof(struct ArgusTCPObject) - sizeof(struct ArgusTCPObjectMetrics))+3)/4 + 1)) {
                        tcp->dst.lasttime.tv_sec  = ntohl(tcp->dst.lasttime.tv_sec);
                        tcp->dst.lasttime.tv_usec = ntohl(tcp->dst.lasttime.tv_usec);
                        tcp->dst.status = ntohl(tcp->dst.status);
                        tcp->dst.seqbase = ntohl(tcp->dst.seqbase);
                        tcp->dst.seq = ntohl(tcp->dst.seq);
                        tcp->dst.ack = ntohl(tcp->dst.ack);
                        tcp->dst.winnum = ntohl(tcp->dst.winnum);
                        tcp->dst.bytes = ntohl(tcp->dst.bytes);
                        tcp->dst.retrans = ntohl(tcp->dst.retrans);
                        tcp->dst.ackbytes = ntohl(tcp->dst.ackbytes);
                        tcp->dst.state = ntohs(tcp->dst.state);
                        tcp->dst.win = ntohs(tcp->dst.win);
                        tcp->dst.winbytes = ntohs(tcp->dst.winbytes);
                     }
                     break;
                  }

                  case ARGUS_ESP_DSR: {
                     struct ArgusESPObject *espObj = (struct ArgusESPObject *)&net->net_union.esp;
                     espObj->status  = ntohl(espObj->status);
                     espObj->spi     = ntohl(espObj->spi);
                     espObj->lastseq = ntohl(espObj->lastseq);
                     espObj->lostseq = ntohl(espObj->lostseq);
                     break;
                  }
                  case ARGUS_RTP_FLOW: {
                     struct ArgusRTPObject *rtpObj = (struct ArgusRTPObject *)&net->net_union.rtp;
                     rtpObj->state       = ntohl(rtpObj->state);
                     rtpObj->src.rh_seq  = ntohs(rtpObj->src.rh_seq);
                     rtpObj->src.rh_time = ntohl(rtpObj->src.rh_time);
                     rtpObj->src.rh_ssrc = ntohl(rtpObj->src.rh_ssrc);

                     rtpObj->dst.rh_seq  = ntohs(rtpObj->dst.rh_seq);
                     rtpObj->dst.rh_time = ntohl(rtpObj->dst.rh_time);
                     rtpObj->dst.rh_ssrc = ntohl(rtpObj->dst.rh_ssrc);

                     rtpObj->sdrop       = ntohs(rtpObj->sdrop);
                     rtpObj->ddrop       = ntohs(rtpObj->ddrop);
                     rtpObj->ssdev       = ntohs(rtpObj->ssdev);
                     rtpObj->dsdev       = ntohs(rtpObj->dsdev);
                     break;
                  }
                  case ARGUS_RTCP_FLOW: {
                     struct ArgusRTCPObject *rtcpObj = (struct ArgusRTCPObject *)&net->net_union.rtcp;
                     rtcpObj->src.rh_len   = ntohs(rtcpObj->src.rh_len);
                     rtcpObj->src.rh_ssrc  = ntohl(rtcpObj->src.rh_ssrc);

                     rtcpObj->dst.rh_len   = ntohs(rtcpObj->dst.rh_len);
                     rtcpObj->dst.rh_ssrc  = ntohl(rtcpObj->dst.rh_ssrc);

                     rtcpObj->sdrop = ntohs(rtcpObj->sdrop);
                     rtcpObj->ddrop = ntohs(rtcpObj->ddrop);
                     break;
                  }
               }
               break;
            }

            case ARGUS_VLAN_DSR: {
               struct ArgusVlanStruct *vlan = (struct ArgusVlanStruct *) dsr;
               vlan->sid = ntohs(vlan->sid);
               vlan->did = ntohs(vlan->did);
               break;
            }

            case ARGUS_MPLS_DSR: {
               struct ArgusMplsStruct *mpls = (struct ArgusMplsStruct *) dsr;
               unsigned int *label = (unsigned int *)(dsr + 1);
               int num, i;

               if ((num = ((mpls->hdr.argus_dsrvl8.qual & 0xF0) >> 4)) > 0) {
                  for (i = 0; i < num; i++) {
                     *label = ntohl(*label);
                     label++;
                  }
               }
               if ((num = (mpls->hdr.argus_dsrvl8.qual & 0x0F)) > 0) {
                  for (i = 0; i < num; i++) {
                     *label = ntohl(*label);
                     label++;
                  }
               }
               break;
            }
             
            case ARGUS_JITTER_DSR: {
               struct ArgusJitterStruct *jit = (struct ArgusJitterStruct *) dsr;
               struct ArgusStatsObject *stat = (struct ArgusStatsObject *) (dsr + 1);
               int len = (jit->hdr.argus_dsrvl8.len - 1) * 4;

               while (len > 0) {
                  u_int32_t value;
                  stat->n        = ntohl(stat->n);
                  stat->minval   = ntohl(stat->minval);
                  bcopy((char *)&stat->meanval, (char *)&value, sizeof(value));
                  value          = ntohl(value);
                  bcopy((char *)&value, (char *)&stat->meanval, sizeof(value));
                  bcopy((char *)&stat->stdev, (char *)&value, sizeof(value));
                  value          = ntohl(value);
                  bcopy((char *)&value, (char *)&stat->stdev, sizeof(value));
                  stat->maxval   = ntohl(stat->maxval);

                  stat++;

                  len -= sizeof (*stat);
               }
               break;
            }

            case ARGUS_DATA_DSR: {
               struct ArgusDataStruct *data = (struct ArgusDataStruct *) dsr;
               data->size  = ntohs(data->size);
               data->count = ntohs(data->count);
               break;
            }
         }

         if ((cnt = ((dsr->type & 0x80) ? 1 : 
                    ((dsr->type == ARGUS_DATA_DSR) ? ntohs(dsr->argus_dsrvl16.len) :
                                                    dsr->argus_dsrvl8.len)) * 4) > 0) {
            if (dsr->type == ARGUS_DATA_DSR)
               dsr->argus_dsrvl16.len = ntohs(dsr->argus_dsrvl16.len);

            dsr = (struct ArgusDSRHeader *)((char *)dsr + cnt);

         } else
            break;
      }
   }
#endif
}


void
ArgusHtoN (struct ArgusRecord *argus)
{
#if defined(_LITTLE_ENDIAN)
   struct ArgusRecordHeader *hdr = &argus->hdr;
   struct ArgusDSRHeader *dsr = (struct ArgusDSRHeader *) (hdr + 1);

   if (argus->hdr.type & ARGUS_MAR) {
      argus->hdr.len                    = htons(argus->hdr.len);
      argus->argus_mar.status            = htonl(argus->argus_mar.status);
      argus->argus_mar.argusid            = htonl(argus->argus_mar.argusid);
      argus->argus_mar.localnet          = htonl(argus->argus_mar.localnet);
      argus->argus_mar.netmask           = htonl(argus->argus_mar.netmask);
      argus->argus_mar.nextMrSequenceNum = htonl(argus->argus_mar.nextMrSequenceNum);
      argus->argus_mar.startime.tv_sec   = htonl(argus->argus_mar.startime.tv_sec);
      argus->argus_mar.startime.tv_usec  = htonl(argus->argus_mar.startime.tv_usec);
      argus->argus_mar.now.tv_sec        = htonl(argus->argus_mar.now.tv_sec);
      argus->argus_mar.now.tv_usec       = htonl(argus->argus_mar.now.tv_usec);
      argus->argus_mar.reportInterval    = htons(argus->argus_mar.reportInterval);
      argus->argus_mar.argusMrInterval    = htons(argus->argus_mar.argusMrInterval);

      argus->argus_mar.pktsRcvd          = htonll(argus->argus_mar.pktsRcvd);
      argus->argus_mar.bytesRcvd         = htonll(argus->argus_mar.bytesRcvd);
      argus->argus_mar.drift             = htonll(argus->argus_mar.drift);

      argus->argus_mar.records           = htonl(argus->argus_mar.records);
      argus->argus_mar.flows             = htonl(argus->argus_mar.flows);
      argus->argus_mar.dropped           = htonl(argus->argus_mar.dropped);
      argus->argus_mar.queue             = htonl(argus->argus_mar.queue);
      argus->argus_mar.output            = htonl(argus->argus_mar.output);
      argus->argus_mar.clients           = htonl(argus->argus_mar.clients);
      argus->argus_mar.bufs              = htonl(argus->argus_mar.bufs);
      argus->argus_mar.bytes             = htonl(argus->argus_mar.bytes);

      argus->argus_mar.thisid            = htonl(argus->argus_mar.thisid);
      argus->argus_mar.record_len        = htonl(argus->argus_mar.record_len);
      return;
   }

   if (argus->hdr.len > 1) {
      int cnt;
      while ((char *) dsr < ((char *) argus + (hdr->len * 4))) {
         switch (dsr->type & 0x7F) {
            case ARGUS_FLOW_DSR: {
               struct ArgusFlow *flow = (struct ArgusFlow *) dsr;

               switch (flow->hdr.subtype & 0x3F) {
                  case ARGUS_FLOW_CLASSIC5TUPLE: {
                     switch (flow->hdr.argus_dsrvl8.qual & 0x7F) {
                        case ARGUS_TYPE_IPV4:
                           flow->ip_flow.ip_src = htonl(flow->ip_flow.ip_src);
                           flow->ip_flow.ip_dst = htonl(flow->ip_flow.ip_dst);
                           switch (flow->ip_flow.ip_p) {
                              case IPPROTO_TCP:
                              case IPPROTO_UDP:
                                 flow->ip_flow.sport = htons(flow->ip_flow.sport);
                                 flow->ip_flow.dport = htons(flow->ip_flow.dport);
                                 break;
                              case IPPROTO_ESP:
                                 flow->esp_flow.spi = htonl(flow->esp_flow.spi);
                                 break;
                              case IPPROTO_IGMP:
                                 flow->igmp_flow.ip_id = htons(flow->igmp_flow.ip_id);
                                 break;
                           }
                           break; 

                        case ARGUS_TYPE_IPV6: {
                           unsigned int *iptr = (unsigned int *)&flow->ipv6_flow;
                           if (flow->hdr.argus_dsrvl8.qual & ARGUS_FRAGMENT) {
                              flow->fragv6_flow.ip_id = htonl(flow->fragv6_flow.ip_id);
                           } else {
                              switch (flow->ipv6_flow.ip_p) {
                                 case IPPROTO_TCP:
                                 case IPPROTO_UDP:
                                    flow->ipv6_flow.sport = htons(flow->ipv6_flow.sport);
                                    flow->ipv6_flow.dport = htons(flow->ipv6_flow.dport);
                                    break;
                                 case IPPROTO_ESP:
                                    flow->esp6_flow.spi = htonl(flow->esp6_flow.spi);
                                    break;
                              }
                           } 
                           iptr[8] = htonl(iptr[8]);
                           break; 
                        }

                        case ARGUS_TYPE_ETHER: {
                           struct ArgusMacFlow *mac = (struct ArgusMacFlow *) &flow->mac_flow;
                           mac->ehdr.ether_type = htons(mac->ehdr.ether_type);
                           break;
                        }
                     
                        case ARGUS_TYPE_RARP: {
                           struct ArgusRarpFlow *rarp = (struct ArgusRarpFlow *) &flow->rarp_flow;
                           rarp->arp_tpa = htonl(rarp->arp_tpa);
                           break;
                        }
                     
                        case ARGUS_TYPE_ARP: {
                           struct ArgusArpFlow *arp = (struct ArgusArpFlow *) &flow->arp_flow;
                           arp->arp_spa = htonl(arp->arp_spa);
                           arp->arp_tpa = htonl(arp->arp_tpa);
                           break;
                        } 
                     }
                     break; 
                  }
               }
               break;
            }

            case ARGUS_IPATTR_DSR: {
               struct ArgusIPAttrStruct *attr = (struct ArgusIPAttrStruct *) dsr;
               unsigned int *dsrptr = (unsigned int *)(dsr + 1);

               if (attr->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_SRC) {
                  struct ArgusIPAttrObject *aobj = (struct ArgusIPAttrObject *) dsrptr;
                  aobj->ip_id = htons(aobj->ip_id);
                  dsrptr++;
               }
               if (attr->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_SRC_OPTIONS) {
                  *dsrptr = htonl(*dsrptr);
                  dsrptr++;
               }
               if (attr->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_DST) {
                  struct ArgusIPAttrObject *aobj = (struct ArgusIPAttrObject *) dsrptr;
                  aobj->ip_id = htons(aobj->ip_id);
                  dsrptr++;
               }
               if (attr->hdr.argus_dsrvl8.qual & ARGUS_IPATTR_DST_OPTIONS) {
                  *dsrptr = htonl(*dsrptr);
                  dsrptr++;
               }
               break;
            }

            case ARGUS_TRANSPORT_DSR: {
               struct ArgusTransportStruct *trans = (struct ArgusTransportStruct *) dsr;

               switch (trans->hdr.subtype) {
                  case ARGUS_SRCSEQ:
                     trans->seqnum = htonl(trans->seqnum);
                  case ARGUS_SRCID:
                     trans->srcid.value = htonl(trans->srcid.value);
                     break;
               }
               break;
            }

            case ARGUS_TIME_DSR: {
               unsigned int x, *dtime = (unsigned int *) dsr; 
 
               for (x = 1; x < dsr->argus_dsrvl8.len; x++)
                  dtime[x] = htonl(dtime[x]);
               break; 
            }

            case ARGUS_METER_DSR: {
               if (dsr->subtype & ARGUS_METER_PKTS_BYTES) {
                  switch (dsr->argus_dsrvl8.qual & 0x0F) {
                     case ARGUS_SRCDST_BYTE:
                        break;
                     case ARGUS_SRCDST_SHORT:
                        ((unsigned short *)(dsr + 1))[0] = htons(((unsigned short *)(dsr + 1))[0]);
                        ((unsigned short *)(dsr + 1))[1] = htons(((unsigned short *)(dsr + 1))[1]);
                        ((unsigned short *)(dsr + 1))[2] = htons(((unsigned short *)(dsr + 1))[2]);
                        ((unsigned short *)(dsr + 1))[3] = htons(((unsigned short *)(dsr + 1))[3]);
                        break;
                     case ARGUS_SRCDST_INT:
                        ((unsigned int *)(dsr + 1))[0] = htonl(((unsigned int *)(dsr + 1))[0]);
                        ((unsigned int *)(dsr + 1))[1] = htonl(((unsigned int *)(dsr + 1))[1]);
                        ((unsigned int *)(dsr + 1))[2] = htonl(((unsigned int *)(dsr + 1))[2]);
                        ((unsigned int *)(dsr + 1))[3] = htonl(((unsigned int *)(dsr + 1))[3]);
                        break;
                     case ARGUS_SRCDST_LONGLONG:
                        break;
                     case ARGUS_SRC_SHORT:
                        ((unsigned short *)(dsr + 1))[0] = htons(((unsigned short *)(dsr + 1))[0]);
                        ((unsigned short *)(dsr + 1))[1] = htons(((unsigned short *)(dsr + 1))[1]);
                        break;
                     case ARGUS_SRC_INT:
                        ((unsigned int *)(dsr + 1))[0] = htonl(((unsigned int *)(dsr + 1))[0]);
                        ((unsigned int *)(dsr + 1))[1] = htonl(((unsigned int *)(dsr + 1))[1]);
                        break;
                     case ARGUS_SRC_LONGLONG:
                        break;
                     case ARGUS_DST_SHORT:
                        ((unsigned short *)(dsr + 1))[0] = htons(((unsigned short *)(dsr + 1))[0]);
                        ((unsigned short *)(dsr + 1))[1] = htons(((unsigned short *)(dsr + 1))[1]);
                        break;
                     case ARGUS_DST_INT:
                        ((unsigned int *)(dsr + 1))[0] = htonl(((unsigned int *)(dsr + 1))[0]);
                        ((unsigned int *)(dsr + 1))[1] = htonl(((unsigned int *)(dsr + 1))[1]);
                        break;
                     case ARGUS_DST_LONGLONG:
                        break;

                  }
               } else 
               if (dsr->subtype & ARGUS_METER_PKTS_BYTES_APP) {
                  switch (dsr->argus_dsrvl8.qual & 0x0F) {
                     case ARGUS_SRCDST_BYTE:
                     case ARGUS_SRC_BYTE:
                     case ARGUS_DST_BYTE:
                        break;
                     case ARGUS_SRCDST_SHORT:
                        ((unsigned short *)(dsr + 1))[0] = htons(((unsigned short *)(dsr + 1))[0]);
                        ((unsigned short *)(dsr + 1))[1] = htons(((unsigned short *)(dsr + 1))[1]);
                        ((unsigned short *)(dsr + 1))[2] = htons(((unsigned short *)(dsr + 1))[2]);
                        ((unsigned short *)(dsr + 1))[3] = htons(((unsigned short *)(dsr + 1))[3]);
                        ((unsigned short *)(dsr + 1))[4] = htons(((unsigned short *)(dsr + 1))[4]);
                        ((unsigned short *)(dsr + 1))[5] = htons(((unsigned short *)(dsr + 1))[5]);
                        break;
                     case ARGUS_SRCDST_INT:
                        ((unsigned int *)(dsr + 1))[0] = htonl(((unsigned int *)(dsr + 1))[0]);
                        ((unsigned int *)(dsr + 1))[1] = htonl(((unsigned int *)(dsr + 1))[1]);
                        ((unsigned int *)(dsr + 1))[2] = htonl(((unsigned int *)(dsr + 1))[2]);
                        ((unsigned int *)(dsr + 1))[3] = htonl(((unsigned int *)(dsr + 1))[3]);
                        ((unsigned int *)(dsr + 1))[4] = htonl(((unsigned int *)(dsr + 1))[4]);
                        ((unsigned int *)(dsr + 1))[5] = htonl(((unsigned int *)(dsr + 1))[5]);
                        break;
                     case ARGUS_SRC_SHORT:
                        ((unsigned short *)(dsr + 1))[0] = htons(((unsigned short *)(dsr + 1))[0]);
                        ((unsigned short *)(dsr + 1))[1] = htons(((unsigned short *)(dsr + 1))[1]);
                        ((unsigned short *)(dsr + 1))[2] = htons(((unsigned short *)(dsr + 1))[2]);
                        break;
                     case ARGUS_SRC_INT:
                        ((unsigned int *)(dsr + 1))[0] = htonl(((unsigned int *)(dsr + 1))[0]);
                        ((unsigned int *)(dsr + 1))[1] = htonl(((unsigned int *)(dsr + 1))[1]);
                        ((unsigned int *)(dsr + 1))[2] = htonl(((unsigned int *)(dsr + 1))[2]);
                        break;
                     case ARGUS_DST_SHORT:
                        ((unsigned short *)(dsr + 1))[0] = htons(((unsigned short *)(dsr + 1))[0]);
                        ((unsigned short *)(dsr + 1))[1] = htons(((unsigned short *)(dsr + 1))[1]);
                        ((unsigned short *)(dsr + 1))[2] = htons(((unsigned short *)(dsr + 1))[2]);
                        break;
                     case ARGUS_DST_INT:
                        ((unsigned int *)(dsr + 1))[0] = htonl(((unsigned int *)(dsr + 1))[0]);
                        ((unsigned int *)(dsr + 1))[1] = htonl(((unsigned int *)(dsr + 1))[1]);
                        ((unsigned int *)(dsr + 1))[2] = htonl(((unsigned int *)(dsr + 1))[2]);
                        break;
                  }
               }
               break;
            }

            case ARGUS_NETWORK_DSR: {
               struct ArgusNetworkStruct *net = (struct ArgusNetworkStruct *)dsr;
               switch (net->hdr.subtype) {
                  case ARGUS_METER_TCP_PERF: {
                     struct ArgusTCPObject *tcp = (struct ArgusTCPObject *)&net->net_union.tcp;
                     tcp->status       = htonl(tcp->status);
                     tcp->state        = htonl(tcp->state);
                     tcp->options      = htonl(tcp->options);
                     tcp->synAckuSecs  = htonl(tcp->synAckuSecs);
                     tcp->ackDatauSecs = htonl(tcp->ackDatauSecs);

                     tcp->src.lasttime.tv_sec  = htonl(tcp->src.lasttime.tv_sec);
                     tcp->src.lasttime.tv_usec = htonl(tcp->src.lasttime.tv_usec);
                     tcp->src.status = htonl(tcp->src.status);
                     tcp->src.seqbase = htonl(tcp->src.seqbase);
                     tcp->src.seq = htonl(tcp->src.seq);
                     tcp->src.ack = htonl(tcp->src.ack);
                     tcp->src.winnum = htonl(tcp->src.winnum);
                     tcp->src.bytes = htonl(tcp->src.bytes);
                     tcp->src.retrans = htonl(tcp->src.retrans);
                     tcp->src.ackbytes = htonl(tcp->src.ackbytes);
                     tcp->src.state = htons(tcp->src.state);
                     tcp->src.win = htons(tcp->src.win);
                     tcp->src.winbytes = htons(tcp->src.winbytes);

                     if (dsr->argus_dsrvl8.len > (((sizeof(struct ArgusTCPObject) - sizeof(struct ArgusTCPObjectMetrics))+3)/4 + 1)) {
                        tcp->dst.lasttime.tv_sec  = htonl(tcp->dst.lasttime.tv_sec);
                        tcp->dst.lasttime.tv_usec = htonl(tcp->dst.lasttime.tv_usec);
                        tcp->dst.status = htonl(tcp->dst.status);
                        tcp->dst.seqbase = htonl(tcp->dst.seqbase);
                        tcp->dst.seq = htonl(tcp->dst.seq);
                        tcp->dst.ack = htonl(tcp->dst.ack);
                        tcp->dst.winnum = htonl(tcp->dst.winnum);
                        tcp->dst.bytes = htonl(tcp->dst.bytes);
                        tcp->dst.retrans = htonl(tcp->dst.retrans);
                        tcp->dst.ackbytes = htonl(tcp->dst.ackbytes);
                        tcp->dst.state = htons(tcp->dst.state);
                        tcp->dst.win = htons(tcp->dst.win);
                        tcp->dst.winbytes = htons(tcp->dst.winbytes);
                     }
                     break;
                  }

                  case ARGUS_ESP_DSR: {
                     struct ArgusESPObject *espObj = (struct ArgusESPObject *)&net->net_union.esp;
                     espObj->status  = htonl(espObj->status);
                     espObj->spi     = htonl(espObj->spi);
                     espObj->lastseq = htonl(espObj->lastseq);
                     espObj->lostseq = htonl(espObj->lostseq);
                     break;
                  }
                  case ARGUS_RTP_FLOW: {
                     struct ArgusRTPObject *rtpObj = (struct ArgusRTPObject *)&net->net_union.rtp;
                     rtpObj->state       = htonl(rtpObj->state);
                     rtpObj->src.rh_seq  = htons(rtpObj->src.rh_seq);
                     rtpObj->src.rh_time = htonl(rtpObj->src.rh_time);
                     rtpObj->src.rh_ssrc = htonl(rtpObj->src.rh_ssrc);

                     rtpObj->dst.rh_seq  = htons(rtpObj->dst.rh_seq);
                     rtpObj->dst.rh_time = htonl(rtpObj->dst.rh_time);
                     rtpObj->dst.rh_ssrc = htonl(rtpObj->dst.rh_ssrc);

                     rtpObj->sdrop       = htons(rtpObj->sdrop);
                     rtpObj->ddrop       = htons(rtpObj->ddrop);
                     rtpObj->ssdev       = htons(rtpObj->ssdev);
                     rtpObj->dsdev       = htons(rtpObj->dsdev);
                     break;
                  }
                  case ARGUS_RTCP_FLOW: {
                     struct ArgusRTCPObject *rtcpObj = (struct ArgusRTCPObject *)&net->net_union.rtcp;
                     rtcpObj->src.rh_len   = htons(rtcpObj->src.rh_len);
                     rtcpObj->src.rh_ssrc  = htonl(rtcpObj->src.rh_ssrc);

                     rtcpObj->dst.rh_len   = htons(rtcpObj->dst.rh_len);
                     rtcpObj->dst.rh_ssrc  = htonl(rtcpObj->dst.rh_ssrc);

                     rtcpObj->sdrop = htons(rtcpObj->sdrop);
                     rtcpObj->ddrop = htons(rtcpObj->ddrop);
                     break;
                  }
               }
               break;
            }

            case ARGUS_VLAN_DSR: {
               struct ArgusVlanStruct *vlan = (struct ArgusVlanStruct *) dsr;
               vlan->sid = htons(vlan->sid);
               vlan->did = htons(vlan->did);
               break;
            }

            case ARGUS_MPLS_DSR: {
               struct ArgusMplsStruct *mpls = (struct ArgusMplsStruct *) dsr;
               unsigned int *label = (unsigned int *)(dsr + 1);
               int num, i;

               if ((num = ((mpls->hdr.argus_dsrvl8.qual & 0xF0) >> 4)) > 0) {
                  for (i = 0; i < num; i++) {
                     *label = htonl(*label);
                     label++;
                  }
               }
               if ((num = (mpls->hdr.argus_dsrvl8.qual & 0x0F)) > 0) {
                  for (i = 0; i < num; i++) {
                     *label = htonl(*label);
                     label++;
                  }
               }
               break;
            }

            case ARGUS_JITTER_DSR: {
               struct ArgusJitterStruct *jit = (struct ArgusJitterStruct *) dsr;
               struct ArgusStatsObject *stat = (struct ArgusStatsObject *) (dsr + 1);
               int len = (jit->hdr.argus_dsrvl8.len - 1) * 4;

               while (len > 0) {
                  u_int32_t value;
                  stat->n        = htonl(stat->n);
                  stat->minval   = htonl(stat->minval);
                  bcopy((char *)&stat->meanval, (char *)&value, sizeof(value));
                  value          = htonl(value);
                  bcopy((char *)&value, (char *)&stat->meanval, sizeof(value));
                  bcopy((char *)&stat->stdev, (char *)&value, sizeof(value));
                  value          = htonl(value);
                  bcopy((char *)&value, (char *)&stat->stdev, sizeof(value));
                  stat->maxval   = htonl(stat->maxval);
                  stat++;
                  len -= sizeof (*stat);
               }
               break;
            }

            case ARGUS_DATA_DSR: {
               struct ArgusDataStruct *data = (struct ArgusDataStruct *) dsr;
               data->size  = htons(data->size);
               data->count = htons(data->count);
               break;
            }
         }

         if ((cnt = ((dsr->type & 0x80) ? 1 : 
                    ((dsr->type == ARGUS_DATA_DSR) ? dsr->argus_dsrvl16.len :
                                                    dsr->argus_dsrvl8.len)) * 4) > 0) {
            if (dsr->type == ARGUS_DATA_DSR)
               dsr->argus_dsrvl16.len = htons(dsr->argus_dsrvl16.len);

            dsr = (struct ArgusDSRHeader *)((char *)dsr + cnt);

         } else
            break;
      }
   }

   hdr->len = htons(hdr->len);
#endif
}


void
ArgusPrintHex (const u_char *bp, u_int length)
{
   const u_short *sp;
   u_int i;
   int nshorts;

   sp = (u_short *)bp;
   nshorts = (u_int) length / sizeof(u_short);
   i = 0;
   while (--nshorts >= 0) {
      if ((i++ % 8) == 0) {
         (void)printf("\n\t");
      }
      (void)printf(" %04x", ntohs(*sp++));
   }

   if (length & 1) {
      if ((i % 8) == 0)
         (void)printf("\n\t");

      (void)printf(" %02x", *(u_char *)sp);
   }
   (void)printf("\n");
   fflush(stdout);
}


struct ArgusMemoryHeader {
   struct ArgusMemoryHeader *nxt, *prv;
#if defined(__GNUC__)
   void *frame[4];
#endif
   unsigned int tag;
   unsigned short len;
   unsigned short offset;
};

struct ArgusMemoryList {
   struct ArgusMemoryHeader *start, *end;
   int total, count, size;
};

int ArgusAllocTotal = 0;
int ArgusAllocBytes = 0;
int ArgusFreeTotal = 0;

struct ArgusMemoryList memory = {NULL, 0};

#define ARGUS_ALLOC	0x45672381
/*
#define ARGUS_ALIGN	128
*/

void *     
ArgusMalloc (int bytes) 
{          
   void *retn = NULL; 
   int offset;
 
   if (bytes) {
      ArgusAllocTotal++;
      ArgusAllocBytes += bytes;

#if defined(ARGUS_ALIGN)
      offset = ARGUS_ALIGN;
#else
      offset = 0;
#endif

#if !defined(ARGUSMEMDEBUG)
      retn = (void *) malloc (bytes + offset);
#else
      if ((retn = (u_int *) malloc (bytes + sizeof(struct ArgusMemoryHeader) + offset)) != NULL) {
         struct ArgusMemoryHeader *mem = (struct ArgusMemoryHeader *)retn;
         mem->tag = ARGUS_ALLOC;
         mem->len = bytes;
         mem->offset = offset;
#if defined(__GNUC__)
         mem->frame[0] = __builtin_return_address(0);
         mem->frame[1] = __builtin_return_address(1);
         mem->frame[2] = __builtin_return_address(2);
#endif
         if (memory.start) {
            mem->nxt = memory.start;
            mem->prv = memory.start->prv;
            mem->prv->nxt = mem;
            mem->nxt->prv = mem;
            memory.end = mem;
         } else {
            memory.start = mem;
            memory.end = mem;
            mem->nxt = mem;
            mem->prv = mem;
         }
         memory.count++;
         memory.total++;
         retn = (void *)(mem + 1);
      }
#endif

#if defined(ARGUS_ALIGN)
      if (retn != NULL) {
         unsigned short toff;
         toff = ((unsigned long)retn & (offset - 1));
         toff = offset - toff;
         retn = (void *)((char *)retn + toff);
         ((unsigned short *)retn)[-1] = toff;
      }
#endif
   }
#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusMalloc (%d) returning 0x%x\n", bytes, retn); 
#endif
   return (retn); 
}

void *
ArgusCalloc (int nitems, int bytes)
{
   int offset, total = nitems * bytes;
   void *retn = NULL;

   if (total) {
      ArgusAllocTotal++;
      ArgusAllocBytes += total;

#if defined(ARGUS_ALIGN)
      offset = ARGUS_ALIGN;
#else
      offset = 0;
#endif

#if !defined(ARGUSMEMDEBUG)
      retn = calloc (1, total + offset);
#else
      if ((retn = calloc (1, total + sizeof(struct ArgusMemoryHeader) + offset)) != NULL) {
         struct ArgusMemoryHeader *mem = retn;
         mem->tag = ARGUS_ALLOC;
         mem->len = total;
         mem->offset = offset;
#if defined(__GNUC__)
         mem->frame[0] = __builtin_return_address(0);
         mem->frame[1] = __builtin_return_address(1);
         mem->frame[2] = __builtin_return_address(2);
#endif
         if (memory.start) {
            mem->nxt = memory.start;
            mem->prv = memory.start->prv;
            mem->prv->nxt = mem;
            mem->nxt->prv = mem;
            memory.end = mem;
         } else {
            memory.start = mem;
            memory.end = mem;
            mem->nxt = mem;
            mem->prv = mem;
         }
         memory.total++;
         memory.count++;
         retn = (void *)(mem + 1);
      }
#endif

#if defined(ARGUS_ALIGN)
      if (retn != NULL) {
         unsigned short toff;
         toff = ((unsigned long)retn & (offset - 1));
         toff = offset - toff;
         retn = (void *)((char *)retn + toff);
         ((unsigned short *)retn)[-1] = toff;
      }
#endif
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusCalloc (%d, %d) returning 0x%x\n", nitems, bytes, retn);
#endif
   return (retn);
}


void
ArgusFree (void *buf)
{
   void *ptr = buf;

   if (ptr) {
      ArgusFreeTotal++;
#if defined(ARGUSMEMDEBUG)
      {
         struct ArgusMemoryHeader *mem = ptr;
#if defined(ARGUS_ALIGN)
         unsigned short offset = ((unsigned short *)mem)[-1];
         mem = (void *)((char *)mem - offset);
#endif
         mem--;
         if (mem->tag != ARGUS_ALLOC)
            ArgusLog (LOG_ERR, "ArgusFree: buffer error 0x%x", ptr);
         if (memory.count == 1) {
            memory.start = NULL;
            memory.end = NULL;
         } else {
            mem->prv->nxt = mem->nxt;
            mem->nxt->prv = mem->prv;
            if (mem == memory.start) {
               memory.start = mem->nxt;
            } else if (mem == memory.end) {
               memory.end = mem->prv;
            }
         }
         ArgusAllocBytes -= mem->len;
         memory.count--;
         ptr = mem;
      }
#else
#if defined(ARGUS_ALIGN)
      {
         unsigned short offset;
         if ((offset = ((unsigned short *)ptr)[-1]) > 0)
            ptr = (void *)((char *)ptr - offset);
      }
#endif
#endif
      free (ptr);
   }
#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusFree (0x%x)\n", buf);
#endif
}


#define ARGUS_MEMORY_MAX		1000000
#define ARGUS_MEMORY_HI_THRESH	2000
#define ARGUS_MEMORY_LOW_THRESH	20

struct ArgusMemoryList *ArgusMallocList = NULL;

void
ArgusInitMallocList (int length)
{
   struct ArgusMemoryHeader *mem;

   if (ArgusMallocList != NULL) {
      if (length == ArgusMallocList->size)
         return;
      else
         ArgusLog(LOG_ERR, "ArgusInitMallocList called with multiple sizes");
   }
   
   if ((ArgusMallocList = (struct ArgusMemoryList *) ArgusCalloc(1, sizeof(*ArgusMallocList))) == NULL)
         ArgusLog(LOG_ERR, "ArgusMallocListRecord ArgusCalloc %s", strerror(errno));

   ArgusMallocList->size = length;

   while ((ArgusMallocList->count < ARGUS_MEMORY_LOW_THRESH) && (ArgusMallocList->total < ARGUS_MEMORY_MAX)) {
      if ((mem = (struct ArgusMemoryHeader *) ArgusCalloc (1, length)) != NULL) {
         if (ArgusMallocList->end) {
            ArgusMallocList->end->nxt = mem;
         } else {
            ArgusMallocList->start = mem;
            ArgusMallocList->count = 0;
         }
         ArgusMallocList->end = mem;
         ArgusMallocList->count++;
         ArgusMallocList->total++;
      }
   }

#ifdef ARGUSDEBUG 
   ArgusDebug (6, "ArgusInitMallocList (%d) returning 0x%x\n", length);
#endif
   return;
}

void
ArgusDeleteMallocList (void)
{
   struct ArgusMemoryHeader *crt, *rel;

   if (ArgusMallocList != NULL) {
      crt = ArgusMallocList->start;
      while (crt != NULL) {
         rel = crt;
         crt = crt->nxt;
         ArgusFree(rel);
      }

      ArgusFree(ArgusMallocList);
      ArgusMallocList = NULL;
   }
}

void *
ArgusMallocListRecord (int length)
{
   void *retn = NULL;
   struct ArgusMemoryHeader *mem;

   if (ArgusMallocList == NULL) {
      if ((ArgusMallocList = (struct ArgusMemoryList *) ArgusCalloc(1, sizeof(*ArgusMallocList))) == NULL)
            ArgusLog(LOG_ERR, "ArgusMallocListRecord ArgusCalloc %s", strerror(errno));

      ArgusMallocList->size = length;
   }

   if (length == ArgusMallocList->size) {
      while ((ArgusMallocList->count < ARGUS_MEMORY_LOW_THRESH) && (ArgusMallocList->total < ARGUS_MEMORY_MAX)) {
         if ((mem = (struct ArgusMemoryHeader *) ArgusCalloc (1, length)) != NULL) {
            if (ArgusMallocList->end) {
               ArgusMallocList->end->nxt = mem;
            } else {
               ArgusMallocList->start = mem;
               ArgusMallocList->count = 0;
            }
            ArgusMallocList->end = mem;
            ArgusMallocList->count++;
            ArgusMallocList->total++;
         }
      }

      if (ArgusMallocList->count > 0) {
         if ((retn = ArgusMallocList->start) != NULL)
            ArgusMallocList->start = ((struct ArgusMemoryHeader *)retn)->nxt;

         if (ArgusMallocList->start == NULL) {
            ArgusMallocList->end = NULL;
            ArgusMallocList->count = 0;

         } else
            ArgusMallocList->count--;
      }
   }

#ifdef ARGUSDEBUG 
   ArgusDebug (6, "ArgusMallocListRecord (%d) returning 0x%x\n", length, retn);
#endif
   return (retn);
}

void
ArgusFreeListRecord (void *buf)
{
   struct ArgusMemoryHeader *mem = (struct ArgusMemoryHeader *)buf;
   struct ArgusRecordStruct *rec = buf;

   if (rec->dsrs[ARGUS_SRCUSERDATA_INDEX] != NULL) {
      ArgusFree(rec->dsrs[ARGUS_SRCUSERDATA_INDEX]);
      rec->dsrs[ARGUS_SRCUSERDATA_INDEX] = NULL;
   }

   if (rec->dsrs[ARGUS_DSTUSERDATA_INDEX] != NULL) {
      ArgusFree(rec->dsrs[ARGUS_DSTUSERDATA_INDEX]);
      rec->dsrs[ARGUS_DSTUSERDATA_INDEX] = NULL;
   }

   if (ArgusMallocList == NULL) {
      ArgusMallocList->total--;
      ArgusFree(rec);

   } else {
      if (ArgusMallocList->count < ARGUS_MEMORY_HI_THRESH) {
         mem->nxt = NULL;
         if (ArgusMallocList->end != NULL)
            ArgusMallocList->end->nxt = mem;

         ArgusMallocList->end = mem;

         if (ArgusMallocList->start == NULL)
            ArgusMallocList->start = mem;

         ArgusMallocList->count++;

      } else {
         ArgusMallocList->total--;
         ArgusFree(rec);
      }
   }
   
#ifdef ARGUSDEBUG 
   ArgusDebug (6, "ArgusFreeListRecord (0x%x) returning\n", buf);
#endif
   return;
}

extern void ArgusShutDown (int);

#include <syslog.h>

struct ArgusLogPriorityStruct {
   int priority;
   char *label;
};

#define ARGUSPRIORITYSTR	8
struct ArgusLogPriorityStruct ArgusPriorityStr[ARGUSPRIORITYSTR] =
{
   { LOG_EMERG,   "ArgusEmergency" },
   { LOG_ALERT,   "    ArgusAlert" },
   { LOG_CRIT,    " ArgusCritical" },
   { LOG_ERR,     "    ArgusError" },
   { LOG_WARNING, "  ArgusWarning" },
   { LOG_NOTICE,  "   ArgusNotice" },
   { LOG_INFO,    "     ArgusInfo" },
   { LOG_DEBUG,   "    ArgusDebug" },
};


#include <sys/time.h>

void
ArgusLog (int priority, char *fmt, ...)
{
   va_list ap;
   char buf[1024], *ptr = buf;
   struct timeval now;

#ifdef ARGUS_SYSLOG
   gettimeofday (&now, 0L);
   (void) snprintf (buf, 1024, "%s ", print_time(&now));
   ptr = &buf[strlen(buf)];
#else
   int i;
   char *label;

   if (priority == LOG_NOTICE)
      return;

   (void) snprintf (buf, 1024, "%s[%d]: %s ", ArgusProgramName, (int)getpid(), print_time(&now));
   ptr = &buf[strlen(buf)];
#endif

   va_start (ap, fmt);
   (void) vsnprintf (ptr, 1024, fmt, ap);
   ptr = &buf[strlen(buf)];
   va_end (ap);

   if (*fmt) {
      fmt += (int) strlen (fmt);
      if (fmt[-1] != '\n')
         snprintf (ptr, 2, "\n");
   }

#ifdef ARGUS_SYSLOG
   syslog (priority, buf, strlen(buf));
#else

   for (i = 0; i < ARGUSPRIORITYSTR; i++)
      if (ArgusPriorityStr[i].priority == priority) {
         label = ArgusPriorityStr[i].label;
         break;
      }

   fprintf (stderr, "%s: %s", label, buf);
#endif

   switch (priority) {
      case LOG_ERR: ArgusShutDown(1); break;
      default: break;
   }
}


void ArgusRecordDump (struct ArgusRecord *);
void ArgusDump (const u_char *, int);

#define HEXDUMP_BYTES_PER_LINE 16
#define HEXDUMP_SHORTS_PER_LINE (HEXDUMP_BYTES_PER_LINE / 2)
#define HEXDUMP_HEXSTUFF_PER_SHORT 5 /* 4 hex digits and a space */
#define HEXDUMP_HEXSTUFF_PER_LINE \
                (HEXDUMP_HEXSTUFF_PER_SHORT * HEXDUMP_SHORTS_PER_LINE)


void
ArgusRecordDump (struct ArgusRecord *argus)
{
   int length = argus->hdr.len;
   const u_char *cp = (const u_char *) argus;

   ArgusDump (cp, length);
}


#include <ctype.h>

void
ArgusDump (const u_char *cp, int length)
{
   u_int oset = 0;
   register u_int i;
   register int s1, s2;
   register int nshorts;
   char hexstuff[HEXDUMP_SHORTS_PER_LINE*HEXDUMP_HEXSTUFF_PER_SHORT+1], *hsp;
   char asciistuff[HEXDUMP_BYTES_PER_LINE+1], *asp;

#ifdef ARGUSDEBUG
   ArgusDebug (2, "ArgusDump (0x%x, %d)\n", cp, length);
#endif

   nshorts = length / sizeof(u_short);
   i = 0;
   hsp = hexstuff; asp = asciistuff;
   while (--nshorts >= 0) {
           s1 = *cp++;
           s2 = *cp++;
           (void)snprintf(hsp, sizeof(hexstuff) - (hsp - hexstuff),
               " %02x%02x", s1, s2);
           hsp += HEXDUMP_HEXSTUFF_PER_SHORT;
           *(asp++) = (isgraph(s1) ? s1 : '.');
           *(asp++) = (isgraph(s2) ? s2 : '.');
           if (++i >= HEXDUMP_SHORTS_PER_LINE) {
               *hsp = *asp = '\0';
               (void)printf("\n0x%04x\t%-*s\t%s",
                            oset, HEXDUMP_HEXSTUFF_PER_LINE,
                            hexstuff, asciistuff);
               i = 0; hsp = hexstuff; asp = asciistuff;
               oset += HEXDUMP_BYTES_PER_LINE;
           }
   }
   if (length & 1) {
      s1 = *cp++;
      (void)snprintf(hsp, sizeof(hexstuff) - (hsp - hexstuff),
               " %02x", s1);
           hsp += 3;
           *(asp++) = (isgraph(s1) ? s1 : '.');
           ++i;
   }
   if (i > 0) {
           *hsp = *asp = '\0';
           (void)printf("\n0x%04x\t%-*s\t%s",
                        oset, HEXDUMP_HEXSTUFF_PER_LINE,
                        hexstuff, asciistuff);
   }
}


#include <stdarg.h>

extern char *ArgusProgramName;
extern int uflag, pflag;

char ArgusPrintTimeBuf[64];

char *
print_time(struct timeval *tvp)
{
   char timeZoneBuf[32];
   char *retn = ArgusPrintTimeBuf, *ptr;
   struct tm *tm, tmbuf;

   bzero (timeZoneBuf, sizeof(timeZoneBuf));
   bzero (ArgusPrintTimeBuf, sizeof(ArgusPrintTimeBuf));

   if ((tm = localtime_r ((time_t *)&tvp->tv_sec, &tmbuf)) != NULL) {
      if (uflag)
         snprintf (retn, 32, "%9d", (int) tvp->tv_sec);
      else
         strftime ((char *) retn, 64, RaTimeFormat, tm);

      if (pflag) {
         ptr = &retn[strlen(retn)];
         snprintf (ptr, 32, ".%06d", (int) tvp->tv_usec);
         ptr[pflag + 1] = '\0';

      }
   } else
      retn = NULL;

   return (retn);
}

#define ArgusAddrtoName

#include <netdb.h>
#include <argus_namedb.h>
#include <ethernames.h>

/* Find the hash node that corresponds the ether address 'ep'. */

static inline struct enamemem *
lookup_emem(struct ArgusParserStruct *parser, const u_char *ep)
{
   u_int i, j, k;
   struct enamemem *tp;

   k = (ep[0] << 8) | ep[1];
   j = (ep[2] << 8) | ep[3];
   i = (ep[4] << 8) | ep[5];

   tp = &parser->enametable[(i ^ j) % (HASHNAMESIZE-1)];
   while (tp->e_nxt)
      if (tp->e_addr0 == i &&
          tp->e_addr1 == j &&
          tp->e_addr2 == k)
         return tp;
      else
         tp = tp->e_nxt;
   tp->e_addr0 = i;
   tp->e_addr1 = j;
   tp->e_addr2 = k;
   tp->e_nxt = (struct enamemem *)calloc(1, sizeof(*tp));

   return tp;
}

/* Find the hash node that corresponds the NSAP 'nsap'. */

static inline struct enamemem *
lookup_nsap(struct ArgusParserStruct *parser, const u_char *nsap)
{
   u_int i, j, k;
   int nlen = *nsap;
   struct enamemem *tp;
   const u_char *ensap = nsap + nlen - 6;

   if (nlen > 6) {
      k = (ensap[0] << 8) | ensap[1];
      j = (ensap[2] << 8) | ensap[3];
      i = (ensap[4] << 8) | ensap[5];
   }
   else
      i = j = k = 0;

   tp = &parser->nsaptable[(i ^ j) % (HASHNAMESIZE-1)];
   while (tp->e_nxt)
      if (tp->e_addr0 == i &&
          tp->e_addr1 == j &&
          tp->e_addr2 == k &&
          tp->e_nsap[0] == nlen &&
          bcmp((char *)&(nsap[1]),
         (char *)&(tp->e_nsap[1]), nlen) == 0)
         return tp;
      else
         tp = tp->e_nxt;
   tp->e_addr0 = i;
   tp->e_addr1 = j;
   tp->e_addr2 = k;
   tp->e_nsap = (u_char *) calloc(1, nlen + 1);
   bcopy(nsap, tp->e_nsap, nlen + 1);
   tp->e_nxt = (struct enamemem *)calloc(1, sizeof(*tp));

   return tp;
}

/* Find the hash node that corresponds the protoid 'pi'. */

static inline struct protoidmem *
lookup_protoid(struct ArgusParserStruct *parser, const u_char *pi)
{
   u_int i, j;
   struct protoidmem *tp;

   /* 5 octets won't be aligned */
   i = (((pi[0] << 8) + pi[1]) << 8) + pi[2];
   j =   (pi[3] << 8) + pi[4];
   /* XXX should be endian-insensitive, but do big-endian testing  XXX */

   tp = &parser->protoidtable[(i ^ j) % (HASHNAMESIZE-1)];
   while (tp->p_nxt)
      if (tp->p_oui == i && tp->p_proto == j)
         return tp;
      else
         tp = tp->p_nxt;
   tp->p_oui = i;
   tp->p_proto = j;
   tp->p_nxt = (struct protoidmem *)calloc(1, sizeof(*tp));

   return tp;
}


void
ArgusInitServarray(struct ArgusParserStruct *parser)
{
#if !defined(CYGWIN)
   struct servent *sv;
   struct hnamemem *table;
   int i;

   while ((sv = getservent()) != NULL) {
      int port = ntohs(sv->s_port);
      i = port % (HASHNAMESIZE-1);
      if (strcmp(sv->s_proto, "tcp") == 0)
         table = &parser->tporttable[i];
      else if (strcmp(sv->s_proto, "udp") == 0)
         table = &parser->uporttable[i];
      else
         continue;

      while (table->name)
         table = table->nxt;
      if (parser->nflag > 1) {
         char buf[32];

         (void)snprintf(buf, 32, "%d", port);
         table->name = strdup(buf);
      } else
         table->name = strdup(sv->s_name);
      table->addr = port;
      table->nxt = (struct hnamemem *)calloc(1, sizeof(*table));
   }
   endservent();
#endif
}


void
ArgusInitEprotoarray(struct ArgusParserStruct *parser)
{
   struct ArgusEtherTypeStruct *p = argus_ethertype_names;

   bzero ((char *)parser->argus_eproto_db, sizeof (parser->argus_eproto_db));

   while (p->range != NULL) {
      int i, start, end;
      char *ptr;
      
      start = atoi(p->range);

      if ((ptr = strchr(p->range, '-')) != NULL)
         end = atoi(ptr + 1);
      else
         end = start;

      for (i = start; i < (end + 1); i++)
         parser->argus_eproto_db[i] = p;

      p++;
   }
}


/*
 * SNAP proto IDs with org code 0:0:0 are actually encapsulated Ethernet
 * types.
 */



void
ArgusInitProtoidarray(struct ArgusParserStruct *parser)
{
   struct ArgusEtherTypeStruct *p;
   int i;
   struct protoidmem *tp;
   u_char protoid[5];

   protoid[0] = 0;
   protoid[1] = 0;
   protoid[2] = 0;

   for (i = 0; i < ARGUS_MAXEPROTODB; i++) {
      if ((p = parser->argus_eproto_db[i]) != NULL) {
         protoid[3] = i;
         tp = lookup_protoid(parser, protoid);
         tp->p_name = p->tag;
      }
   }
}

static struct etherlist {
   u_char addr[6];
   char *name;
} etherlist[] = {
   {{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff }, "Broadcast" },
   {{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }, NULL }
};

/*
 * Initialize the ethers hash table.  We take two different approaches
 * depending on whether or not the system provides the ethers name
 * service.  If it does, we just wire in a few names at startup,
 * and etheraddr_string() fills in the table on demand.  If it doesn't,
 * then we suck in the entire /etc/ethers file at startup.  The idea
 * is that parsing the local file will be fast, but spinning through
 * all the ethers entries via NIS & next_etherent might be very slow.
 *
 * XXX argus_next_etherent doesn't belong in the pcap interface, but
 * since the pcap module already does name-to-address translation,
 * it's already does most of the work for the ethernet address-to-name
 * translation, so we just argus_next_etherent as a convenience.
 */


void
ArgusInitEtherarray(struct ArgusParserStruct *parser)
{
   struct etherlist *el;
   struct enamemem *tp;
#ifndef ETHER_SERVICE
   struct argus_etherent *ep;
   FILE *fp;

   /* Suck in entire ethers file */
   fp = fopen(PCAP_ETHERS_FILE, "r");
   if (fp != NULL) {
      while ((ep = argus_next_etherent(fp)) != NULL) {
         tp = lookup_emem(parser, ep->addr);
         tp->e_name = strdup(ep->name);
      }
      (void)fclose(fp);
   }
#endif

   /* Hardwire some ethernet names */
   for (el = etherlist; el->name != NULL; ++el) {
#if defined(ETHER_SERVICE) && !defined(linux) && !defined(CYGWIN)
                /* Use yp/nis version of name if available */
      char wrk[256];
                if (ether_ntohost(wrk, (struct ether_addr *)el->addr) == 0) {
         tp = lookup_emem(parser, el->addr);
                        tp->e_name = strdup(wrk);
      }
#else
      /* install if not already present */
      tp = lookup_emem(parser, el->addr);
      if (tp->e_name == NULL)
         tp->e_name = el->name;
#endif

   }
}

#include <argus_int.h>

static struct ArgusTokenStruct llcsap_db[] = {
   { LLCSAP_NULL,   "null" },
   { LLCSAP_8021B_I,   "gsap" },
   { LLCSAP_8021B_G,   "isap" },
   { LLCSAP_SNAPATH,   "snapath" },
   { LLCSAP_IP,      "ipsap" },
   { LLCSAP_SNA1,   "sna1" },
   { LLCSAP_SNA2,   "sna2" },
   { LLCSAP_PROWAYNM,   "p-nm" },
   { LLCSAP_TI,      "ti" },
   { LLCSAP_BPDU,   "stp" },
   { LLCSAP_RS511,   "eia" },
   { LLCSAP_ISO8208,   "x25" },
   { LLCSAP_XNS,   "xns" },
   { LLCSAP_NESTAR,   "nestar" },
   { LLCSAP_PROWAYASLM,   "p-aslm" },
   { LLCSAP_ARP,   "arp" },
   { LLCSAP_SNAP,   "snap" },
   { LLCSAP_VINES1,   "vine1" },
   { LLCSAP_VINES2,   "vine2" },
   { LLCSAP_NETWARE,   "netware" },
   { LLCSAP_NETBIOS,   "netbios" },
   { LLCSAP_IBMNM,   "ibmnm" },
   { LLCSAP_RPL1,   "rpl1" },
   { LLCSAP_UB,      "ub" },
   { LLCSAP_RPL2,   "rpl2" },
   { LLCSAP_ISONS,   "clns" },
   { LLCSAP_GLOBAL,   "gbl" },
   { 0,             NULL }
};

void
ArgusInitLlcsaparray(struct ArgusParserStruct *parser)
{
   int i;
   struct hnamemem *table;

   for (i = 0; llcsap_db[i].s != NULL; i++) {
      table = &parser->llcsaptable[llcsap_db[i].v];
      while (table->name)
         table = table->nxt;
      table->name = llcsap_db[i].s;
      table->addr = llcsap_db[i].v;
      table->nxt = (struct hnamemem *)calloc(1, sizeof(*table));
   }
}

/*
 * Initialize the address to name translation machinery.  We map all
 * non-local IP addresses to numeric addresses if fflag is true (i.e.,
 * to prevent blocking on the nameserver).  localnet is the IP address
 * of the local network.  mask is its subnet mask.
 */



void
ArgusInitAddrtoname(struct ArgusParserStruct *parser, u_int localnet, u_int mask)
{
   if (parser->fflag) {
      parser->f_localnet = localnet;
      parser->f_netmask = mask;
   }

   if (parser->nflag > 1)
      /*
       * Simplest way to suppress names.
       */
      return;

   ArgusInitEtherarray(parser);
   ArgusInitServarray(parser);
   ArgusInitEprotoarray(parser);
   ArgusInitLlcsaparray(parser);
   ArgusInitProtoidarray(parser);
}


#ifndef __GNUC__
#define inline
#endif

/*
 * Convert a port name to its port and protocol numbers.
 * We assume only TCP or UDP.
 * Return 0 upon failure.
 */
int
argus_nametoport(char *name, int *port, int *proto)
{
   struct protoent *pp = NULL;
   struct servent *sp = NULL;
   char *pname = NULL, *other;

#ifdef ARGUSDEBUG
   ArgusDebug (8, "argus_nametoport (%s, .., ..) starting\n", name);
#endif

   if ((proto != NULL) && (*proto != -1)) {
#ifdef ARGUSDEBUG
      ArgusDebug (8, "argus_nametoport (%s, .., %d) calling getprotobynumber\n", name, *proto);
#endif
      if ((pp = getprotobynumber(*proto)) != NULL) {
         pname = pp->p_name;
      } else
         ArgusLog(LOG_ERR, "getprotobynumber(%d) returned NULL %s", *proto, strerror(errno));
   }

   if (name != NULL) {
#ifdef ARGUSDEBUG
      ArgusDebug (8, "argus_nametoport: calling getservbyname(%s, %s)\n", name, pname);
#endif
      sp = getservbyname(name, pname);

#ifdef ARGUSDEBUG
      ArgusDebug (8, "argus_nametoport: getservbyname() returned 0x%x\n", sp);
#endif
   }

   if (sp != NULL) {
#ifdef ARGUSDEBUG
      ArgusDebug (8, "argus_nametoport: sp is 0x%x\n", sp);
#endif
      *port = ntohs(sp->s_port);

#ifdef ARGUSDEBUG
      ArgusDebug (8, "argus_nametoport (%s, .., ..) calling argus_nametoproto(%s)\n", sp->s_proto);
#endif

      *proto = argus_nametoproto(sp->s_proto);
      /*
       * We need to check /etc/services for ambiguous entries.
       * If we find the ambiguous entry, and it has the
       * same port number, change the proto to PROTO_UNDEF
       * so both TCP and UDP will be checked.
       */
      if (*proto == IPPROTO_TCP)
         other = "udp";
      else
         other = "tcp";

      sp = getservbyname(name, other);
      if (sp != 0) {
         if (*port != ntohs(sp->s_port))
            /* Can't handle ambiguous names that refer
               to different port numbers. */
            ArgusLog(LOG_ERR, "ambiguous port %s in /etc/services values %d and %d", name, *port, ntohs(sp->s_port));
         *proto = PROTO_UNDEF;
      }

#ifdef ARGUSDEBUG
      ArgusDebug (8, "argus_nametoport (%s, %d, %d)\n", name, *port, *proto);
#endif
      return 1;
   }

#if defined(ultrix) || defined(__osf__)
   /* Special hack in case NFS isn't in /etc/services */
   if (strcmp(name, "nfs") == 0) {
      *port = 2049;
      *proto = PROTO_UNDEF;
      return 1;
   }
#endif

#ifdef ARGUSDEBUG
   ArgusDebug (8, "argus_nametoport (%s, %d, %d)\n", name, *port, *proto);
#endif

   return 0;
}

int
argus_nametoproto(char *str)
{
   struct protoent *p;

   p = getprotobyname(str);
   if (p != 0)
      return p->p_proto;
   else
      return PROTO_UNDEF;
}


int
argus_nametoeproto(char *s)
{
   struct ArgusEtherTypeStruct *p = argus_ethertype_names;

   while (p->tag != 0) {
      if (strcmp(p->tag, s) == 0) {
         return atoi(p->range);
      }
      p += 1;
   }

   return PROTO_UNDEF;
}

u_int
__argus_atoin(char *s, u_int *addr)
{
   u_int n;
   int len;

   *addr = 0;
   len = 0;
   while (1) {
      n = 0;
      while (*s && *s != '.')
         n = n * 10 + *s++ - '0';
      *addr <<= 8;
      *addr |= n & 0xff;
      len += 8;
      if (*s == '\0') {
         *addr = *addr;
         return len;
      }
      ++s;
   }
   /* NOTREACHED */
}

u_int
__argus_atodn(char *s)
{
#define AREASHIFT 10
#define AREAMASK 0176000
#define NODEMASK 01777

   u_int addr = 0;
   u_int node, area;

   if (sscanf((char *)s, "%d.%d", (int *) &area, (int *) &node) != 2)
      ArgusLog (LOG_ERR,"malformed decnet address '%s'", s);

   addr = (area << AREASHIFT) & AREAMASK;
   addr |= (node & NODEMASK);

   return(addr);
}

/*
 * Convert 's' which has the form "xx:xx:xx:xx:xx:xx" into a new
 * ethernet address.  Assumes 's' is well formed.
 */

/* Hex digit to integer. */
 
inline int xdtoi(int);
 
inline int
xdtoi(int c)
{
   if (isdigit(c))
      return c - '0';
   else if (islower(c))
      return c - 'a' + 10;
   else
      return c - 'A' + 10;
}


u_char *
argus_ether_aton(char *s)
{
   register u_char *ep, *e;
   register u_int d;

   e = ep = (u_char *)malloc(6);

   while (*s) {
      if (*s == ':')
         s += 1;
      d = xdtoi(*s++);
      if (isxdigit((int)*s)) {
         d <<= 4;
         d |= xdtoi(*s++);
      }
      *ep++ = d;
   }

   return (e);
}

#ifndef HAVE_ETHER_HOSTTON 
u_char *
argus_ether_hostton(char *name)
{
   register struct argus_etherent *ep;
   register u_char *ap;
   static FILE *fp = NULL;
   static int init = 0;

   if (!init) {
      fp = fopen(PCAP_ETHERS_FILE, "r");
      ++init;
      if (fp == NULL)
         return (NULL);
   } else if (fp == NULL)
      return (NULL);
   else
      rewind(fp);
   
   while ((ep = argus_next_etherent(fp)) != NULL) {
      if (strcmp(ep->name, name) == 0) {
         ap = (u_char *)malloc(6);
         if (ap != NULL) {
            memcpy(ap, ep->addr, 6);
            return (ap);
         }
         break;
      }
   }
   return (NULL);
}
#else
#if !defined(HAVE_DECL_ETHER_HOSTTON) || !HAVE_DECL_ETHER_HOSTTON
#ifndef HAVE_STRUCT_ETHER_ADDR
struct ether_addr {
   unsigned char ether_addr_octet[6];
};
#endif
#if !defined(__APPLE_CC__) && !defined(__APPLE__)
extern int ether_hostton(const char *, struct ether_addr *);
#endif
#endif

u_char *
argus_ether_hostton(char *name)
{
   register u_char *ap;
   u_char a[6];

   ap = NULL;
   if (ether_hostton((char*)name, (struct ether_addr *)a) == 0) {
      ap = (u_char *)malloc(6);
      if (ap != NULL)
         memcpy(ap, a, 6);
   }
   return (ap);
}
#endif

u_short
__argus_nametodnaddr(char *name)
{
#ifndef   DECNETLIB
   ArgusLog (LOG_ERR,"decnet name support not included, '%s' cannot be translated\n", name);
   return(0);
#else
   struct nodeent *getnodebyname();
   struct nodeent *nep;
   u_short res;

   nep = getnodebyname(name);
   if (nep == ((struct nodeent *)0))
      ArgusLog (LOG_ERR,"unknown decnet host name '%s'\n", name);

   memcpy((char *)&res, (char *)nep->n_addr, sizeof(u_short));

   return(res);
#endif
}

 
u_int
ipaddrtonetmask(u_int addr)
{
   if (IN_CLASSA (addr)) return IN_CLASSA_NET;
   if (IN_CLASSB (addr)) return IN_CLASSB_NET;
   if (IN_CLASSC (addr)) return IN_CLASSC_NET;
   if (IN_CLASSD (addr)) return 0xFFFFFFFF;
   else return 0;
}
 
 
u_int
getnetnumber( u_int addr)
{
   if (IN_CLASSA (addr)) return (addr >> 24 );
   if (IN_CLASSB (addr)) return (addr >> 16 );
   if (IN_CLASSC (addr)) return (addr >>  8 );
   if (IN_CLASSD (addr)) return (addr >>  0 );
   else return 0;
}

char *etheraddr_string(u_char *);

static char hex[] = "0123456789abcdef"; 
char etheraddrbuf[32];

char *
etheraddr_string(u_char *ep)
{
   char *cp = etheraddrbuf;
   u_int i, j;

   bzero (cp, sizeof(etheraddrbuf));
   if ((j = *ep >> 4) != 0)
      *cp++ = hex[j];
   *cp++ = hex[*ep++ & 0xf];
   for (i = 5; (int)--i >= 0;) {
      *cp++ = ':';
      if ((j = *ep >> 4) != 0)
         *cp++ = hex[j];
      *cp++ = hex[*ep++ & 0xf];
   }
   return (etheraddrbuf);
}
