/*
 * Argus Software.  Common library routines - Parsing
 * Copyright (c) 2000-2006 QoSient, LLC
 * All Rights Reserved
 *
 * QoSIENT, LLC DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL QoSIENT, LLC BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
 * THIS SOFTWARE.
 *
 */

/* 
 * $Id: $
 * $DateTime: $
 * $Change: $
 */


#define ArgusParse

#include <stdio.h>
#include <errno.h>
#include <fcntl.h>

#include <string.h>
#include <sys/stat.h>
#include <netinet/in.h>

#include <compat.h>
#include <pcap.h>

#include <netinet/tcp.h>

#include <interface.h>

#include <argus_client.h>
#include <argus_util.h>
#include <argus_out.h>
#include <argus_filter.h>

int ArgusParseResourceFile (char *);
void clearRaConfiguration (void);

unsigned char *ArgusRemoteFilter = NULL;

extern void ArgusLog (int, char *, ...);

int ArgusParseInit = 0;

extern void ArgusClientTimeout (void);

#define ARGUS_READINGPREHDR	1
#define ARGUS_READINGHDR	2
#define ARGUS_READINGBLOCK	4
#define ARGUS_READINGDATAGRAM	8

struct ArgusOutputStruct {
   char *filename;
   struct stat statbuf;
   FILE *fd;
};

struct ArgusOutputStruct ArgusOutputFile;

void
argus_parse_init (struct ARGUS_INPUT *input)
{
   char errbuf[MAXSTRLEN];
   char *device = NULL;
   struct tm *tm, tmbuf;
   struct argtimeval tvpbuf, *tvp = &tvpbuf;
   unsigned int net, mask;
   int i, fd = 0;

   if (input != NULL)
      fd = input->fd;

   bzero((char *)&ArgusOutputFile, sizeof(ArgusOutputFile));

   if (initCon) {
      if (initCon->ahdr.status & ARGUS_DETAIL) detail++;
      input->ArgusLocalNet = htonl(initCon->argus_mar.localnet);
      input->ArgusNetMask = htonl(initCon->argus_mar.netmask);

      if (tflag && timearg) {
         tvp->tv_sec  = ntohl(initCon->argus_mar.now.tv_sec);
         tm = localtime_r((time_t *) &tvp->tv_sec, &tmbuf);
         if (check_time_format (tm, timearg))
            ArgusLog (LOG_ERR, "time syntax error %s\n", timearg);
      }

   } else {
      if ((device = argus_lookupdev (errbuf)) != NULL) {
         argus_lookupnet(device, &net, &mask, errbuf);
         input->ArgusLocalNet = net;
         input->ArgusNetMask = mask;
      }
   }

   input->ArgusLastTime = ArgusGlobalTime;
   input->ArgusMarInterval = ntohs(input->ArgusInitCon.argus_mar.argusMrInterval);

   if (Cflag) {
      input->ArgusReadSocketState = ARGUS_READINGDATAGRAM;
      input->ArgusReadSize = k_maxFlowPacketSize;

      if (input->ArgusReadBuffer != NULL)
         ArgusFree (input->ArgusReadBuffer);
      if ((input->ArgusReadBuffer = (unsigned char *) ArgusCalloc (1, k_maxFlowPacketSize)) == NULL)
         ArgusLog (LOG_ERR, "ArgusCalloc error %s\n", strerror(errno));

      if (input->ArgusConvBuffer != NULL)
         ArgusFree (input->ArgusConvBuffer);

      if ((input->ArgusConvBuffer = (u_char *) ArgusCalloc (1, k_maxFlowPacketSize)) == NULL)
         ArgusLog (LOG_ERR, "ArgusCalloc error %s\n", strerror(errno));

   } else {
      if (input->ArgusReadBuffer != NULL)
         ArgusFree (input->ArgusReadBuffer);

      if ((input->ArgusReadBuffer = (unsigned char *) ArgusCalloc (1, MAXSTRLEN)) == NULL)
         ArgusLog (LOG_ERR, "ArgusCalloc error %s\n", strerror(errno));

      if (input->ArgusConvBuffer != NULL)
         ArgusFree (input->ArgusConvBuffer);

      if ((input->ArgusConvBuffer = (u_char *) ArgusCalloc (1, MAXSTRLEN)) == NULL)
         ArgusLog (LOG_ERR, "ArgusCalloc error %s\n", strerror(errno));

#ifdef ARGUS_SASL
      if (input->ArgusSaslBuffer != NULL)
         ArgusFree (input->ArgusSaslBuffer);

      if ((input->ArgusSaslBuffer = (u_char *) ArgusCalloc (1, MAXSTRLEN)) == NULL)
         ArgusLog (LOG_ERR, "ArgusCalloc error %s\n", strerror(errno));
#endif /* ARGUS_SASL */

      if (major_version > 1)
         input->ArgusReadSocketState = ARGUS_READINGHDR;
      else {
         input->ArgusReadSocketState = ARGUS_READINGBLOCK;
         input->ArgusReadSize = 60;
      }
   }

   input->ArgusReadPtr = input->ArgusReadBuffer;
   input->ArgusConvPtr = input->ArgusConvBuffer;

   input->ArgusReadSocketSize = (input->ArgusReadSize < 0) ?
                    sizeof(struct ArgusRecordHeader) : input->ArgusReadSize;

   if (!ArgusParseInit++) {
      for (i = 0; i < ARGUS_MAX_REMOTE_CONN; i++)
         ArgusRemoteFDs[i] = NULL;

      init_addrtoname (fflag, ArgusLocalNet, ArgusNetMask);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusParseInit (0x%x) returning\n", input);
#endif
}



int ArgusPassNum = 1;
int ArgusPortNum = 0;

#if defined(CYGWIN)
#include <getopt.h>
#endif

char *getoptStr = "aAbB:cCd:D:E:e:f:F:gGhH:iIL:lmM:nN:oO:p:P:qr:RS:s:t:T:uU:Vvw:xXzZ:";

#define RaEnvItems	2

char *RaResourceEnvStr [] = {
   "HOME",
   "ARGUSHOME",
};

char *RaOutputFilter = NULL;
char *RaHomePath = NULL;

int
main (int argc, char **argv)
{
   int i, cc, op, retn = 0, fd = 0, Scmdline = 0, rcmdline = 0;
   char *cmdbuf = NULL, *infile = NULL;
   char *envstr = NULL, *filter = NULL;
   static char path[MAXPATHNAMELEN];
   struct stat statbuf;
   char *tmparg;
   struct timeval now;
   struct timezone tz;
   pcap_t pbuf;
   extern char *optarg;
   extern int optind, opterr;

   opterr = 0;

   for (i = 0, cc = 0; i < argc; i++)
      cc += strlen(argv[i]);

   if (cc > 0) {
      int len = cc + (argc + 1);

      if ((ArgusProgramArgs = (char *) ArgusCalloc (len, sizeof(char))) != NULL) {
         for (i = 0, *ArgusProgramArgs = '\0'; i < argc; i++) {
            strcat (ArgusProgramArgs, argv[i]);
            strcat (ArgusProgramArgs, " ");
         }
      } else
         ArgusLog (LOG_ERR, "ArgusCalloc(%d, %d) failed %s\n", len, sizeof(char), strerror(errno));
   }

   if (strchr (argv[0], '/'))
      argv[0] = strrchr(argv[0], '/') + 1;

   ArgusProgramName = argv[0];

   if (gettimeofday(&now, &tz) < 0)
      ArgusLog (LOG_ERR, "gettimeofday failed %s\n", strerror(errno));

   ArgusGlobalTime = now;
   ArgusNowTime = now;

   thiszone = tz.tz_minuteswest * -60;

   if ((RaTmStruct = localtime_r((time_t *)&now.tv_sec, &RaTmStructBuf))) {
      if (RaTmStruct->tm_isdst)
         thiszone += 3600;
   } else
      ArgusLog (LOG_ERR, "%s: localtime: error %s \n", *argv, strerror(errno));

   snprintf (path, MAXPATHNAMELEN - 1, "/etc/ra.conf");

   if (stat (path, &statbuf) == 0)
      ArgusParseResourceFile (path);

   if ((RaHomePath = getenv("ARGUSHOME")) != NULL) {
      snprintf (path, MAXPATHNAMELEN - 1, "%s/ra.conf", RaHomePath);
      if (stat (path, &statbuf) == 0) {
         ArgusParseResourceFile (path);
      }
   }

   if ((envstr = getenv("ARGUSPATH")) != NULL) {
      while ((RaHomePath = strtok(envstr, ":")) != NULL) {
         snprintf (path, MAXPATHNAMELEN - 1, "%s/.rarc", RaHomePath);
         if (stat (path, &statbuf) == 0) {
            ArgusParseResourceFile (path);
            break;
         }
         envstr = NULL;
      }

   } else {
      for (i = 0; i < RaEnvItems; i++) {
         envstr = RaResourceEnvStr[i];
         if ((RaHomePath = getenv(envstr)) != NULL) {
            sprintf (path, "%s/.rarc", RaHomePath);
            if (stat (path, &statbuf) == 0) {
               ArgusParseResourceFile (path);
               break;
            }
         }
      }
   }

   if ((argv[optind]) != NULL)
      ArgusProgramOptions =  strdup(copy_argv (&argv[optind]));

   while ((op = getopt (argc, argv, getoptStr)) != EOF) {
      switch (op) {
         case 'a': ++aflag; break;
         case 'A': ++Aflag; break;
         case 'b': ++bflag; break;
         case 'B': Bflag = atoi(optarg); break;
         case 'c': ++RaPrintCounts; ++RaPrintBytes; break;
         case 'C': {
            struct ARGUS_INPUT *addr = NULL;

            if (Sflag)
               ArgusLog (LOG_ERR, "usage: -C and -S not compatible.\n");

            if (ArgusInputFileList != NULL)
               ArgusDeleteFileList();

            if (ArgusRemoteHostList != NULL)
               ArgusDeleteHostList();

            Cflag++;
            if ((addr = (struct ARGUS_INPUT *) ArgusCalloc (1, sizeof (struct ARGUS_INPUT))) != NULL) {
               addr->nxt = ArgusRemoteHostList;
               ArgusRemoteHostList = addr;
               addr->status |= ARGUS_CISCO_DATA_SOURCE;
            }
            break;
         }

         case 'D': Argusdflag = atoi (optarg); break;
         case 'd': ++dflag;
            if ((strncmp(ArgusProgramName, "rahistogram", 11)) &&
                (strncmp(ArgusProgramName, "racricket", 9)) &&
                (strncmp(ArgusProgramName, "rabins", 6))) {
               if ((dataarg = optarg) != NULL) {
                  if ((retn = parseUserDataArg (&dataarg, argv, optind)) < 0) {
                     usage ();
                  } else {
                     optind += retn;
                  }
               }
            } else
               dflag = atoi(optarg);
            break;

         case 'e':
            estr = optarg;
            if (strncmp(ArgusProgramName, "ragrep", 6)) {
               if (!(strncasecmp(optarg, "ascii", 5)))
                  eflag = ARGUS_ENCODE_ASCII;
               else
               if (!(strncasecmp(optarg, "encode64", 8)))
                  eflag = ARGUS_ENCODE_64;
               else
               if (!(strncasecmp(optarg, "encode32", 8)))
                  eflag = ARGUS_ENCODE_32;
               else
                  usage();
            } else {
               ArgusGrepSource++;
               ArgusGrepDestination++;

               if ((estr[0] == 's') && (estr[1] == ':')) {
                  ArgusGrepDestination = 0;
                  estr = &estr[2];
               }

               if ((estr[0] == 'd') && (estr[1] == ':')) {
                  ArgusGrepSource = 0;
                  estr = &estr[2];
               }
            }
            break;

         case 'E': exceptfile = optarg; break;
         case 'f': ArgusFlowModelFile = optarg; break;
         case 'F': 
            if (!(ArgusParseResourceFile (optarg)))
               ArgusLog(LOG_ERR, "%s: %s\n", optarg, strerror(errno));
            break;

         case 'g': ++RaPrintDuration; Gflag = 0; break;
         case 'G':
             ++Gflag;
             RaPrintLastTime = 1;
             RaPrintDuration = 0;
             break;
	 case 'H':
            ++Hflag;
            Hstr = optarg;
            break;
         case 'i': ++RaPrintSource; break;
         case 'I': ++Iflag; break;
         case 'L': 
            Lflag = atoi(optarg);
            if (Lflag == 0)
               Lflag = -1;
            break;
         case 'l': ++lflag; break;
         case 'm': ++RaPrintMACAddress; break;

         case 'M':
            do {
               if (!(ArgusAddModeList (optarg)))
                  ArgusLog (LOG_ERR, "error: file arg %s \n", optarg);

               if ((optarg = argv[optind]) != NULL)
                  if (*optarg != '-')
                     optind++;
            } while (optarg && (*optarg != '-'));
            break;

         case 'n': ++nflag; break;
         case 'N': Nflag = atoi (optarg); break;
         case 'o': oflag++; break;
         case 'O': Oflag = atoi (optarg); break;
         case 'p': pflag = atoi (optarg); break;
         case 'P':  ArgusPortNum = atoi (optarg); break;
         case 'q': ++qflag; break;
         case 'r': ++rflag; 
            Sflag = 0;
            if ((!rcmdline++) && (ArgusInputFileList != NULL))
               ArgusDeleteFileList();

            if (optarg == NULL)
               optarg = "-";
            do {
               if (!(ArgusAddFileList (optarg)))
                  ArgusLog (LOG_ERR, "error: file arg %s \n", optarg);

               if ((optarg = argv[optind]) != NULL)
                  if (*optarg != '-')
                     optind++;
            } while (optarg && (*optarg != '-'));
            break;

         case 'R': ++Rflag; break;
         case 's': 
            if (RaSOptionIndex < ARGUS_MAX_S_OPTIONS)
               RaSOptionStrings[RaSOptionIndex++] = optarg;
            else
               ArgusLog (LOG_ERR, "usage: number of -s options exceeds %d\n", ARGUS_MAX_S_OPTIONS);
            break;

         case 'S':
            if (Cflag)
               ArgusLog (LOG_ERR, "usage: -C and -S not compatible.\n");

             ++Sflag;
            if ((!Scmdline++) && (ArgusRemoteHostList != NULL))
               ArgusDeleteHostList();
                              
            if (!(ArgusAddHostList (optarg)))
               ArgusLog (LOG_ERR, "host %s unknown\n", optarg);

            break;

         case 't': ++tflag; 
            if ((timearg = strdup(optarg)) != NULL) {
               if ((retn = parseTimeArg (&timearg, argv, optind, RaTmStruct)) < 0) {
                  usage ();
               } else {
                  optind += retn;
               }
            }
            break;
         case 'T': Tflag = atoi(optarg); break;
         case 'u': uflag++;  break;
         case 'U': ustr = strdup(optarg);  break;
         case 'v': vflag++; break;
         case 'V': Vflag++; break;
         case 'w':  
            if ((tmparg = optarg) != NULL) {
               if ((*tmparg != '-') || ((*tmparg == '-') &&
                                       (!(strcmp (tmparg, "-"))))) {
                  if (argc == optind)
                     filter = NULL;
                  else {
                     filter = argv[optind];
                     if (*filter == '-') {
                        filter = NULL;
                     } else
                        optind++;
                     }
                  wfile =  strdup(tmparg);
                  break;
               }
            }
            break;

	 case 'x': ++xflag; break;
         case 'X': clearRaConfiguration (); break;
	 case 'z': ++zflag; break;
	 case 'Z': Zflag = *optarg; break;
         case 'h':
            default:  
               usage ();
            /* NOTREACHED */
      }
   }
 
   if (infile)
      cmdbuf = read_infile (infile);
   else {
      char *str;

      if ((str = argv[optind]) != NULL) {
         if (strcmp(str, "-") == 0)
            optind++;
         cmdbuf = copy_argv (&argv[optind]);
      }
   }

   if (cmdbuf) {
      if (*RaInputFilter != NULL)
         free(*RaInputFilter);

      *RaInputFilter = cmdbuf;
   }

   bzero ((char *) &ArgusFilterCode, sizeof (ArgusFilterCode));
   bzero ((char *) &pbuf, sizeof (pbuf));

   if (ArgusFilterCompile (&pbuf, &ArgusFilterCode, *RaInputFilter, 1, ArgusNetMask) < 0) {
      if (pbuf.errbuf[0] != '\0') {
         ArgusLog (LOG_ERR, "expression: %s\n", pbuf.errbuf);
      } else
         ArgusLog (LOG_ERR, "%s error\n", *RaInputFilter);

   } else {
      if (cmdbuf)
         ArgusRemoteFilter = (unsigned char *) strdup(cmdbuf);
      else
         ArgusRemoteFilter = NULL;
   }

   if (bflag) {
      bpf_dump(&ArgusFilterCode, bflag);
      exit (0);
   }

   if (RaSOptionIndex > 0)
      RaProcessSOptions();

   ArgusClientInit ();

   if (Sflag || Cflag) {
      register struct ARGUS_INPUT *addr;

      if ((addr = ArgusRemoteHostList) != NULL) {
         while (addr != NULL) {
            if ((addr->fd = ArgusGetServerSocket (addr)) >= 0)
               if ((ArgusReadConnection (addr, NULL)) >= 0)
                  ArgusRemoteFDs[ArgusActiveServers++] = addr;

            addr = addr->nxt;
         }
      }

      ArgusReadStream();

   } else {
      struct ARGUS_INPUT *addr;

      if (ArgusInputFileList != NULL) {
         for (i = 0; i < ArgusPassNum; i++) {
            addr = ArgusInputFileList;
            while (addr) {
               if (strcmp (addr->filename, "-")) {
                  if (addr->fd < 0) {
                     if ((addr->fd = open(addr->filename, O_RDONLY)) < 0)
                        ArgusLog (LOG_ERR, "open '%s': %s\n", addr->filename, strerror(errno));
                  } else
                     if (lseek(addr->fd, 0, SEEK_SET) < 0)
                        ArgusLog (LOG_ERR, "lseek '%s': %s\n", addr->filename, strerror(errno));

                  if (((ArgusReadConnection (addr, addr->filename)) >= 0)) {
                     ArgusRemoteFDs[0] = addr;
                     ArgusReadStream();
                     if (RaCloseInputFd)
                        close(addr->fd);
                  }
               } else {
                  addr->fd = 0;
                  if (((ArgusReadConnection (addr, NULL)) >= 0)) {
                     ArgusRemoteFDs[0] = addr;
                     ArgusReadStream();
                  }
               }
               addr = addr->nxt;
            }

            RaParseComplete(ArgusPassNum - i);
         }

      } else {
         struct ARGUS_INPUT addrbuf, *addr = &addrbuf;

         bzero ((char *) addr, sizeof (*addr));
         addr->fd = 0;

         if (((ArgusReadConnection (addr, NULL)) >= 0)) {
            ArgusRemoteFDs[0] = addr;
            ArgusReadStream();
         }
      }
   }

   if (fd >= 0) {
      ArgusShutDown (0);
   } else
      retn = 1;

#ifdef ARGUSDEBUG
   ArgusDebug (1, "main () exiting with %d\n", retn);
#endif

   exit (retn);
}


void
ArgusShutDown (int value)
{

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusShutDown (%d)\n", value);
#endif

   if (value >= 0)
      RaParseComplete (value);

   exit (value);
}

void ArgusGenerateCanonicalRecord (struct ArgusRecord *, struct ArgusCanonicalRecord *);

unsigned int ArgusTotalCount = 0;
unsigned int ArgusTotalBytes = 0;
static int firstWrite = 0;

int
ArgusHandleDatum (struct ArgusRecord *ptr, struct bpf_program *filter)
{
   int retn = 0;

   if (ptr != NULL) {
      int len = ntohs(ptr->ahdr.length);
      struct bpf_insn *fcode = filter->bf_insns;
      u_char buf[MAXSTRLEN];

      totalrecords++;
      bcopy ((char *)ptr, (char *)ArgusOriginal, len);
      bcopy ((char *)ptr, (char *)&buf, len);
   
      switch (ptr->ahdr.type) {
         case ARGUS_MAR:
         case (ARGUS_MAR | ARGUS_CISCO_NETFLOW):
         case ARGUS_INDEX:
         case ARGUS_EVENT:
            marrecords++;
            break;
   
         case ARGUS_FAR:
         case ARGUS_DATASUP:
            farrecords++;
            break;
      }

      ArgusInput->ArgusLastTime = ArgusGlobalTime;

      if ((retn = argus_filter (fcode, (unsigned char *) ptr)) != 0) {
#ifdef _LITTLE_ENDIAN
         ArgusNtoH ((struct ArgusRecord *)&buf);	 
#endif
         ArgusThisFarStatus = ArgusIndexRecord ((struct ArgusRecord *)&buf, ArgusThisFarHdrs);
   
         if ((retn = check_time ((struct ArgusRecord *)&buf)) != 0) {
            struct ArgusRecord *argus = (struct ArgusRecord *)&buf;

   
            if (!(ptr->ahdr.type & ARGUS_MAR)) {
               unsigned int count, bytes;

#define ARGUSMAXPACKETSIZE      65536       /* correct for 1.8x byte count bug */

               if ((count = argus->argus_far.src.count) > 0)
                  if ((bytes = argus->argus_far.src.bytes) > 0)
                     if ((bytes/count) > ARGUSMAXPACKETSIZE)
                        argus->argus_far.src.bytes = 0;
               if ((count = argus->argus_far.dst.count) > 0)
                  if ((bytes = argus->argus_far.dst.bytes) > 0)
                     if ((bytes/count) > ARGUSMAXPACKETSIZE)
                        argus->argus_far.dst.bytes = 0;

               ArgusTotalCount += (argus->argus_far.src.count + argus->argus_far.dst.count);
               if (Aflag)
                  ArgusTotalBytes += (argus->argus_far.src.appbytes + argus->argus_far.dst.appbytes);
               else
                  ArgusTotalBytes += (argus->argus_far.src.bytes + argus->argus_far.dst.bytes);
            }

            if (wfile) {
               if (RaWriteOut) {
                  if (!(firstWrite && ((argus->ahdr.type & ARGUS_MAR) && (argus->ahdr.cause & ARGUS_START))))
                     if (ArgusWriteNewLogfile (wfile, ArgusOriginal))
                        ArgusLog (LOG_ERR, "ArgusWriteNewLogfile: error\n");
               } else
                  RaProcessRecord ((struct ArgusRecord *)&buf);
            } else
               RaProcessRecord ((struct ArgusRecord *)&buf);
         }
      } else {
         if (RaWriteOut && exceptfile) {
            if (ArgusWriteNewLogfile (exceptfile, ArgusOriginal)) 
               ArgusLog (LOG_ERR, "ArgusWriteNewLogfile: error using file %s\n", exceptfile);
         }
      }
   
      retn = 0;
   
      if (ptr->ahdr.type & ARGUS_MAR) {
         switch (ptr->ahdr.cause) {
            case ARGUS_STOP:
            case ARGUS_SHUTDOWN:
            case ARGUS_ERROR: {
#ifdef ARGUSDEBUG
               ArgusDebug (3, "ArgusHandleDatum (0x%x, 0x%x) received closing Mar\n", ptr, filter);
#endif
               if (Sflag)
                  retn = 1;
               break;
            }
         }
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusHandleDatum (0x%x, 0x%x) returning %d\n", ptr, filter, retn);
#endif

   return (retn);
}


void
ArgusGenerateCanonicalRecord (struct ArgusRecord *ptr, struct ArgusCanonicalRecord *canon)
{
   int i, index = 0;
   struct ArgusFarHeaderStruct **hdrs = NULL;
   struct ArgusRecord *ar = (struct ArgusRecord *) canon;

   ptr->ahdr.length = ntohs(ptr->ahdr.length);

   ArgusThisFarStatus = ArgusIndexRecord (ptr, ArgusThisFarHdrs);

   ptr->ahdr.length = htons(ptr->ahdr.length);

   hdrs = ArgusThisFarHdrs;

   bzero ((char *)canon, sizeof(*canon));

   if (ptr->ahdr.type & ARGUS_MAR) {
      bcopy ((char *)ptr, (char *)canon, sizeof(*ptr));

   } else {
      bcopy ((char *)&ptr->ahdr, (char *)&canon->ahdr, sizeof(canon->ahdr));
   
      for (i = 1; i < 33; i++) {
         index = 1 << (i - 1);
         switch (index) {
            case ARGUS_FAR_DSR_STATUS:
               if (ArgusThisFarStatus & ARGUS_FAR_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_FAR_DSR_INDEX], (char *)&ar->argus_far, sizeof (ar->argus_far));
               break;
            case ARGUS_MAC_DSR_STATUS:
               if (ArgusThisFarStatus & ARGUS_MAC_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_MAC_DSR_INDEX], (char *)&canon->mac, sizeof(canon->mac));
               break;
            case ARGUS_VLAN_DSR_STATUS:
               if (ArgusThisFarStatus & ARGUS_VLAN_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_VLAN_DSR_INDEX], (char *)&canon->vlan, sizeof(canon->vlan));
               break;
            case ARGUS_MPLS_DSR_STATUS:
               if (ArgusThisFarStatus & ARGUS_MPLS_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_MPLS_DSR_INDEX], (char *)&canon->mpls, sizeof(canon->mpls));
               break;
            case ARGUS_AGR_DSR_STATUS:
               if (ArgusThisFarStatus & ARGUS_AGR_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_AGR_DSR_INDEX], (char *)&canon->agr, sizeof(canon->agr));
               break;
            case ARGUS_TIME_DSR_STATUS:
               if (ArgusThisFarStatus & ARGUS_TIME_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_TIME_DSR_INDEX], (char *)&canon->time, sizeof(canon->time));
               break;
            case ARGUS_TCP_DSR_STATUS:   
               if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_TCP_DSR_INDEX], (char *)&canon->acr_tcp, sizeof(canon->acr_tcp));
               break;
            case ARGUS_ICMP_DSR_STATUS:  
               if (ArgusThisFarStatus & ARGUS_ICMP_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_ICMP_DSR_INDEX], (char *)&canon->acr_icmp, sizeof(canon->acr_icmp));
               break;
            case ARGUS_RTP_DSR_STATUS:   
               if (ArgusThisFarStatus & ARGUS_RTP_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_RTP_DSR_INDEX], (char *)&canon->acr_rtp, sizeof(canon->acr_rtp));
               break;
            case ARGUS_IGMP_DSR_STATUS:  
               if (ArgusThisFarStatus & ARGUS_IGMP_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_IGMP_DSR_INDEX], (char *)&canon->acr_igmp, sizeof(canon->acr_igmp));
               break;
            case ARGUS_ARP_DSR_STATUS:   
               if (ArgusThisFarStatus & ARGUS_ARP_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_ARP_DSR_INDEX], (char *)&canon->acr_arp, sizeof(canon->acr_arp));
               break;
            case ARGUS_FRG_DSR_STATUS:   
               if (ArgusThisFarStatus & ARGUS_FRG_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_FRG_DSR_INDEX], (char *)&canon->acr_frag, sizeof(canon->acr_frag));
               break;
            case ARGUS_ESP_DSR_STATUS:   
               if (ArgusThisFarStatus & ARGUS_ESP_DSR_STATUS)
                  bcopy((char *) hdrs[ARGUS_ESP_DSR_INDEX], (char *)&canon->acr_esp, sizeof(canon->acr_esp));
               break;
         }
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (7, "ArgusGenerateCanonicalRecord (0x%x, 0x%x) returning\n", ptr, canon);
#endif
}

struct ArgusRecord *ArgusNetFlowCallRecord (u_char **);
struct ArgusRecord *ArgusNetFlowDetailInt  (u_char **);
struct ArgusRecord *ArgusParseCiscoRecord (u_char **);

struct ArgusRecord *ArgusParseCiscoRecordV1 (u_char **);
struct ArgusRecord *ArgusParseCiscoRecordV5 (u_char **);
struct ArgusRecord *ArgusParseCiscoRecordV6 (u_char **);

unsigned char *ArgusNetFlowRecordHeader = NULL;

unsigned char ArgusNetFlowArgusRecordBuf[1024];
struct ArgusRecord *ArgusNetFlowArgusRecord = (struct ArgusRecord *) ArgusNetFlowArgusRecordBuf;

struct ArgusRecord * 
ArgusParseCiscoRecordV1 (u_char **ptr)
{
   CiscoFlowEntryV1_t  *entryPtrV1 = ((CiscoFlowEntryV1_t *) *ptr)++;
   CiscoFlowHeaderV1_t *hdrPtrV1   = (CiscoFlowHeaderV1_t *) ArgusNetFlowRecordHeader;
   struct ArgusRecord *argus = ArgusNetFlowArgusRecord;
   struct ArgusMacStruct mac;

   bzero ((char *) argus, sizeof (*argus));
   argus->ahdr.type    = ARGUS_FAR | ARGUS_CISCO_NETFLOW;
   argus->ahdr.cause   = ARGUS_STATUS;
   argus->ahdr.length  = sizeof(argus->ahdr) + sizeof(argus->argus_far);

   argus->ahdr.status |= ETHERTYPE_IP;

   argus->argus_far.type   = ARGUS_FAR;
   argus->argus_far.length = sizeof(argus->argus_far);

   if (hdrPtrV1) {
      long time;
      time = ntohl(entryPtrV1->first);
      argus->argus_far.time.start.tv_sec  = (time - (long)hdrPtrV1->sysUptime)/1000;
      argus->argus_far.time.start.tv_sec += hdrPtrV1->unix_secs;

      argus->argus_far.time.start.tv_usec = ((time - (long)hdrPtrV1->sysUptime)%1000) * 1000;
      argus->argus_far.time.start.tv_usec += hdrPtrV1->unix_nsecs/1000;

      time = ntohl(entryPtrV1->last);
      argus->argus_far.time.last.tv_sec  = (time - (long)hdrPtrV1->sysUptime)/1000;
      argus->argus_far.time.last.tv_sec += hdrPtrV1->unix_secs;

      argus->argus_far.time.last.tv_usec = ((time - (long)hdrPtrV1->sysUptime)%1000) * 1000;
      argus->argus_far.time.last.tv_usec += hdrPtrV1->unix_nsecs/1000;
   }

   argus->argus_far.flow.ip_flow.ip_src = ntohl(entryPtrV1->srcaddr);
   argus->argus_far.flow.ip_flow.ip_dst = ntohl(entryPtrV1->dstaddr);
   argus->argus_far.flow.ip_flow.ip_p   = entryPtrV1->prot;
   argus->argus_far.attr_ip.stos        = entryPtrV1->tos;
   argus->argus_far.src.count    = ntohl(entryPtrV1->pkts);
   argus->argus_far.src.bytes    = ntohl(entryPtrV1->bytes);
   argus->argus_far.src.appbytes = 0;

   switch (argus->argus_far.flow.ip_flow.ip_p) {
      case IPPROTO_TCP: { 
         struct ArgusTCPObject tcpbuf, *tcp = &tcpbuf;

         bzero ((char *) tcp, sizeof(*tcp));
         tcp->type = ARGUS_TCP_DSR;
         tcp->length = sizeof(struct ArgusTCPObject);
         tcp->src.flags    = entryPtrV1->flags;

         if (tcp->src.flags & TH_RST)
            tcp->status |= ARGUS_RESET;

         if (tcp->src.flags & TH_FIN)
            tcp->status |= ARGUS_FIN;

         if ((tcp->src.flags & TH_ACK) || (tcp->src.flags & TH_PUSH) || (tcp->src.flags & TH_URG))
            tcp->status |= ARGUS_CON_ESTABLISHED;

         switch (tcp->src.flags & (TH_SYN|TH_ACK|TH_FIN|TH_PUSH|TH_URG)) {
            case (TH_SYN):
               tcp->status |= ARGUS_SAW_SYN;
               break;
         }

         bcopy ((char *)tcp, &((char *)argus)[argus->ahdr.length], sizeof(*tcp));
         argus->ahdr.length += sizeof(*tcp);
      }

/* fall through to UDP switch to get the ports */

      case IPPROTO_UDP:
         argus->argus_far.flow.ip_flow.sport  = ntohs(entryPtrV1->srcport);
         argus->argus_far.flow.ip_flow.dport  = ntohs(entryPtrV1->dstport);
      break;

      case IPPROTO_ICMP: { 
         argus->argus_far.flow.icmp_flow.type  = ((char *)&entryPtrV1->dstport)[0];
         argus->argus_far.flow.icmp_flow.code  = ((char *)&entryPtrV1->dstport)[1];
      }
      break;
   }

   bzero ((char *)&mac, sizeof (mac));
   mac.type   = ARGUS_MAC_DSR;
   mac.length = sizeof(mac);
   mac.status = 0;
   entryPtrV1->input = ntohs(entryPtrV1->input);
   entryPtrV1->output = ntohs(entryPtrV1->output);
   
   bcopy((char *)&entryPtrV1->input, (char *)&mac.phys_union.ether.ethersrc[4], 2);
   bcopy((char *)&entryPtrV1->output,(char *)&mac.phys_union.ether.etherdst[4], 2);

   bcopy ((char *)&mac, &((char *)argus)[argus->ahdr.length], sizeof(mac));
   argus->ahdr.length += sizeof(mac);

#ifdef _LITTLE_ENDIAN
   ArgusHtoN(argus);
#endif

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusParseCiscoRecordV1 (0x%x) returning 0x%x\n", *ptr, argus);
#endif

   return(argus);
}


struct ArgusRecord * 
ArgusParseCiscoRecordV5 (u_char **ptr)
{
   CiscoFlowEntryV5_t  *entryPtrV5 = ((CiscoFlowEntryV5_t *) *ptr)++;
   CiscoFlowHeaderV5_t *hdrPtrV5   = (CiscoFlowHeaderV5_t *) ArgusNetFlowRecordHeader;
   struct ArgusRecord *argus = ArgusNetFlowArgusRecord;
   struct ArgusMacStruct mac;

   bzero ((char *) argus, sizeof (*argus));
   argus->ahdr.type    = ARGUS_FAR | ARGUS_CISCO_NETFLOW;
   argus->ahdr.cause   = ARGUS_STATUS;
   argus->ahdr.length  = sizeof(argus->ahdr) + sizeof(argus->argus_far);

   argus->ahdr.status |= ETHERTYPE_IP;

   argus->argus_far.type   = ARGUS_FAR;
   argus->argus_far.length = sizeof(argus->argus_far);

   if (hdrPtrV5) {
      long time; 
      time = ntohl(entryPtrV5->first);
      argus->argus_far.time.start.tv_sec  = (time - (long)hdrPtrV5->sysUptime)/1000;
      argus->argus_far.time.start.tv_sec += hdrPtrV5->unix_secs;

      argus->argus_far.time.start.tv_usec = ((time - (long)hdrPtrV5->sysUptime)%1000) * 1000;
      argus->argus_far.time.start.tv_usec += hdrPtrV5->unix_nsecs/1000;
      
      time = ntohl(entryPtrV5->last);
      argus->argus_far.time.last.tv_sec  = (time - (long)hdrPtrV5->sysUptime)/1000;
      argus->argus_far.time.last.tv_sec += hdrPtrV5->unix_secs;
      
      argus->argus_far.time.last.tv_usec = ((time - (long)hdrPtrV5->sysUptime)%1000) * 1000;
      argus->argus_far.time.last.tv_usec += hdrPtrV5->unix_nsecs/1000;
   }

   argus->argus_far.flow.ip_flow.ip_src = ntohl(entryPtrV5->srcaddr);
   argus->argus_far.flow.ip_flow.ip_dst = ntohl(entryPtrV5->dstaddr);
   argus->argus_far.flow.ip_flow.sport  = ntohs(entryPtrV5->srcport);
   argus->argus_far.flow.ip_flow.dport  = ntohs(entryPtrV5->dstport);
   argus->argus_far.flow.ip_flow.ip_p   = entryPtrV5->prot;
   argus->argus_far.attr_ip.stos        = entryPtrV5->tos;
   argus->argus_far.src.count    = ntohl(entryPtrV5->pkts);
   argus->argus_far.src.bytes    = ntohl(entryPtrV5->bytes);
   argus->argus_far.src.appbytes = 0;

   switch (argus->argus_far.flow.ip_flow.ip_p) {
      case IPPROTO_TCP: {
         struct ArgusTCPObject tcpbuf, *tcp = &tcpbuf;

         bzero ((char *) tcp, sizeof(*tcp));
         tcp->type = ARGUS_TCP_DSR;
         tcp->length = sizeof(struct ArgusTCPObject);
         tcp->src.flags    = entryPtrV5->tcp_flags;

         if (tcp->src.flags & TH_RST)
            tcp->status |= ARGUS_RESET;

         if (tcp->src.flags & TH_FIN)
            tcp->status |= ARGUS_FIN;

         if ((tcp->src.flags & TH_ACK) || (tcp->src.flags & TH_PUSH) || (tcp->src.flags & TH_URG))
            tcp->status |= ARGUS_CON_ESTABLISHED;

         switch (tcp->src.flags & (TH_SYN|TH_ACK|TH_FIN|TH_PUSH|TH_URG)) {
            case (TH_SYN):
               tcp->status |= ARGUS_SAW_SYN;
               break;
         }

         bcopy ((char *)tcp, &((char *)argus)[argus->ahdr.length], sizeof(*tcp));
         argus->ahdr.length += sizeof(*tcp);
      }
      break;
   }

   bzero ((char *)&mac, sizeof (mac));
   mac.type   = ARGUS_MAC_DSR;
   mac.length = sizeof(mac);
   mac.status = 0;
   entryPtrV5->input = ntohs(entryPtrV5->input);
   entryPtrV5->output = ntohs(entryPtrV5->output);

   bcopy((char *)&entryPtrV5->input, (char *)&mac.phys_union.ether.ethersrc[4], 2);
   bcopy((char *)&entryPtrV5->output,(char *)&mac.phys_union.ether.etherdst[4], 2);

   bcopy ((char *)&mac, &((char *)argus)[argus->ahdr.length], sizeof(mac));
   argus->ahdr.length += sizeof(mac);

#ifdef _LITTLE_ENDIAN
   ArgusHtoN(argus);
#endif

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusParseCiscoRecordV5 (0x%x) returning 0x%x\n", *ptr, argus);
#endif

   return (argus);
}


struct ArgusRecord * 
ArgusParseCiscoRecordV6 (u_char **ptr)
{
   CiscoFlowEntryV6_t  *entryPtrV6 = ((CiscoFlowEntryV6_t *) *ptr)++;
   CiscoFlowHeaderV6_t *hdrPtrV6   = (CiscoFlowHeaderV6_t *) ArgusNetFlowRecordHeader;
   struct ArgusRecord *argus = ArgusNetFlowArgusRecord;
   struct ArgusMacStruct mac;

   bzero ((char *) argus, sizeof (*argus));
   argus->ahdr.type    = ARGUS_FAR | ARGUS_CISCO_NETFLOW;
   argus->ahdr.cause   = ARGUS_STATUS;
   argus->ahdr.length  = sizeof(argus->ahdr) + sizeof(argus->argus_far);

   argus->ahdr.status |= ETHERTYPE_IP;

   argus->argus_far.type   = ARGUS_FAR;
   argus->argus_far.length = sizeof(argus->argus_far);

   if (hdrPtrV6) {
      long time; 
      time = ntohl(entryPtrV6->first);
      argus->argus_far.time.start.tv_sec  = (time - (long)hdrPtrV6->sysUptime)/1000;
      argus->argus_far.time.start.tv_sec += hdrPtrV6->unix_secs;

      argus->argus_far.time.start.tv_usec = ((time - (long)hdrPtrV6->sysUptime)%1000) * 1000;
      argus->argus_far.time.start.tv_usec += hdrPtrV6->unix_nsecs/1000;
      
      time = ntohl(entryPtrV6->last);
      argus->argus_far.time.last.tv_sec  = (time - (long)hdrPtrV6->sysUptime)/1000;
      argus->argus_far.time.last.tv_sec += hdrPtrV6->unix_secs;
      
      argus->argus_far.time.last.tv_usec = ((time - (long)hdrPtrV6->sysUptime)%1000) * 1000;
      argus->argus_far.time.last.tv_usec += hdrPtrV6->unix_nsecs/1000;
   }

   argus->argus_far.flow.ip_flow.ip_src = ntohl(entryPtrV6->srcaddr);
   argus->argus_far.flow.ip_flow.ip_dst = ntohl(entryPtrV6->dstaddr);
   argus->argus_far.flow.ip_flow.sport  = ntohs(entryPtrV6->srcport);
   argus->argus_far.flow.ip_flow.dport  = ntohs(entryPtrV6->dstport);
   argus->argus_far.flow.ip_flow.ip_p   = entryPtrV6->prot;
   argus->argus_far.attr_ip.stos        = entryPtrV6->tos;
   argus->argus_far.src.count    = ntohl(entryPtrV6->pkts);
   argus->argus_far.src.bytes    = ntohl(entryPtrV6->bytes);
   argus->argus_far.src.appbytes = 0;

   switch (argus->argus_far.flow.ip_flow.ip_p) {
      case IPPROTO_TCP: {
         struct ArgusTCPObject tcpbuf, *tcp = &tcpbuf;

         bzero ((char *) tcp, sizeof(*tcp));
         tcp->type = ARGUS_TCP_DSR;
         tcp->length = sizeof(struct ArgusTCPObject);
         tcp->src.flags    = entryPtrV6->tcp_flags;

         if (tcp->src.flags & TH_RST)
            tcp->status |= ARGUS_RESET;

         if (tcp->src.flags & TH_FIN)
            tcp->status |= ARGUS_FIN;

         if ((tcp->src.flags & TH_ACK) || (tcp->src.flags & TH_PUSH) || (tcp->src.flags & TH_URG))
            tcp->status |= ARGUS_CON_ESTABLISHED;

         switch (tcp->src.flags & (TH_SYN|TH_ACK|TH_FIN|TH_PUSH|TH_URG)) {
            case (TH_SYN):
               tcp->status |= ARGUS_SAW_SYN;
               break;
         }

         bcopy ((char *)tcp, &((char *)argus)[argus->ahdr.length], sizeof(*tcp));
         argus->ahdr.length += sizeof(*tcp);
      }
      break;
   }

   bzero ((char *)&mac, sizeof (mac));
   mac.type   = ARGUS_MAC_DSR;
   mac.length = sizeof(mac);
   mac.status = 0;
   entryPtrV6->input = ntohs(entryPtrV6->input);
   entryPtrV6->output = ntohs(entryPtrV6->output);

   bcopy((char *)&entryPtrV6->input, (char *)&mac.phys_union.ether.ethersrc[4], 2);
   bcopy((char *)&entryPtrV6->output,(char *)&mac.phys_union.ether.etherdst[4], 2);

   bcopy ((char *)&mac, &((char *)argus)[argus->ahdr.length], sizeof(mac));
   argus->ahdr.length += sizeof(mac);

#ifdef _LITTLE_ENDIAN
   ArgusHtoN(argus);
#endif

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusParseCiscoRecord (0x%x) returning 0x%x\n", *ptr, argus);
#endif

   return(argus);
}

struct ArgusRecord *
ArgusParseCiscoRecord (u_char **ptr)
{
   struct ArgusRecord *argus = ArgusNetFlowArgusRecord;
   unsigned short *sptr = (unsigned short *) *ptr;

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusParseCiscoRecord (0x%x) version %h\n", *ptr, *sptr);
#endif

   switch (*sptr) {
      case Version1: {
         CiscoFlowHeaderV1_t *hdrPtrV1   = (CiscoFlowHeaderV1_t *) *ptr;
         CiscoFlowEntryV1_t  *entryPtrV1 = (CiscoFlowEntryV1_t *) (hdrPtrV1 + 1);

         bzero ((char *) argus, sizeof (*argus));
         argus->ahdr.type    = ARGUS_FAR | ARGUS_CISCO_NETFLOW;
         argus->ahdr.cause   = ARGUS_STATUS;
         argus->ahdr.length  = sizeof(argus->ahdr) + sizeof(argus->argus_far);

         argus->ahdr.status |= ETHERTYPE_IP;

         argus->argus_far.type   = ARGUS_FAR;
         argus->argus_far.length = sizeof(argus->argus_far);

         if (hdrPtrV1) {
            long time; 
            time = ntohl(entryPtrV1->first);
            argus->argus_far.time.start.tv_sec  = (time - (long)hdrPtrV1->sysUptime)/1000;
            argus->argus_far.time.start.tv_sec += hdrPtrV1->unix_secs;
      
            argus->argus_far.time.start.tv_usec = ((time - (long)hdrPtrV1->sysUptime)%1000) * 1000;
            argus->argus_far.time.start.tv_usec += hdrPtrV1->unix_nsecs/1000;
      
            time = ntohl(entryPtrV1->last);
            argus->argus_far.time.last.tv_sec  = (time - (long)hdrPtrV1->sysUptime)/1000;
            argus->argus_far.time.last.tv_sec += hdrPtrV1->unix_secs;
      
            argus->argus_far.time.last.tv_usec = ((time - (long)hdrPtrV1->sysUptime)%1000) * 1000;
            argus->argus_far.time.last.tv_usec += hdrPtrV1->unix_nsecs/1000;
         }

         argus->argus_far.flow.ip_flow.ip_src = ntohl(entryPtrV1->srcaddr);
         argus->argus_far.flow.ip_flow.ip_dst = ntohl(entryPtrV1->dstaddr);
         argus->argus_far.flow.ip_flow.sport  = ntohs(entryPtrV1->srcport);
         argus->argus_far.flow.ip_flow.dport  = ntohs(entryPtrV1->dstport);
         argus->argus_far.flow.ip_flow.ip_p   = entryPtrV1->prot;
         argus->argus_far.attr_ip.stos        = entryPtrV1->tos;
         argus->argus_far.src.count    = ntohl(entryPtrV1->pkts);
         argus->argus_far.src.bytes    = ntohl(entryPtrV1->bytes);

#ifdef _LITTLE_ENDIAN
         ArgusHtoN(argus);
#endif
         break;
      }

      case Version5: {
         CiscoFlowHeaderV5_t *hdrPtrV5   = (CiscoFlowHeaderV5_t *) ptr;
         CiscoFlowEntryV5_t  *entryPtrV5 = (CiscoFlowEntryV5_t *) (hdrPtrV5 + 1);

         bzero ((char *) argus, sizeof (*argus));
         argus->ahdr.type    = ARGUS_FAR | ARGUS_CISCO_NETFLOW;
         argus->ahdr.cause   = ARGUS_STATUS;
         argus->ahdr.length  = sizeof(argus->ahdr) + sizeof(argus->argus_far);

         argus->ahdr.status |= ETHERTYPE_IP; 
   
         argus->argus_far.type   = ARGUS_FAR;
         argus->argus_far.length = sizeof(argus->argus_far);

         if (hdrPtrV5) {
            long time;
            time = ntohl(entryPtrV5->first);
            argus->argus_far.time.start.tv_sec  = (time - (long)hdrPtrV5->sysUptime)/1000;
            argus->argus_far.time.start.tv_sec += hdrPtrV5->unix_secs;
      
            argus->argus_far.time.start.tv_usec = ((time - (long)hdrPtrV5->sysUptime)%1000) * 1000;
            argus->argus_far.time.start.tv_usec += hdrPtrV5->unix_nsecs/1000;
      
            time = ntohl(entryPtrV5->last);
            argus->argus_far.time.last.tv_sec  = (time - (long)hdrPtrV5->sysUptime)/1000;
            argus->argus_far.time.last.tv_sec += hdrPtrV5->unix_secs;
      
            argus->argus_far.time.last.tv_usec = ((time - (long)hdrPtrV5->sysUptime)%1000) * 1000;
            argus->argus_far.time.last.tv_usec += hdrPtrV5->unix_nsecs/1000;
         }

         argus->argus_far.flow.ip_flow.ip_src = ntohl(entryPtrV5->srcaddr);
         argus->argus_far.flow.ip_flow.ip_dst = ntohl(entryPtrV5->dstaddr);
         argus->argus_far.flow.ip_flow.sport  = ntohs(entryPtrV5->srcport);
         argus->argus_far.flow.ip_flow.dport  = ntohs(entryPtrV5->dstport);
         argus->argus_far.flow.ip_flow.ip_p   = entryPtrV5->prot;
         argus->argus_far.attr_ip.stos        = entryPtrV5->tos;
         argus->argus_far.src.count    = ntohl(entryPtrV5->pkts);
         argus->argus_far.src.bytes    = ntohl(entryPtrV5->bytes);
         argus->argus_far.src.appbytes = 0;

#ifdef _LITTLE_ENDIAN
         ArgusHtoN(argus);
#endif
         break;
      }

      case Version6: {
         CiscoFlowHeaderV6_t *hdrPtrV6   = (CiscoFlowHeaderV6_t *) ptr;
         CiscoFlowEntryV6_t  *entryPtrV6 = (CiscoFlowEntryV6_t *) (hdrPtrV6 + 1);

         bzero ((char *) argus, sizeof (*argus));
         argus->ahdr.type    = ARGUS_FAR | ARGUS_CISCO_NETFLOW;
         argus->ahdr.cause   = ARGUS_STATUS;
         argus->ahdr.length  = sizeof(argus->ahdr) + sizeof(argus->argus_far);

         argus->ahdr.status |= ETHERTYPE_IP; 
   
         argus->argus_far.type   = ARGUS_FAR;
         argus->argus_far.length = sizeof(argus->argus_far);

         if (hdrPtrV6) {
            long time;
            time = ntohl(entryPtrV6->first);
            argus->argus_far.time.start.tv_sec  = (time - (long)hdrPtrV6->sysUptime)/1000;
            argus->argus_far.time.start.tv_sec += hdrPtrV6->unix_secs;
      
            argus->argus_far.time.start.tv_usec = ((time - (long)hdrPtrV6->sysUptime)%1000) * 1000;
            argus->argus_far.time.start.tv_usec += hdrPtrV6->unix_nsecs/1000;
      
            time = ntohl(entryPtrV6->last);
            argus->argus_far.time.last.tv_sec  = (time - (long)hdrPtrV6->sysUptime)/1000;
            argus->argus_far.time.last.tv_sec += hdrPtrV6->unix_secs;
      
            argus->argus_far.time.last.tv_usec = ((time - (long)hdrPtrV6->sysUptime)%1000) * 1000;
            argus->argus_far.time.last.tv_usec += hdrPtrV6->unix_nsecs/1000;
         }

         argus->argus_far.flow.ip_flow.ip_src = ntohl(entryPtrV6->srcaddr);
         argus->argus_far.flow.ip_flow.ip_dst = ntohl(entryPtrV6->dstaddr);
         argus->argus_far.flow.ip_flow.sport  = ntohs(entryPtrV6->srcport);
         argus->argus_far.flow.ip_flow.dport  = ntohs(entryPtrV6->dstport);
         argus->argus_far.flow.ip_flow.ip_p   = entryPtrV6->prot;
         argus->argus_far.attr_ip.stos        = entryPtrV6->tos;
         argus->argus_far.src.count    = ntohl(entryPtrV6->pkts);
         argus->argus_far.src.bytes    = ntohl(entryPtrV6->bytes);
         argus->argus_far.src.appbytes = 0;

#ifdef _LITTLE_ENDIAN
         ArgusHtoN(argus);
#endif
         break;
      }

      case Version8: {
         break;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusParseCiscoRecord (0x%x) returning 0x%x\n", *ptr, argus);
#endif

   return (argus);
}


struct ArgusRecord *
ArgusNetFlowCallRecord (u_char **ptr)
{
   struct ArgusRecord *argus = ArgusNetFlowArgusRecord;
   BinaryRecord_CallRecord_V1 *call = (BinaryRecord_CallRecord_V1 *) *ptr;

   if (*ptr) {
      bzero ((char *) argus, sizeof (*argus));
      argus->ahdr.type    = ARGUS_FAR | ARGUS_CISCO_NETFLOW;
      argus->ahdr.cause   = ARGUS_STATUS;
      argus->ahdr.length  = sizeof(argus->ahdr) + sizeof(argus->argus_far);

      argus->ahdr.status |= ETHERTYPE_IP;

      argus->argus_far.type   = ARGUS_FAR;
      argus->argus_far.length = sizeof(argus->argus_far);

      argus->argus_far.time.start.tv_sec = ntohl(call->starttime);
      argus->argus_far.time.last.tv_sec = ntohl(call->endtime);

      argus->argus_far.time.last.tv_usec = ntohl(call->activetime) % 1000000;
      argus->argus_far.time.last.tv_sec += ntohl(call->activetime) / 1000000;

      argus->argus_far.flow.ip_flow.ip_src = ntohl(call->srcaddr);
      argus->argus_far.flow.ip_flow.ip_dst = ntohl(call->dstaddr);
      argus->argus_far.flow.ip_flow.sport  = ntohs(call->srcport);
      argus->argus_far.flow.ip_flow.dport  = ntohs(call->dstport);
      argus->argus_far.flow.ip_flow.ip_p   = call->prot;
      argus->argus_far.attr_ip.stos        = call->tos;
      argus->argus_far.src.count    = ntohl(call->pkts);
      argus->argus_far.src.bytes    = ntohl(call->octets);
      argus->argus_far.src.appbytes = 0;

#ifdef _LITTLE_ENDIAN
      ArgusHtoN(argus);
#endif
   }
   
#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusNetFlowCallRecord (0x%x) returns 0x%x\n", *ptr, argus);
#endif

   return (argus);
}


struct ArgusRecord *
ArgusNetFlowDetailInt (u_char **ptr)
{
   struct ArgusRecord *argus = ArgusNetFlowArgusRecord;
   BinaryRecord_DetailInterface_V1  *dint = (BinaryRecord_DetailInterface_V1 *) *ptr;

   if (*ptr) {
      dint = NULL;
      bzero ((char *) argus, sizeof (*argus));
   }


#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusNetFlowDetailInt (0x%x) returns 0x%x\n", *ptr, argus);
#endif

   return (argus);
}


ArgusNetFlowHandler ArgusLookUpNetFlow(struct ARGUS_INPUT *, int);

struct ArgusNetFlowParsers {
   int type, size;
   ArgusNetFlowHandler proc;
};

struct ArgusNetFlowParsers ArgusNetFlowParsers [] = {
   { SourceNode, 0, NULL },
   { DestNode, 0, NULL },
   { HostMatrix, 0, NULL },
   { SourcePort, 0, NULL },
   { DestPort, 0, NULL },
   { Protocol, 0, NULL },
   { DetailDestNode, 0, NULL },
   { DetailHostMatrix, 0, NULL },
   { DetailInterface, sizeof(BinaryRecord_DetailInterface_V1), ArgusNetFlowDetailInt },
   { CallRecord, sizeof(BinaryRecord_CallRecord_V1), ArgusNetFlowCallRecord },
   { ASMatrix, 0, NULL },
   { NetMatrix, 0, NULL },
   { DetailSourceNode, 0, NULL },
   { DetailASMatrix, 0, NULL },
   { ASHostMatrix, 0, NULL },
   { HostMatrixInterface, 0, NULL },
   { DetailCallRecord, 0, NULL },
   { RouterAS, 0, NULL },
   { RouterProtoPort, 0, NULL },
   { RouterSrcPrefix, 0, NULL },
   { RouterDstPrefix, 0, NULL },
   { RouterPrefix, 0, NULL },
   { -1, 0, NULL },
};
   
ArgusNetFlowHandler ArgusCiscoNetFlowParse = NULL;
int ArgusWriteConnection (struct ARGUS_INPUT *, unsigned char *, int);

ArgusNetFlowHandler 
ArgusLookUpNetFlow(struct ARGUS_INPUT *input, int type)
{
   ArgusNetFlowHandler retn = NULL;
   struct ArgusNetFlowParsers *p = ArgusNetFlowParsers;

   do {
      if (type == p->type) {
         retn = p->proc;
         input->ArgusReadSize = p->size;
         break;
      }
      p++;
   } while (p->type != -1);

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusLookUpNetFlow (0x%x, %d) returning 0x%x\n", input, type, retn);
#endif

   return (retn);
}


extern char *ArgusVersionStr;

int
ArgusReadConnection (struct ARGUS_INPUT *input, char *filename)
{
   struct ArgusCanonicalRecord canonbuf, *canon = &canonbuf;
   struct ArgusRecord argus;
   u_char *ptr = (u_char *)&argus;
   unsigned char buf[MAXSTRLEN];
   int cnt, fd = -1;

   if (input != NULL) {
      fd = input->fd;
   } else {
      fd = 0;
   }

   if (fd >= 0) {
      if (!(Cflag)) {
         bzero ((char *) &argus, sizeof(argus));
         if ((cnt = read (fd, &argus, sizeof(argus.ahdr))) == sizeof(argus.ahdr)) {
#ifdef ARGUSDEBUG
            ArgusDebug (5, "ArgusReadConnection() read %d bytes\n", cnt);
#endif
            if (filename) {
               if (((ptr[0] == 0x1F) && ((ptr[1] == 0x8B) || (ptr[1] == 0x9D))) ||
                   ((ptr[0] == 'B') && (ptr[1] == 'Z') && (ptr[2] == 'h'))) {
                  char cmd[256];
                  bzero(cmd, 256);
                  close(fd);
                  if (ptr[0] == 'B')
                     strcpy(cmd, "bzip2 -dc ");
                  else
                  if (ptr[1] == 0x8B)
                     strcpy(cmd, "gzip -dc ");
                  else
                     strcpy(cmd, "zcat ");
      
                  strcat(cmd, filename);
       
                  if ((input->pipe = popen(cmd, "r")) == NULL) {
                     ArgusLog (LOG_ERR, "ArgusReadConnection: popen(%s) failed. %s\n", cmd, strerror(errno));
                     close (fd);
                     return (-1);
                  } else {
                     fd = fileno(input->pipe);
                     if ((cnt = read (fd, &argus, sizeof(argus.ahdr))) != sizeof(argus.ahdr)) {
                        ArgusLog (LOG_ERR, "ArgusReadConnection: read from '%s' failed. %s\n", cmd, strerror(errno));
                        pclose(input->pipe);
                        input->pipe = NULL;
                        close (fd);
                        return (-1);
                     } else {
#ifdef ARGUSDEBUG
                        ArgusDebug (5, "ArgusReadConnection() read %d bytes from pipe\n", cnt);
#endif
                     }
                  }
               }
            }
   
            if (argus.ahdr.type & ARGUS_MAR) {
               unsigned short length = ntohs(argus.ahdr.length);
               unsigned int argusid  = ntohl(argus.ahdr.argusid);
               unsigned int sequence = ntohl(argus.ahdr.seqNumber);
               unsigned int status   = ntohl(argus.ahdr.status);

               if (argus.ahdr.cause & ARGUS_ERROR) {
#ifdef ARGUSDEBUG
                  ArgusDebug (5, "ArgusReadConnection() ARGUS_ERROR Mar.\n");
#endif
                  if (status & ARGUS_MAXLISTENEXCD) {
                     ArgusLog (LOG_ALERT, "remote exceed listen error.\n");
                     close (fd);
                     return (-1);
                  }
               }

               if (argus.ahdr.cause == ARGUS_START) {
#ifdef ARGUSDEBUG
                  ArgusDebug (5, "ArgusReadConnection() ARGUS_START Mar.\n");
#endif
                  input->status |= ARGUS_DATA_SOURCE;
                  if ((argusid == ARGUS_COOKIE) && (sequence == 0)) {
                     int size = length - sizeof(argus.ahdr);
      
                     if ((cnt = read (fd, &argus.argus_mar, size)) != size) {
#ifdef ARGUSDEBUG
                        ArgusDebug (5, "ArgusReadConnection() read failed for ARGUS_START Mar %s.\n",
                                        strerror(errno));
#endif
                        close (fd);
                        return (-1);
                     }
      
                     bcopy ((char *) &argus, (char *)&input->ArgusInitCon, sizeof (argus));
                     bcopy ((char *) &argus, (char *) ArgusOriginal, length);
                     ArgusInput = input;
      
                     ArgusHandleDatum ((struct ArgusRecord *)&argus, &ArgusFilterCode);
      
#ifdef _LITTLE_ENDIAN
                     ArgusNtoH(&argus);
#endif
      
                     bcopy ((char *) &argus, (char *)&input->ArgusManStart, sizeof (argus));
                     input->major_version = MAJOR_VERSION_2;
                     input->minor_version = MINOR_VERSION_0;
                     input->ArgusReadSize = argus.argus_mar.record_len;
      
                     argus_parse_init (input);
      
                     if (Sflag && (input->major_version >= MAJOR_VERSION_2)) {
                        if (ntohl(argus.ahdr.status) & ARGUS_SASL_AUTHENTICATE) {
                           if (!(ArgusAuthenticate(input))) {
                              ArgusLog (LOG_ALERT, "incorrect password\n", ArgusProgramName);
                              close(fd);
                              return (-1);
                           }
                        }
   
                        if ((ArgusRemoteFilter != NULL) && (filename == NULL) && (fd != 0)) {
                           int len;

                           bzero(buf, MAXSTRLEN);
                           snprintf ((char *) buf, MAXSTRLEN-1, "FILTER: man or %s", (char *) ArgusRemoteFilter);
                           len = strlen((char *) buf);
                           if (ArgusWriteConnection (input, buf, len) < 0) {
                              ArgusLog (LOG_ALERT, "%s: write remote filter error %s.", strerror(errno));
                              close(fd);
                              return (-1);
                           }
                        }
                     }
                  } else {
                     ArgusLog (LOG_ALERT, "ArgusReadConnection: not Argus-2.0 data stream.");
                     close(fd);
                     fd = -1;
                  }
               } else {

                  struct WriteStruct *ws = NULL;
                  char *ptr;
                  int size;
   
                  bcopy ((char *)&argus, buf, sizeof(argus.ahdr));

                  size = sizeof(*ws) - sizeof(argus.ahdr);
   
                  if ((cnt = read (fd, &buf[sizeof(argus.ahdr)], size)) != size) {
                     fprintf (stderr, "%s: reading %d bytes, got %d bytes. %s", ArgusProgramName, size, cnt, strerror(errno));
                     close (fd);
                     return (-1);

                  } else
                     ws = (struct WriteStruct *) buf;

                  if ((ptr = strstr (ws->ws_init.initString, ArgusVersionStr)) != NULL) {
                     ArgusConvertInitialWriteStruct (ws, &argus);
                     input->major_version = argus.argus_mar.major_version;
                     input->minor_version = argus.argus_mar.minor_version;
                     input->ArgusReadSize = sizeof(*ws);

                     if (initCon == NULL) {
                        if ((initCon = (struct ArgusRecord *) calloc (1, sizeof (argus))) != NULL)
                           bcopy ((char *) &argus, (char *) initCon, sizeof (argus));
                     }

                     bcopy ((char *) &argus, (char *)&input->ArgusInitCon, sizeof (argus));
                     bcopy ((char *)&argus, (char *) ArgusOriginal, sizeof(argus));

                     ArgusInput = input;

                     ArgusHandleDatum ((struct ArgusRecord *)&argus, &ArgusFilterCode);
#ifdef _LITTLE_ENDIAN
                     ArgusNtoH(&argus);
#endif

                     argus_parse_init (input);

                     input->status |= ARGUS_DATA_SOURCE;

                  } else {
                     ArgusLog (LOG_ALERT, "ArgusReadConnection: not Argus-2.0 data stream.");
                     close(fd);
                     fd = -1;
                  }
               }
            } else {
               char *ptr = (char *)&argus;
   
#ifdef ARGUSDEBUG
               ArgusDebug (2, "ArgusReadConnection() testing for CISCO records\n");
#endif
               if (!(strncmp(&ptr[3], "SOURCE", 6))) {
                  BinaryHeaderF2 *ArgusNetFlow = (BinaryHeaderF2 *) buf;
                  int size;
   
                  bcopy ((char *)&argus, buf, sizeof(argus.ahdr));
                  size = sizeof(*ArgusNetFlow) - sizeof(argus.ahdr);
   
                  if ((cnt = read (fd, &buf[sizeof(argus.ahdr)], size)) != size) {
                     fprintf (stderr, "%s: reading %d bytes, got %d bytes. %s", ArgusProgramName, size, cnt, strerror(errno));
                     close (fd);
                     return (-1);
   
                  } else {
#ifdef _LITTLE_ENDIAN
                     ArgusNetFlow->starttime = ntohl(ArgusNetFlow->starttime);
                     ArgusNetFlow->endtime   = ntohl(ArgusNetFlow->endtime);
                     ArgusNetFlow->flows     = ntohl(ArgusNetFlow->flows);
                     ArgusNetFlow->missed    = ntohl(ArgusNetFlow->missed);
                     ArgusNetFlow->records   = ntohl(ArgusNetFlow->records);
#endif
                     bzero ((char *)&argus, sizeof(argus));
   
                     argus.ahdr.type                 = ARGUS_MAR | ARGUS_CISCO_NETFLOW;
                     argus.ahdr.length               = sizeof (argus);
                     argus.ahdr.cause                = ARGUS_START;
                     argus.ahdr.argusid              = ARGUS_COOKIE;
                     argus.argus_mar.startime.tv_sec = ArgusNetFlow->starttime;
                     argus.argus_mar.now.tv_sec      = ArgusNetFlow->starttime;
                     argus.argus_mar.major_version   = major_version;
                     argus.argus_mar.minor_version   = minor_version;
                     argus.argus_mar.flows           = ArgusNetFlow->flows;
                     argus.argus_mar.pktsDrop        = ArgusNetFlow->missed;
                     argus.argus_mar.record_len      = -1;
   
                     input->major_version = argus.argus_mar.major_version;
                     input->minor_version = argus.argus_mar.minor_version;
   
                     if ((input->ArgusCiscoNetFlowParse =
                            ArgusLookUpNetFlow(input, ArgusNetFlow->aggregation)) != NULL) {
#ifdef _LITTLE_ENDIAN
                        ArgusHtoN(&argus);
#endif
                        if (initCon == NULL) {
                           if ((initCon = (struct ArgusRecord *) calloc (1, sizeof (argus))) != NULL)
                              bcopy ((char *) &argus, (char *) initCon, sizeof (argus));
                        }

                        bcopy ((char *) &argus, (char *)&input->ArgusInitCon, sizeof (argus));
                        bcopy ((char *) &argus, (char *) ArgusOriginal, sizeof(argus));
                        ArgusInput = input;

                        ArgusGenerateCanonicalRecord (&argus, canon);
#ifdef _LITTLE_ENDIAN
                        ArgusNtoH(&argus);
#endif
                        argus_parse_init (input);
   
                        if (check_time (&argus)) {
                           if (!(wfile) || !(wfile || RaWriteOut))
                              RaProcessRecord(&argus);
                        }

                        input->status |= ARGUS_CISCO_DATA_SOURCE;

                     } else {
                        fprintf (stderr, "%s: not supported Cisco data stream.\n", ArgusProgramName);
                        close(fd);
                        fd = -1;
                     }
                  }
               } else {
                  ArgusLog (LOG_ALERT, "ArgusReadConnection: not Argus-2.0 data stream.");
                  close(fd);
                  fd = -1;
               }
            }
         } else {
            ArgusLog (LOG_ALERT, "ArgusReadConnection: no data in data stream.\n");
            close(fd);
            fd = -1;
         }
      } else {
         if (!(input->status & ARGUS_CISCO_DATA_SOURCE))
            ArgusLog (LOG_ERR, "ArgusReadConnection(0x%x) expecting Cisco netflow source!\n", input);

         else {
#ifdef ARGUSDEBUG
            ArgusDebug (2, "ArgusReadConnection(0x%x) reading from Cisco Router.\n", input);
#endif
            bzero ((char *)&argus, sizeof(argus));
            argus_parse_init (input);
         }
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (3, "ArgusReadConnection() returning %d\n", fd);
#endif

   return (fd);
}


int
ArgusWriteConnection (struct ARGUS_INPUT *input, unsigned char *buf, int cnt)
{
   int retn = 0, fd = 0, len = 0;
   unsigned char *output = NULL;

   if (input && (!input->filename)) {
      if ((fd = input->fd) > 2) {
#ifdef ARGUS_SASL
         unsigned char outputbuf[MAXSTRLEN];

         output = outputbuf;
         len = cnt;

         if (input->sasl_conn) {
#ifdef ARGUSDEBUG
            ArgusDebug (5, "ArgusWriteConnection: sasl_encode(0x%x, 0x%x, %d, 0x%x, 0x%x)\n",
                                         input->sasl_conn, buf, cnt, &output, &len);
#endif
            if ((retn = sasl_encode(input->sasl_conn, buf, (unsigned int) cnt, (char **) &output, &len)) != SASL_OK)
               ArgusLog (LOG_ERR, "sasl_encode: failed returned %d\n", retn);
         }
#else
         output = buf;
         len = cnt;
#endif /* ARGUS_SASL */

#ifdef ARGUSDEBUG
         ArgusDebug (3, "ArgusWriteConnection: write(%d, 0x%x, %d)\n", fd, output, len);
#endif

         if ((retn = write (fd, output, len)) > 0) {
         } else
            ArgusLog (LOG_ERR, "ArgusWriteConnection(0x%x) error %s\n", input, strerror(errno));

#ifdef ARGUS_SASL
         if (output != outputbuf)
            free(output);
#endif /* ARGUS_SASL */

      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (3, "ArgusWriteConnection(0x%x, 0x%x, %d) returning %d\n", input, buf, cnt, len);
#endif

   return (retn);
}


void ArgusCloseInput(struct ARGUS_INPUT *);
int ArgusReadStreamSocket (struct ARGUS_INPUT *);
int ArgusReadCiscoStreamSocket (struct ARGUS_INPUT *);
int ArgusReadCiscoDatagramSocket (struct ARGUS_INPUT *);

void
ArgusCloseInput(struct ARGUS_INPUT *input)
{
   if (input->pipe) {
      pclose(input->pipe);
      input->pipe = NULL;
   }

   if (!input->filename)
      ArgusWriteConnection (input, (unsigned char *)"DONE: ", strlen("DONE: "));

   if (input->in != NULL)
      fclose(input->in);

   if (input->out != NULL)
      fclose(input->out);

   if (input->ArgusReadBuffer != NULL)
      ArgusFree(input->ArgusReadBuffer);

   if (input->ArgusConvBuffer != NULL)
     ArgusFree(input->ArgusConvBuffer);

   if (RaCloseInputFd) {
      close (input->fd);
      input->fd = -1;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusCloseInput(0x%x) done\n", input);
#endif
}

void ArgusRecordDump (struct ArgusRecord *);

#define HEXDUMP_BYTES_PER_LINE 16
#define HEXDUMP_SHORTS_PER_LINE (HEXDUMP_BYTES_PER_LINE / 2)
#define HEXDUMP_HEXSTUFF_PER_SHORT 5 /* 4 hex digits and a space */
#define HEXDUMP_HEXSTUFF_PER_LINE \
                (HEXDUMP_HEXSTUFF_PER_SHORT * HEXDUMP_SHORTS_PER_LINE)

void
ArgusRecordDump (struct ArgusRecord *argus)
{
   int length = argus->ahdr.length;
   const u_char *cp = (const u_char *) argus;
   u_int oset = 0;
   register u_int i;
   register int s1, s2;
   register int nshorts;
   char hexstuff[HEXDUMP_SHORTS_PER_LINE*HEXDUMP_HEXSTUFF_PER_SHORT+1], *hsp;
   char asciistuff[HEXDUMP_BYTES_PER_LINE+1], *asp;

   nshorts = length / sizeof(u_short);
   i = 0;
   hsp = hexstuff; asp = asciistuff;
   while (--nshorts >= 0) {
           s1 = *cp++;
           s2 = *cp++;
           (void)snprintf(hsp, sizeof(hexstuff) - (hsp - hexstuff),
               " %02x%02x", s1, s2);
           hsp += HEXDUMP_HEXSTUFF_PER_SHORT;
           *(asp++) = (isgraph(s1) ? s1 : '.');
           *(asp++) = (isgraph(s2) ? s2 : '.');
           if (++i >= HEXDUMP_SHORTS_PER_LINE) {
               *hsp = *asp = '\0';
               (void)printf("\n0x%04x\t%-*s\t%s",
                            oset, HEXDUMP_HEXSTUFF_PER_LINE,
                            hexstuff, asciistuff);
               i = 0; hsp = hexstuff; asp = asciistuff;
               oset += HEXDUMP_BYTES_PER_LINE;
           }
   }
   if (length & 1) {
      s1 = *cp++;
      (void)snprintf(hsp, sizeof(hexstuff) - (hsp - hexstuff),
               " %02x", s1);
           hsp += 3;
           *(asp++) = (isgraph(s1) ? s1 : '.');
           ++i;
   }
   if (i > 0) {
           *hsp = *asp = '\0';
           (void)printf("\n0x%04x\t%-*s\t%s",
                        oset, HEXDUMP_HEXSTUFF_PER_LINE,
                        hexstuff, asciistuff);
   }
}


#ifdef ARGUS_SASL
#include <saslint.h>

int ArgusReadSaslStreamSocket (struct ARGUS_INPUT *);

int
ArgusReadSaslStreamSocket (struct ARGUS_INPUT *input)
{
   int retn = 0, fd = input->fd, cnt;
   unsigned int value = 0, *pvalue = &value;
   struct ArgusRecord *argus = NULL;
   char *output = NULL, *end = NULL, *ptr = NULL;
   unsigned int outputlen = 0;


   if ((retn = sasl_getprop(input->sasl_conn, SASL_MAXOUTBUF, (void **) &pvalue)) != SASL_OK)
      ArgusLog (LOG_ERR, "ArgusReadSaslStreamSocket: sasl_getprop %s\n", strerror(errno));

   if (value == 0)
      value = MAXSTRLEN;

   if ((cnt = read (fd, input->ArgusSaslBuffer + input->ArgusSaslBufCnt, MAXSTRLEN)) > 0) {
      input->ArgusSaslBufCnt = cnt;
      ptr = input->ArgusSaslBuffer;

      do {
         cnt = (input->ArgusSaslBufCnt > value) ? value : input->ArgusSaslBufCnt; 

         if (sasl_decode (input->sasl_conn, ptr, cnt, &output, &outputlen) == SASL_OK) {
#ifdef ARGUSDEBUG
            ArgusDebug (5, "ArgusReadSaslStreamSocket (0x%x) sasl_decoded %d bytes\n", input, outputlen);
#endif
            ptr += cnt;

            if (outputlen) {
               argus = (struct ArgusRecord *) output;
               end = output + outputlen;

               while ((char *)argus < end) {
                  input->ArgusReadSocketCnt = ntohs(argus->ahdr.length);
                  bcopy (argus, input->ArgusReadBuffer, input->ArgusReadSocketCnt);

                  if (ArgusHandleDatum (argus, &ArgusFilterCode) == 1) {
                     if (!input->filename)
                        ArgusWriteConnection (input, "DONE: ", strlen("DONE: "));

                     retn = 1;
                     break;

                  } else 
                     (char *)argus += input->ArgusReadSocketCnt;
               }

               free (output);
               input->ArgusSaslBufCnt -= cnt;

            } else {
               input->ArgusSaslBufCnt = 0;
               break;
            }

         } else {
            ArgusLog (LOG_ERR, "ArgusReadSaslStreamSocket: sasl_decode () failed");
            break;
         }

      } while (input->ArgusSaslBufCnt > 0);

   } else {
      retn = 1;

      if ((cnt < 0) && ((errno == EAGAIN) || (errno == EINTR))) {
         retn = 0;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusReadSaslStreamSocket (0x%x) returning %d\n", input, retn);
#endif

   return (retn);
}

#endif /* ARGUS_SASL */


int
ArgusReadStreamSocket (struct ARGUS_INPUT *input)
{
   int retn = 0, fd = input->fd, cnt = 0;
   unsigned short length;

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusReadStreamSocket (0x%x) starting\n", input);
#endif

   if ((cnt = read (fd, input->ArgusReadPtr + input->ArgusReadSocketCnt,
                            (input->ArgusReadSocketSize - input->ArgusReadSocketCnt))) > 0) {
      input->ArgusReadSocketCnt += cnt;

#ifdef ARGUSDEBUG
      ArgusDebug (8, "ArgusReadStreamSocket (0x%x) read %d bytes\n", input, cnt);
#endif

      if (input->ArgusReadSocketCnt == input->ArgusReadSocketSize) {
         if (input->ArgusReadSocketState == ARGUS_READINGHDR) {
            input->ArgusReadSocketState = ARGUS_READINGBLOCK;
            
            bcopy ((char *) &((struct ArgusRecordHeader *)input->ArgusReadPtr)->length, (char *) &length, sizeof(length));
            input->ArgusReadSocketSize = ntohs(length) - sizeof(struct ArgusRecordHeader);
            if ((input->ArgusReadSocketSize > 0) && (input->ArgusReadSocketSize < MAXSTRLEN)) {
               input->ArgusReadPtr = &input->ArgusReadBuffer[input->ArgusReadSocketCnt];
               input->ArgusReadSocketCnt = 0;

            } else {
               ArgusLog(LOG_WARNING, "ArgusReadSocketStream: malformed argus record len %d\n", 
                                  input->ArgusReadSocketSize + sizeof(struct ArgusRecordHeader));
               return(1);
            }

         } else {
            if (input->major_version < 2) {
               ArgusConvertWriteStruct ((struct WriteStruct *)input->ArgusReadBuffer,
                                        (struct ArgusRecord *)input->ArgusConvBuffer);
               bcopy ((char *) input->ArgusConvBuffer, input->ArgusReadBuffer, MAXSTRLEN);
            }

            if (ArgusHandleDatum ((struct ArgusRecord *)input->ArgusReadBuffer, &ArgusFilterCode) == 1) {
               if (!input->filename) {
                  ArgusWriteConnection (input, (unsigned char *)"DONE: ", strlen("DONE: "));
                  retn = 1;
               }
            }

            if (input->major_version >= 2) {
               input->ArgusReadSocketState = ARGUS_READINGHDR;
               input->ArgusReadSocketSize = sizeof(struct ArgusRecordHeader);
            }

            input->ArgusReadPtr = input->ArgusReadBuffer;
            bzero (input->ArgusReadBuffer, MAXSTRLEN);
            input->ArgusReadSocketCnt = 0;
         }
      }
   } else {
#ifdef ARGUSDEBUG
      ArgusDebug (3, "ArgusReadStreamSocket (0x%x) read returned %d\n", input, cnt);
#endif

      retn = 1;

      if ((cnt < 0) && ((errno == EAGAIN) || (errno == EINTR))) {
         retn = 0;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusReadStreamSocket (0x%x) returning %d\n", input, retn);
#endif

   return (retn);
}

int
ArgusReadCiscoStreamSocket (struct ARGUS_INPUT *input)
{
   int cnt = 0, retn = 0;

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusReadCiscoStreamSocket (0x%x) starting\n", input);
#endif

   if ((cnt = read (input->fd, input->ArgusReadPtr + input->ArgusReadSocketCnt,
                 (input->ArgusReadSocketSize - input->ArgusReadSocketCnt))) > 0) {
      input->ArgusReadSocketCnt += cnt;

#ifdef ARGUSDEBUG
      ArgusDebug (8, "ArgusReadCiscoStreamSocket (0x%x) read %d bytes, total %d need %d\n",
                      input, cnt, input->ArgusReadSocketCnt, input->ArgusReadSocketSize);
#endif

      if (input->ArgusReadSocketCnt == input->ArgusReadSocketSize) {
         switch (input->ArgusReadSocketState) {

            case ARGUS_READINGPREHDR: {
               unsigned short *sptr = (unsigned short *) input->ArgusReadPtr;

               input->ArgusReadCiscoVersion = ntohs(*sptr);
               input->ArgusReadSocketNum  = ntohs(*sptr);
               
#define CISCO_VERSION_1		1

               switch (input->ArgusReadCiscoVersion) {
                  case CISCO_VERSION_1:
                     input->ArgusReadSocketSize  = sizeof(CiscoFlowHeaderV1_t) - 4;
                     break;

                  default: {
#ifdef ARGUSDEBUG
                     ArgusDebug (8, "ArgusReadCiscoStreamSocket (0x%x) read version %d preheader num %d\n",
                                       input, input->ArgusReadCiscoVersion, input->ArgusReadSocketNum);
#endif
                  }
               }

               input->ArgusReadSocketState = ARGUS_READINGHDR;
               input->ArgusReadPtr = &input->ArgusReadBuffer[input->ArgusReadSocketCnt];
               input->ArgusReadSocketCnt = 0;
               break;
            }

            case ARGUS_READINGHDR: {
#ifdef ARGUSDEBUG
               ArgusDebug (7, "ArgusReadCiscoStreamSocket (0x%x) read record header\n", input);
#endif
               switch (input->ArgusReadCiscoVersion) {
                  case CISCO_VERSION_1:
                     input->ArgusReadSocketSize  = sizeof(CiscoFlowEntryV1_t);
                     break;

                  default: {
#ifdef ARGUSDEBUG
                     ArgusDebug (7, "ArgusReadCiscoStreamSocket (0x%x) read header\n", input);
#endif
                  }
               }
               
               input->ArgusReadSocketState = ARGUS_READINGBLOCK;
               input->ArgusReadPtr = &input->ArgusReadBuffer[input->ArgusReadSocketCnt];
               input->ArgusReadBlockPtr = input->ArgusReadPtr;
               input->ArgusReadSocketCnt = 0;
               break;
            }

            default:
#ifdef ARGUSDEBUG
               ArgusDebug (7, "ArgusReadCiscoStreamSocket (0x%x) read record complete\n", input);
#endif
               if (ArgusHandleDatum (input->ArgusCiscoNetFlowParse (&input->ArgusReadPtr), &ArgusFilterCode))
                  return(1);

               if (!(input->ArgusReadSocketNum--)) {
                  input->ArgusReadPtr = input->ArgusReadBuffer;
                  bzero (input->ArgusReadBuffer, MAXSTRLEN);

               } else {
                  input->ArgusReadPtr = input->ArgusReadBlockPtr;
               }
   
               input->ArgusReadSocketCnt = 0;
               break;
         }
      }

   } else {
#ifdef ARGUSDEBUG
     ArgusDebug (3, "ArgusReadCiscoStreamSocket (0x%x) read returned %d error %s\n", input, cnt, strerror(errno));
#endif

      retn = 1;

      if ((cnt < 0) && ((errno == EAGAIN) || (errno == EINTR))) {
         retn = 0;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusReadCiscoStreamSocket (0x%x) returning %d\n", input, retn);
#endif

   return (retn);
}


int ArgusCiscoDatagramSocketStart = 1;
int
ArgusReadCiscoDatagramSocket (struct ARGUS_INPUT *input)
{
   int retn = 0, cnt = 0, count = 0, i = 0;
   unsigned short *sptr = NULL;
   unsigned char *ptr = NULL;

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusReadCiscoDatagramSocket (0x%x) starting\n", input);
#endif

   if ((cnt = read (input->fd, input->ArgusReadPtr, input->ArgusReadSocketSize)) > 0) {
      input->ArgusReadSocketCnt = cnt;
      sptr = (unsigned short *) input->ArgusReadPtr;
      ptr = (unsigned char *) input->ArgusReadPtr;

#ifdef ARGUSDEBUG
      ArgusDebug (8, "ArgusReadCiscoDatagramSocket (0x%x) read %d bytes, capacity %d\n",
                      input, cnt, input->ArgusReadSocketCnt, input->ArgusReadSocketSize);
#endif

#define CISCO_VERSION_1		1
#define CISCO_VERSION_5		5
#define CISCO_VERSION_6		6
#define CISCO_VERSION_8		8

      switch (input->ArgusReadCiscoVersion = ntohs(*sptr)) {
         case CISCO_VERSION_1: {
            CiscoFlowHeaderV1_t *ArgusNetFlow = (CiscoFlowHeaderV1_t *) ptr;
            CiscoFlowHeaderV1_t *nfptr = (CiscoFlowHeaderV1_t *) sptr;

            input->ArgusCiscoNetFlowParse = ArgusParseCiscoRecordV1;
            ArgusNetFlow->version    = ntohs(nfptr->version);
            ArgusNetFlow->count      = ntohs(nfptr->count);
            ArgusNetFlow->sysUptime  = ntohl(nfptr->sysUptime);
            ArgusNetFlow->unix_secs  = ntohl(nfptr->unix_secs);
            ArgusNetFlow->unix_nsecs = ntohl(nfptr->unix_nsecs);
            ArgusNetFlowRecordHeader = ptr;
            ptr = (unsigned char *) (nfptr + 1);
            count = ArgusNetFlow->count;
         }
         break;

         case CISCO_VERSION_5: {
            CiscoFlowHeaderV5_t *ArgusNetFlow = (CiscoFlowHeaderV5_t *) ptr;
            CiscoFlowHeaderV5_t *nfptr = (CiscoFlowHeaderV5_t *) sptr;

            input->ArgusCiscoNetFlowParse = ArgusParseCiscoRecordV5;
            ArgusNetFlow->version       = ntohs(nfptr->version);
            ArgusNetFlow->count         = ntohs(nfptr->count);
            ArgusNetFlow->sysUptime     = ntohl(nfptr->sysUptime);
            ArgusNetFlow->unix_secs     = ntohl(nfptr->unix_secs);
            ArgusNetFlow->unix_nsecs    = ntohl(nfptr->unix_nsecs);
            ArgusNetFlow->flow_sequence = ntohl(nfptr->flow_sequence);
            ArgusNetFlowRecordHeader = ptr;
            ptr = (unsigned char *) (nfptr + 1);
            count = ArgusNetFlow->count;
         }

         case CISCO_VERSION_6: {
            CiscoFlowHeaderV6_t *ArgusNetFlow = (CiscoFlowHeaderV6_t *) ptr;
            CiscoFlowHeaderV6_t *nfptr = (CiscoFlowHeaderV6_t *) sptr;

            input->ArgusCiscoNetFlowParse = ArgusParseCiscoRecordV6;
            ArgusNetFlow->version       = ntohs(nfptr->version);
            ArgusNetFlow->count         = ntohs(nfptr->count);
            ArgusNetFlow->sysUptime     = ntohl(nfptr->sysUptime);
            ArgusNetFlow->unix_secs     = ntohl(nfptr->unix_secs);
            ArgusNetFlow->unix_nsecs    = ntohl(nfptr->unix_nsecs);
            ArgusNetFlow->flow_sequence = ntohl(nfptr->flow_sequence);
            ArgusNetFlowRecordHeader = ptr;
            ptr = (unsigned char *) (nfptr + 1);
            count = ArgusNetFlow->count;
         }
         break;

         case CISCO_VERSION_8: {
            CiscoFlowHeaderV8_t *ArgusNetFlow = (CiscoFlowHeaderV8_t *) ptr;
            CiscoFlowHeaderV8_t *nfptr = (CiscoFlowHeaderV8_t *) sptr;

            ArgusNetFlow->version       = ntohs(nfptr->version);
            ArgusNetFlow->count         = ntohs(nfptr->count);
            ArgusNetFlow->sysUptime     = ntohl(nfptr->sysUptime);
            ArgusNetFlow->unix_secs     = ntohl(nfptr->unix_secs);
            ArgusNetFlow->unix_nsecs    = ntohl(nfptr->unix_nsecs);
            ArgusNetFlow->flow_sequence = ntohl(nfptr->flow_sequence);
            ArgusNetFlowRecordHeader = ptr;
            ptr = (unsigned char *) (nfptr + 1);
            count = ArgusNetFlow->count;

            if ((input->ArgusCiscoNetFlowParse =
                   ArgusLookUpNetFlow(input, ArgusNetFlow->agg_method)) != NULL) {
            }
         }
         break;
      }
               
      for (i = 0; i < count; i++) {
         if (ArgusHandleDatum (input->ArgusCiscoNetFlowParse (&ptr), &ArgusFilterCode))
            return(1);
      }

   } else {
#ifdef ARGUSDEBUG
     ArgusDebug (3, "ArgusReadCiscoDatagramSocket (0x%x) read returned %d error %s\n", input, cnt, strerror(errno));
#endif


      if ((cnt < 0) && ((errno == EAGAIN) || (errno == EINTR))) {
         retn = 0;
      } else
         retn = 1;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusReadCiscoDatagramSocket (0x%x) returning %d\n", input, retn);
#endif

   return (retn);
}


void
ArgusReadStream ()
{
   int retn = 0, width = -1, i, found = 0;
   struct timeval now, wait, timeoutValue;
   struct ARGUS_INPUT *input = NULL;
   fd_set readmask;

   for (i = 0; i < ARGUS_MAX_REMOTE_CONN; i++)
      if (ArgusRemoteFDs[i] != NULL)
         found++;

   if (!(found)) {
#ifdef ARGUSDEBUG
      ArgusDebug (4, "ArgusReadStream() ArgusRemoteFDs is empty\n");
#endif

      return;
   }

   bzero ((char *)&timeoutValue, sizeof(timeoutValue));

   if (gettimeofday (&now, NULL) == 0) {
      ArgusAdjustGlobalTime(&ArgusGlobalTime, &ArgusNowTime);

      FD_ZERO (&readmask);
      for (i = 0; i < ARGUS_MAX_REMOTE_CONN; i++)
         if (ArgusRemoteFDs[i] != NULL) {
            FD_SET (ArgusRemoteFDs[i]->fd, &readmask);
            width = (width < ArgusRemoteFDs[i]->fd) ? ArgusRemoteFDs[i]->fd : width;
         }
      width++;

      wait.tv_sec = 0;
      wait.tv_usec = 200000;

#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusReadStream() starting\n");
#endif
   
      for (;;) {
         if ((retn = select (width, &readmask, NULL, NULL, &wait)) >= 0) {
            for (i = 0; i < ARGUS_MAX_REMOTE_CONN; i++) {
               if ((input = ArgusRemoteFDs[i]) != NULL) {
                  if (FD_ISSET (input->fd, &readmask)) {
                     ArgusInput = input;
                     if (input->status & ARGUS_DATA_SOURCE) {
#ifdef ARGUS_SASL
                        if (input->sasl_conn) {
                           if (ArgusReadSaslStreamSocket (input)) {
                              ArgusCloseInput(input);
                              ArgusRemoteFDs[i] = NULL;
                           }

                        } else
#endif /* ARGUS_SASL */
                        if (ArgusReadStreamSocket (input)) {
                           ArgusCloseInput(input);
                           ArgusRemoteFDs[i] = NULL;
                        }
                     } else {
                        if (Cflag) {
                           if (ArgusReadCiscoDatagramSocket (input)) {
                              ArgusCloseInput(input);
                              ArgusRemoteFDs[i] = NULL;
                           }
                        } else {
                           if (ArgusReadCiscoStreamSocket (input)) {
                              ArgusCloseInput(input);
                              ArgusRemoteFDs[i] = NULL;
                           }
                        }
                     }
                  }

                  if (ArgusGlobalTime.tv_usec < 0)
                     ArgusGlobalTime.tv_usec = 0;

                  gettimeofday (&now, NULL);
                  ArgusAdjustGlobalTime(&ArgusGlobalTime, &ArgusNowTime);

                  if (input->hostname && input->ArgusMarInterval) {
                     if (input->ArgusLastTime.tv_sec) {
                        if ((ArgusGlobalTime.tv_sec - input->ArgusLastTime.tv_sec) > (3 * input->ArgusMarInterval)) {
                           ArgusLog (LOG_WARNING, "ArgusReadStream %s: idle stream: closing", input->hostname);
                           ArgusCloseInput(input);
                           ArgusRemoteFDs[i] = NULL;
                        }
                     }
                  }
               }
            }

            if (timeoutValue.tv_sec == 0) {
               timeoutValue = ArgusGlobalTime;

               timeoutValue.tv_sec  += RaClientTimeout.tv_sec;
               timeoutValue.tv_usec += RaClientTimeout.tv_usec;

               if (timeoutValue.tv_usec >= 1000000) {
                  timeoutValue.tv_sec  += 1;
                  timeoutValue.tv_usec -= 1000000;
               }

               if (timeoutValue.tv_usec < 0) {
                  timeoutValue.tv_usec = 0;
               }
            }

            if ((now.tv_sec  > timeoutValue.tv_sec) ||
               ((now.tv_sec == timeoutValue.tv_sec) &&
                (now.tv_usec > timeoutValue.tv_usec))) {

               ArgusClientTimeout ();

               if (Tflag) {
                  if ((Tflag - 1) == 0) {
                  ArgusShutDown(0);
                  }
                  Tflag--;
               }

               timeoutValue = now;
               timeoutValue.tv_sec  += RaClientTimeout.tv_sec;
               timeoutValue.tv_usec += RaClientTimeout.tv_usec;

               if (timeoutValue.tv_usec >= 1000000) {
                  timeoutValue.tv_sec  += 1;
                  timeoutValue.tv_usec -= 1000000;
               }

               if (timeoutValue.tv_usec < 0) {
                  timeoutValue.tv_usec = 0;
               }
            }

            width = -1;
            FD_ZERO (&readmask);
            for (i = 0; i < ARGUS_MAX_REMOTE_CONN; i++)
               if (ArgusRemoteFDs[i] != NULL) {
                  FD_SET (ArgusRemoteFDs[i]->fd, &readmask);
                  width = (width < ArgusRemoteFDs[i]->fd) ? ArgusRemoteFDs[i]->fd : width;
               }

            if (width < 0)
               break;
            else
               width++;

            wait.tv_sec  = 0;
            wait.tv_usec = 200000;

         } else {

#ifdef ARGUSDEBUG
            ArgusDebug (3, "ArgusReadStream() select returned %s\n", strerror(errno));
#endif
            if (errno != EINTR)
               break;
         }
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusReadStream() returning\n");
#endif
}

#include <signal.h>
int ArgusTotalRecords = 0;

/*
void
ArgusProcessRecord (struct ArgusRecord *ptr)
{
   if (ptr->ahdr.type & ARGUS_MAR)
      process_man (ptr);

   else {
      switch (ptr->ahdr.status & 0xFFFF) {
         case ETHERTYPE_IP:
            switch (ptr->argus_far.flow.ip_flow.ip_p) {
               case IPPROTO_TCP:              
                  process_tcp (ptr);
                  break;

               case IPPROTO_UDP:              
                  process_udp (ptr);
                  break;

               case IPPROTO_ICMP:              
                  process_icmp (ptr);
                  break;

               default:
                  process_ip (ptr);
                  break;
            }
            break;

         case ETHERTYPE_ARP:
         case ETHERTYPE_REVARP:
            process_arp (ptr);
            break;

         default:
            process_non_ip (ptr);
            break;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusProcessRecord (0x%x) returning\n", ptr);
#endif
}
*/



#include <netdb.h>

extern void ArgusLog (int, char *, ...);

#define ARGUS_DEFAULTCISCOPORT		9995

char *ArgusRecordType = NULL;

extern int ArgusInitializeAuthentication(void);

#include <netinet/in.h>
#include <arpa/inet.h>

int
ArgusGetServerSocket (struct ARGUS_INPUT *input)
{
   int retn = -1;
   struct sockaddr_in server;
   struct servent *sp;
   struct hostent *hp;
   int s, type = 0;
   unsigned short portnum = 0;

   if (input->status & ARGUS_DATA_SOURCE) {
      ArgusRecordType = "Argus";
      type = SOCK_STREAM;
      if (!input->portnum) {
         if (!ArgusPortNum) {
            if ((sp = getservbyname ("monitor", "tcp")) != NULL)
               portnum = sp->s_port;
            else
               portnum = htons(ARGUS_DEFAULTPORT);
         } else
            portnum = htons(ArgusPortNum);

         input->portnum = ntohs(portnum);

      } else
         portnum = htons(input->portnum);

   } else {
      if (input->status & ARGUS_CISCO_DATA_SOURCE) {
         struct ArgusRecord argus;

         ArgusRecordType = "Netflow";
         type = SOCK_DGRAM;
         if (!input->portnum) {
            if (!ArgusPortNum)
               portnum = htons(ARGUS_DEFAULTCISCOPORT);
            else
               portnum = htons(ArgusPortNum);

            input->portnum = ntohs(portnum);

         } else
            portnum = htons(input->portnum);

         bzero ((char *)&argus, sizeof(argus));
         argus.ahdr.type                 = ARGUS_MAR | ARGUS_CISCO_NETFLOW;
         argus.ahdr.length               = sizeof (argus);
         argus.ahdr.cause                = ARGUS_START;
         argus.ahdr.argusid              = ARGUS_COOKIE;
         argus.argus_mar.startime.tv_sec = ArgusGlobalTime.tv_sec;
         argus.argus_mar.now.tv_sec      = ArgusGlobalTime.tv_sec;
         argus.argus_mar.major_version   = major_version;
         argus.argus_mar.minor_version   = minor_version;
         argus.argus_mar.record_len      = -1;

         input->major_version = argus.argus_mar.major_version;
         input->minor_version = argus.argus_mar.minor_version;

         bcopy ((char *) &argus, (char *)&input->ArgusInitCon, sizeof (argus));
         bcopy ((char *) &argus, (char *) ArgusOriginal, sizeof(argus));
         ArgusInput = input;

      } else
         ArgusLog (LOG_ERR, "ArgusGetServerSocket(0x%x) unknown type\n", input);
   }

   bzero ((char *) &server, sizeof (server));

   if ((s = socket (AF_INET, type, 0)) >= 0) {
      if (type == SOCK_DGRAM) {
         server.sin_addr.s_addr = INADDR_ANY;
         server.sin_family = AF_INET;
         server.sin_port = portnum;
         fprintf (stderr, "%s: Binding port %d Expecting %s records\n", ArgusProgramName,
                             ntohs(portnum), ArgusRecordType); 
         if ((bind (s, (struct sockaddr *)&server, sizeof(server))) < 0)
            ArgusLog (LOG_ERR, "bind (%d, %s:%hu, %d) failed %s\n", s, inet_ntoa(server.sin_addr),
                                           server.sin_port, sizeof(server), strerror(errno));
         retn = s;
         input->fd = s;

      } else {
         int optval = 1;
         if (setsockopt(s, SOL_SOCKET, SO_KEEPALIVE, (char *)&optval, sizeof(int)) < 0) {
#ifdef ARGUSDEBUG
            ArgusDebug (2, "setsockopt(%d, SOL_SOCKET, SO_KEEPALIVE, 0x%x, %d) failed:", s, optval, sizeof(int));
#endif
         }

         if ((hp = gethostbyaddr ((char *)&input->addr, sizeof (input->addr), AF_INET)) != NULL) {
            input->hostname = strdup(hp->h_name);
            bcopy ((char *) hp->h_addr, (char *)&server.sin_addr, hp->h_length);
            server.sin_family = hp->h_addrtype;
            server.sin_port = portnum;
#ifdef ARGUSDEBUG
            ArgusDebug (1, "Trying %s port %d Expecting %s records\n", (hp->h_name) ?
                        (hp->h_name) : intoa (input->addr), ntohs(portnum), ArgusRecordType); 
#endif
        } else {
            server.sin_addr.s_addr = input->addr;
            server.sin_family = AF_INET;
            server.sin_port = portnum;
#ifdef ARGUSDEBUG
            ArgusDebug (1, "Trying %s port %d Expecting %s records\n", 
                          intoa (input->addr), ntohs(portnum), ArgusRecordType); 
#endif
         }

         if ((connect (s, (struct sockaddr *)&server, sizeof(server))) < 0) {
            ArgusLog (LOG_WARNING, "connect to %s:%hu failed %s\n", 
                      inet_ntoa(server.sin_addr), ntohs(server.sin_port), 
                      strerror(errno));
         } else {
            retn = s;
            input->fd = s;

#ifdef ARGUSDEBUG
            if (type == SOCK_DGRAM)
               ArgusDebug (1, "receiving\n");
            else
               ArgusDebug (1, "connected\n");
#endif
         }
      }

   } else {
      ArgusLog (LOG_ERR, "ArgusGetServerSocket: socket() failed. %s\n", strerror(errno));
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusGetServerSocket (0x%x) returning %d\n", input, retn);
#endif

   return (retn);
}


int
ArgusAddModeList (char *ptr)
{
   int retn = 0;
   struct ArgusModeStruct *mode, *list;

   if (ptr) {
      if ((mode = (struct ArgusModeStruct *) ArgusCalloc (1, sizeof(struct ArgusModeStruct))) != NULL) {
         if ((list = ArgusModeList) != NULL) {
            while (list->nxt)
               list = list->nxt;
            list->nxt = mode;
         } else
            ArgusModeList = mode;

         mode->mode = strdup(ptr);
         retn = 1;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (2, "ArgusAddFileList (%s) returning %d\n", ptr, retn);
#endif

   return (retn);
}


int
ArgusAddFileList (char *ptr)
{
   register int retn = 0;
   register struct ARGUS_INPUT *file, *list;

   if (ptr) {
      if ((file = (struct ARGUS_INPUT *) ArgusCalloc (1, sizeof(struct ARGUS_INPUT))) != NULL) {
         if ((list = ArgusInputFileList) != NULL) {
            while (list->nxt) list = list->nxt;
            list->nxt = file;
         } else
            ArgusInputFileList = file;

         file->filename = strdup(ptr);
         file->fd = -1;
         retn = 1;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (2, "ArgusAddFileList (%s) returning %d\n", ptr, retn);
#endif

   return (retn);
}

void
ArgusDeleteFileList ()
{
   struct ARGUS_INPUT *addr = ArgusInputFileList;

   while (addr) {
     if (addr->filename)
        free(addr->filename);

     addr = addr->nxt;
     ArgusFree(ArgusInputFileList);
     ArgusInputFileList = addr;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (2, "ArgusDeleteFileList () returning\n");
#endif
}


#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>


int
ArgusAddHostList (char *str)
{
   int retn = 0;
   struct ARGUS_INPUT *addr = NULL;
   unsigned int ipaddr, **name;
   long int portnum = 0;
   char *ptr = NULL, *endptr = NULL;

   if ((ptr = strchr (str, (int)':')) != NULL) {
      *ptr++ = '\0';

      portnum = strtol(ptr, &endptr, 10);
   }

   if ((ipaddr = (unsigned int) inet_addr (str)) == (unsigned int) -1) {
      if ((name = (unsigned int **) argus_nametoaddr (str)) != NULL) {
         if (*name) {
            if ((addr = (struct ARGUS_INPUT *) ArgusCalloc (1, sizeof (struct ARGUS_INPUT))) != NULL) {
               addr->nxt = ArgusRemoteHostList;
               ArgusRemoteHostList = addr;
               addr->addr = ntohl(**name);
               addr->hostname = strdup(str);
               addr->portnum = portnum;
               retn = 1;
            }
         }
      }
   } else
      if ((addr = (struct ARGUS_INPUT *) ArgusCalloc (1, sizeof (struct ARGUS_INPUT))) != NULL) {
         addr->nxt = ArgusRemoteHostList;
         ArgusRemoteHostList = addr;
         addr->addr = ipaddr;
         addr->portnum = portnum;
         retn = 1;
      } else
         ArgusLog (LOG_ERR, "ArgusAddHostList(%s) ArgusCalloc %s", str, strerror(errno));

   if (addr)
      addr->status |= ARGUS_DATA_SOURCE;

#ifdef ARGUSDEBUG
   ArgusDebug (2, "ArgusAddHostList (%s) returning %d\n", str, retn);
#endif

   return (retn);
}

void
ArgusDeleteHostList ()
{
   struct ARGUS_INPUT *addr = ArgusRemoteHostList;

   while (addr) {
     if (addr->hostname)
        free(addr->hostname);

     addr = addr->nxt; 
     ArgusFree(ArgusRemoteHostList);
     ArgusRemoteHostList = addr;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (2, "ArgusDeleteHostList () returning\n");
#endif
}


int
ArgusWriteNewLogfile (char *file, struct ArgusRecord *argus)
{
   int retn = 0;

   if (file) {
      if (ArgusOutputFile.filename != NULL) {
         if (strcmp(file, ArgusOutputFile.filename)) {
            free(ArgusOutputFile.filename);
            ArgusOutputFile.filename = strdup(file);
            if (ArgusOutputFile.fd != NULL)
               fclose (ArgusOutputFile.fd);
            ArgusOutputFile.fd = NULL;
            firstWrite++;
         }
      } else
         ArgusOutputFile.filename = strdup(file);

/*
      if (strcmp (file, "-")) {
         if ((stat (file, &ArgusOutputFile.statbuf) < 0))
            if (ArgusOutputFile.fd != NULL) {
               fclose (ArgusOutputFile.fd);
               ArgusOutputFile.fd = NULL;
            }

         if (ArgusOutputFile.fd == NULL) {
            if ((ArgusOutputFile.fd = fopen (file, "a+")) == NULL)
               ArgusLog (LOG_ERR, "ArgusWriteNewLogfile(%s, 0x%x) fopen %s", strerror(errno));
            
            firstWrite++;
         }

*/

      if (strcmp (file, "-")) {
         if ((stat (file, &ArgusOutputFile.statbuf) < 0))
            firstWrite++;

         if ((ArgusOutputFile.fd = fopen (file, "a+")) == NULL)
            ArgusLog (LOG_ERR, "ArgusWriteNewLogfile(%s, 0x%x) fopen %s", strerror(errno));

      } else {
         if (ArgusOutputFile.fd == NULL) {
            ArgusOutputFile.fd = stdout;
            firstWrite++;
         }
      }

      if (firstWrite) {
         if (!(fwrite ((char *)&ArgusInput->ArgusInitCon,
                            ntohs(ArgusInput->ArgusInitCon.ahdr.length), 1, ArgusOutputFile.fd)))
            ArgusLog (LOG_ERR, "ArgusWriteNewLogfile(%s, 0x%x) fwrite error %s", file, argus, strerror(errno));
      }

      if (argus) {
         if (!(fwrite (argus, ntohs(argus->ahdr.length), 1, ArgusOutputFile.fd)))
            ArgusLog (LOG_ERR, "ArgusWriteNewLogfile(%s, 0x%x) fwrite error %s", file, argus, strerror(errno));
      }


      fflush (ArgusOutputFile.fd);

      if (strcmp (file, "-")) {
         fclose(ArgusOutputFile.fd);
         ArgusOutputFile.fd = NULL;
      }

      if (firstWrite)
         firstWrite = 0;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusWriteNewLogFile (%s, 0x%x) returning %d\n", file, argus, retn);
#endif

   return (retn);
}


int
check_time (struct ArgusRecord *ptr)
{
   struct tm tmbuf, *tm;
   int retn = 0;
   struct timeval *start, *last, lastbuf;

   if (ptr->ahdr.type & ARGUS_MAR) {
      start = &ptr->argus_mar.startime;
      last  = &ptr->argus_mar.now;
      if (!(start->tv_sec))
         start  = &ptr->argus_mar.now;

   } else {
      start = &ptr->argus_far.time.start;
      last  = &ptr->argus_far.time.last;

      if (!(start->tv_sec))
         *start  = *last;

      if (ArgusThisFarStatus & ARGUS_AGR_DSR_STATUS) {
         lastbuf = ((struct ArgusAGRStruct *)ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX])->lasttime;
      }

      ArgusGlobalTime = *last;
   }

   gettimeofday (&ArgusNowTime, 0L);
 
   if (tflag) {
      if (!explicit_date) {
         tm = localtime_r((time_t *)&start->tv_sec, &tmbuf);
         if (tm->tm_yday != starTimeFilter.tm_yday) {
            bcopy ((char *) tm, (char *) &tmbuf, sizeof (struct tm));
            if (check_time_format (&tmbuf, timearg))
               ArgusLog (LOG_ERR, "time syntax error %s\n", timearg);
         }
      }

      if (ptr->ahdr.type & ARGUS_MAR) {
         if (ptr->ahdr.status & ARGUS_START) {
            if ((ptr->argus_mar.now.tv_sec >= startime_t) &&
                (ptr->argus_mar.now.tv_sec <= lasttime_t))
               retn++;
         } else {
            if ((ptr->argus_mar.now.tv_sec >= startime_t) &&
                (ptr->argus_mar.now.tv_sec <= lasttime_t))
               retn++;
         }
      } else {
         if (((start->tv_sec >= startime_t) && (start->tv_sec <= lasttime_t)) ||
             ((last->tv_sec >= startime_t)  && (last->tv_sec <= lasttime_t)) ||
             ((start->tv_sec <  startime_t) && (last->tv_sec >  lasttime_t)))
            retn++;
      }
   } else
      retn++;

   return (retn);
}


#include <ctype.h>

int
parseUserDataArg (char **arg, char *args[], int ind)
{
   int retn = -1;
   char buf[64], *ptr = buf;
   
   bzero (buf, 64);
   strcpy (buf, *arg);
   ptr += strlen (buf);

   if ((ptr = strchr(*arg, ':')) && (*(ptr + 1) != '\0')) {
      retn = 0;
   } else {
      if (args) {
         if (args[ind] && (*args[ind] == ':')) {
            if (strlen (args[ind]) == 1) {
               strcat (buf, ":");
               strcat (buf, args[ind + 1]);
               retn = 2;
            } else {
               ptr = args[ind];
               if (isdigit((int)*(ptr + 1))) {
                  strcat (buf, args[ind]);
                  retn = 1;
               } else
                  retn = 0;
            }
         } else
            retn = 0;
      } else
         retn = 0;
   }

   *arg = savestr(buf);

   if ((ptr = strchr (buf, ':')) != NULL) {
      ptr++;
      if (*buf == 's')
         ArgusSrcUserDataLen = atoi(buf + 1);
      else
         ArgusLog (LOG_ERR, "user data syntax error %s\n", buf);

      if (*ptr == 'd')
         ArgusDstUserDataLen = atoi(ptr + 1);
      else
         ArgusLog (LOG_ERR, "user data syntax error %s\n", buf);

   } else {
      if (isdigit((int)*buf)) {
         ArgusSrcUserDataLen = atoi(buf);
         ArgusDstUserDataLen = atoi(buf);
      } else {
         if (*buf == 's')
            ArgusSrcUserDataLen = atoi(buf + 1);

         if (*buf == 'd')
            ArgusDstUserDataLen = atoi(buf + 1);
      }
   }

   if (retn < 0)
      ArgusLog (LOG_ERR, "user data syntax error %s\n", buf);

   return (retn);
}

int
parseTimeArg ( char **arg, char *args[], int ind, struct tm *tm)
{
   int retn = -1;
   char buf[64], *ptr = buf;

   bzero (buf, 64);
   strcpy (buf, *arg);
   ptr += strlen (buf);
   if ((ptr = strchr(*arg, '-')) && (*(ptr + 1) != '\0')) {
      retn = 0;
   } else {
      if (args) {
         if (args[ind] && (*args[ind] == '-')) {
            if (strlen (args[ind]) == 1) {
               strcat (buf, "-");
               strcat (buf, args[ind + 1]);
               retn = 2;
            } else {
               ptr = args[ind];
               if (isdigit((int)*(ptr + 1))) {
                  strcat (buf, args[ind]);
                  retn = 1;
               } else
                  retn = 0;
            }
         } else
            retn = 0;
      }
   }

   if ((ptr = strchr(*arg, '.')) || (ptr = strchr(*arg, '/')))
      explicit_date++;

   if (check_time_format (tm, buf))
      ArgusLog (LOG_ERR, "time syntax error %s\n", buf);

   *arg = savestr(buf);
   return (retn);
}


#define ARGUS_YEAR	1
#define ARGUS_MONTH	2
#define ARGUS_DAY	3
#define ARGUS_HOUR	4
#define ARGUS_MIN	5
#define ARGUS_SEC	6


int
check_time_format (struct tm *tm, char *str)
{
   int retn = 0;
   char *ptr, buf[64];

   /*[[[yyyy/]mm/]dd.]hh[:mm[:ss]] - [[[yyyy/]mm/]dd.]hh[:mm[:ss]]*/

   strcpy (buf, str);

   if ((ptr = strchr(buf, '-')) != NULL) {
      *ptr = '\0';
      if ((retn = parseTime (&starTimeFilter, tm, buf)) > 0)
         if ((retn = parseTime (&lastTimeFilter, &starTimeFilter, ptr + 1)) > 0)
            retn = 0;

   } else {
      if ((retn = parseTime (&starTimeFilter, tm, buf)) > 0) {
         bcopy ((char *)&starTimeFilter, (char *)&lastTimeFilter, sizeof(struct tm));
         switch (retn) {
            case ARGUS_YEAR:  lastTimeFilter.tm_year++; break;
            case ARGUS_MONTH: lastTimeFilter.tm_mon++; break;
            case ARGUS_DAY:   lastTimeFilter.tm_mday++; break;
            case ARGUS_HOUR:  lastTimeFilter.tm_hour++; break;
            case ARGUS_MIN:   lastTimeFilter.tm_min++; break;
            case ARGUS_SEC:   lastTimeFilter.tm_sec++; break;
         }

         retn = 0;
      }
   }

   if (retn == 0) {
      startime_t = timelocal (&starTimeFilter);
      lasttime_t = timelocal (&lastTimeFilter);

      if (!(lasttime_t >= startime_t)) {
         fprintf (stderr, "error: invalid time range\n");
         retn++;
      }
   }
      
   return (retn);
}

int
parseTime (struct tm *tm, struct tm *ctm, char *str)
{
   char *hptr = NULL, *dptr = NULL, *mptr = NULL, *yptr = NULL;
   char *minptr = NULL, *secptr = NULL, *ptr;
   int retn = 0, hour = 0, mins = 0, sec = 0, i;
   time_t thistime;

   /*[[[yyyy/]mm/]dd].]hh[:mm[:ss]]*/

   bcopy ((u_char *) ctm, (u_char *) tm, sizeof (struct tm));

   if ((hptr = strchr (str, '.')) != NULL) {
      *hptr++ = '\0';
      if (!(isdigit((int)*hptr)))
         return -1;
   }

   if ((dptr = strrchr (str, '/')) != NULL) {  /* mm/dd */
                                               /*   ^   */
      *dptr++ = '\0';
      if ((mptr = strrchr (str, '/')) != NULL) {  /* yyyy/mm/dd */
                                                  /*     ^      */
         *mptr++ = '\0';
         yptr = str;
      } else
         mptr = str;
   } else {
      if (hptr != NULL)
         dptr = str;
      else
         hptr = str;
   }

   if (yptr) {
      if (strlen(yptr) != 4)
         return -1;
      for (ptr = yptr, i = 0; i < strlen(yptr); i++)
         if (!(isdigit((int)*ptr++)))
            return -1;
      tm->tm_year = atoi(yptr) - 1900;
      retn = ARGUS_YEAR;
   }
   if (mptr) {
      if (strlen(mptr) != 2)
         return -1;
      for (ptr = mptr, i = 0; i < strlen(mptr); i++)
         if (!(isdigit((int)*ptr++)))
            return -1;
      tm->tm_mon  = atoi(mptr) - 1;
      retn = ARGUS_MONTH;
   }

   if (dptr) {
      if (strlen(dptr) != 2)
         return -1;
      for (ptr = dptr, i = 0; i < strlen(dptr); i++)
         if (!(isdigit((int)*ptr++)))
            return -1;
      tm->tm_mday = atoi(dptr);
      retn = ARGUS_DAY;
   }

   if (hptr) {
      if ((minptr = strchr (hptr, ':')) != NULL) {
         *minptr++ = '\0';
         if ((secptr = strchr (minptr, ':')) != NULL) {
            *secptr++ = '\0';

            for (ptr = secptr, i = 0; i < strlen(secptr); i++)
               if (!(isdigit((int)*ptr++)))
                  return -1;

            sec = atoi(secptr);
            retn = ARGUS_SEC;
         }
         for (ptr = minptr, i = 0; i < strlen(minptr); i++)
            if (!(isdigit((int)*ptr++)))
               return -1;

         mins = atoi(minptr);
         retn = ARGUS_MIN;
      }

      for (ptr = hptr, i = 0; i < strlen(hptr); i++)
         if (!(isdigit((int)*ptr++)))
            return -1;

      hour = atoi(hptr);
      retn = ARGUS_HOUR;
   }

   tm->tm_hour = hour;
   tm->tm_min  = mins;
   tm->tm_sec  = sec;

#if !defined(HAVE_SOLARIS) && !defined(__sgi) && !defined(linux) && !defined(AIX) && !defined(CYGWIN)
   tm->tm_zone = NULL;
   tm->tm_gmtoff = 0;
#endif

   if (tm->tm_year < 0)
      retn = -1;
   if ((tm->tm_mon > 11) || (tm->tm_mon < 0))
      retn = -1;
   if ((tm->tm_mday > 31) || (tm->tm_mday < 0))
      retn = -1;
   if ((tm->tm_hour > 23) || (tm->tm_hour < 0))
      retn = -1;
   if ((tm->tm_min > 60) || (tm->tm_min < 0))
      retn = -1;
   if ((tm->tm_sec > 60) || (tm->tm_sec < 0))
      retn = -1;

   if (retn >= 0) {
      thistime = timelocal (tm);
      localtime_r ((time_t *)&thistime, tm);
   }

   return (retn);
}


#define ARGUS_RCITEMS				46

#define RA_ARGUS_SERVER				0
#define RA_CISCONETFLOW_SOURCE			1
#define RA_ARGUS_SERVERPORT			2
#define RA_INPUT_FILE				3
#define RA_NO_OUTPUT				4
#define RA_USER_AUTH				5
#define RA_AUTH_PASS				6
#define RA_OUTPUT_FILE				7
#define RA_EXCEPTION_OUTPUT_FILE		8
#define RA_TIMERANGE				9
#define RA_RUNTIME				10
#define RA_FLOW_MODEL				11
#define RA_FIELD_DELIMITER			12
#define RA_TIME_FORMAT				13
#define RA_USEC_PRECISION			14
#define RA_PRINT_LABELS				15
#define RA_PRINT_SUMMARY			16
#define RA_PRINT_ARGUSID			17
#define RA_PRINT_MACADDRS			18
#define RA_PRINT_HOSTNAMES			19
#define RA_PRINT_LOCALONLY			20
#define RA_PRINT_COUNTS				21
#define RA_PRINT_BYTES				22
#define RA_PRINT_APPLICATION_BYTES		23
#define RA_PRINT_RESPONSE_DATA			24
#define RA_PRINT_UNIX_TIME			25
#define RA_PRINT_STARTIME			26
#define RA_PRINT_LASTIME			27
#define RA_PRINT_INDICATORS			28
#define RA_PRINT_DURATION			29
#define RA_PRINT_TCPSTATES			30
#define RA_PRINT_TCPFLAGS			31
#define RAGATOR_TIME_SERIES			32
#define RAGATOR_VALIDATE			33
#define RAMON_MODE    				34
#define RAMON_NUMBER  				35
#define RA_DEBUG_LEVEL				36
#define RA_PRINT_USERDATA			37
#define RA_USERDATA_ENCODE			38
#define RA_FILTER				39
#define RA_HOST_FIELD_LENGTH			40
#define RA_PORT_FIELD_LENGTH			41
#define RA_PRINT_TRANSACTIONS			42
#define RA_PRINT_LOAD				43
#define RA_PRINT_RATE				44
#define RA_PRINT_LOSS				45


char *ArgusResourceFileStr [] = {
   "RA_ARGUS_SERVER=",
   "RA_CISCONETFLOW_SOURCE=",
   "RA_ARGUS_SERVERPORT=",
   "RA_INPUT_FILE=",
   "RA_NO_OUTPUT=",
   "RA_USER_AUTH=",
   "RA_AUTH_PASS=",
   "RA_OUTPUT_FILE=",
   "RA_EXCEPTION_OUTPUT_FILE=",
   "RA_TIMERANGE=",
   "RA_RUN_TIME=",
   "RA_FLOW_MODEL=",
   "RA_FIELD_DELIMITER=",
   "RA_TIME_FORMAT=",
   "RA_USEC_PRECISION=",
   "RA_PRINT_LABELS=",
   "RA_PRINT_SUMMARY=",
   "RA_PRINT_ARGUSID=",
   "RA_PRINT_MACADDRS=",
   "RA_PRINT_HOSTNAMES=",
   "RA_PRINT_LOCALONLY=",
   "RA_PRINT_COUNTS=",
   "RA_PRINT_BYTES=",
   "RA_PRINT_APPLICATION_BYTES=",
   "RA_PRINT_RESPONSE_DATA=",
   "RA_PRINT_UNIX_TIME=",
   "RA_PRINT_STARTIME=",
   "RA_PRINT_LASTIME=",
   "RA_PRINT_INDICATORS=",
   "RA_PRINT_DURATION=",
   "RA_PRINT_TCPSTATES=",
   "RA_PRINT_TCPFLAGS=",
   "RAGATOR_TIME_SERIES=",
   "RAGATOR_VALIDATE=",
   "RAMON_MODE=",
   "RAMON_NUMBER=",
   "RA_DEBUG_LEVEL=",
   "RA_PRINT_USERDATA=",
   "RA_USERDATA_ENCODE=",
   "RA_FILTER=",
   "RA_HOST_FIELD_LENGTH=",
   "RA_PORT_FIELD_LENGTH=",
   "RA_PRINT_TRANSACTIONS=",
   "RA_PRINT_LOAD=",
   "RA_PRINT_RATE=",
   "RA_PRINT_LOSS=",
};

#include <ctype.h>

int
ArgusParseResourceFile (char *file)
{
   int retn = 0, i, len, Soption = 0, roption = 0, found = 0, lines = 0;
   char strbuf[MAXSTRLEN], *str = strbuf, *optarg = NULL, *ptr = NULL;
   FILE *fd;

   if (file) {
      if ((fd = fopen (file, "r")) != NULL) {
         retn = 1;
         while ((fgets(str, MAXSTRLEN, fd)) != NULL)  {
            lines++;
            while (*str && isspace((int)*str))
                str++;

            if (*str && (*str != '#') && (*str != '\n') && (*str != '!')) {
               found = 0;
               for (i = 0; i < ARGUS_RCITEMS; i++) {
                  len = strlen(ArgusResourceFileStr[i]);
                  if (!(strncmp (str, ArgusResourceFileStr[i], len))) {

                     optarg = &str[len];

                     if (optarg[strlen(optarg) - 1] == '\n')
                        optarg[strlen(optarg) - 1] = '\0';

                     if (*optarg == '\"')
                        optarg++;

                     if (optarg[strlen(optarg) - 1] == '\"')
                        optarg[strlen(optarg) - 1] = '\0';
                        
                     if (*optarg == '\0')
                        optarg = NULL;

                     if (optarg) {
                        switch (i) {
                           case RA_ARGUS_SERVER:
                              ++Sflag;
                              if (!Soption++ && (ArgusRemoteHostList != NULL))
                                 ArgusDeleteHostList();
                              
                              if (!(ArgusAddHostList (optarg)))
                                 ArgusLog (LOG_ERR, "ArgusParseResourceFile: line %d host %s unknown\n",
                                     lines, optarg);
                              break;

                           case RA_CISCONETFLOW_SOURCE:
                              ++Sflag; ++Cflag;
                              if (!Soption++ && (ArgusRemoteHostList != NULL))
                                 ArgusDeleteHostList();
                              
                              if (!(ArgusAddHostList (optarg)))
                                 ArgusLog (LOG_ERR, "ArgusParseResourceFile: line %d host %s unknown\n",
                                     lines, optarg);
                              break;

                           case RA_ARGUS_SERVERPORT:
                              ArgusPortNum = atoi (optarg); break;
                              break;

                           case RA_INPUT_FILE:
                              if ((!roption++) && (ArgusInputFileList != NULL))
                                 ArgusDeleteFileList();

                              if (!(ArgusAddFileList (optarg)))
                                 ArgusLog (LOG_ERR, "ArgusParseResourceFile: line %d error file arg %s\n",
                                     lines, optarg);
                              break;

                           case RA_NO_OUTPUT:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 qflag++;
                              else
                                 qflag = 0;
                              break;

                           case RA_USER_AUTH:
                              ustr = strdup(optarg);
                              break;

                           case RA_AUTH_PASS:
                              pstr = strdup(optarg);
                              break;

                           case RA_OUTPUT_FILE:
                              wfile = strdup(optarg);
                              break;

                           case RA_EXCEPTION_OUTPUT_FILE:
                              exceptfile = optarg;
                              break;

                           case RA_TIMERANGE:
                              timearg = strdup(optarg);
                              if ((parseTimeArg (&timearg, NULL, 0, RaTmStruct)) < 0)
                                 usage ();
                              break;

                           case RA_RUNTIME:
                              Tflag = atoi (optarg);
                              break;

                           case RA_FIELD_DELIMITER:
                              ptr = optarg;
                              if ((ptr = strchr (optarg, '\'')) != NULL) {
                                 ptr++;
                                 if (ptr[0] == '\'')
                                    break;
                              }

                              if (ptr[0] == '\\') {
                                 switch (ptr[1]) {
                                    case  'a': RaFieldDelimiter = '\a'; break;
                                    case  'b': RaFieldDelimiter = '\b'; break;
                                    case  't': RaFieldDelimiter = '\t'; break;
                                    case  'n': RaFieldDelimiter = '\n'; break;
                                    case  'v': RaFieldDelimiter = '\v'; break;
                                    case  'f': RaFieldDelimiter = '\f'; break;
                                    case  'r': RaFieldDelimiter = '\r'; break;
                                    case '\\': RaFieldDelimiter = '\\'; break;
                                 }
                                 if (RaFieldDelimiter != '\0')
                                    break;
                              } else
                                 RaFieldDelimiter = *ptr;

                              break;

                           case RA_TIME_FORMAT:
                              RaTimeFormat = strdup(optarg);
 
                           case RA_USEC_PRECISION:
                              pflag = atoi (optarg);
                              break;
 
                           case RA_PRINT_SUMMARY:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 aflag = 1;
                              else
                                 aflag = 0;
                              break;

                           case RA_PRINT_ARGUSID:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 RaPrintSource = 1;
                              else
                                 RaPrintSource = 0;
                              break;

                           case RA_PRINT_MACADDRS:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 RaPrintMACAddress = 1;
                              else
                                 RaPrintMACAddress = 0;
                              break;

                           case RA_PRINT_HOSTNAMES:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 nflag = 0;
                              else
                                 nflag = 1;
                              break;

                           case RA_PRINT_LOCALONLY:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 ++fflag;
                              else
                                 fflag = 0;
                              break;

                           case RA_FLOW_MODEL:
                              ArgusFlowModelFile = strdup(optarg);
                              break;
 
                           case RA_PRINT_LABELS:
                              Lflag = atoi(optarg);
                              if (Lflag == 0)
                                 Lflag = -1;
                              break;

                           case RA_PRINT_COUNTS:
                              if (!(strncasecmp(optarg, "yes", 3))) {
                                 RaPrintCounts++;
                              } else {
                                 RaPrintCounts = 0;
                              }
                              break;

                           case RA_PRINT_BYTES:
                              if (!(strncasecmp(optarg, "yes", 3))) {
                                 RaPrintBytes++;
                              } else {
                                 RaPrintBytes = 0;
                              }
                              break;

                           case RA_PRINT_APPLICATION_BYTES:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 ++Aflag;
                              else
                                 Aflag = 0;
                              break;

                           case RA_PRINT_RESPONSE_DATA:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 Rflag++;
                              else
                                 Rflag = 0;
                              break;

                           case RA_PRINT_UNIX_TIME:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 ++uflag;
                              else
                                 uflag = 0;
                              break;

                           case RA_PRINT_STARTIME:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 ++RaPrintStartTime;
                              else
                                 RaPrintStartTime = 0;
                              break;

                           case RA_PRINT_LASTIME:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 ++RaPrintLastTime;
                              else
                                 RaPrintLastTime = 0;
                              break;

                           case RA_PRINT_INDICATORS:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 RaPrintIndicator++;
                              else
                                 RaPrintIndicator = 0;
                              break;

                           case RA_PRINT_DURATION:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 RaPrintDuration++;
                              break;

                           case RA_PRINT_TRANSACTIONS:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 RaPrintTransactions++;
                              else
                                 RaPrintTransactions = 0;
                              break;

                           case RA_PRINT_LOAD:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 RaPrintLoad++;
                              else
                                 RaPrintLoad = 0;
                              break;

                           case RA_PRINT_RATE:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 RaPrintRate++;
                              else
                                 RaPrintRate = 0;
                              break;


                           case RA_PRINT_LOSS:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 RaPrintLoss++;
                              else
                                 RaPrintLoss = 0;
                              break;

                           case RA_PRINT_TCPSTATES:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 zflag++;
                              else
                                 zflag = 0;
                              break;

                           case RA_PRINT_TCPFLAGS:
                                 Zflag = *optarg;
                              break;

                           case RAGATOR_TIME_SERIES:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 Hflag++;
                              else
                                 Hflag = 0;
                              break;

                           case RAGATOR_VALIDATE:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 Vflag++;
                              else
                                 Vflag = 0;
                              break;

                           case RAMON_MODE:
                              Mflag = optarg;
                              break;

                           case RAMON_NUMBER:
                              Nflag = atoi (optarg);
                              break;

                           case RA_DEBUG_LEVEL:
                              Argusdflag = (atoi(optarg));
                              break;

                           case RA_PRINT_USERDATA:
                              dflag++;
                              if ((parseUserDataArg (&optarg, NULL, 0)) < 0)
                                 usage ();
                              break;

                           case RA_USERDATA_ENCODE:
                              if (!(strncasecmp(optarg, "ascii", 5)))
                                 eflag = ARGUS_ENCODE_ASCII;
                              else
                              if (!(strncasecmp(optarg, "encode32", 8)))
                                 eflag = ARGUS_ENCODE_32;
                              else
                                 eflag = ARGUS_ENCODE_64;
                              break;

                           case RA_FILTER: {
                              char *ptr;

                              if (*RaInputFilter != NULL)
                                 free(*RaInputFilter);

                              if ((*RaInputFilter = calloc (1, MAXSTRLEN)) != NULL) {
                                 ptr = *RaInputFilter;
                                 str = optarg;
                                 while (*str) {
                                    if ((*str == '\\') && (str[1] == '\n')) {
                                       fgets(str, MAXSTRLEN, fd);
                                       while (*str && (isspace((int)*str) && (str[1] && isspace((int)str[1]))))
                                          str++;
                                    }
                                    
                                    if ((*str != '\n') && (*str != '"'))
                                       *ptr++ = *str++;
                                    else
                                       str++;
                                 }
                              }
#ifdef ARGUSDEBUG
                              ArgusDebug (1, "ArgusParseResourceFile: ArgusFilter \"%s\" \n", RaInputFilter);
#endif
                              break;
                           }

                           case RA_HOST_FIELD_LENGTH:
                              hfield = atoi (optarg);
                              break;

                           case RA_PORT_FIELD_LENGTH:
                              pfield = atoi (optarg);
                              break;
                        }

                     }
                     found++;
                     break;
                  }
               }
               if (!found) {
                  ArgusLog (LOG_ERR, "%s: syntax error line %d\n", file, lines);
               }
            }
         }

      } else {
#ifdef ARGUSDEBUG
         ArgusDebug (2, "%s: %s\n", file, strerror(errno));
#endif
      }

      if (RaPrintStartTime && RaPrintLastTime)
         Gflag++;
      else
      if (RaPrintLastTime)
         lflag++;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusParseResourceFile (%s) returning %d\n", file, retn);
#endif

   return (retn);
}


void
clearRaConfiguration ()
{
   aflag = 0;
   Aflag = 0;
   Argusdflag = 0;
   bflag = 0;
   Bflag = 0;
   cflag = 0;
   Cflag = 0;
   dflag = 0;
   Dflag = 0;
   eflag = 0;
   Eflag = 0;
   estr = NULL;
   fflag = 0;
   Fflag = 0;
   gflag = 0;
   Gflag = 0;
   hfield = 15;
   Hflag = 0;
   Hstr = NULL;
   idflag = 0;
   jflag = 0;
   lflag = 0;
   Lflag = 0;
   mflag = 0;
   Mflag = NULL;
   Netflag = 0;
   nflag = 0;
   Nflag = 0;
   Normflag = 0;
   notNetflag = 0;
   oflag = 0;
   Oflag = 0;
   pfield = 5;
   pflag = 0;
   Pflag = 0;
   qflag = 0;
   sflag = NULL;
   tflag = 0;
   uflag = 0;
   Wflag = 0;

   Uflag = 6;
   vflag = 0;
   Vflag = 0;
   iflag = 0;

   Iflag = 0;
   Tflag = 0;
   rflag = 0;
   Rflag = 0;
   Sflag = 0;
   xflag = 0;
   Xflag = 0;
   XMLflag = 0;

   zflag = 0;
   Zflag = 0;

   RaPrintDate = 1;
   RaPrintTime = 1;

   RaCumulativeMerge = 1;

   RaFlowModelFile = 0;
   RaAllocHashTableHeaders = 0;
   RaAllocArgusRecordStore = 0;
   RaAllocArgusRecord      = 0;

   ArgusMinuteUpdate = 1;
   ArgusHourlyUpdate = 1;
   RaHistoTimeSeries = 1;

   RaThisActiveIndex = 0;

   RaThisFlowNum = 0;
   RaThisModelNum = 0;
   RaParseError = 0;

   if (wfile != NULL)
      free (wfile);
   wfile = NULL;
 
   if (ustr != NULL)
      free (ustr);
   ustr = NULL;
 
   if (pstr != NULL)
      free (pstr);
   pstr = NULL;
   if (timearg != NULL)
      free(timearg);
   timearg = NULL;

   if (ArgusRemoteFilter != NULL)
      free (ArgusRemoteFilter);
   ArgusRemoteFilter = NULL;

   ArgusSrcUserDataLen = 0;
   ArgusDstUserDataLen = 0;

   if (ArgusInputFileList != NULL)
      ArgusDeleteFileList();

   if (ArgusRemoteHostList != NULL)
      ArgusDeleteHostList();

#ifdef ARGUSDEBUG
   ArgusDebug (1, "clearArgusConfiguration () returning\n");
#endif 

}
