/*
 * Argus Software.  Argus files - TCP protocol
 * Copyright (c) 2000-2006 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* 
 * $Id: $
 * $DateTime: $
 * $Change: $
 */

#if !defined(ArgusTcp)
#define ArgusTcp
#endif


#include <stdio.h>
#include <ArgusModeler.h>

#include <netinet/tcp.h>
#include <netinet/tcp_fsm.h>

extern void ArgusZeroRecord(struct ArgusFlowStruct *);

static struct ArgusTCPObjectMetrics *ArgusThisTCPsrc, *ArgusThisTCPdst;

/* These tcp optinos do not have the size octet */
#define ZEROLENOPT(o) ((o) == TCPOPT_EOL || (o) == TCPOPT_NOP)

#if !defined(TH_ECE)
#define TH_ECE  0x40
#endif
#if !defined(TH_CWR)
#define TH_CWR  0x80
#endif

void ArgusParseTCPOptions(struct ArgusModelerStruct *, struct tcphdr *, int, u_int *);
void ArgusInitializeTCP (struct ArgusModelerStruct *, struct ArgusFlowStruct *);
void ArgusUpdateTCPState (struct ArgusModelerStruct *, struct ArgusFlowStruct *, unsigned char *);
int ArgusUpdateTCPSequence (struct ArgusModelerStruct *, struct ArgusFlowStruct *, struct tcphdr *);
int ArgusUpdateTCPStateMachine (struct ArgusModelerStruct *, struct ArgusFlowStruct *, struct tcphdr *);
void ArgusTCPFlowRecord (struct ArgusNetworkStruct *, unsigned char);


#include <errno.h>
#include <string.h>


void
ArgusUpdateTCPState (struct ArgusModelerStruct *model, struct ArgusFlowStruct *flowstr, unsigned char *state)
{
   struct tcphdr *tcp = (struct tcphdr *) model->ArgusThisUpHdr;
   struct ArgusTCPObject *tcpExt = NULL;

   if (tcp && STRUCTCAPTURED(model, *tcp)) {
      int tcplen = model->ArgusThisLength;
      int tcphlen = tcp->th_off * 4;
      int tcpdatalen = tcplen - tcphlen;
      unsigned char flags = tcp->th_flags;

#ifdef _LITTLE_ENDIAN
      tcp->th_dport = ntohs(tcp->th_dport);
      tcp->th_sport = ntohs(tcp->th_sport);
      tcp->th_win   = ntohs(tcp->th_win);
      tcp->th_seq   = ntohl(tcp->th_seq);
      tcp->th_ack   = ntohl(tcp->th_ack);
#endif
      if (*state == ARGUS_START) {
         struct ArgusNetworkStruct *net = (struct ArgusNetworkStruct *) &flowstr->canon.net;
         net->hdr.type             = ARGUS_NETWORK_DSR;
         net->hdr.subtype          = ARGUS_METER_TCP_PERF;
         net->hdr.argus_dsrvl8.qual = 0;
         net->hdr.argus_dsrvl8.len  = ((sizeof(struct ArgusTCPObject)+3))/4 + 1;
         flowstr->dsrs[ARGUS_NETWORK_INDEX] = (struct ArgusDSRHeader *) net;
         tcpExt                    = &net->net_union.tcp;
         bzero ((char *)tcpExt, sizeof(*tcpExt));

         model->ArgusSnapLength -= tcphlen;
         model->ArgusThisLength  = tcpdatalen;
         model->ArgusThisUpHdr  += tcphlen;

         ArgusThisTCPsrc = &tcpExt->src;
         ArgusThisTCPdst = &tcpExt->dst;

         if ((tcphlen -= sizeof(*tcp)) > 0)
            ArgusParseTCPOptions (model, tcp, tcphlen, &tcpExt->options);

         if (flags & TH_RST) {
            tcpExt->status |= ARGUS_RESET;
            ArgusThisTCPsrc->status  |= ARGUS_RESET;
            tcpExt->state        = TCPS_LISTEN;
            ArgusThisTCPsrc->bytes   += model->ArgusThisLength;
            ArgusThisTCPsrc->flags    = tcp->th_flags;
            ArgusThisTCPsrc->seqbase  = tcp->th_seq - 1;
            ArgusThisTCPsrc->ackbytes = tcp->th_seq - 1;

            ArgusThisTCPsrc->lasttime.tv_sec  = model->ArgusGlobalTime.tv_sec;
            ArgusThisTCPsrc->lasttime.tv_usec = model->ArgusGlobalTime.tv_usec;

            ArgusThisTCPsrc->seq      = tcp->th_seq + model->ArgusThisLength;

         } else {
            switch (flags & (TH_SYN|TH_ACK|TH_FIN|TH_PUSH|TH_URG)) {
               case (TH_SYN):
                  tcpExt->status      |= ARGUS_SAW_SYN;
                  tcpExt->state        = TCPS_SYN_SENT;
                  ArgusThisTCPsrc->bytes   += model->ArgusThisLength;
                  ArgusThisTCPsrc->seqbase  = tcp->th_seq; 
                  ArgusThisTCPsrc->ackbytes = tcp->th_seq; 
                  ArgusThisTCPsrc->seq      = tcp->th_seq; 
                  ArgusThisTCPsrc->win      = tcp->th_win;
                  ArgusThisTCPsrc->flags    = tcp->th_flags; 

                  ArgusThisTCPsrc->lasttime.tv_sec  = model->ArgusGlobalTime.tv_sec;
                  ArgusThisTCPsrc->lasttime.tv_usec = model->ArgusGlobalTime.tv_usec;
      
                  if ((flags & (TH_ECE|TH_CWR)) == (TH_ECE|TH_CWR))
                     tcpExt->options |= ARGUS_TCP_SRC_ECN;
                  break;
         
               case (TH_SYN|TH_ACK): {
                  tcpExt->status      |= ARGUS_SAW_SYN_SENT;
                  tcpExt->state        = TCPS_SYN_RECEIVED;
                  ArgusThisTCPsrc->bytes   += model->ArgusThisLength;
                  ArgusThisTCPsrc->seqbase  = tcp->th_seq;
                  ArgusThisTCPsrc->ackbytes = tcp->th_seq;
                  ArgusThisTCPsrc->seq      = tcp->th_seq;
                  ArgusThisTCPsrc->win      = tcp->th_win;
                  ArgusThisTCPsrc->flags    = tcp->th_flags; 

                  ArgusThisTCPsrc->lasttime.tv_sec  = model->ArgusGlobalTime.tv_sec;
                  ArgusThisTCPsrc->lasttime.tv_usec = model->ArgusGlobalTime.tv_usec;

                  ArgusThisTCPdst->ack      = tcp->th_ack - 1;

                  if ((tcp->th_flags & (TH_ECE|TH_CWR)) == TH_ECE)
                     tcpExt->options |= ARGUS_TCP_DST_ECN;
                  break;
               }
      
               case (TH_ACK):
               case (TH_PUSH|TH_ACK):
               case (TH_URG|TH_ACK):
               case (TH_PUSH|TH_URG|TH_ACK):
                  ArgusThisTCPdst->ack      = tcp->th_ack - 1;

               case (TH_PUSH):
               case (TH_URG):
               case (TH_PUSH|TH_URG):

                  tcpExt->status      |= ARGUS_CON_ESTABLISHED;
                  tcpExt->state        = TCPS_ESTABLISHED;
                  ArgusThisTCPsrc->bytes   += model->ArgusThisLength;
                  ArgusThisTCPsrc->flags    = tcp->th_flags; 
                  ArgusThisTCPsrc->seqbase  = tcp->th_seq - 1;
                  ArgusThisTCPsrc->ackbytes = tcp->th_seq - 1;

                  ArgusThisTCPsrc->lasttime.tv_sec  = model->ArgusGlobalTime.tv_sec;
                  ArgusThisTCPsrc->lasttime.tv_usec = model->ArgusGlobalTime.tv_usec;

                  ArgusThisTCPsrc->seq      = tcp->th_seq + model->ArgusThisLength;
                  ArgusThisTCPsrc->win      = tcp->th_win;
                  break;
      
               case (TH_FIN):
               case (TH_FIN|TH_ACK):
                  tcpExt->status      |= ARGUS_FIN;
                  tcpExt->state        = TCPS_FIN_WAIT_1;
                  ArgusThisTCPsrc->bytes   += model->ArgusThisLength;
                  ArgusThisTCPsrc->flags    = tcp->th_flags; 
                  ArgusThisTCPsrc->seqbase  = tcp->th_seq - 1;
                  ArgusThisTCPsrc->ackbytes = tcp->th_seq - 1;

                  ArgusThisTCPsrc->lasttime.tv_sec  = model->ArgusGlobalTime.tv_sec;
                  ArgusThisTCPsrc->lasttime.tv_usec = model->ArgusGlobalTime.tv_usec;

                  ArgusThisTCPsrc->seq      = tcp->th_seq + model->ArgusThisLength;
                  ArgusThisTCPsrc->win      = tcp->th_win;
                  break;
      
               default:
                  tcpExt->status      |= ARGUS_CON_ESTABLISHED;
                  tcpExt->state        = TCPS_CLOSING;
                  ArgusThisTCPsrc->bytes   += model->ArgusThisLength;
                  ArgusThisTCPsrc->flags    = tcp->th_flags; 
                  ArgusThisTCPsrc->seqbase  = tcp->th_seq - 1;
                  ArgusThisTCPsrc->ackbytes = tcp->th_seq - 1;

                  ArgusThisTCPsrc->lasttime.tv_sec  = model->ArgusGlobalTime.tv_sec;
                  ArgusThisTCPsrc->lasttime.tv_usec = model->ArgusGlobalTime.tv_usec;

                  ArgusThisTCPsrc->seq      = model->ArgusThisLength;
                  ArgusThisTCPsrc->win      = tcp->th_win;
                  break;
            }
         }

      } else {
         tcpExt = (struct ArgusTCPObject *)&flowstr->canon.net.net_union.tcp;
         switch (tcpExt->state) {
            case TCPS_LISTEN:
               if (flags == TH_SYN) {
                  ArgusSendFlowRecord (model, flowstr, ARGUS_STOP);
                  ArgusZeroRecord (flowstr);
                  *state = ARGUS_START;
                  ArgusUpdateTCPState (model, flowstr, state);
                  break;
               }

            default: {
               if (model->ArgusThisDir) {
                  ArgusThisTCPsrc = &tcpExt->src;
                  ArgusThisTCPdst = &tcpExt->dst;
               } else {
                  ArgusThisTCPsrc = &tcpExt->dst;
                  ArgusThisTCPdst = &tcpExt->src;
               }

               model->ArgusSnapLength -= tcphlen;
               model->ArgusThisLength  = tcpdatalen;
               model->ArgusThisUpHdr  += tcphlen;

               if ((tcphlen - sizeof(*tcp)) > 0)
                  ArgusParseTCPOptions (model, tcp, (tcphlen - sizeof(*tcp)), &tcpExt->options);
   
               ArgusThisTCPsrc->flags |= tcp->th_flags;

#define ARGUS_ECN_FLAGS		(ARGUS_TCP_SRC_ECN | ARGUS_TCP_DST_ECN)
   
               if ((tcpExt->options & ARGUS_ECN_FLAGS) && (flags & TH_ECE)) {
                  if (flags & TH_ACK) {
                     tcpExt->status |= ARGUS_ECN_CONGESTED;
                     ArgusThisTCPdst->state = ARGUS_ECN_CONGESTED;
                  }
               }

               ArgusUpdateTCPSequence(model, flowstr, tcp);
   
               switch (ArgusUpdateTCPStateMachine(model, flowstr, tcp)) {
                  case TCPS_LISTEN:
                     if (flags == TH_SYN) {
                        ArgusThisTCPsrc->bytes -= model->ArgusThisLength;
                        model->ArgusThisUpHdr  -= tcphlen;
                        model->ArgusThisLength = tcplen;
                        model->ArgusSnapLength += tcphlen;

                        ArgusSendFlowRecord (model, flowstr, ARGUS_STOP);
                        ArgusInitializeTCP (model, flowstr);
                        return;
                     }
                     break;
   
                  case TCPS_CLOSED:
                  case TCPS_TIME_WAIT:
                     if (!(tcpExt->status & ARGUS_RESET))
                        tcpExt->status |= ARGUS_NORMAL_CLOSE;
                     flowstr->timeout = 10;
                     break;
               }
   
               ArgusThisTCPsrc->lasttime.tv_sec  = model->ArgusGlobalTime.tv_sec;
               ArgusThisTCPsrc->lasttime.tv_usec = model->ArgusGlobalTime.tv_usec;
            }
         }
      }
   }
}

void
ArgusInitializeTCP (struct ArgusModelerStruct *model, struct ArgusFlowStruct *flow)
{
   struct ArgusTCPObject *tcpExt = (struct ArgusTCPObject *)&flow->canon.net.net_union.tcp;

   bzero ((char *)tcpExt, sizeof(*tcpExt));

   flow->qhdr.lasttime.tv_sec  = 0;
   flow->qhdr.lasttime.tv_usec = 0;

   flow->qhdr.qtime = model->ArgusGlobalTime;

   ArgusUpdateFlow (model, flow, ARGUS_START);
}


int
ArgusUpdateTCPStateMachine (struct ArgusModelerStruct *model, struct ArgusFlowStruct *flowstr, struct tcphdr *tcp)
{
   unsigned char flags = tcp->th_flags;
   struct ArgusTCPObject *tcpExt = (struct ArgusTCPObject *)&flowstr->canon.net.net_union.tcp;
   unsigned int state = tcpExt->state;
   int len = model->ArgusThisLength;

   if (flags & TH_RST) {
      tcpExt->status |= ARGUS_RESET;
      ArgusThisTCPsrc->status |= ARGUS_RESET;

      if (state == TCPS_SYN_SENT) {
         if ((ArgusThisTCPdst->seq == ArgusThisTCPsrc->ack))
             state = TCPS_LISTEN;
      } else
         if ((tcp->th_seq >= ArgusThisTCPsrc->ack) &&
                    (tcp->th_seq < (ArgusThisTCPsrc->ack + ArgusThisTCPsrc->win)))
            state = TCPS_CLOSED;

   } else {
      switch (state) {
         case TCPS_LISTEN:
         case TCPS_SYN_SENT:
            if (flags == TH_SYN) {
               ArgusThisTCPsrc->status |= ARGUS_PKTS_RETRANS;
               ArgusThisTCPsrc->retrans++;
            } else
            if (flags == (TH_SYN|TH_ACK)) {
               if (ArgusThisTCPsrc->status & ARGUS_SAW_SYN) {
                  state = TCPS_LISTEN;
                  tcpExt->status |= ARGUS_SAW_SYN_SENT;
                  ArgusThisTCPsrc->status |= ARGUS_SAW_SYN_SENT;

               } else {
                  state = TCPS_SYN_RECEIVED;
                  tcpExt->status |= ARGUS_SAW_SYN_SENT;
                  ArgusThisTCPsrc->status |= ARGUS_SAW_SYN_SENT;

                  if ((ArgusThisTCPdst->seq == ArgusThisTCPsrc->ack)) {
                     struct timeval lasttime;
                     lasttime.tv_sec  = ArgusThisTCPdst->lasttime.tv_sec;
                     lasttime.tv_usec = ArgusThisTCPdst->lasttime.tv_usec;

                     tcpExt->synAckuSecs = ArgusAbsTimeDiff (&model->ArgusGlobalTime, &lasttime);
                  }
               }

            } else
            if (flags & TH_FIN) {
               state = TCPS_FIN_WAIT_1;
               tcpExt->status |= ARGUS_FIN;
               ArgusThisTCPsrc->status |= ARGUS_FIN;
            } else 
            if (flags & TH_ACK) {
               state = TCPS_ESTABLISHED;
               tcpExt->status |= ARGUS_CON_ESTABLISHED;
               ArgusThisTCPsrc->status |= ARGUS_CON_ESTABLISHED;
/*
               flowstr->ArgusTimeout = ARGUS_IPTIMEOUT;
*/
            }
            break;
    
         case TCPS_SYN_RECEIVED:
            if (flags & TH_FIN) {
               state = TCPS_FIN_WAIT_1;
               tcpExt->status |= ARGUS_FIN;
               ArgusThisTCPsrc->status |= ARGUS_FIN;

            } else
            if (!(flags & TH_SYN)) {
               if (flags & TH_ACK) {
                  state = TCPS_ESTABLISHED;
                  tcpExt->status |= ARGUS_CON_ESTABLISHED;
/*
                  flowstr->ArgusTimeout = ARGUS_IPTIMEOUT;
*/
                  ArgusThisTCPsrc->status |= ARGUS_CON_ESTABLISHED;
                  if ((ArgusThisTCPsrc->seq == ArgusThisTCPsrc->ack)) {
                     struct timeval lasttime;
                     lasttime.tv_sec  = ArgusThisTCPdst->lasttime.tv_sec;
                     lasttime.tv_usec = ArgusThisTCPdst->lasttime.tv_usec;

                     tcpExt->ackDatauSecs = ArgusAbsTimeDiff (&model->ArgusGlobalTime, &lasttime);
                  }
               }
            } else {
               ArgusThisTCPsrc->status |= ARGUS_PKTS_RETRANS;
               ArgusThisTCPsrc->retrans++;
            }

            break;
    
         case TCPS_ESTABLISHED:
            if (flags & TH_FIN) {
               state = TCPS_FIN_WAIT_1;
               tcpExt->status |= ARGUS_FIN;
               ArgusThisTCPsrc->status |= ARGUS_FIN;

            } else {
               if (flags & TH_SYN) {
                  if (flags & TH_ACK) {
                     tcpExt->status |= ARGUS_SAW_SYN_SENT;
                     tcpExt->status |= ARGUS_CON_ESTABLISHED;
                  }
                  ArgusThisTCPsrc->status |= ARGUS_PKTS_RETRANS;
                  ArgusThisTCPsrc->retrans++;
               }
            }
            break;
    
         case TCPS_CLOSE_WAIT:
         case TCPS_FIN_WAIT_1:
            if ((flags & TH_SYN) && !(flags & TH_ACK)) {
               state = TCPS_LISTEN;
            } else

         case TCPS_LAST_ACK:
         case TCPS_FIN_WAIT_2:
            if (flags & TH_FIN) {
               if (!(flags & TH_ACK)) {
                  if (ArgusThisTCPdst->status & ARGUS_FIN_ACK) {
                     ArgusThisTCPsrc->status |= ARGUS_PKTS_RETRANS;
                     ArgusThisTCPsrc->retrans++;
                  }
               } else {
                  tcpExt->status |= ARGUS_FIN;
                  ArgusThisTCPsrc->status |= ARGUS_FIN;
               }
            }

            if ((flags & TH_ACK) && !(len)) {
               if (ArgusThisTCPdst->status & ARGUS_FIN) {
                  if (ArgusThisTCPdst->seq == ArgusThisTCPsrc->ack) {
                     state = TCPS_FIN_WAIT_2;
                     tcpExt->status |= ARGUS_FIN_ACK;
                     ArgusThisTCPdst->status |= ARGUS_FIN_ACK;
                  }
               }
            }

            break;
      
         case TCPS_CLOSING:
         case TCPS_TIME_WAIT:
            if ((flags & TH_SYN) && !(flags & TH_ACK))
               state = TCPS_LISTEN;
            else
            if (flags & TH_ACK)
               if ((ArgusThisTCPsrc->seq == ArgusThisTCPsrc->ack) &&
                         (ArgusThisTCPdst->seq == ArgusThisTCPsrc->ack))
                  state = TCPS_CLOSED;
            break;
         
         case TCPS_CLOSED:
            if ((flags & TH_SYN) && !(flags & TH_ACK))
               state = TCPS_LISTEN;
            break;
      }
   }

   if (state != TCPS_LISTEN)
      tcpExt->state = state;
   
   return (state);
}


int
ArgusUpdateTCPSequence (struct ArgusModelerStruct *model, struct ArgusFlowStruct *flowstr, struct tcphdr *tcp)
{
   unsigned char flags = tcp->th_flags;
   int len = model->ArgusThisLength;

   int retn = 1, win;
   unsigned int maxseq = 0;
   unsigned int seq = tcp->th_seq;
   unsigned int newseq = seq + len;

   model->ArgusInProtocol = 1;

   if (!(tcp->th_win) && !(flags & (TH_FIN|TH_RST))) {
      ArgusThisTCPsrc->status |= ARGUS_WINDOW_SHUT;
      model->ArgusInProtocol = 0;
   } else {
      if (!(flags & (TH_FIN|TH_RST))) {
         ArgusThisTCPsrc->win = tcp->th_win;
      }
   }

   if (len && (ArgusThisTCPdst->win != 0)) {
      ArgusThisTCPsrc->bytes += len;

      if (ArgusThisTCPsrc->winbytes == 0)
         model->ArgusInProtocol = 0;

      if (ArgusThisTCPsrc->flags & TH_FIN) {
         if (tcp->th_seq < ArgusThisTCPsrc->seq) {
            ArgusThisTCPsrc->retrans++;
            ArgusThisTCPsrc->status |= ARGUS_PKTS_RETRANS;
         } else
            if (!(flags & TH_FIN))
               ArgusThisTCPsrc->status |= ARGUS_OUTOFORDER;
      }

      if (!(flags & TH_SYN))
         ArgusThisTCPsrc->winbytes += len;

   } else
      model->ArgusInProtocol = 0;

   if ((newseq < seq) || (flags == TH_SYN)) {          /* we rolled over or started over */
      ArgusThisTCPsrc->seqbase = newseq;
      ArgusThisTCPsrc->ackbytes = newseq;
      ArgusThisTCPsrc->seq = newseq;

   } else {
      if (!ArgusThisTCPsrc->seqbase) {                 /* first packet in this direction */
         ArgusThisTCPsrc->seqbase = seq;
         ArgusThisTCPsrc->ackbytes = seq;
         ArgusThisTCPsrc->seq = newseq;
      } else {
         if (len) {
            if (ArgusThisTCPdst->win != 0) {     /* not first packet seen in this direction */
               if (tcp->th_seq < ArgusThisTCPsrc->ack) {
                  if ((ArgusThisTCPsrc->ack - tcp->th_seq) < ArgusThisTCPsrc->win) {
                     ArgusThisTCPsrc->retrans++;
                     ArgusThisTCPsrc->status |= ARGUS_PKTS_RETRANS;
                     ArgusThisTCPsrc->winbytes -= len;
                     model->ArgusInProtocol = 0;
                  }

               } else {
                  if (newseq > ArgusThisTCPsrc->seq) {
                     maxseq = newseq;
                  } else {
                     maxseq = ArgusThisTCPsrc->seq;
                  }

                  if (ArgusThisTCPsrc->win) {
                     if (ArgusThisTCPsrc->winbytes > ((maxseq - 1) - ArgusThisTCPdst->ack)) {
                        ArgusThisTCPsrc->retrans++;
                        ArgusThisTCPsrc->status |= ARGUS_PKTS_RETRANS;
                        ArgusThisTCPsrc->winbytes -= len;
                        model->ArgusInProtocol = 0;

                     } else {
                        if (newseq < ArgusThisTCPsrc->seq)
                           ArgusThisTCPsrc->status |= ARGUS_OUTOFORDER;
                     }
                  }

                  ArgusThisTCPsrc->seq = maxseq;
               }
            }

         } else {
         }
      }
   }

   if (flags & TH_FIN)
      ArgusThisTCPdst->seq = tcp->th_seq;

   if (tcp->th_ack && (flags & TH_ACK)) {
      if (ArgusThisTCPsrc->ack) {
         if (ArgusThisTCPdst->seq > ArgusThisTCPsrc->ack)
            ArgusThisTCPdst->winbytes = (ArgusThisTCPdst->seq - 1) - ArgusThisTCPsrc->ack;  
      }

      if (!(ArgusThisTCPsrc->ack == (tcp->th_ack - 1))) {
         if (!(ArgusThisTCPsrc->ack) || (ArgusThisTCPdst->seq == tcp->th_ack)) {

            ArgusThisTCPdst->winbytes = 0;
            if (!(ArgusThisTCPsrc->ack == (tcp->th_ack - 1)))
               if (ArgusThisTCPdst->seq == tcp->th_ack)
                  ArgusThisTCPdst->winnum++;

         } else {
            if (!(flags & TH_SYN))
               if (ArgusThisTCPsrc->ack) {
                  win = (tcp->th_ack - 1) - ArgusThisTCPsrc->ack;
                  win = (ArgusThisTCPdst->winbytes < win) ? ArgusThisTCPdst->winbytes : win;
                  ArgusThisTCPdst->winbytes -= win;
                  ArgusThisTCPdst->winnum++;
               }
         }

         ArgusThisTCPsrc->ack = tcp->th_ack - 1;
      }
   }

/* ArgusInProtocol = 1; */

   return (retn);
}


#include <argus_out.h>

void
ArgusTCPFlowRecord (struct ArgusNetworkStruct *net, unsigned char state)
{
   struct ArgusTCPObject *tcp = (struct ArgusTCPObject *)&net->net_union.tcp;

   if (tcp) {
      net->hdr.argus_dsrvl8.qual = 0;

      tcp->status &= ~ARGUS_RESET;
      if (tcp->src.status & ARGUS_RESET)
         tcp->status |= ARGUS_SRC_RESET;
      if (tcp->dst.status & ARGUS_RESET)
         tcp->status |= ARGUS_DST_RESET;
   
      tcp->status &= ~ARGUS_PKTS_RETRANS;
      if (tcp->src.status & ARGUS_PKTS_RETRANS)
         tcp->status |= ARGUS_SRC_PKTS_RETRANS;
      if (tcp->dst.status & ARGUS_PKTS_RETRANS)
         tcp->status |= ARGUS_DST_PKTS_RETRANS;

      tcp->status &= ~ARGUS_WINDOW_SHUT;
      if (tcp->src.status & ARGUS_WINDOW_SHUT)
         tcp->status |= ARGUS_SRC_WINDOW_SHUT;
      if (tcp->dst.status & ARGUS_WINDOW_SHUT)
         tcp->status |= ARGUS_DST_WINDOW_SHUT;

      tcp->status &= ~ARGUS_OUTOFORDER;
      if (tcp->src.status & ARGUS_OUTOFORDER)
         tcp->status |= ARGUS_SRC_OUTOFORDER;
      if (tcp->dst.status & ARGUS_OUTOFORDER)
         tcp->status |= ARGUS_DST_OUTOFORDER;
   
      net->hdr.argus_dsrvl8.len  = ((sizeof(struct ArgusTCPObject)+3))/4 + 1;

  /* 
      tcp->synAckuSecs  = tcpExt->synAckuSecs;
      tcp->ackDatauSecs = tcpExt->ackDatauSecs;
      tcp->options      = tcpExt->options;
      tcp->src.seqbase  = tcpExt->src.seqbase;
      tcp->dst.seqbase  = tcpExt->dst.seqbase;

      tcpExt->src.seqbase = 0;
      tcpExt->dst.seqbase = 0;

      if (tcpExt->src.ack && tcpExt->src.ackbytes) {
         if (tcpExt->src.ack != tcpExt->src.ackbytes) {
            if (tcpExt->src.ack > tcpExt->src.ackbytes)
               tcp->src.ackbytes = (tcpExt->src.ack - 1) - tcpExt->src.ackbytes;
         }
      }

      if (tcpExt->dst.ack && tcpExt->dst.ackbytes) {
         if (tcpExt->dst.ack > tcpExt->dst.ackbytes) {
            if (tcpExt->dst.ack > tcpExt->dst.ackbytes)
               tcp->dst.ackbytes = (tcpExt->dst.ack - 1) - tcpExt->dst.ackbytes;
         }
      }

      tcp->src.retrans  = tcpExt->src.retrans;
      tcp->dst.retrans  = tcpExt->dst.retrans;

      if ((tcp->src.bytes = tcpExt->src.bytes) < 0)
         tcp->src.bytes = 0;

      if ((tcp->dst.bytes = tcpExt->dst.bytes) < 0)
         tcp->dst.bytes = 0;

      tcp->src.win     = tcpExt->src.win;
      tcp->dst.win     = tcpExt->dst.win;
      tcp->src.flags   = tcpExt->src.flags;
      tcp->dst.flags   = tcpExt->dst.flags;

      if ((tcpExt->src.ackbytes = tcpExt->src.ack) < 0)
         tcpExt->src.ackbytes = 0;

      if (tcpExt->src.ackbytes == (tcp->src.bytes - 1))
         tcpExt->src.ackbytes++;

      if ((tcpExt->dst.ackbytes = tcpExt->dst.ack) < 0)
         tcpExt->dst.ackbytes = 0;

      if (tcpExt->dst.ackbytes == (tcp->dst.bytes - 1))
         tcpExt->dst.ackbytes++;

      if (tcp && ((length = argus->hdr.len) > 0)) {
         bcopy ((char *)tcp, &((char *)argus)[length], sizeof(*tcp));
         argus->hdr.len += sizeof(*tcp);
      }

      tcpExt->src.pkts  = 0;
      tcpExt->dst.pkts  = 0;
      tcpExt->src.bytes = 0;
      tcpExt->dst.bytes = 0;
      tcpExt->src.retrans = 0;
      tcpExt->dst.retrans = 0;
      tcpExt->src.flags = 0;
      tcpExt->dst.flags = 0;

      tcp->status &= ~(ARGUS_RESET|ARGUS_PKTS_RETRANS|ARGUS_WINDOW_SHUT|ARGUS_OUTOFORDER|ARGUS_ECN_CONGESTED);
      tcp->src.status &= ~(ARGUS_RESET|ARGUS_PKTS_RETRANS|ARGUS_WINDOW_SHUT|ARGUS_OUTOFORDER|ARGUS_ECN_CONGESTED);
      tcp->dst.status &= ~(ARGUS_RESET|ARGUS_PKTS_RETRANS|ARGUS_WINDOW_SHUT|ARGUS_OUTOFORDER|ARGUS_ECN_CONGESTED);
*/
   }
}

void
ArgusParseTCPOptions(struct ArgusModelerStruct *model, struct tcphdr *tcp, int len, u_int *options)
{
   register const u_char *cp;
   register int i, opt, alen, datalen;

   if ((tcp != NULL)) {
      cp = (const u_char *)tcp + sizeof(*tcp);

      while (len > 0) {
         STRUCTCHECK(model,*cp);
         opt = *cp++;
         if (ZEROLENOPT(opt))
            alen = 1;

         else {
            STRUCTCHECK(model,*cp);
            alen = *cp++;   /* total including type, len */
            if (alen < 2 || alen > len)
               goto bad;
            --len;      /* account for length byte */
         }
         --len;         /* account for type byte */
         datalen = 0;

         switch (opt) {
            case TCPOPT_MAXSEG:
               *options |= ARGUS_TCP_MAXSEG;
               datalen = 2;
               LENCHECK(model, datalen);
               break;

            case TCPOPT_EOL:
               break;

            case TCPOPT_NOP:
               break;

            case TCPOPT_WSCALE:
               *options |= ARGUS_TCP_WSCALE;
               datalen = 1;
               LENCHECK(model, datalen);
               ArgusThisTCPsrc->winshift = *cp;
               break;

            case TCPOPT_SACKOK:
               *options |= ARGUS_TCP_SACKOK;
               break;

            case TCPOPT_SACK:
               *options |= ARGUS_TCP_SACK;
               datalen = alen - 2;
               for (i = 0; i < datalen; i += 4) {
                  LENCHECK(model, i + 4);
               }
               break;

            case TCPOPT_ECHO:
               *options |= ARGUS_TCP_ECHO;
               datalen = 4;
               LENCHECK(model, datalen);
               break;

            case TCPOPT_ECHOREPLY:
               *options |= ARGUS_TCP_ECHOREPLY;
               datalen = 4;
               LENCHECK(model, datalen);
               break;

            case TCPOPT_TIMESTAMP:
               *options |= ARGUS_TCP_TIMESTAMP;
               datalen = 8;
               LENCHECK(model, 4);
               LENCHECK(model, datalen);
               break;

            case TCPOPT_CC:
               *options |= ARGUS_TCP_CC;
               datalen = 4;
               LENCHECK(model, datalen);
               break;

            case TCPOPT_CCNEW:
               *options |= ARGUS_TCP_CCNEW;
               datalen = 4;
               LENCHECK(model, datalen);
               break;

            case TCPOPT_CCECHO:
               *options |= ARGUS_TCP_CCECHO;
               datalen = 4;
               LENCHECK(model, datalen);
               break;

            default:
               datalen = alen - 2;
               for (i = 0; i < datalen; ++i)
                  LENCHECK(model, i);
               break;
            }

            cp += datalen;
            len -= datalen;

            ++datalen;         /* option octet */
            if (!ZEROLENOPT(opt))
               ++datalen;      /* size octet */

            if (opt == TCPOPT_EOL)
               break;
      }
   }

bad:
trunc: {
   }
}
